var commonjsGlobal = typeof globalThis !== 'undefined' ? globalThis : typeof window !== 'undefined' ? window : typeof global !== 'undefined' ? global : typeof self !== 'undefined' ? self : {};

var rangyCore = {exports: {}};

/**
 * Rangy, a cross-browser JavaScript range and selection library
 * https://github.com/timdown/rangy
 *
 * Copyright 2015, Tim Down
 * Licensed under the MIT license.
 * Version: 1.3.0
 * Build date: 10 May 2015
 */

(function (module, exports) {
(function(factory, root) {
    {
        // Node/CommonJS style
        module.exports = factory();
    }
})(function() {

    var OBJECT = "object", FUNCTION = "function", UNDEFINED = "undefined";

    // Minimal set of properties required for DOM Level 2 Range compliance. Comparison constants such as START_TO_START
    // are omitted because ranges in KHTML do not have them but otherwise work perfectly well. See issue 113.
    var domRangeProperties = ["startContainer", "startOffset", "endContainer", "endOffset", "collapsed",
        "commonAncestorContainer"];

    // Minimal set of methods required for DOM Level 2 Range compliance
    var domRangeMethods = ["setStart", "setStartBefore", "setStartAfter", "setEnd", "setEndBefore",
        "setEndAfter", "collapse", "selectNode", "selectNodeContents", "compareBoundaryPoints", "deleteContents",
        "extractContents", "cloneContents", "insertNode", "surroundContents", "cloneRange", "toString", "detach"];

    var textRangeProperties = ["boundingHeight", "boundingLeft", "boundingTop", "boundingWidth", "htmlText", "text"];

    // Subset of TextRange's full set of methods that we're interested in
    var textRangeMethods = ["collapse", "compareEndPoints", "duplicate", "moveToElementText", "parentElement", "select",
        "setEndPoint", "getBoundingClientRect"];

    /*----------------------------------------------------------------------------------------------------------------*/

    // Trio of functions taken from Peter Michaux's article:
    // http://peter.michaux.ca/articles/feature-detection-state-of-the-art-browser-scripting
    function isHostMethod(o, p) {
        var t = typeof o[p];
        return t == FUNCTION || (!!(t == OBJECT && o[p])) || t == "unknown";
    }

    function isHostObject(o, p) {
        return !!(typeof o[p] == OBJECT && o[p]);
    }

    function isHostProperty(o, p) {
        return typeof o[p] != UNDEFINED;
    }

    // Creates a convenience function to save verbose repeated calls to tests functions
    function createMultiplePropertyTest(testFunc) {
        return function(o, props) {
            var i = props.length;
            while (i--) {
                if (!testFunc(o, props[i])) {
                    return false;
                }
            }
            return true;
        };
    }

    // Next trio of functions are a convenience to save verbose repeated calls to previous two functions
    var areHostMethods = createMultiplePropertyTest(isHostMethod);
    var areHostObjects = createMultiplePropertyTest(isHostObject);
    var areHostProperties = createMultiplePropertyTest(isHostProperty);

    function isTextRange(range) {
        return range && areHostMethods(range, textRangeMethods) && areHostProperties(range, textRangeProperties);
    }

    function getBody(doc) {
        return isHostObject(doc, "body") ? doc.body : doc.getElementsByTagName("body")[0];
    }

    var forEach = [].forEach ?
        function(arr, func) {
            arr.forEach(func);
        } :
        function(arr, func) {
            for (var i = 0, len = arr.length; i < len; ++i) {
                func(arr[i], i);
            }
        };

    var modules = {};

    var isBrowser = (typeof window != UNDEFINED && typeof document != UNDEFINED);

    var util = {
        isHostMethod: isHostMethod,
        isHostObject: isHostObject,
        isHostProperty: isHostProperty,
        areHostMethods: areHostMethods,
        areHostObjects: areHostObjects,
        areHostProperties: areHostProperties,
        isTextRange: isTextRange,
        getBody: getBody,
        forEach: forEach
    };

    var api = {
        version: "1.3.0",
        initialized: false,
        isBrowser: isBrowser,
        supported: true,
        util: util,
        features: {},
        modules: modules,
        config: {
            alertOnFail: false,
            alertOnWarn: false,
            preferTextRange: false,
            autoInitialize: (typeof rangyAutoInitialize == UNDEFINED) ? true : rangyAutoInitialize
        }
    };

    function consoleLog(msg) {
        if (typeof console != UNDEFINED && isHostMethod(console, "log")) {
            console.log(msg);
        }
    }

    function alertOrLog(msg, shouldAlert) {
        if (isBrowser && shouldAlert) {
            alert(msg);
        } else  {
            consoleLog(msg);
        }
    }

    function fail(reason) {
        api.initialized = true;
        api.supported = false;
        alertOrLog("Rangy is not supported in this environment. Reason: " + reason, api.config.alertOnFail);
    }

    api.fail = fail;

    function warn(msg) {
        alertOrLog("Rangy warning: " + msg, api.config.alertOnWarn);
    }

    api.warn = warn;

    // Add utility extend() method
    var extend;
    if ({}.hasOwnProperty) {
        util.extend = extend = function(obj, props, deep) {
            var o, p;
            for (var i in props) {
                if (props.hasOwnProperty(i)) {
                    o = obj[i];
                    p = props[i];
                    if (deep && o !== null && typeof o == "object" && p !== null && typeof p == "object") {
                        extend(o, p, true);
                    }
                    obj[i] = p;
                }
            }
            // Special case for toString, which does not show up in for...in loops in IE <= 8
            if (props.hasOwnProperty("toString")) {
                obj.toString = props.toString;
            }
            return obj;
        };

        util.createOptions = function(optionsParam, defaults) {
            var options = {};
            extend(options, defaults);
            if (optionsParam) {
                extend(options, optionsParam);
            }
            return options;
        };
    } else {
        fail("hasOwnProperty not supported");
    }

    // Test whether we're in a browser and bail out if not
    if (!isBrowser) {
        fail("Rangy can only run in a browser");
    }

    // Test whether Array.prototype.slice can be relied on for NodeLists and use an alternative toArray() if not
    (function() {
        var toArray;

        if (isBrowser) {
            var el = document.createElement("div");
            el.appendChild(document.createElement("span"));
            var slice = [].slice;
            try {
                if (slice.call(el.childNodes, 0)[0].nodeType == 1) {
                    toArray = function(arrayLike) {
                        return slice.call(arrayLike, 0);
                    };
                }
            } catch (e) {}
        }

        if (!toArray) {
            toArray = function(arrayLike) {
                var arr = [];
                for (var i = 0, len = arrayLike.length; i < len; ++i) {
                    arr[i] = arrayLike[i];
                }
                return arr;
            };
        }

        util.toArray = toArray;
    })();

    // Very simple event handler wrapper function that doesn't attempt to solve issues such as "this" handling or
    // normalization of event properties
    var addListener;
    if (isBrowser) {
        if (isHostMethod(document, "addEventListener")) {
            addListener = function(obj, eventType, listener) {
                obj.addEventListener(eventType, listener, false);
            };
        } else if (isHostMethod(document, "attachEvent")) {
            addListener = function(obj, eventType, listener) {
                obj.attachEvent("on" + eventType, listener);
            };
        } else {
            fail("Document does not have required addEventListener or attachEvent method");
        }

        util.addListener = addListener;
    }

    var initListeners = [];

    function getErrorDesc(ex) {
        return ex.message || ex.description || String(ex);
    }

    // Initialization
    function init() {
        if (!isBrowser || api.initialized) {
            return;
        }
        var testRange;
        var implementsDomRange = false, implementsTextRange = false;

        // First, perform basic feature tests

        if (isHostMethod(document, "createRange")) {
            testRange = document.createRange();
            if (areHostMethods(testRange, domRangeMethods) && areHostProperties(testRange, domRangeProperties)) {
                implementsDomRange = true;
            }
        }

        var body = getBody(document);
        if (!body || body.nodeName.toLowerCase() != "body") {
            fail("No body element found");
            return;
        }

        if (body && isHostMethod(body, "createTextRange")) {
            testRange = body.createTextRange();
            if (isTextRange(testRange)) {
                implementsTextRange = true;
            }
        }

        if (!implementsDomRange && !implementsTextRange) {
            fail("Neither Range nor TextRange are available");
            return;
        }

        api.initialized = true;
        api.features = {
            implementsDomRange: implementsDomRange,
            implementsTextRange: implementsTextRange
        };

        // Initialize modules
        var module, errorMessage;
        for (var moduleName in modules) {
            if ( (module = modules[moduleName]) instanceof Module ) {
                module.init(module, api);
            }
        }

        // Call init listeners
        for (var i = 0, len = initListeners.length; i < len; ++i) {
            try {
                initListeners[i](api);
            } catch (ex) {
                errorMessage = "Rangy init listener threw an exception. Continuing. Detail: " + getErrorDesc(ex);
                consoleLog(errorMessage);
            }
        }
    }

    function deprecationNotice(deprecated, replacement, module) {
        if (module) {
            deprecated += " in module " + module.name;
        }
        api.warn("DEPRECATED: " + deprecated + " is deprecated. Please use " +
        replacement + " instead.");
    }

    function createAliasForDeprecatedMethod(owner, deprecated, replacement, module) {
        owner[deprecated] = function() {
            deprecationNotice(deprecated, replacement, module);
            return owner[replacement].apply(owner, util.toArray(arguments));
        };
    }

    util.deprecationNotice = deprecationNotice;
    util.createAliasForDeprecatedMethod = createAliasForDeprecatedMethod;

    // Allow external scripts to initialize this library in case it's loaded after the document has loaded
    api.init = init;

    // Execute listener immediately if already initialized
    api.addInitListener = function(listener) {
        if (api.initialized) {
            listener(api);
        } else {
            initListeners.push(listener);
        }
    };

    var shimListeners = [];

    api.addShimListener = function(listener) {
        shimListeners.push(listener);
    };

    function shim(win) {
        win = win || window;
        init();

        // Notify listeners
        for (var i = 0, len = shimListeners.length; i < len; ++i) {
            shimListeners[i](win);
        }
    }

    if (isBrowser) {
        api.shim = api.createMissingNativeApi = shim;
        createAliasForDeprecatedMethod(api, "createMissingNativeApi", "shim");
    }

    function Module(name, dependencies, initializer) {
        this.name = name;
        this.dependencies = dependencies;
        this.initialized = false;
        this.supported = false;
        this.initializer = initializer;
    }

    Module.prototype = {
        init: function() {
            var requiredModuleNames = this.dependencies || [];
            for (var i = 0, len = requiredModuleNames.length, requiredModule, moduleName; i < len; ++i) {
                moduleName = requiredModuleNames[i];

                requiredModule = modules[moduleName];
                if (!requiredModule || !(requiredModule instanceof Module)) {
                    throw new Error("required module '" + moduleName + "' not found");
                }

                requiredModule.init();

                if (!requiredModule.supported) {
                    throw new Error("required module '" + moduleName + "' not supported");
                }
            }

            // Now run initializer
            this.initializer(this);
        },

        fail: function(reason) {
            this.initialized = true;
            this.supported = false;
            throw new Error(reason);
        },

        warn: function(msg) {
            api.warn("Module " + this.name + ": " + msg);
        },

        deprecationNotice: function(deprecated, replacement) {
            api.warn("DEPRECATED: " + deprecated + " in module " + this.name + " is deprecated. Please use " +
                replacement + " instead");
        },

        createError: function(msg) {
            return new Error("Error in Rangy " + this.name + " module: " + msg);
        }
    };

    function createModule(name, dependencies, initFunc) {
        var newModule = new Module(name, dependencies, function(module) {
            if (!module.initialized) {
                module.initialized = true;
                try {
                    initFunc(api, module);
                    module.supported = true;
                } catch (ex) {
                    var errorMessage = "Module '" + name + "' failed to load: " + getErrorDesc(ex);
                    consoleLog(errorMessage);
                    if (ex.stack) {
                        consoleLog(ex.stack);
                    }
                }
            }
        });
        modules[name] = newModule;
        return newModule;
    }

    api.createModule = function(name) {
        // Allow 2 or 3 arguments (second argument is an optional array of dependencies)
        var initFunc, dependencies;
        if (arguments.length == 2) {
            initFunc = arguments[1];
            dependencies = [];
        } else {
            initFunc = arguments[2];
            dependencies = arguments[1];
        }

        var module = createModule(name, dependencies, initFunc);

        // Initialize the module immediately if the core is already initialized
        if (api.initialized && api.supported) {
            module.init();
        }
    };

    api.createCoreModule = function(name, dependencies, initFunc) {
        createModule(name, dependencies, initFunc);
    };

    /*----------------------------------------------------------------------------------------------------------------*/

    // Ensure rangy.rangePrototype and rangy.selectionPrototype are available immediately

    function RangePrototype() {}
    api.RangePrototype = RangePrototype;
    api.rangePrototype = new RangePrototype();

    function SelectionPrototype() {}
    api.selectionPrototype = new SelectionPrototype();

    /*----------------------------------------------------------------------------------------------------------------*/

    // DOM utility methods used by Rangy
    api.createCoreModule("DomUtil", [], function(api, module) {
        var UNDEF = "undefined";
        var util = api.util;
        var getBody = util.getBody;

        // Perform feature tests
        if (!util.areHostMethods(document, ["createDocumentFragment", "createElement", "createTextNode"])) {
            module.fail("document missing a Node creation method");
        }

        if (!util.isHostMethod(document, "getElementsByTagName")) {
            module.fail("document missing getElementsByTagName method");
        }

        var el = document.createElement("div");
        if (!util.areHostMethods(el, ["insertBefore", "appendChild", "cloneNode"] ||
                !util.areHostObjects(el, ["previousSibling", "nextSibling", "childNodes", "parentNode"]))) {
            module.fail("Incomplete Element implementation");
        }

        // innerHTML is required for Range's createContextualFragment method
        if (!util.isHostProperty(el, "innerHTML")) {
            module.fail("Element is missing innerHTML property");
        }

        var textNode = document.createTextNode("test");
        if (!util.areHostMethods(textNode, ["splitText", "deleteData", "insertData", "appendData", "cloneNode"] ||
                !util.areHostObjects(el, ["previousSibling", "nextSibling", "childNodes", "parentNode"]) ||
                !util.areHostProperties(textNode, ["data"]))) {
            module.fail("Incomplete Text Node implementation");
        }

        /*----------------------------------------------------------------------------------------------------------------*/

        // Removed use of indexOf because of a bizarre bug in Opera that is thrown in one of the Acid3 tests. I haven't been
        // able to replicate it outside of the test. The bug is that indexOf returns -1 when called on an Array that
        // contains just the document as a single element and the value searched for is the document.
        var arrayContains = /*Array.prototype.indexOf ?
            function(arr, val) {
                return arr.indexOf(val) > -1;
            }:*/

            function(arr, val) {
                var i = arr.length;
                while (i--) {
                    if (arr[i] === val) {
                        return true;
                    }
                }
                return false;
            };

        // Opera 11 puts HTML elements in the null namespace, it seems, and IE 7 has undefined namespaceURI
        function isHtmlNamespace(node) {
            var ns;
            return typeof node.namespaceURI == UNDEF || ((ns = node.namespaceURI) === null || ns == "http://www.w3.org/1999/xhtml");
        }

        function parentElement(node) {
            var parent = node.parentNode;
            return (parent.nodeType == 1) ? parent : null;
        }

        function getNodeIndex(node) {
            var i = 0;
            while( (node = node.previousSibling) ) {
                ++i;
            }
            return i;
        }

        function getNodeLength(node) {
            switch (node.nodeType) {
                case 7:
                case 10:
                    return 0;
                case 3:
                case 8:
                    return node.length;
                default:
                    return node.childNodes.length;
            }
        }

        function getCommonAncestor(node1, node2) {
            var ancestors = [], n;
            for (n = node1; n; n = n.parentNode) {
                ancestors.push(n);
            }

            for (n = node2; n; n = n.parentNode) {
                if (arrayContains(ancestors, n)) {
                    return n;
                }
            }

            return null;
        }

        function isAncestorOf(ancestor, descendant, selfIsAncestor) {
            var n = selfIsAncestor ? descendant : descendant.parentNode;
            while (n) {
                if (n === ancestor) {
                    return true;
                } else {
                    n = n.parentNode;
                }
            }
            return false;
        }

        function isOrIsAncestorOf(ancestor, descendant) {
            return isAncestorOf(ancestor, descendant, true);
        }

        function getClosestAncestorIn(node, ancestor, selfIsAncestor) {
            var p, n = selfIsAncestor ? node : node.parentNode;
            while (n) {
                p = n.parentNode;
                if (p === ancestor) {
                    return n;
                }
                n = p;
            }
            return null;
        }

        function isCharacterDataNode(node) {
            var t = node.nodeType;
            return t == 3 || t == 4 || t == 8 ; // Text, CDataSection or Comment
        }

        function isTextOrCommentNode(node) {
            if (!node) {
                return false;
            }
            var t = node.nodeType;
            return t == 3 || t == 8 ; // Text or Comment
        }

        function insertAfter(node, precedingNode) {
            var nextNode = precedingNode.nextSibling, parent = precedingNode.parentNode;
            if (nextNode) {
                parent.insertBefore(node, nextNode);
            } else {
                parent.appendChild(node);
            }
            return node;
        }

        // Note that we cannot use splitText() because it is bugridden in IE 9.
        function splitDataNode(node, index, positionsToPreserve) {
            var newNode = node.cloneNode(false);
            newNode.deleteData(0, index);
            node.deleteData(index, node.length - index);
            insertAfter(newNode, node);

            // Preserve positions
            if (positionsToPreserve) {
                for (var i = 0, position; position = positionsToPreserve[i++]; ) {
                    // Handle case where position was inside the portion of node after the split point
                    if (position.node == node && position.offset > index) {
                        position.node = newNode;
                        position.offset -= index;
                    }
                    // Handle the case where the position is a node offset within node's parent
                    else if (position.node == node.parentNode && position.offset > getNodeIndex(node)) {
                        ++position.offset;
                    }
                }
            }
            return newNode;
        }

        function getDocument(node) {
            if (node.nodeType == 9) {
                return node;
            } else if (typeof node.ownerDocument != UNDEF) {
                return node.ownerDocument;
            } else if (typeof node.document != UNDEF) {
                return node.document;
            } else if (node.parentNode) {
                return getDocument(node.parentNode);
            } else {
                throw module.createError("getDocument: no document found for node");
            }
        }

        function getWindow(node) {
            var doc = getDocument(node);
            if (typeof doc.defaultView != UNDEF) {
                return doc.defaultView;
            } else if (typeof doc.parentWindow != UNDEF) {
                return doc.parentWindow;
            } else {
                throw module.createError("Cannot get a window object for node");
            }
        }

        function getIframeDocument(iframeEl) {
            if (typeof iframeEl.contentDocument != UNDEF) {
                return iframeEl.contentDocument;
            } else if (typeof iframeEl.contentWindow != UNDEF) {
                return iframeEl.contentWindow.document;
            } else {
                throw module.createError("getIframeDocument: No Document object found for iframe element");
            }
        }

        function getIframeWindow(iframeEl) {
            if (typeof iframeEl.contentWindow != UNDEF) {
                return iframeEl.contentWindow;
            } else if (typeof iframeEl.contentDocument != UNDEF) {
                return iframeEl.contentDocument.defaultView;
            } else {
                throw module.createError("getIframeWindow: No Window object found for iframe element");
            }
        }

        // This looks bad. Is it worth it?
        function isWindow(obj) {
            return obj && util.isHostMethod(obj, "setTimeout") && util.isHostObject(obj, "document");
        }

        function getContentDocument(obj, module, methodName) {
            var doc;

            if (!obj) {
                doc = document;
            }

            // Test if a DOM node has been passed and obtain a document object for it if so
            else if (util.isHostProperty(obj, "nodeType")) {
                doc = (obj.nodeType == 1 && obj.tagName.toLowerCase() == "iframe") ?
                    getIframeDocument(obj) : getDocument(obj);
            }

            // Test if the doc parameter appears to be a Window object
            else if (isWindow(obj)) {
                doc = obj.document;
            }

            if (!doc) {
                throw module.createError(methodName + "(): Parameter must be a Window object or DOM node");
            }

            return doc;
        }

        function getRootContainer(node) {
            var parent;
            while ( (parent = node.parentNode) ) {
                node = parent;
            }
            return node;
        }

        function comparePoints(nodeA, offsetA, nodeB, offsetB) {
            // See http://www.w3.org/TR/DOM-Level-2-Traversal-Range/ranges.html#Level-2-Range-Comparing
            var nodeC, root, childA, childB, n;
            if (nodeA == nodeB) {
                // Case 1: nodes are the same
                return offsetA === offsetB ? 0 : (offsetA < offsetB) ? -1 : 1;
            } else if ( (nodeC = getClosestAncestorIn(nodeB, nodeA, true)) ) {
                // Case 2: node C (container B or an ancestor) is a child node of A
                return offsetA <= getNodeIndex(nodeC) ? -1 : 1;
            } else if ( (nodeC = getClosestAncestorIn(nodeA, nodeB, true)) ) {
                // Case 3: node C (container A or an ancestor) is a child node of B
                return getNodeIndex(nodeC) < offsetB  ? -1 : 1;
            } else {
                root = getCommonAncestor(nodeA, nodeB);
                if (!root) {
                    throw new Error("comparePoints error: nodes have no common ancestor");
                }

                // Case 4: containers are siblings or descendants of siblings
                childA = (nodeA === root) ? root : getClosestAncestorIn(nodeA, root, true);
                childB = (nodeB === root) ? root : getClosestAncestorIn(nodeB, root, true);

                if (childA === childB) {
                    // This shouldn't be possible
                    throw module.createError("comparePoints got to case 4 and childA and childB are the same!");
                } else {
                    n = root.firstChild;
                    while (n) {
                        if (n === childA) {
                            return -1;
                        } else if (n === childB) {
                            return 1;
                        }
                        n = n.nextSibling;
                    }
                }
            }
        }

        /*----------------------------------------------------------------------------------------------------------------*/

        // Test for IE's crash (IE 6/7) or exception (IE >= 8) when a reference to garbage-collected text node is queried
        var crashyTextNodes = false;

        function isBrokenNode(node) {
            var n;
            try {
                n = node.parentNode;
                return false;
            } catch (e) {
                return true;
            }
        }

        (function() {
            var el = document.createElement("b");
            el.innerHTML = "1";
            var textNode = el.firstChild;
            el.innerHTML = "<br />";
            crashyTextNodes = isBrokenNode(textNode);

            api.features.crashyTextNodes = crashyTextNodes;
        })();

        /*----------------------------------------------------------------------------------------------------------------*/

        function inspectNode(node) {
            if (!node) {
                return "[No node]";
            }
            if (crashyTextNodes && isBrokenNode(node)) {
                return "[Broken node]";
            }
            if (isCharacterDataNode(node)) {
                return '"' + node.data + '"';
            }
            if (node.nodeType == 1) {
                var idAttr = node.id ? ' id="' + node.id + '"' : "";
                return "<" + node.nodeName + idAttr + ">[index:" + getNodeIndex(node) + ",length:" + node.childNodes.length + "][" + (node.innerHTML || "[innerHTML not supported]").slice(0, 25) + "]";
            }
            return node.nodeName;
        }

        function fragmentFromNodeChildren(node) {
            var fragment = getDocument(node).createDocumentFragment(), child;
            while ( (child = node.firstChild) ) {
                fragment.appendChild(child);
            }
            return fragment;
        }

        var getComputedStyleProperty;
        if (typeof window.getComputedStyle != UNDEF) {
            getComputedStyleProperty = function(el, propName) {
                return getWindow(el).getComputedStyle(el, null)[propName];
            };
        } else if (typeof document.documentElement.currentStyle != UNDEF) {
            getComputedStyleProperty = function(el, propName) {
                return el.currentStyle ? el.currentStyle[propName] : "";
            };
        } else {
            module.fail("No means of obtaining computed style properties found");
        }

        function createTestElement(doc, html, contentEditable) {
            var body = getBody(doc);
            var el = doc.createElement("div");
            el.contentEditable = "" + !!contentEditable;
            if (html) {
                el.innerHTML = html;
            }

            // Insert the test element at the start of the body to prevent scrolling to the bottom in iOS (issue #292)
            var bodyFirstChild = body.firstChild;
            if (bodyFirstChild) {
                body.insertBefore(el, bodyFirstChild);
            } else {
                body.appendChild(el);
            }

            return el;
        }

        function removeNode(node) {
            return node.parentNode.removeChild(node);
        }

        function NodeIterator(root) {
            this.root = root;
            this._next = root;
        }

        NodeIterator.prototype = {
            _current: null,

            hasNext: function() {
                return !!this._next;
            },

            next: function() {
                var n = this._current = this._next;
                var child, next;
                if (this._current) {
                    child = n.firstChild;
                    if (child) {
                        this._next = child;
                    } else {
                        next = null;
                        while ((n !== this.root) && !(next = n.nextSibling)) {
                            n = n.parentNode;
                        }
                        this._next = next;
                    }
                }
                return this._current;
            },

            detach: function() {
                this._current = this._next = this.root = null;
            }
        };

        function createIterator(root) {
            return new NodeIterator(root);
        }

        function DomPosition(node, offset) {
            this.node = node;
            this.offset = offset;
        }

        DomPosition.prototype = {
            equals: function(pos) {
                return !!pos && this.node === pos.node && this.offset == pos.offset;
            },

            inspect: function() {
                return "[DomPosition(" + inspectNode(this.node) + ":" + this.offset + ")]";
            },

            toString: function() {
                return this.inspect();
            }
        };

        function DOMException(codeName) {
            this.code = this[codeName];
            this.codeName = codeName;
            this.message = "DOMException: " + this.codeName;
        }

        DOMException.prototype = {
            INDEX_SIZE_ERR: 1,
            HIERARCHY_REQUEST_ERR: 3,
            WRONG_DOCUMENT_ERR: 4,
            NO_MODIFICATION_ALLOWED_ERR: 7,
            NOT_FOUND_ERR: 8,
            NOT_SUPPORTED_ERR: 9,
            INVALID_STATE_ERR: 11,
            INVALID_NODE_TYPE_ERR: 24
        };

        DOMException.prototype.toString = function() {
            return this.message;
        };

        api.dom = {
            arrayContains: arrayContains,
            isHtmlNamespace: isHtmlNamespace,
            parentElement: parentElement,
            getNodeIndex: getNodeIndex,
            getNodeLength: getNodeLength,
            getCommonAncestor: getCommonAncestor,
            isAncestorOf: isAncestorOf,
            isOrIsAncestorOf: isOrIsAncestorOf,
            getClosestAncestorIn: getClosestAncestorIn,
            isCharacterDataNode: isCharacterDataNode,
            isTextOrCommentNode: isTextOrCommentNode,
            insertAfter: insertAfter,
            splitDataNode: splitDataNode,
            getDocument: getDocument,
            getWindow: getWindow,
            getIframeWindow: getIframeWindow,
            getIframeDocument: getIframeDocument,
            getBody: getBody,
            isWindow: isWindow,
            getContentDocument: getContentDocument,
            getRootContainer: getRootContainer,
            comparePoints: comparePoints,
            isBrokenNode: isBrokenNode,
            inspectNode: inspectNode,
            getComputedStyleProperty: getComputedStyleProperty,
            createTestElement: createTestElement,
            removeNode: removeNode,
            fragmentFromNodeChildren: fragmentFromNodeChildren,
            createIterator: createIterator,
            DomPosition: DomPosition
        };

        api.DOMException = DOMException;
    });

    /*----------------------------------------------------------------------------------------------------------------*/

    // Pure JavaScript implementation of DOM Range
    api.createCoreModule("DomRange", ["DomUtil"], function(api, module) {
        var dom = api.dom;
        var util = api.util;
        var DomPosition = dom.DomPosition;
        var DOMException = api.DOMException;

        var isCharacterDataNode = dom.isCharacterDataNode;
        var getNodeIndex = dom.getNodeIndex;
        var isOrIsAncestorOf = dom.isOrIsAncestorOf;
        var getDocument = dom.getDocument;
        var comparePoints = dom.comparePoints;
        var splitDataNode = dom.splitDataNode;
        var getClosestAncestorIn = dom.getClosestAncestorIn;
        var getNodeLength = dom.getNodeLength;
        var arrayContains = dom.arrayContains;
        var getRootContainer = dom.getRootContainer;
        var crashyTextNodes = api.features.crashyTextNodes;

        var removeNode = dom.removeNode;

        /*----------------------------------------------------------------------------------------------------------------*/

        // Utility functions

        function isNonTextPartiallySelected(node, range) {
            return (node.nodeType != 3) &&
                   (isOrIsAncestorOf(node, range.startContainer) || isOrIsAncestorOf(node, range.endContainer));
        }

        function getRangeDocument(range) {
            return range.document || getDocument(range.startContainer);
        }

        function getRangeRoot(range) {
            return getRootContainer(range.startContainer);
        }

        function getBoundaryBeforeNode(node) {
            return new DomPosition(node.parentNode, getNodeIndex(node));
        }

        function getBoundaryAfterNode(node) {
            return new DomPosition(node.parentNode, getNodeIndex(node) + 1);
        }

        function insertNodeAtPosition(node, n, o) {
            var firstNodeInserted = node.nodeType == 11 ? node.firstChild : node;
            if (isCharacterDataNode(n)) {
                if (o == n.length) {
                    dom.insertAfter(node, n);
                } else {
                    n.parentNode.insertBefore(node, o == 0 ? n : splitDataNode(n, o));
                }
            } else if (o >= n.childNodes.length) {
                n.appendChild(node);
            } else {
                n.insertBefore(node, n.childNodes[o]);
            }
            return firstNodeInserted;
        }

        function rangesIntersect(rangeA, rangeB, touchingIsIntersecting) {
            assertRangeValid(rangeA);
            assertRangeValid(rangeB);

            if (getRangeDocument(rangeB) != getRangeDocument(rangeA)) {
                throw new DOMException("WRONG_DOCUMENT_ERR");
            }

            var startComparison = comparePoints(rangeA.startContainer, rangeA.startOffset, rangeB.endContainer, rangeB.endOffset),
                endComparison = comparePoints(rangeA.endContainer, rangeA.endOffset, rangeB.startContainer, rangeB.startOffset);

            return touchingIsIntersecting ? startComparison <= 0 && endComparison >= 0 : startComparison < 0 && endComparison > 0;
        }

        function cloneSubtree(iterator) {
            var partiallySelected;
            for (var node, frag = getRangeDocument(iterator.range).createDocumentFragment(), subIterator; node = iterator.next(); ) {
                partiallySelected = iterator.isPartiallySelectedSubtree();
                node = node.cloneNode(!partiallySelected);
                if (partiallySelected) {
                    subIterator = iterator.getSubtreeIterator();
                    node.appendChild(cloneSubtree(subIterator));
                    subIterator.detach();
                }

                if (node.nodeType == 10) { // DocumentType
                    throw new DOMException("HIERARCHY_REQUEST_ERR");
                }
                frag.appendChild(node);
            }
            return frag;
        }

        function iterateSubtree(rangeIterator, func, iteratorState) {
            var it, n;
            iteratorState = iteratorState || { stop: false };
            for (var node, subRangeIterator; node = rangeIterator.next(); ) {
                if (rangeIterator.isPartiallySelectedSubtree()) {
                    if (func(node) === false) {
                        iteratorState.stop = true;
                        return;
                    } else {
                        // The node is partially selected by the Range, so we can use a new RangeIterator on the portion of
                        // the node selected by the Range.
                        subRangeIterator = rangeIterator.getSubtreeIterator();
                        iterateSubtree(subRangeIterator, func, iteratorState);
                        subRangeIterator.detach();
                        if (iteratorState.stop) {
                            return;
                        }
                    }
                } else {
                    // The whole node is selected, so we can use efficient DOM iteration to iterate over the node and its
                    // descendants
                    it = dom.createIterator(node);
                    while ( (n = it.next()) ) {
                        if (func(n) === false) {
                            iteratorState.stop = true;
                            return;
                        }
                    }
                }
            }
        }

        function deleteSubtree(iterator) {
            var subIterator;
            while (iterator.next()) {
                if (iterator.isPartiallySelectedSubtree()) {
                    subIterator = iterator.getSubtreeIterator();
                    deleteSubtree(subIterator);
                    subIterator.detach();
                } else {
                    iterator.remove();
                }
            }
        }

        function extractSubtree(iterator) {
            for (var node, frag = getRangeDocument(iterator.range).createDocumentFragment(), subIterator; node = iterator.next(); ) {

                if (iterator.isPartiallySelectedSubtree()) {
                    node = node.cloneNode(false);
                    subIterator = iterator.getSubtreeIterator();
                    node.appendChild(extractSubtree(subIterator));
                    subIterator.detach();
                } else {
                    iterator.remove();
                }
                if (node.nodeType == 10) { // DocumentType
                    throw new DOMException("HIERARCHY_REQUEST_ERR");
                }
                frag.appendChild(node);
            }
            return frag;
        }

        function getNodesInRange(range, nodeTypes, filter) {
            var filterNodeTypes = !!(nodeTypes && nodeTypes.length), regex;
            var filterExists = !!filter;
            if (filterNodeTypes) {
                regex = new RegExp("^(" + nodeTypes.join("|") + ")$");
            }

            var nodes = [];
            iterateSubtree(new RangeIterator(range, false), function(node) {
                if (filterNodeTypes && !regex.test(node.nodeType)) {
                    return;
                }
                if (filterExists && !filter(node)) {
                    return;
                }
                // Don't include a boundary container if it is a character data node and the range does not contain any
                // of its character data. See issue 190.
                var sc = range.startContainer;
                if (node == sc && isCharacterDataNode(sc) && range.startOffset == sc.length) {
                    return;
                }

                var ec = range.endContainer;
                if (node == ec && isCharacterDataNode(ec) && range.endOffset == 0) {
                    return;
                }

                nodes.push(node);
            });
            return nodes;
        }

        function inspect(range) {
            var name = (typeof range.getName == "undefined") ? "Range" : range.getName();
            return "[" + name + "(" + dom.inspectNode(range.startContainer) + ":" + range.startOffset + ", " +
                    dom.inspectNode(range.endContainer) + ":" + range.endOffset + ")]";
        }

        /*----------------------------------------------------------------------------------------------------------------*/

        // RangeIterator code partially borrows from IERange by Tim Ryan (http://github.com/timcameronryan/IERange)

        function RangeIterator(range, clonePartiallySelectedTextNodes) {
            this.range = range;
            this.clonePartiallySelectedTextNodes = clonePartiallySelectedTextNodes;


            if (!range.collapsed) {
                this.sc = range.startContainer;
                this.so = range.startOffset;
                this.ec = range.endContainer;
                this.eo = range.endOffset;
                var root = range.commonAncestorContainer;

                if (this.sc === this.ec && isCharacterDataNode(this.sc)) {
                    this.isSingleCharacterDataNode = true;
                    this._first = this._last = this._next = this.sc;
                } else {
                    this._first = this._next = (this.sc === root && !isCharacterDataNode(this.sc)) ?
                        this.sc.childNodes[this.so] : getClosestAncestorIn(this.sc, root, true);
                    this._last = (this.ec === root && !isCharacterDataNode(this.ec)) ?
                        this.ec.childNodes[this.eo - 1] : getClosestAncestorIn(this.ec, root, true);
                }
            }
        }

        RangeIterator.prototype = {
            _current: null,
            _next: null,
            _first: null,
            _last: null,
            isSingleCharacterDataNode: false,

            reset: function() {
                this._current = null;
                this._next = this._first;
            },

            hasNext: function() {
                return !!this._next;
            },

            next: function() {
                // Move to next node
                var current = this._current = this._next;
                if (current) {
                    this._next = (current !== this._last) ? current.nextSibling : null;

                    // Check for partially selected text nodes
                    if (isCharacterDataNode(current) && this.clonePartiallySelectedTextNodes) {
                        if (current === this.ec) {
                            (current = current.cloneNode(true)).deleteData(this.eo, current.length - this.eo);
                        }
                        if (this._current === this.sc) {
                            (current = current.cloneNode(true)).deleteData(0, this.so);
                        }
                    }
                }

                return current;
            },

            remove: function() {
                var current = this._current, start, end;

                if (isCharacterDataNode(current) && (current === this.sc || current === this.ec)) {
                    start = (current === this.sc) ? this.so : 0;
                    end = (current === this.ec) ? this.eo : current.length;
                    if (start != end) {
                        current.deleteData(start, end - start);
                    }
                } else {
                    if (current.parentNode) {
                        removeNode(current);
                    }
                }
            },

            // Checks if the current node is partially selected
            isPartiallySelectedSubtree: function() {
                var current = this._current;
                return isNonTextPartiallySelected(current, this.range);
            },

            getSubtreeIterator: function() {
                var subRange;
                if (this.isSingleCharacterDataNode) {
                    subRange = this.range.cloneRange();
                    subRange.collapse(false);
                } else {
                    subRange = new Range(getRangeDocument(this.range));
                    var current = this._current;
                    var startContainer = current, startOffset = 0, endContainer = current, endOffset = getNodeLength(current);

                    if (isOrIsAncestorOf(current, this.sc)) {
                        startContainer = this.sc;
                        startOffset = this.so;
                    }
                    if (isOrIsAncestorOf(current, this.ec)) {
                        endContainer = this.ec;
                        endOffset = this.eo;
                    }

                    updateBoundaries(subRange, startContainer, startOffset, endContainer, endOffset);
                }
                return new RangeIterator(subRange, this.clonePartiallySelectedTextNodes);
            },

            detach: function() {
                this.range = this._current = this._next = this._first = this._last = this.sc = this.so = this.ec = this.eo = null;
            }
        };

        /*----------------------------------------------------------------------------------------------------------------*/

        var beforeAfterNodeTypes = [1, 3, 4, 5, 7, 8, 10];
        var rootContainerNodeTypes = [2, 9, 11];
        var readonlyNodeTypes = [5, 6, 10, 12];
        var insertableNodeTypes = [1, 3, 4, 5, 7, 8, 10, 11];
        var surroundNodeTypes = [1, 3, 4, 5, 7, 8];

        function createAncestorFinder(nodeTypes) {
            return function(node, selfIsAncestor) {
                var t, n = selfIsAncestor ? node : node.parentNode;
                while (n) {
                    t = n.nodeType;
                    if (arrayContains(nodeTypes, t)) {
                        return n;
                    }
                    n = n.parentNode;
                }
                return null;
            };
        }

        var getDocumentOrFragmentContainer = createAncestorFinder( [9, 11] );
        var getReadonlyAncestor = createAncestorFinder(readonlyNodeTypes);
        var getDocTypeNotationEntityAncestor = createAncestorFinder( [6, 10, 12] );

        function assertNoDocTypeNotationEntityAncestor(node, allowSelf) {
            if (getDocTypeNotationEntityAncestor(node, allowSelf)) {
                throw new DOMException("INVALID_NODE_TYPE_ERR");
            }
        }

        function assertValidNodeType(node, invalidTypes) {
            if (!arrayContains(invalidTypes, node.nodeType)) {
                throw new DOMException("INVALID_NODE_TYPE_ERR");
            }
        }

        function assertValidOffset(node, offset) {
            if (offset < 0 || offset > (isCharacterDataNode(node) ? node.length : node.childNodes.length)) {
                throw new DOMException("INDEX_SIZE_ERR");
            }
        }

        function assertSameDocumentOrFragment(node1, node2) {
            if (getDocumentOrFragmentContainer(node1, true) !== getDocumentOrFragmentContainer(node2, true)) {
                throw new DOMException("WRONG_DOCUMENT_ERR");
            }
        }

        function assertNodeNotReadOnly(node) {
            if (getReadonlyAncestor(node, true)) {
                throw new DOMException("NO_MODIFICATION_ALLOWED_ERR");
            }
        }

        function assertNode(node, codeName) {
            if (!node) {
                throw new DOMException(codeName);
            }
        }

        function isValidOffset(node, offset) {
            return offset <= (isCharacterDataNode(node) ? node.length : node.childNodes.length);
        }

        function isRangeValid(range) {
            return (!!range.startContainer && !!range.endContainer &&
                    !(crashyTextNodes && (dom.isBrokenNode(range.startContainer) || dom.isBrokenNode(range.endContainer))) &&
                    getRootContainer(range.startContainer) == getRootContainer(range.endContainer) &&
                    isValidOffset(range.startContainer, range.startOffset) &&
                    isValidOffset(range.endContainer, range.endOffset));
        }

        function assertRangeValid(range) {
            if (!isRangeValid(range)) {
                throw new Error("Range error: Range is not valid. This usually happens after DOM mutation. Range: (" + range.inspect() + ")");
            }
        }

        /*----------------------------------------------------------------------------------------------------------------*/

        // Test the browser's innerHTML support to decide how to implement createContextualFragment
        var styleEl = document.createElement("style");
        var htmlParsingConforms = false;
        try {
            styleEl.innerHTML = "<b>x</b>";
            htmlParsingConforms = (styleEl.firstChild.nodeType == 3); // Opera incorrectly creates an element node
        } catch (e) {
            // IE 6 and 7 throw
        }

        api.features.htmlParsingConforms = htmlParsingConforms;

        var createContextualFragment = htmlParsingConforms ?

            // Implementation as per HTML parsing spec, trusting in the browser's implementation of innerHTML. See
            // discussion and base code for this implementation at issue 67.
            // Spec: http://html5.org/specs/dom-parsing.html#extensions-to-the-range-interface
            // Thanks to Aleks Williams.
            function(fragmentStr) {
                // "Let node the context object's start's node."
                var node = this.startContainer;
                var doc = getDocument(node);

                // "If the context object's start's node is null, raise an INVALID_STATE_ERR
                // exception and abort these steps."
                if (!node) {
                    throw new DOMException("INVALID_STATE_ERR");
                }

                // "Let element be as follows, depending on node's interface:"
                // Document, Document Fragment: null
                var el = null;

                // "Element: node"
                if (node.nodeType == 1) {
                    el = node;

                // "Text, Comment: node's parentElement"
                } else if (isCharacterDataNode(node)) {
                    el = dom.parentElement(node);
                }

                // "If either element is null or element's ownerDocument is an HTML document
                // and element's local name is "html" and element's namespace is the HTML
                // namespace"
                if (el === null || (
                    el.nodeName == "HTML" &&
                    dom.isHtmlNamespace(getDocument(el).documentElement) &&
                    dom.isHtmlNamespace(el)
                )) {

                // "let element be a new Element with "body" as its local name and the HTML
                // namespace as its namespace.""
                    el = doc.createElement("body");
                } else {
                    el = el.cloneNode(false);
                }

                // "If the node's document is an HTML document: Invoke the HTML fragment parsing algorithm."
                // "If the node's document is an XML document: Invoke the XML fragment parsing algorithm."
                // "In either case, the algorithm must be invoked with fragment as the input
                // and element as the context element."
                el.innerHTML = fragmentStr;

                // "If this raises an exception, then abort these steps. Otherwise, let new
                // children be the nodes returned."

                // "Let fragment be a new DocumentFragment."
                // "Append all new children to fragment."
                // "Return fragment."
                return dom.fragmentFromNodeChildren(el);
            } :

            // In this case, innerHTML cannot be trusted, so fall back to a simpler, non-conformant implementation that
            // previous versions of Rangy used (with the exception of using a body element rather than a div)
            function(fragmentStr) {
                var doc = getRangeDocument(this);
                var el = doc.createElement("body");
                el.innerHTML = fragmentStr;

                return dom.fragmentFromNodeChildren(el);
            };

        function splitRangeBoundaries(range, positionsToPreserve) {
            assertRangeValid(range);

            var sc = range.startContainer, so = range.startOffset, ec = range.endContainer, eo = range.endOffset;
            var startEndSame = (sc === ec);

            if (isCharacterDataNode(ec) && eo > 0 && eo < ec.length) {
                splitDataNode(ec, eo, positionsToPreserve);
            }

            if (isCharacterDataNode(sc) && so > 0 && so < sc.length) {
                sc = splitDataNode(sc, so, positionsToPreserve);
                if (startEndSame) {
                    eo -= so;
                    ec = sc;
                } else if (ec == sc.parentNode && eo >= getNodeIndex(sc)) {
                    eo++;
                }
                so = 0;
            }
            range.setStartAndEnd(sc, so, ec, eo);
        }

        function rangeToHtml(range) {
            assertRangeValid(range);
            var container = range.commonAncestorContainer.parentNode.cloneNode(false);
            container.appendChild( range.cloneContents() );
            return container.innerHTML;
        }

        /*----------------------------------------------------------------------------------------------------------------*/

        var rangeProperties = ["startContainer", "startOffset", "endContainer", "endOffset", "collapsed",
            "commonAncestorContainer"];

        var s2s = 0, s2e = 1, e2e = 2, e2s = 3;
        var n_b = 0, n_a = 1, n_b_a = 2, n_i = 3;

        util.extend(api.rangePrototype, {
            compareBoundaryPoints: function(how, range) {
                assertRangeValid(this);
                assertSameDocumentOrFragment(this.startContainer, range.startContainer);

                var nodeA, offsetA, nodeB, offsetB;
                var prefixA = (how == e2s || how == s2s) ? "start" : "end";
                var prefixB = (how == s2e || how == s2s) ? "start" : "end";
                nodeA = this[prefixA + "Container"];
                offsetA = this[prefixA + "Offset"];
                nodeB = range[prefixB + "Container"];
                offsetB = range[prefixB + "Offset"];
                return comparePoints(nodeA, offsetA, nodeB, offsetB);
            },

            insertNode: function(node) {
                assertRangeValid(this);
                assertValidNodeType(node, insertableNodeTypes);
                assertNodeNotReadOnly(this.startContainer);

                if (isOrIsAncestorOf(node, this.startContainer)) {
                    throw new DOMException("HIERARCHY_REQUEST_ERR");
                }

                // No check for whether the container of the start of the Range is of a type that does not allow
                // children of the type of node: the browser's DOM implementation should do this for us when we attempt
                // to add the node

                var firstNodeInserted = insertNodeAtPosition(node, this.startContainer, this.startOffset);
                this.setStartBefore(firstNodeInserted);
            },

            cloneContents: function() {
                assertRangeValid(this);

                var clone, frag;
                if (this.collapsed) {
                    return getRangeDocument(this).createDocumentFragment();
                } else {
                    if (this.startContainer === this.endContainer && isCharacterDataNode(this.startContainer)) {
                        clone = this.startContainer.cloneNode(true);
                        clone.data = clone.data.slice(this.startOffset, this.endOffset);
                        frag = getRangeDocument(this).createDocumentFragment();
                        frag.appendChild(clone);
                        return frag;
                    } else {
                        var iterator = new RangeIterator(this, true);
                        clone = cloneSubtree(iterator);
                        iterator.detach();
                    }
                    return clone;
                }
            },

            canSurroundContents: function() {
                assertRangeValid(this);
                assertNodeNotReadOnly(this.startContainer);
                assertNodeNotReadOnly(this.endContainer);

                // Check if the contents can be surrounded. Specifically, this means whether the range partially selects
                // no non-text nodes.
                var iterator = new RangeIterator(this, true);
                var boundariesInvalid = (iterator._first && (isNonTextPartiallySelected(iterator._first, this)) ||
                        (iterator._last && isNonTextPartiallySelected(iterator._last, this)));
                iterator.detach();
                return !boundariesInvalid;
            },

            surroundContents: function(node) {
                assertValidNodeType(node, surroundNodeTypes);

                if (!this.canSurroundContents()) {
                    throw new DOMException("INVALID_STATE_ERR");
                }

                // Extract the contents
                var content = this.extractContents();

                // Clear the children of the node
                if (node.hasChildNodes()) {
                    while (node.lastChild) {
                        node.removeChild(node.lastChild);
                    }
                }

                // Insert the new node and add the extracted contents
                insertNodeAtPosition(node, this.startContainer, this.startOffset);
                node.appendChild(content);

                this.selectNode(node);
            },

            cloneRange: function() {
                assertRangeValid(this);
                var range = new Range(getRangeDocument(this));
                var i = rangeProperties.length, prop;
                while (i--) {
                    prop = rangeProperties[i];
                    range[prop] = this[prop];
                }
                return range;
            },

            toString: function() {
                assertRangeValid(this);
                var sc = this.startContainer;
                if (sc === this.endContainer && isCharacterDataNode(sc)) {
                    return (sc.nodeType == 3 || sc.nodeType == 4) ? sc.data.slice(this.startOffset, this.endOffset) : "";
                } else {
                    var textParts = [], iterator = new RangeIterator(this, true);
                    iterateSubtree(iterator, function(node) {
                        // Accept only text or CDATA nodes, not comments
                        if (node.nodeType == 3 || node.nodeType == 4) {
                            textParts.push(node.data);
                        }
                    });
                    iterator.detach();
                    return textParts.join("");
                }
            },

            // The methods below are all non-standard. The following batch were introduced by Mozilla but have since
            // been removed from Mozilla.

            compareNode: function(node) {
                assertRangeValid(this);

                var parent = node.parentNode;
                var nodeIndex = getNodeIndex(node);

                if (!parent) {
                    throw new DOMException("NOT_FOUND_ERR");
                }

                var startComparison = this.comparePoint(parent, nodeIndex),
                    endComparison = this.comparePoint(parent, nodeIndex + 1);

                if (startComparison < 0) { // Node starts before
                    return (endComparison > 0) ? n_b_a : n_b;
                } else {
                    return (endComparison > 0) ? n_a : n_i;
                }
            },

            comparePoint: function(node, offset) {
                assertRangeValid(this);
                assertNode(node, "HIERARCHY_REQUEST_ERR");
                assertSameDocumentOrFragment(node, this.startContainer);

                if (comparePoints(node, offset, this.startContainer, this.startOffset) < 0) {
                    return -1;
                } else if (comparePoints(node, offset, this.endContainer, this.endOffset) > 0) {
                    return 1;
                }
                return 0;
            },

            createContextualFragment: createContextualFragment,

            toHtml: function() {
                return rangeToHtml(this);
            },

            // touchingIsIntersecting determines whether this method considers a node that borders a range intersects
            // with it (as in WebKit) or not (as in Gecko pre-1.9, and the default)
            intersectsNode: function(node, touchingIsIntersecting) {
                assertRangeValid(this);
                if (getRootContainer(node) != getRangeRoot(this)) {
                    return false;
                }

                var parent = node.parentNode, offset = getNodeIndex(node);
                if (!parent) {
                    return true;
                }

                var startComparison = comparePoints(parent, offset, this.endContainer, this.endOffset),
                    endComparison = comparePoints(parent, offset + 1, this.startContainer, this.startOffset);

                return touchingIsIntersecting ? startComparison <= 0 && endComparison >= 0 : startComparison < 0 && endComparison > 0;
            },

            isPointInRange: function(node, offset) {
                assertRangeValid(this);
                assertNode(node, "HIERARCHY_REQUEST_ERR");
                assertSameDocumentOrFragment(node, this.startContainer);

                return (comparePoints(node, offset, this.startContainer, this.startOffset) >= 0) &&
                       (comparePoints(node, offset, this.endContainer, this.endOffset) <= 0);
            },

            // The methods below are non-standard and invented by me.

            // Sharing a boundary start-to-end or end-to-start does not count as intersection.
            intersectsRange: function(range) {
                return rangesIntersect(this, range, false);
            },

            // Sharing a boundary start-to-end or end-to-start does count as intersection.
            intersectsOrTouchesRange: function(range) {
                return rangesIntersect(this, range, true);
            },

            intersection: function(range) {
                if (this.intersectsRange(range)) {
                    var startComparison = comparePoints(this.startContainer, this.startOffset, range.startContainer, range.startOffset),
                        endComparison = comparePoints(this.endContainer, this.endOffset, range.endContainer, range.endOffset);

                    var intersectionRange = this.cloneRange();
                    if (startComparison == -1) {
                        intersectionRange.setStart(range.startContainer, range.startOffset);
                    }
                    if (endComparison == 1) {
                        intersectionRange.setEnd(range.endContainer, range.endOffset);
                    }
                    return intersectionRange;
                }
                return null;
            },

            union: function(range) {
                if (this.intersectsOrTouchesRange(range)) {
                    var unionRange = this.cloneRange();
                    if (comparePoints(range.startContainer, range.startOffset, this.startContainer, this.startOffset) == -1) {
                        unionRange.setStart(range.startContainer, range.startOffset);
                    }
                    if (comparePoints(range.endContainer, range.endOffset, this.endContainer, this.endOffset) == 1) {
                        unionRange.setEnd(range.endContainer, range.endOffset);
                    }
                    return unionRange;
                } else {
                    throw new DOMException("Ranges do not intersect");
                }
            },

            containsNode: function(node, allowPartial) {
                if (allowPartial) {
                    return this.intersectsNode(node, false);
                } else {
                    return this.compareNode(node) == n_i;
                }
            },

            containsNodeContents: function(node) {
                return this.comparePoint(node, 0) >= 0 && this.comparePoint(node, getNodeLength(node)) <= 0;
            },

            containsRange: function(range) {
                var intersection = this.intersection(range);
                return intersection !== null && range.equals(intersection);
            },

            containsNodeText: function(node) {
                var nodeRange = this.cloneRange();
                nodeRange.selectNode(node);
                var textNodes = nodeRange.getNodes([3]);
                if (textNodes.length > 0) {
                    nodeRange.setStart(textNodes[0], 0);
                    var lastTextNode = textNodes.pop();
                    nodeRange.setEnd(lastTextNode, lastTextNode.length);
                    return this.containsRange(nodeRange);
                } else {
                    return this.containsNodeContents(node);
                }
            },

            getNodes: function(nodeTypes, filter) {
                assertRangeValid(this);
                return getNodesInRange(this, nodeTypes, filter);
            },

            getDocument: function() {
                return getRangeDocument(this);
            },

            collapseBefore: function(node) {
                this.setEndBefore(node);
                this.collapse(false);
            },

            collapseAfter: function(node) {
                this.setStartAfter(node);
                this.collapse(true);
            },

            getBookmark: function(containerNode) {
                var doc = getRangeDocument(this);
                var preSelectionRange = api.createRange(doc);
                containerNode = containerNode || dom.getBody(doc);
                preSelectionRange.selectNodeContents(containerNode);
                var range = this.intersection(preSelectionRange);
                var start = 0, end = 0;
                if (range) {
                    preSelectionRange.setEnd(range.startContainer, range.startOffset);
                    start = preSelectionRange.toString().length;
                    end = start + range.toString().length;
                }

                return {
                    start: start,
                    end: end,
                    containerNode: containerNode
                };
            },

            moveToBookmark: function(bookmark) {
                var containerNode = bookmark.containerNode;
                var charIndex = 0;
                this.setStart(containerNode, 0);
                this.collapse(true);
                var nodeStack = [containerNode], node, foundStart = false, stop = false;
                var nextCharIndex, i, childNodes;

                while (!stop && (node = nodeStack.pop())) {
                    if (node.nodeType == 3) {
                        nextCharIndex = charIndex + node.length;
                        if (!foundStart && bookmark.start >= charIndex && bookmark.start <= nextCharIndex) {
                            this.setStart(node, bookmark.start - charIndex);
                            foundStart = true;
                        }
                        if (foundStart && bookmark.end >= charIndex && bookmark.end <= nextCharIndex) {
                            this.setEnd(node, bookmark.end - charIndex);
                            stop = true;
                        }
                        charIndex = nextCharIndex;
                    } else {
                        childNodes = node.childNodes;
                        i = childNodes.length;
                        while (i--) {
                            nodeStack.push(childNodes[i]);
                        }
                    }
                }
            },

            getName: function() {
                return "DomRange";
            },

            equals: function(range) {
                return Range.rangesEqual(this, range);
            },

            isValid: function() {
                return isRangeValid(this);
            },

            inspect: function() {
                return inspect(this);
            },

            detach: function() {
                // In DOM4, detach() is now a no-op.
            }
        });

        function copyComparisonConstantsToObject(obj) {
            obj.START_TO_START = s2s;
            obj.START_TO_END = s2e;
            obj.END_TO_END = e2e;
            obj.END_TO_START = e2s;

            obj.NODE_BEFORE = n_b;
            obj.NODE_AFTER = n_a;
            obj.NODE_BEFORE_AND_AFTER = n_b_a;
            obj.NODE_INSIDE = n_i;
        }

        function copyComparisonConstants(constructor) {
            copyComparisonConstantsToObject(constructor);
            copyComparisonConstantsToObject(constructor.prototype);
        }

        function createRangeContentRemover(remover, boundaryUpdater) {
            return function() {
                assertRangeValid(this);

                var sc = this.startContainer, so = this.startOffset, root = this.commonAncestorContainer;

                var iterator = new RangeIterator(this, true);

                // Work out where to position the range after content removal
                var node, boundary;
                if (sc !== root) {
                    node = getClosestAncestorIn(sc, root, true);
                    boundary = getBoundaryAfterNode(node);
                    sc = boundary.node;
                    so = boundary.offset;
                }

                // Check none of the range is read-only
                iterateSubtree(iterator, assertNodeNotReadOnly);

                iterator.reset();

                // Remove the content
                var returnValue = remover(iterator);
                iterator.detach();

                // Move to the new position
                boundaryUpdater(this, sc, so, sc, so);

                return returnValue;
            };
        }

        function createPrototypeRange(constructor, boundaryUpdater) {
            function createBeforeAfterNodeSetter(isBefore, isStart) {
                return function(node) {
                    assertValidNodeType(node, beforeAfterNodeTypes);
                    assertValidNodeType(getRootContainer(node), rootContainerNodeTypes);

                    var boundary = (isBefore ? getBoundaryBeforeNode : getBoundaryAfterNode)(node);
                    (isStart ? setRangeStart : setRangeEnd)(this, boundary.node, boundary.offset);
                };
            }

            function setRangeStart(range, node, offset) {
                var ec = range.endContainer, eo = range.endOffset;
                if (node !== range.startContainer || offset !== range.startOffset) {
                    // Check the root containers of the range and the new boundary, and also check whether the new boundary
                    // is after the current end. In either case, collapse the range to the new position
                    if (getRootContainer(node) != getRootContainer(ec) || comparePoints(node, offset, ec, eo) == 1) {
                        ec = node;
                        eo = offset;
                    }
                    boundaryUpdater(range, node, offset, ec, eo);
                }
            }

            function setRangeEnd(range, node, offset) {
                var sc = range.startContainer, so = range.startOffset;
                if (node !== range.endContainer || offset !== range.endOffset) {
                    // Check the root containers of the range and the new boundary, and also check whether the new boundary
                    // is after the current end. In either case, collapse the range to the new position
                    if (getRootContainer(node) != getRootContainer(sc) || comparePoints(node, offset, sc, so) == -1) {
                        sc = node;
                        so = offset;
                    }
                    boundaryUpdater(range, sc, so, node, offset);
                }
            }

            // Set up inheritance
            var F = function() {};
            F.prototype = api.rangePrototype;
            constructor.prototype = new F();

            util.extend(constructor.prototype, {
                setStart: function(node, offset) {
                    assertNoDocTypeNotationEntityAncestor(node, true);
                    assertValidOffset(node, offset);

                    setRangeStart(this, node, offset);
                },

                setEnd: function(node, offset) {
                    assertNoDocTypeNotationEntityAncestor(node, true);
                    assertValidOffset(node, offset);

                    setRangeEnd(this, node, offset);
                },

                /**
                 * Convenience method to set a range's start and end boundaries. Overloaded as follows:
                 * - Two parameters (node, offset) creates a collapsed range at that position
                 * - Three parameters (node, startOffset, endOffset) creates a range contained with node starting at
                 *   startOffset and ending at endOffset
                 * - Four parameters (startNode, startOffset, endNode, endOffset) creates a range starting at startOffset in
                 *   startNode and ending at endOffset in endNode
                 */
                setStartAndEnd: function() {
                    var args = arguments;
                    var sc = args[0], so = args[1], ec = sc, eo = so;

                    switch (args.length) {
                        case 3:
                            eo = args[2];
                            break;
                        case 4:
                            ec = args[2];
                            eo = args[3];
                            break;
                    }

                    boundaryUpdater(this, sc, so, ec, eo);
                },

                setBoundary: function(node, offset, isStart) {
                    this["set" + (isStart ? "Start" : "End")](node, offset);
                },

                setStartBefore: createBeforeAfterNodeSetter(true, true),
                setStartAfter: createBeforeAfterNodeSetter(false, true),
                setEndBefore: createBeforeAfterNodeSetter(true, false),
                setEndAfter: createBeforeAfterNodeSetter(false, false),

                collapse: function(isStart) {
                    assertRangeValid(this);
                    if (isStart) {
                        boundaryUpdater(this, this.startContainer, this.startOffset, this.startContainer, this.startOffset);
                    } else {
                        boundaryUpdater(this, this.endContainer, this.endOffset, this.endContainer, this.endOffset);
                    }
                },

                selectNodeContents: function(node) {
                    assertNoDocTypeNotationEntityAncestor(node, true);

                    boundaryUpdater(this, node, 0, node, getNodeLength(node));
                },

                selectNode: function(node) {
                    assertNoDocTypeNotationEntityAncestor(node, false);
                    assertValidNodeType(node, beforeAfterNodeTypes);

                    var start = getBoundaryBeforeNode(node), end = getBoundaryAfterNode(node);
                    boundaryUpdater(this, start.node, start.offset, end.node, end.offset);
                },

                extractContents: createRangeContentRemover(extractSubtree, boundaryUpdater),

                deleteContents: createRangeContentRemover(deleteSubtree, boundaryUpdater),

                canSurroundContents: function() {
                    assertRangeValid(this);
                    assertNodeNotReadOnly(this.startContainer);
                    assertNodeNotReadOnly(this.endContainer);

                    // Check if the contents can be surrounded. Specifically, this means whether the range partially selects
                    // no non-text nodes.
                    var iterator = new RangeIterator(this, true);
                    var boundariesInvalid = (iterator._first && isNonTextPartiallySelected(iterator._first, this) ||
                            (iterator._last && isNonTextPartiallySelected(iterator._last, this)));
                    iterator.detach();
                    return !boundariesInvalid;
                },

                splitBoundaries: function() {
                    splitRangeBoundaries(this);
                },

                splitBoundariesPreservingPositions: function(positionsToPreserve) {
                    splitRangeBoundaries(this, positionsToPreserve);
                },

                normalizeBoundaries: function() {
                    assertRangeValid(this);

                    var sc = this.startContainer, so = this.startOffset, ec = this.endContainer, eo = this.endOffset;

                    var mergeForward = function(node) {
                        var sibling = node.nextSibling;
                        if (sibling && sibling.nodeType == node.nodeType) {
                            ec = node;
                            eo = node.length;
                            node.appendData(sibling.data);
                            removeNode(sibling);
                        }
                    };

                    var mergeBackward = function(node) {
                        var sibling = node.previousSibling;
                        if (sibling && sibling.nodeType == node.nodeType) {
                            sc = node;
                            var nodeLength = node.length;
                            so = sibling.length;
                            node.insertData(0, sibling.data);
                            removeNode(sibling);
                            if (sc == ec) {
                                eo += so;
                                ec = sc;
                            } else if (ec == node.parentNode) {
                                var nodeIndex = getNodeIndex(node);
                                if (eo == nodeIndex) {
                                    ec = node;
                                    eo = nodeLength;
                                } else if (eo > nodeIndex) {
                                    eo--;
                                }
                            }
                        }
                    };

                    var normalizeStart = true;
                    var sibling;

                    if (isCharacterDataNode(ec)) {
                        if (eo == ec.length) {
                            mergeForward(ec);
                        } else if (eo == 0) {
                            sibling = ec.previousSibling;
                            if (sibling && sibling.nodeType == ec.nodeType) {
                                eo = sibling.length;
                                if (sc == ec) {
                                    normalizeStart = false;
                                }
                                sibling.appendData(ec.data);
                                removeNode(ec);
                                ec = sibling;
                            }
                        }
                    } else {
                        if (eo > 0) {
                            var endNode = ec.childNodes[eo - 1];
                            if (endNode && isCharacterDataNode(endNode)) {
                                mergeForward(endNode);
                            }
                        }
                        normalizeStart = !this.collapsed;
                    }

                    if (normalizeStart) {
                        if (isCharacterDataNode(sc)) {
                            if (so == 0) {
                                mergeBackward(sc);
                            } else if (so == sc.length) {
                                sibling = sc.nextSibling;
                                if (sibling && sibling.nodeType == sc.nodeType) {
                                    if (ec == sibling) {
                                        ec = sc;
                                        eo += sc.length;
                                    }
                                    sc.appendData(sibling.data);
                                    removeNode(sibling);
                                }
                            }
                        } else {
                            if (so < sc.childNodes.length) {
                                var startNode = sc.childNodes[so];
                                if (startNode && isCharacterDataNode(startNode)) {
                                    mergeBackward(startNode);
                                }
                            }
                        }
                    } else {
                        sc = ec;
                        so = eo;
                    }

                    boundaryUpdater(this, sc, so, ec, eo);
                },

                collapseToPoint: function(node, offset) {
                    assertNoDocTypeNotationEntityAncestor(node, true);
                    assertValidOffset(node, offset);
                    this.setStartAndEnd(node, offset);
                }
            });

            copyComparisonConstants(constructor);
        }

        /*----------------------------------------------------------------------------------------------------------------*/

        // Updates commonAncestorContainer and collapsed after boundary change
        function updateCollapsedAndCommonAncestor(range) {
            range.collapsed = (range.startContainer === range.endContainer && range.startOffset === range.endOffset);
            range.commonAncestorContainer = range.collapsed ?
                range.startContainer : dom.getCommonAncestor(range.startContainer, range.endContainer);
        }

        function updateBoundaries(range, startContainer, startOffset, endContainer, endOffset) {
            range.startContainer = startContainer;
            range.startOffset = startOffset;
            range.endContainer = endContainer;
            range.endOffset = endOffset;
            range.document = dom.getDocument(startContainer);

            updateCollapsedAndCommonAncestor(range);
        }

        function Range(doc) {
            this.startContainer = doc;
            this.startOffset = 0;
            this.endContainer = doc;
            this.endOffset = 0;
            this.document = doc;
            updateCollapsedAndCommonAncestor(this);
        }

        createPrototypeRange(Range, updateBoundaries);

        util.extend(Range, {
            rangeProperties: rangeProperties,
            RangeIterator: RangeIterator,
            copyComparisonConstants: copyComparisonConstants,
            createPrototypeRange: createPrototypeRange,
            inspect: inspect,
            toHtml: rangeToHtml,
            getRangeDocument: getRangeDocument,
            rangesEqual: function(r1, r2) {
                return r1.startContainer === r2.startContainer &&
                    r1.startOffset === r2.startOffset &&
                    r1.endContainer === r2.endContainer &&
                    r1.endOffset === r2.endOffset;
            }
        });

        api.DomRange = Range;
    });

    /*----------------------------------------------------------------------------------------------------------------*/

    // Wrappers for the browser's native DOM Range and/or TextRange implementation
    api.createCoreModule("WrappedRange", ["DomRange"], function(api, module) {
        var WrappedRange, WrappedTextRange;
        var dom = api.dom;
        var util = api.util;
        var DomPosition = dom.DomPosition;
        var DomRange = api.DomRange;
        var getBody = dom.getBody;
        var getContentDocument = dom.getContentDocument;
        var isCharacterDataNode = dom.isCharacterDataNode;


        /*----------------------------------------------------------------------------------------------------------------*/

        if (api.features.implementsDomRange) {
            // This is a wrapper around the browser's native DOM Range. It has two aims:
            // - Provide workarounds for specific browser bugs
            // - provide convenient extensions, which are inherited from Rangy's DomRange

            (function() {
                var rangeProto;
                var rangeProperties = DomRange.rangeProperties;

                function updateRangeProperties(range) {
                    var i = rangeProperties.length, prop;
                    while (i--) {
                        prop = rangeProperties[i];
                        range[prop] = range.nativeRange[prop];
                    }
                    // Fix for broken collapsed property in IE 9.
                    range.collapsed = (range.startContainer === range.endContainer && range.startOffset === range.endOffset);
                }

                function updateNativeRange(range, startContainer, startOffset, endContainer, endOffset) {
                    var startMoved = (range.startContainer !== startContainer || range.startOffset != startOffset);
                    var endMoved = (range.endContainer !== endContainer || range.endOffset != endOffset);
                    var nativeRangeDifferent = !range.equals(range.nativeRange);

                    // Always set both boundaries for the benefit of IE9 (see issue 35)
                    if (startMoved || endMoved || nativeRangeDifferent) {
                        range.setEnd(endContainer, endOffset);
                        range.setStart(startContainer, startOffset);
                    }
                }

                var createBeforeAfterNodeSetter;

                WrappedRange = function(range) {
                    if (!range) {
                        throw module.createError("WrappedRange: Range must be specified");
                    }
                    this.nativeRange = range;
                    updateRangeProperties(this);
                };

                DomRange.createPrototypeRange(WrappedRange, updateNativeRange);

                rangeProto = WrappedRange.prototype;

                rangeProto.selectNode = function(node) {
                    this.nativeRange.selectNode(node);
                    updateRangeProperties(this);
                };

                rangeProto.cloneContents = function() {
                    return this.nativeRange.cloneContents();
                };

                // Due to a long-standing Firefox bug that I have not been able to find a reliable way to detect,
                // insertNode() is never delegated to the native range.

                rangeProto.surroundContents = function(node) {
                    this.nativeRange.surroundContents(node);
                    updateRangeProperties(this);
                };

                rangeProto.collapse = function(isStart) {
                    this.nativeRange.collapse(isStart);
                    updateRangeProperties(this);
                };

                rangeProto.cloneRange = function() {
                    return new WrappedRange(this.nativeRange.cloneRange());
                };

                rangeProto.refresh = function() {
                    updateRangeProperties(this);
                };

                rangeProto.toString = function() {
                    return this.nativeRange.toString();
                };

                // Create test range and node for feature detection

                var testTextNode = document.createTextNode("test");
                getBody(document).appendChild(testTextNode);
                var range = document.createRange();

                /*--------------------------------------------------------------------------------------------------------*/

                // Test for Firefox 2 bug that prevents moving the start of a Range to a point after its current end and
                // correct for it

                range.setStart(testTextNode, 0);
                range.setEnd(testTextNode, 0);

                try {
                    range.setStart(testTextNode, 1);

                    rangeProto.setStart = function(node, offset) {
                        this.nativeRange.setStart(node, offset);
                        updateRangeProperties(this);
                    };

                    rangeProto.setEnd = function(node, offset) {
                        this.nativeRange.setEnd(node, offset);
                        updateRangeProperties(this);
                    };

                    createBeforeAfterNodeSetter = function(name) {
                        return function(node) {
                            this.nativeRange[name](node);
                            updateRangeProperties(this);
                        };
                    };

                } catch(ex) {

                    rangeProto.setStart = function(node, offset) {
                        try {
                            this.nativeRange.setStart(node, offset);
                        } catch (ex) {
                            this.nativeRange.setEnd(node, offset);
                            this.nativeRange.setStart(node, offset);
                        }
                        updateRangeProperties(this);
                    };

                    rangeProto.setEnd = function(node, offset) {
                        try {
                            this.nativeRange.setEnd(node, offset);
                        } catch (ex) {
                            this.nativeRange.setStart(node, offset);
                            this.nativeRange.setEnd(node, offset);
                        }
                        updateRangeProperties(this);
                    };

                    createBeforeAfterNodeSetter = function(name, oppositeName) {
                        return function(node) {
                            try {
                                this.nativeRange[name](node);
                            } catch (ex) {
                                this.nativeRange[oppositeName](node);
                                this.nativeRange[name](node);
                            }
                            updateRangeProperties(this);
                        };
                    };
                }

                rangeProto.setStartBefore = createBeforeAfterNodeSetter("setStartBefore", "setEndBefore");
                rangeProto.setStartAfter = createBeforeAfterNodeSetter("setStartAfter", "setEndAfter");
                rangeProto.setEndBefore = createBeforeAfterNodeSetter("setEndBefore", "setStartBefore");
                rangeProto.setEndAfter = createBeforeAfterNodeSetter("setEndAfter", "setStartAfter");

                /*--------------------------------------------------------------------------------------------------------*/

                // Always use DOM4-compliant selectNodeContents implementation: it's simpler and less code than testing
                // whether the native implementation can be trusted
                rangeProto.selectNodeContents = function(node) {
                    this.setStartAndEnd(node, 0, dom.getNodeLength(node));
                };

                /*--------------------------------------------------------------------------------------------------------*/

                // Test for and correct WebKit bug that has the behaviour of compareBoundaryPoints round the wrong way for
                // constants START_TO_END and END_TO_START: https://bugs.webkit.org/show_bug.cgi?id=20738

                range.selectNodeContents(testTextNode);
                range.setEnd(testTextNode, 3);

                var range2 = document.createRange();
                range2.selectNodeContents(testTextNode);
                range2.setEnd(testTextNode, 4);
                range2.setStart(testTextNode, 2);

                if (range.compareBoundaryPoints(range.START_TO_END, range2) == -1 &&
                        range.compareBoundaryPoints(range.END_TO_START, range2) == 1) {
                    // This is the wrong way round, so correct for it

                    rangeProto.compareBoundaryPoints = function(type, range) {
                        range = range.nativeRange || range;
                        if (type == range.START_TO_END) {
                            type = range.END_TO_START;
                        } else if (type == range.END_TO_START) {
                            type = range.START_TO_END;
                        }
                        return this.nativeRange.compareBoundaryPoints(type, range);
                    };
                } else {
                    rangeProto.compareBoundaryPoints = function(type, range) {
                        return this.nativeRange.compareBoundaryPoints(type, range.nativeRange || range);
                    };
                }

                /*--------------------------------------------------------------------------------------------------------*/

                // Test for IE deleteContents() and extractContents() bug and correct it. See issue 107.

                var el = document.createElement("div");
                el.innerHTML = "123";
                var textNode = el.firstChild;
                var body = getBody(document);
                body.appendChild(el);

                range.setStart(textNode, 1);
                range.setEnd(textNode, 2);
                range.deleteContents();

                if (textNode.data == "13") {
                    // Behaviour is correct per DOM4 Range so wrap the browser's implementation of deleteContents() and
                    // extractContents()
                    rangeProto.deleteContents = function() {
                        this.nativeRange.deleteContents();
                        updateRangeProperties(this);
                    };

                    rangeProto.extractContents = function() {
                        var frag = this.nativeRange.extractContents();
                        updateRangeProperties(this);
                        return frag;
                    };
                }

                body.removeChild(el);
                body = null;

                /*--------------------------------------------------------------------------------------------------------*/

                // Test for existence of createContextualFragment and delegate to it if it exists
                if (util.isHostMethod(range, "createContextualFragment")) {
                    rangeProto.createContextualFragment = function(fragmentStr) {
                        return this.nativeRange.createContextualFragment(fragmentStr);
                    };
                }

                /*--------------------------------------------------------------------------------------------------------*/

                // Clean up
                getBody(document).removeChild(testTextNode);

                rangeProto.getName = function() {
                    return "WrappedRange";
                };

                api.WrappedRange = WrappedRange;

                api.createNativeRange = function(doc) {
                    doc = getContentDocument(doc, module, "createNativeRange");
                    return doc.createRange();
                };
            })();
        }

        if (api.features.implementsTextRange) {
            /*
            This is a workaround for a bug where IE returns the wrong container element from the TextRange's parentElement()
            method. For example, in the following (where pipes denote the selection boundaries):

            <ul id="ul"><li id="a">| a </li><li id="b"> b |</li></ul>

            var range = document.selection.createRange();
            alert(range.parentElement().id); // Should alert "ul" but alerts "b"

            This method returns the common ancestor node of the following:
            - the parentElement() of the textRange
            - the parentElement() of the textRange after calling collapse(true)
            - the parentElement() of the textRange after calling collapse(false)
            */
            var getTextRangeContainerElement = function(textRange) {
                var parentEl = textRange.parentElement();
                var range = textRange.duplicate();
                range.collapse(true);
                var startEl = range.parentElement();
                range = textRange.duplicate();
                range.collapse(false);
                var endEl = range.parentElement();
                var startEndContainer = (startEl == endEl) ? startEl : dom.getCommonAncestor(startEl, endEl);

                return startEndContainer == parentEl ? startEndContainer : dom.getCommonAncestor(parentEl, startEndContainer);
            };

            var textRangeIsCollapsed = function(textRange) {
                return textRange.compareEndPoints("StartToEnd", textRange) == 0;
            };

            // Gets the boundary of a TextRange expressed as a node and an offset within that node. This function started
            // out as an improved version of code found in Tim Cameron Ryan's IERange (http://code.google.com/p/ierange/)
            // but has grown, fixing problems with line breaks in preformatted text, adding workaround for IE TextRange
            // bugs, handling for inputs and images, plus optimizations.
            var getTextRangeBoundaryPosition = function(textRange, wholeRangeContainerElement, isStart, isCollapsed, startInfo) {
                var workingRange = textRange.duplicate();
                workingRange.collapse(isStart);
                var containerElement = workingRange.parentElement();

                // Sometimes collapsing a TextRange that's at the start of a text node can move it into the previous node, so
                // check for that
                if (!dom.isOrIsAncestorOf(wholeRangeContainerElement, containerElement)) {
                    containerElement = wholeRangeContainerElement;
                }


                // Deal with nodes that cannot "contain rich HTML markup". In practice, this means form inputs, images and
                // similar. See http://msdn.microsoft.com/en-us/library/aa703950%28VS.85%29.aspx
                if (!containerElement.canHaveHTML) {
                    var pos = new DomPosition(containerElement.parentNode, dom.getNodeIndex(containerElement));
                    return {
                        boundaryPosition: pos,
                        nodeInfo: {
                            nodeIndex: pos.offset,
                            containerElement: pos.node
                        }
                    };
                }

                var workingNode = dom.getDocument(containerElement).createElement("span");

                // Workaround for HTML5 Shiv's insane violation of document.createElement(). See Rangy issue 104 and HTML5
                // Shiv issue 64: https://github.com/aFarkas/html5shiv/issues/64
                if (workingNode.parentNode) {
                    dom.removeNode(workingNode);
                }

                var comparison, workingComparisonType = isStart ? "StartToStart" : "StartToEnd";
                var previousNode, nextNode, boundaryPosition, boundaryNode;
                var start = (startInfo && startInfo.containerElement == containerElement) ? startInfo.nodeIndex : 0;
                var childNodeCount = containerElement.childNodes.length;
                var end = childNodeCount;

                // Check end first. Code within the loop assumes that the endth child node of the container is definitely
                // after the range boundary.
                var nodeIndex = end;

                while (true) {
                    if (nodeIndex == childNodeCount) {
                        containerElement.appendChild(workingNode);
                    } else {
                        containerElement.insertBefore(workingNode, containerElement.childNodes[nodeIndex]);
                    }
                    workingRange.moveToElementText(workingNode);
                    comparison = workingRange.compareEndPoints(workingComparisonType, textRange);
                    if (comparison == 0 || start == end) {
                        break;
                    } else if (comparison == -1) {
                        if (end == start + 1) {
                            // We know the endth child node is after the range boundary, so we must be done.
                            break;
                        } else {
                            start = nodeIndex;
                        }
                    } else {
                        end = (end == start + 1) ? start : nodeIndex;
                    }
                    nodeIndex = Math.floor((start + end) / 2);
                    containerElement.removeChild(workingNode);
                }


                // We've now reached or gone past the boundary of the text range we're interested in
                // so have identified the node we want
                boundaryNode = workingNode.nextSibling;

                if (comparison == -1 && boundaryNode && isCharacterDataNode(boundaryNode)) {
                    // This is a character data node (text, comment, cdata). The working range is collapsed at the start of
                    // the node containing the text range's boundary, so we move the end of the working range to the
                    // boundary point and measure the length of its text to get the boundary's offset within the node.
                    workingRange.setEndPoint(isStart ? "EndToStart" : "EndToEnd", textRange);

                    var offset;

                    if (/[\r\n]/.test(boundaryNode.data)) {
                        /*
                        For the particular case of a boundary within a text node containing rendered line breaks (within a
                        <pre> element, for example), we need a slightly complicated approach to get the boundary's offset in
                        IE. The facts:

                        - Each line break is represented as \r in the text node's data/nodeValue properties
                        - Each line break is represented as \r\n in the TextRange's 'text' property
                        - The 'text' property of the TextRange does not contain trailing line breaks

                        To get round the problem presented by the final fact above, we can use the fact that TextRange's
                        moveStart() and moveEnd() methods return the actual number of characters moved, which is not
                        necessarily the same as the number of characters it was instructed to move. The simplest approach is
                        to use this to store the characters moved when moving both the start and end of the range to the
                        start of the document body and subtracting the start offset from the end offset (the
                        "move-negative-gazillion" method). However, this is extremely slow when the document is large and
                        the range is near the end of it. Clearly doing the mirror image (i.e. moving the range boundaries to
                        the end of the document) has the same problem.

                        Another approach that works is to use moveStart() to move the start boundary of the range up to the
                        end boundary one character at a time and incrementing a counter with the value returned by the
                        moveStart() call. However, the check for whether the start boundary has reached the end boundary is
                        expensive, so this method is slow (although unlike "move-negative-gazillion" is largely unaffected
                        by the location of the range within the document).

                        The approach used below is a hybrid of the two methods above. It uses the fact that a string
                        containing the TextRange's 'text' property with each \r\n converted to a single \r character cannot
                        be longer than the text of the TextRange, so the start of the range is moved that length initially
                        and then a character at a time to make up for any trailing line breaks not contained in the 'text'
                        property. This has good performance in most situations compared to the previous two methods.
                        */
                        var tempRange = workingRange.duplicate();
                        var rangeLength = tempRange.text.replace(/\r\n/g, "\r").length;

                        offset = tempRange.moveStart("character", rangeLength);
                        while ( (comparison = tempRange.compareEndPoints("StartToEnd", tempRange)) == -1) {
                            offset++;
                            tempRange.moveStart("character", 1);
                        }
                    } else {
                        offset = workingRange.text.length;
                    }
                    boundaryPosition = new DomPosition(boundaryNode, offset);
                } else {

                    // If the boundary immediately follows a character data node and this is the end boundary, we should favour
                    // a position within that, and likewise for a start boundary preceding a character data node
                    previousNode = (isCollapsed || !isStart) && workingNode.previousSibling;
                    nextNode = (isCollapsed || isStart) && workingNode.nextSibling;
                    if (nextNode && isCharacterDataNode(nextNode)) {
                        boundaryPosition = new DomPosition(nextNode, 0);
                    } else if (previousNode && isCharacterDataNode(previousNode)) {
                        boundaryPosition = new DomPosition(previousNode, previousNode.data.length);
                    } else {
                        boundaryPosition = new DomPosition(containerElement, dom.getNodeIndex(workingNode));
                    }
                }

                // Clean up
                dom.removeNode(workingNode);

                return {
                    boundaryPosition: boundaryPosition,
                    nodeInfo: {
                        nodeIndex: nodeIndex,
                        containerElement: containerElement
                    }
                };
            };

            // Returns a TextRange representing the boundary of a TextRange expressed as a node and an offset within that
            // node. This function started out as an optimized version of code found in Tim Cameron Ryan's IERange
            // (http://code.google.com/p/ierange/)
            var createBoundaryTextRange = function(boundaryPosition, isStart) {
                var boundaryNode, boundaryParent, boundaryOffset = boundaryPosition.offset;
                var doc = dom.getDocument(boundaryPosition.node);
                var workingNode, childNodes, workingRange = getBody(doc).createTextRange();
                var nodeIsDataNode = isCharacterDataNode(boundaryPosition.node);

                if (nodeIsDataNode) {
                    boundaryNode = boundaryPosition.node;
                    boundaryParent = boundaryNode.parentNode;
                } else {
                    childNodes = boundaryPosition.node.childNodes;
                    boundaryNode = (boundaryOffset < childNodes.length) ? childNodes[boundaryOffset] : null;
                    boundaryParent = boundaryPosition.node;
                }

                // Position the range immediately before the node containing the boundary
                workingNode = doc.createElement("span");

                // Making the working element non-empty element persuades IE to consider the TextRange boundary to be within
                // the element rather than immediately before or after it
                workingNode.innerHTML = "&#feff;";

                // insertBefore is supposed to work like appendChild if the second parameter is null. However, a bug report
                // for IERange suggests that it can crash the browser: http://code.google.com/p/ierange/issues/detail?id=12
                if (boundaryNode) {
                    boundaryParent.insertBefore(workingNode, boundaryNode);
                } else {
                    boundaryParent.appendChild(workingNode);
                }

                workingRange.moveToElementText(workingNode);
                workingRange.collapse(!isStart);

                // Clean up
                boundaryParent.removeChild(workingNode);

                // Move the working range to the text offset, if required
                if (nodeIsDataNode) {
                    workingRange[isStart ? "moveStart" : "moveEnd"]("character", boundaryOffset);
                }

                return workingRange;
            };

            /*------------------------------------------------------------------------------------------------------------*/

            // This is a wrapper around a TextRange, providing full DOM Range functionality using rangy's DomRange as a
            // prototype

            WrappedTextRange = function(textRange) {
                this.textRange = textRange;
                this.refresh();
            };

            WrappedTextRange.prototype = new DomRange(document);

            WrappedTextRange.prototype.refresh = function() {
                var start, end, startBoundary;

                // TextRange's parentElement() method cannot be trusted. getTextRangeContainerElement() works around that.
                var rangeContainerElement = getTextRangeContainerElement(this.textRange);

                if (textRangeIsCollapsed(this.textRange)) {
                    end = start = getTextRangeBoundaryPosition(this.textRange, rangeContainerElement, true,
                        true).boundaryPosition;
                } else {
                    startBoundary = getTextRangeBoundaryPosition(this.textRange, rangeContainerElement, true, false);
                    start = startBoundary.boundaryPosition;

                    // An optimization used here is that if the start and end boundaries have the same parent element, the
                    // search scope for the end boundary can be limited to exclude the portion of the element that precedes
                    // the start boundary
                    end = getTextRangeBoundaryPosition(this.textRange, rangeContainerElement, false, false,
                        startBoundary.nodeInfo).boundaryPosition;
                }

                this.setStart(start.node, start.offset);
                this.setEnd(end.node, end.offset);
            };

            WrappedTextRange.prototype.getName = function() {
                return "WrappedTextRange";
            };

            DomRange.copyComparisonConstants(WrappedTextRange);

            var rangeToTextRange = function(range) {
                if (range.collapsed) {
                    return createBoundaryTextRange(new DomPosition(range.startContainer, range.startOffset), true);
                } else {
                    var startRange = createBoundaryTextRange(new DomPosition(range.startContainer, range.startOffset), true);
                    var endRange = createBoundaryTextRange(new DomPosition(range.endContainer, range.endOffset), false);
                    var textRange = getBody( DomRange.getRangeDocument(range) ).createTextRange();
                    textRange.setEndPoint("StartToStart", startRange);
                    textRange.setEndPoint("EndToEnd", endRange);
                    return textRange;
                }
            };

            WrappedTextRange.rangeToTextRange = rangeToTextRange;

            WrappedTextRange.prototype.toTextRange = function() {
                return rangeToTextRange(this);
            };

            api.WrappedTextRange = WrappedTextRange;

            // IE 9 and above have both implementations and Rangy makes both available. The next few lines sets which
            // implementation to use by default.
            if (!api.features.implementsDomRange || api.config.preferTextRange) {
                // Add WrappedTextRange as the Range property of the global object to allow expression like Range.END_TO_END to work
                var globalObj = (function(f) { return f("return this;")(); })(Function);
                if (typeof globalObj.Range == "undefined") {
                    globalObj.Range = WrappedTextRange;
                }

                api.createNativeRange = function(doc) {
                    doc = getContentDocument(doc, module, "createNativeRange");
                    return getBody(doc).createTextRange();
                };

                api.WrappedRange = WrappedTextRange;
            }
        }

        api.createRange = function(doc) {
            doc = getContentDocument(doc, module, "createRange");
            return new api.WrappedRange(api.createNativeRange(doc));
        };

        api.createRangyRange = function(doc) {
            doc = getContentDocument(doc, module, "createRangyRange");
            return new DomRange(doc);
        };

        util.createAliasForDeprecatedMethod(api, "createIframeRange", "createRange");
        util.createAliasForDeprecatedMethod(api, "createIframeRangyRange", "createRangyRange");

        api.addShimListener(function(win) {
            var doc = win.document;
            if (typeof doc.createRange == "undefined") {
                doc.createRange = function() {
                    return api.createRange(doc);
                };
            }
            doc = win = null;
        });
    });

    /*----------------------------------------------------------------------------------------------------------------*/

    // This module creates a selection object wrapper that conforms as closely as possible to the Selection specification
    // in the HTML Editing spec (http://dvcs.w3.org/hg/editing/raw-file/tip/editing.html#selections)
    api.createCoreModule("WrappedSelection", ["DomRange", "WrappedRange"], function(api, module) {
        api.config.checkSelectionRanges = true;

        var BOOLEAN = "boolean";
        var NUMBER = "number";
        var dom = api.dom;
        var util = api.util;
        var isHostMethod = util.isHostMethod;
        var DomRange = api.DomRange;
        var WrappedRange = api.WrappedRange;
        var DOMException = api.DOMException;
        var DomPosition = dom.DomPosition;
        var getNativeSelection;
        var selectionIsCollapsed;
        var features = api.features;
        var CONTROL = "Control";
        var getDocument = dom.getDocument;
        var getBody = dom.getBody;
        var rangesEqual = DomRange.rangesEqual;


        // Utility function to support direction parameters in the API that may be a string ("backward", "backwards",
        // "forward" or "forwards") or a Boolean (true for backwards).
        function isDirectionBackward(dir) {
            return (typeof dir == "string") ? /^backward(s)?$/i.test(dir) : !!dir;
        }

        function getWindow(win, methodName) {
            if (!win) {
                return window;
            } else if (dom.isWindow(win)) {
                return win;
            } else if (win instanceof WrappedSelection) {
                return win.win;
            } else {
                var doc = dom.getContentDocument(win, module, methodName);
                return dom.getWindow(doc);
            }
        }

        function getWinSelection(winParam) {
            return getWindow(winParam, "getWinSelection").getSelection();
        }

        function getDocSelection(winParam) {
            return getWindow(winParam, "getDocSelection").document.selection;
        }

        function winSelectionIsBackward(sel) {
            var backward = false;
            if (sel.anchorNode) {
                backward = (dom.comparePoints(sel.anchorNode, sel.anchorOffset, sel.focusNode, sel.focusOffset) == 1);
            }
            return backward;
        }

        // Test for the Range/TextRange and Selection features required
        // Test for ability to retrieve selection
        var implementsWinGetSelection = isHostMethod(window, "getSelection"),
            implementsDocSelection = util.isHostObject(document, "selection");

        features.implementsWinGetSelection = implementsWinGetSelection;
        features.implementsDocSelection = implementsDocSelection;

        var useDocumentSelection = implementsDocSelection && (!implementsWinGetSelection || api.config.preferTextRange);

        if (useDocumentSelection) {
            getNativeSelection = getDocSelection;
            api.isSelectionValid = function(winParam) {
                var doc = getWindow(winParam, "isSelectionValid").document, nativeSel = doc.selection;

                // Check whether the selection TextRange is actually contained within the correct document
                return (nativeSel.type != "None" || getDocument(nativeSel.createRange().parentElement()) == doc);
            };
        } else if (implementsWinGetSelection) {
            getNativeSelection = getWinSelection;
            api.isSelectionValid = function() {
                return true;
            };
        } else {
            module.fail("Neither document.selection or window.getSelection() detected.");
            return false;
        }

        api.getNativeSelection = getNativeSelection;

        var testSelection = getNativeSelection();

        // In Firefox, the selection is null in an iframe with display: none. See issue #138.
        if (!testSelection) {
            module.fail("Native selection was null (possibly issue 138?)");
            return false;
        }

        var testRange = api.createNativeRange(document);
        var body = getBody(document);

        // Obtaining a range from a selection
        var selectionHasAnchorAndFocus = util.areHostProperties(testSelection,
            ["anchorNode", "focusNode", "anchorOffset", "focusOffset"]);

        features.selectionHasAnchorAndFocus = selectionHasAnchorAndFocus;

        // Test for existence of native selection extend() method
        var selectionHasExtend = isHostMethod(testSelection, "extend");
        features.selectionHasExtend = selectionHasExtend;

        // Test if rangeCount exists
        var selectionHasRangeCount = (typeof testSelection.rangeCount == NUMBER);
        features.selectionHasRangeCount = selectionHasRangeCount;

        var selectionSupportsMultipleRanges = false;
        var collapsedNonEditableSelectionsSupported = true;

        var addRangeBackwardToNative = selectionHasExtend ?
            function(nativeSelection, range) {
                var doc = DomRange.getRangeDocument(range);
                var endRange = api.createRange(doc);
                endRange.collapseToPoint(range.endContainer, range.endOffset);
                nativeSelection.addRange(getNativeRange(endRange));
                nativeSelection.extend(range.startContainer, range.startOffset);
            } : null;

        if (util.areHostMethods(testSelection, ["addRange", "getRangeAt", "removeAllRanges"]) &&
                typeof testSelection.rangeCount == NUMBER && features.implementsDomRange) {

            (function() {
                // Previously an iframe was used but this caused problems in some circumstances in IE, so tests are
                // performed on the current document's selection. See issue 109.

                // Note also that if a selection previously existed, it is wiped and later restored by these tests. This
                // will result in the selection direction begin reversed if the original selection was backwards and the
                // browser does not support setting backwards selections (Internet Explorer, I'm looking at you).
                var sel = window.getSelection();
                if (sel) {
                    // Store the current selection
                    var originalSelectionRangeCount = sel.rangeCount;
                    var selectionHasMultipleRanges = (originalSelectionRangeCount > 1);
                    var originalSelectionRanges = [];
                    var originalSelectionBackward = winSelectionIsBackward(sel);
                    for (var i = 0; i < originalSelectionRangeCount; ++i) {
                        originalSelectionRanges[i] = sel.getRangeAt(i);
                    }

                    // Create some test elements
                    var testEl = dom.createTestElement(document, "", false);
                    var textNode = testEl.appendChild( document.createTextNode("\u00a0\u00a0\u00a0") );

                    // Test whether the native selection will allow a collapsed selection within a non-editable element
                    var r1 = document.createRange();

                    r1.setStart(textNode, 1);
                    r1.collapse(true);
                    sel.removeAllRanges();
                    sel.addRange(r1);
                    collapsedNonEditableSelectionsSupported = (sel.rangeCount == 1);
                    sel.removeAllRanges();

                    // Test whether the native selection is capable of supporting multiple ranges.
                    if (!selectionHasMultipleRanges) {
                        // Doing the original feature test here in Chrome 36 (and presumably later versions) prints a
                        // console error of "Discontiguous selection is not supported." that cannot be suppressed. There's
                        // nothing we can do about this while retaining the feature test so we have to resort to a browser
                        // sniff. I'm not happy about it. See
                        // https://code.google.com/p/chromium/issues/detail?id=399791
                        var chromeMatch = window.navigator.appVersion.match(/Chrome\/(.*?) /);
                        if (chromeMatch && parseInt(chromeMatch[1]) >= 36) {
                            selectionSupportsMultipleRanges = false;
                        } else {
                            var r2 = r1.cloneRange();
                            r1.setStart(textNode, 0);
                            r2.setEnd(textNode, 3);
                            r2.setStart(textNode, 2);
                            sel.addRange(r1);
                            sel.addRange(r2);
                            selectionSupportsMultipleRanges = (sel.rangeCount == 2);
                        }
                    }

                    // Clean up
                    dom.removeNode(testEl);
                    sel.removeAllRanges();

                    for (i = 0; i < originalSelectionRangeCount; ++i) {
                        if (i == 0 && originalSelectionBackward) {
                            if (addRangeBackwardToNative) {
                                addRangeBackwardToNative(sel, originalSelectionRanges[i]);
                            } else {
                                api.warn("Rangy initialization: original selection was backwards but selection has been restored forwards because the browser does not support Selection.extend");
                                sel.addRange(originalSelectionRanges[i]);
                            }
                        } else {
                            sel.addRange(originalSelectionRanges[i]);
                        }
                    }
                }
            })();
        }

        features.selectionSupportsMultipleRanges = selectionSupportsMultipleRanges;
        features.collapsedNonEditableSelectionsSupported = collapsedNonEditableSelectionsSupported;

        // ControlRanges
        var implementsControlRange = false, testControlRange;

        if (body && isHostMethod(body, "createControlRange")) {
            testControlRange = body.createControlRange();
            if (util.areHostProperties(testControlRange, ["item", "add"])) {
                implementsControlRange = true;
            }
        }
        features.implementsControlRange = implementsControlRange;

        // Selection collapsedness
        if (selectionHasAnchorAndFocus) {
            selectionIsCollapsed = function(sel) {
                return sel.anchorNode === sel.focusNode && sel.anchorOffset === sel.focusOffset;
            };
        } else {
            selectionIsCollapsed = function(sel) {
                return sel.rangeCount ? sel.getRangeAt(sel.rangeCount - 1).collapsed : false;
            };
        }

        function updateAnchorAndFocusFromRange(sel, range, backward) {
            var anchorPrefix = backward ? "end" : "start", focusPrefix = backward ? "start" : "end";
            sel.anchorNode = range[anchorPrefix + "Container"];
            sel.anchorOffset = range[anchorPrefix + "Offset"];
            sel.focusNode = range[focusPrefix + "Container"];
            sel.focusOffset = range[focusPrefix + "Offset"];
        }

        function updateAnchorAndFocusFromNativeSelection(sel) {
            var nativeSel = sel.nativeSelection;
            sel.anchorNode = nativeSel.anchorNode;
            sel.anchorOffset = nativeSel.anchorOffset;
            sel.focusNode = nativeSel.focusNode;
            sel.focusOffset = nativeSel.focusOffset;
        }

        function updateEmptySelection(sel) {
            sel.anchorNode = sel.focusNode = null;
            sel.anchorOffset = sel.focusOffset = 0;
            sel.rangeCount = 0;
            sel.isCollapsed = true;
            sel._ranges.length = 0;
        }

        function getNativeRange(range) {
            var nativeRange;
            if (range instanceof DomRange) {
                nativeRange = api.createNativeRange(range.getDocument());
                nativeRange.setEnd(range.endContainer, range.endOffset);
                nativeRange.setStart(range.startContainer, range.startOffset);
            } else if (range instanceof WrappedRange) {
                nativeRange = range.nativeRange;
            } else if (features.implementsDomRange && (range instanceof dom.getWindow(range.startContainer).Range)) {
                nativeRange = range;
            }
            return nativeRange;
        }

        function rangeContainsSingleElement(rangeNodes) {
            if (!rangeNodes.length || rangeNodes[0].nodeType != 1) {
                return false;
            }
            for (var i = 1, len = rangeNodes.length; i < len; ++i) {
                if (!dom.isAncestorOf(rangeNodes[0], rangeNodes[i])) {
                    return false;
                }
            }
            return true;
        }

        function getSingleElementFromRange(range) {
            var nodes = range.getNodes();
            if (!rangeContainsSingleElement(nodes)) {
                throw module.createError("getSingleElementFromRange: range " + range.inspect() + " did not consist of a single element");
            }
            return nodes[0];
        }

        // Simple, quick test which only needs to distinguish between a TextRange and a ControlRange
        function isTextRange(range) {
            return !!range && typeof range.text != "undefined";
        }

        function updateFromTextRange(sel, range) {
            // Create a Range from the selected TextRange
            var wrappedRange = new WrappedRange(range);
            sel._ranges = [wrappedRange];

            updateAnchorAndFocusFromRange(sel, wrappedRange, false);
            sel.rangeCount = 1;
            sel.isCollapsed = wrappedRange.collapsed;
        }

        function updateControlSelection(sel) {
            // Update the wrapped selection based on what's now in the native selection
            sel._ranges.length = 0;
            if (sel.docSelection.type == "None") {
                updateEmptySelection(sel);
            } else {
                var controlRange = sel.docSelection.createRange();
                if (isTextRange(controlRange)) {
                    // This case (where the selection type is "Control" and calling createRange() on the selection returns
                    // a TextRange) can happen in IE 9. It happens, for example, when all elements in the selected
                    // ControlRange have been removed from the ControlRange and removed from the document.
                    updateFromTextRange(sel, controlRange);
                } else {
                    sel.rangeCount = controlRange.length;
                    var range, doc = getDocument(controlRange.item(0));
                    for (var i = 0; i < sel.rangeCount; ++i) {
                        range = api.createRange(doc);
                        range.selectNode(controlRange.item(i));
                        sel._ranges.push(range);
                    }
                    sel.isCollapsed = sel.rangeCount == 1 && sel._ranges[0].collapsed;
                    updateAnchorAndFocusFromRange(sel, sel._ranges[sel.rangeCount - 1], false);
                }
            }
        }

        function addRangeToControlSelection(sel, range) {
            var controlRange = sel.docSelection.createRange();
            var rangeElement = getSingleElementFromRange(range);

            // Create a new ControlRange containing all the elements in the selected ControlRange plus the element
            // contained by the supplied range
            var doc = getDocument(controlRange.item(0));
            var newControlRange = getBody(doc).createControlRange();
            for (var i = 0, len = controlRange.length; i < len; ++i) {
                newControlRange.add(controlRange.item(i));
            }
            try {
                newControlRange.add(rangeElement);
            } catch (ex) {
                throw module.createError("addRange(): Element within the specified Range could not be added to control selection (does it have layout?)");
            }
            newControlRange.select();

            // Update the wrapped selection based on what's now in the native selection
            updateControlSelection(sel);
        }

        var getSelectionRangeAt;

        if (isHostMethod(testSelection, "getRangeAt")) {
            // try/catch is present because getRangeAt() must have thrown an error in some browser and some situation.
            // Unfortunately, I didn't write a comment about the specifics and am now scared to take it out. Let that be a
            // lesson to us all, especially me.
            getSelectionRangeAt = function(sel, index) {
                try {
                    return sel.getRangeAt(index);
                } catch (ex) {
                    return null;
                }
            };
        } else if (selectionHasAnchorAndFocus) {
            getSelectionRangeAt = function(sel) {
                var doc = getDocument(sel.anchorNode);
                var range = api.createRange(doc);
                range.setStartAndEnd(sel.anchorNode, sel.anchorOffset, sel.focusNode, sel.focusOffset);

                // Handle the case when the selection was selected backwards (from the end to the start in the
                // document)
                if (range.collapsed !== this.isCollapsed) {
                    range.setStartAndEnd(sel.focusNode, sel.focusOffset, sel.anchorNode, sel.anchorOffset);
                }

                return range;
            };
        }

        function WrappedSelection(selection, docSelection, win) {
            this.nativeSelection = selection;
            this.docSelection = docSelection;
            this._ranges = [];
            this.win = win;
            this.refresh();
        }

        WrappedSelection.prototype = api.selectionPrototype;

        function deleteProperties(sel) {
            sel.win = sel.anchorNode = sel.focusNode = sel._ranges = null;
            sel.rangeCount = sel.anchorOffset = sel.focusOffset = 0;
            sel.detached = true;
        }

        var cachedRangySelections = [];

        function actOnCachedSelection(win, action) {
            var i = cachedRangySelections.length, cached, sel;
            while (i--) {
                cached = cachedRangySelections[i];
                sel = cached.selection;
                if (action == "deleteAll") {
                    deleteProperties(sel);
                } else if (cached.win == win) {
                    if (action == "delete") {
                        cachedRangySelections.splice(i, 1);
                        return true;
                    } else {
                        return sel;
                    }
                }
            }
            if (action == "deleteAll") {
                cachedRangySelections.length = 0;
            }
            return null;
        }

        var getSelection = function(win) {
            // Check if the parameter is a Rangy Selection object
            if (win && win instanceof WrappedSelection) {
                win.refresh();
                return win;
            }

            win = getWindow(win, "getNativeSelection");

            var sel = actOnCachedSelection(win);
            var nativeSel = getNativeSelection(win), docSel = implementsDocSelection ? getDocSelection(win) : null;
            if (sel) {
                sel.nativeSelection = nativeSel;
                sel.docSelection = docSel;
                sel.refresh();
            } else {
                sel = new WrappedSelection(nativeSel, docSel, win);
                cachedRangySelections.push( { win: win, selection: sel } );
            }
            return sel;
        };

        api.getSelection = getSelection;

        util.createAliasForDeprecatedMethod(api, "getIframeSelection", "getSelection");

        var selProto = WrappedSelection.prototype;

        function createControlSelection(sel, ranges) {
            // Ensure that the selection becomes of type "Control"
            var doc = getDocument(ranges[0].startContainer);
            var controlRange = getBody(doc).createControlRange();
            for (var i = 0, el, len = ranges.length; i < len; ++i) {
                el = getSingleElementFromRange(ranges[i]);
                try {
                    controlRange.add(el);
                } catch (ex) {
                    throw module.createError("setRanges(): Element within one of the specified Ranges could not be added to control selection (does it have layout?)");
                }
            }
            controlRange.select();

            // Update the wrapped selection based on what's now in the native selection
            updateControlSelection(sel);
        }

        // Selecting a range
        if (!useDocumentSelection && selectionHasAnchorAndFocus && util.areHostMethods(testSelection, ["removeAllRanges", "addRange"])) {
            selProto.removeAllRanges = function() {
                this.nativeSelection.removeAllRanges();
                updateEmptySelection(this);
            };

            var addRangeBackward = function(sel, range) {
                addRangeBackwardToNative(sel.nativeSelection, range);
                sel.refresh();
            };

            if (selectionHasRangeCount) {
                selProto.addRange = function(range, direction) {
                    if (implementsControlRange && implementsDocSelection && this.docSelection.type == CONTROL) {
                        addRangeToControlSelection(this, range);
                    } else {
                        if (isDirectionBackward(direction) && selectionHasExtend) {
                            addRangeBackward(this, range);
                        } else {
                            var previousRangeCount;
                            if (selectionSupportsMultipleRanges) {
                                previousRangeCount = this.rangeCount;
                            } else {
                                this.removeAllRanges();
                                previousRangeCount = 0;
                            }
                            // Clone the native range so that changing the selected range does not affect the selection.
                            // This is contrary to the spec but is the only way to achieve consistency between browsers. See
                            // issue 80.
                            var clonedNativeRange = getNativeRange(range).cloneRange();
                            try {
                                this.nativeSelection.addRange(clonedNativeRange);
                            } catch (ex) {
                            }

                            // Check whether adding the range was successful
                            this.rangeCount = this.nativeSelection.rangeCount;

                            if (this.rangeCount == previousRangeCount + 1) {
                                // The range was added successfully

                                // Check whether the range that we added to the selection is reflected in the last range extracted from
                                // the selection
                                if (api.config.checkSelectionRanges) {
                                    var nativeRange = getSelectionRangeAt(this.nativeSelection, this.rangeCount - 1);
                                    if (nativeRange && !rangesEqual(nativeRange, range)) {
                                        // Happens in WebKit with, for example, a selection placed at the start of a text node
                                        range = new WrappedRange(nativeRange);
                                    }
                                }
                                this._ranges[this.rangeCount - 1] = range;
                                updateAnchorAndFocusFromRange(this, range, selectionIsBackward(this.nativeSelection));
                                this.isCollapsed = selectionIsCollapsed(this);
                            } else {
                                // The range was not added successfully. The simplest thing is to refresh
                                this.refresh();
                            }
                        }
                    }
                };
            } else {
                selProto.addRange = function(range, direction) {
                    if (isDirectionBackward(direction) && selectionHasExtend) {
                        addRangeBackward(this, range);
                    } else {
                        this.nativeSelection.addRange(getNativeRange(range));
                        this.refresh();
                    }
                };
            }

            selProto.setRanges = function(ranges) {
                if (implementsControlRange && implementsDocSelection && ranges.length > 1) {
                    createControlSelection(this, ranges);
                } else {
                    this.removeAllRanges();
                    for (var i = 0, len = ranges.length; i < len; ++i) {
                        this.addRange(ranges[i]);
                    }
                }
            };
        } else if (isHostMethod(testSelection, "empty") && isHostMethod(testRange, "select") &&
                   implementsControlRange && useDocumentSelection) {

            selProto.removeAllRanges = function() {
                // Added try/catch as fix for issue #21
                try {
                    this.docSelection.empty();

                    // Check for empty() not working (issue #24)
                    if (this.docSelection.type != "None") {
                        // Work around failure to empty a control selection by instead selecting a TextRange and then
                        // calling empty()
                        var doc;
                        if (this.anchorNode) {
                            doc = getDocument(this.anchorNode);
                        } else if (this.docSelection.type == CONTROL) {
                            var controlRange = this.docSelection.createRange();
                            if (controlRange.length) {
                                doc = getDocument( controlRange.item(0) );
                            }
                        }
                        if (doc) {
                            var textRange = getBody(doc).createTextRange();
                            textRange.select();
                            this.docSelection.empty();
                        }
                    }
                } catch(ex) {}
                updateEmptySelection(this);
            };

            selProto.addRange = function(range) {
                if (this.docSelection.type == CONTROL) {
                    addRangeToControlSelection(this, range);
                } else {
                    api.WrappedTextRange.rangeToTextRange(range).select();
                    this._ranges[0] = range;
                    this.rangeCount = 1;
                    this.isCollapsed = this._ranges[0].collapsed;
                    updateAnchorAndFocusFromRange(this, range, false);
                }
            };

            selProto.setRanges = function(ranges) {
                this.removeAllRanges();
                var rangeCount = ranges.length;
                if (rangeCount > 1) {
                    createControlSelection(this, ranges);
                } else if (rangeCount) {
                    this.addRange(ranges[0]);
                }
            };
        } else {
            module.fail("No means of selecting a Range or TextRange was found");
            return false;
        }

        selProto.getRangeAt = function(index) {
            if (index < 0 || index >= this.rangeCount) {
                throw new DOMException("INDEX_SIZE_ERR");
            } else {
                // Clone the range to preserve selection-range independence. See issue 80.
                return this._ranges[index].cloneRange();
            }
        };

        var refreshSelection;

        if (useDocumentSelection) {
            refreshSelection = function(sel) {
                var range;
                if (api.isSelectionValid(sel.win)) {
                    range = sel.docSelection.createRange();
                } else {
                    range = getBody(sel.win.document).createTextRange();
                    range.collapse(true);
                }

                if (sel.docSelection.type == CONTROL) {
                    updateControlSelection(sel);
                } else if (isTextRange(range)) {
                    updateFromTextRange(sel, range);
                } else {
                    updateEmptySelection(sel);
                }
            };
        } else if (isHostMethod(testSelection, "getRangeAt") && typeof testSelection.rangeCount == NUMBER) {
            refreshSelection = function(sel) {
                if (implementsControlRange && implementsDocSelection && sel.docSelection.type == CONTROL) {
                    updateControlSelection(sel);
                } else {
                    sel._ranges.length = sel.rangeCount = sel.nativeSelection.rangeCount;
                    if (sel.rangeCount) {
                        for (var i = 0, len = sel.rangeCount; i < len; ++i) {
                            sel._ranges[i] = new api.WrappedRange(sel.nativeSelection.getRangeAt(i));
                        }
                        updateAnchorAndFocusFromRange(sel, sel._ranges[sel.rangeCount - 1], selectionIsBackward(sel.nativeSelection));
                        sel.isCollapsed = selectionIsCollapsed(sel);
                    } else {
                        updateEmptySelection(sel);
                    }
                }
            };
        } else if (selectionHasAnchorAndFocus && typeof testSelection.isCollapsed == BOOLEAN && typeof testRange.collapsed == BOOLEAN && features.implementsDomRange) {
            refreshSelection = function(sel) {
                var range, nativeSel = sel.nativeSelection;
                if (nativeSel.anchorNode) {
                    range = getSelectionRangeAt(nativeSel, 0);
                    sel._ranges = [range];
                    sel.rangeCount = 1;
                    updateAnchorAndFocusFromNativeSelection(sel);
                    sel.isCollapsed = selectionIsCollapsed(sel);
                } else {
                    updateEmptySelection(sel);
                }
            };
        } else {
            module.fail("No means of obtaining a Range or TextRange from the user's selection was found");
            return false;
        }

        selProto.refresh = function(checkForChanges) {
            var oldRanges = checkForChanges ? this._ranges.slice(0) : null;
            var oldAnchorNode = this.anchorNode, oldAnchorOffset = this.anchorOffset;

            refreshSelection(this);
            if (checkForChanges) {
                // Check the range count first
                var i = oldRanges.length;
                if (i != this._ranges.length) {
                    return true;
                }

                // Now check the direction. Checking the anchor position is the same is enough since we're checking all the
                // ranges after this
                if (this.anchorNode != oldAnchorNode || this.anchorOffset != oldAnchorOffset) {
                    return true;
                }

                // Finally, compare each range in turn
                while (i--) {
                    if (!rangesEqual(oldRanges[i], this._ranges[i])) {
                        return true;
                    }
                }
                return false;
            }
        };

        // Removal of a single range
        var removeRangeManually = function(sel, range) {
            var ranges = sel.getAllRanges();
            sel.removeAllRanges();
            for (var i = 0, len = ranges.length; i < len; ++i) {
                if (!rangesEqual(range, ranges[i])) {
                    sel.addRange(ranges[i]);
                }
            }
            if (!sel.rangeCount) {
                updateEmptySelection(sel);
            }
        };

        if (implementsControlRange && implementsDocSelection) {
            selProto.removeRange = function(range) {
                if (this.docSelection.type == CONTROL) {
                    var controlRange = this.docSelection.createRange();
                    var rangeElement = getSingleElementFromRange(range);

                    // Create a new ControlRange containing all the elements in the selected ControlRange minus the
                    // element contained by the supplied range
                    var doc = getDocument(controlRange.item(0));
                    var newControlRange = getBody(doc).createControlRange();
                    var el, removed = false;
                    for (var i = 0, len = controlRange.length; i < len; ++i) {
                        el = controlRange.item(i);
                        if (el !== rangeElement || removed) {
                            newControlRange.add(controlRange.item(i));
                        } else {
                            removed = true;
                        }
                    }
                    newControlRange.select();

                    // Update the wrapped selection based on what's now in the native selection
                    updateControlSelection(this);
                } else {
                    removeRangeManually(this, range);
                }
            };
        } else {
            selProto.removeRange = function(range) {
                removeRangeManually(this, range);
            };
        }

        // Detecting if a selection is backward
        var selectionIsBackward;
        if (!useDocumentSelection && selectionHasAnchorAndFocus && features.implementsDomRange) {
            selectionIsBackward = winSelectionIsBackward;

            selProto.isBackward = function() {
                return selectionIsBackward(this);
            };
        } else {
            selectionIsBackward = selProto.isBackward = function() {
                return false;
            };
        }

        // Create an alias for backwards compatibility. From 1.3, everything is "backward" rather than "backwards"
        selProto.isBackwards = selProto.isBackward;

        // Selection stringifier
        // This is conformant to the old HTML5 selections draft spec but differs from WebKit and Mozilla's implementation.
        // The current spec does not yet define this method.
        selProto.toString = function() {
            var rangeTexts = [];
            for (var i = 0, len = this.rangeCount; i < len; ++i) {
                rangeTexts[i] = "" + this._ranges[i];
            }
            return rangeTexts.join("");
        };

        function assertNodeInSameDocument(sel, node) {
            if (sel.win.document != getDocument(node)) {
                throw new DOMException("WRONG_DOCUMENT_ERR");
            }
        }

        // No current browser conforms fully to the spec for this method, so Rangy's own method is always used
        selProto.collapse = function(node, offset) {
            assertNodeInSameDocument(this, node);
            var range = api.createRange(node);
            range.collapseToPoint(node, offset);
            this.setSingleRange(range);
            this.isCollapsed = true;
        };

        selProto.collapseToStart = function() {
            if (this.rangeCount) {
                var range = this._ranges[0];
                this.collapse(range.startContainer, range.startOffset);
            } else {
                throw new DOMException("INVALID_STATE_ERR");
            }
        };

        selProto.collapseToEnd = function() {
            if (this.rangeCount) {
                var range = this._ranges[this.rangeCount - 1];
                this.collapse(range.endContainer, range.endOffset);
            } else {
                throw new DOMException("INVALID_STATE_ERR");
            }
        };

        // The spec is very specific on how selectAllChildren should be implemented and not all browsers implement it as
        // specified so the native implementation is never used by Rangy.
        selProto.selectAllChildren = function(node) {
            assertNodeInSameDocument(this, node);
            var range = api.createRange(node);
            range.selectNodeContents(node);
            this.setSingleRange(range);
        };

        selProto.deleteFromDocument = function() {
            // Sepcial behaviour required for IE's control selections
            if (implementsControlRange && implementsDocSelection && this.docSelection.type == CONTROL) {
                var controlRange = this.docSelection.createRange();
                var element;
                while (controlRange.length) {
                    element = controlRange.item(0);
                    controlRange.remove(element);
                    dom.removeNode(element);
                }
                this.refresh();
            } else if (this.rangeCount) {
                var ranges = this.getAllRanges();
                if (ranges.length) {
                    this.removeAllRanges();
                    for (var i = 0, len = ranges.length; i < len; ++i) {
                        ranges[i].deleteContents();
                    }
                    // The spec says nothing about what the selection should contain after calling deleteContents on each
                    // range. Firefox moves the selection to where the final selected range was, so we emulate that
                    this.addRange(ranges[len - 1]);
                }
            }
        };

        // The following are non-standard extensions
        selProto.eachRange = function(func, returnValue) {
            for (var i = 0, len = this._ranges.length; i < len; ++i) {
                if ( func( this.getRangeAt(i) ) ) {
                    return returnValue;
                }
            }
        };

        selProto.getAllRanges = function() {
            var ranges = [];
            this.eachRange(function(range) {
                ranges.push(range);
            });
            return ranges;
        };

        selProto.setSingleRange = function(range, direction) {
            this.removeAllRanges();
            this.addRange(range, direction);
        };

        selProto.callMethodOnEachRange = function(methodName, params) {
            var results = [];
            this.eachRange( function(range) {
                results.push( range[methodName].apply(range, params || []) );
            } );
            return results;
        };

        function createStartOrEndSetter(isStart) {
            return function(node, offset) {
                var range;
                if (this.rangeCount) {
                    range = this.getRangeAt(0);
                    range["set" + (isStart ? "Start" : "End")](node, offset);
                } else {
                    range = api.createRange(this.win.document);
                    range.setStartAndEnd(node, offset);
                }
                this.setSingleRange(range, this.isBackward());
            };
        }

        selProto.setStart = createStartOrEndSetter(true);
        selProto.setEnd = createStartOrEndSetter(false);

        // Add select() method to Range prototype. Any existing selection will be removed.
        api.rangePrototype.select = function(direction) {
            getSelection( this.getDocument() ).setSingleRange(this, direction);
        };

        selProto.changeEachRange = function(func) {
            var ranges = [];
            var backward = this.isBackward();

            this.eachRange(function(range) {
                func(range);
                ranges.push(range);
            });

            this.removeAllRanges();
            if (backward && ranges.length == 1) {
                this.addRange(ranges[0], "backward");
            } else {
                this.setRanges(ranges);
            }
        };

        selProto.containsNode = function(node, allowPartial) {
            return this.eachRange( function(range) {
                return range.containsNode(node, allowPartial);
            }, true ) || false;
        };

        selProto.getBookmark = function(containerNode) {
            return {
                backward: this.isBackward(),
                rangeBookmarks: this.callMethodOnEachRange("getBookmark", [containerNode])
            };
        };

        selProto.moveToBookmark = function(bookmark) {
            var selRanges = [];
            for (var i = 0, rangeBookmark, range; rangeBookmark = bookmark.rangeBookmarks[i++]; ) {
                range = api.createRange(this.win);
                range.moveToBookmark(rangeBookmark);
                selRanges.push(range);
            }
            if (bookmark.backward) {
                this.setSingleRange(selRanges[0], "backward");
            } else {
                this.setRanges(selRanges);
            }
        };

        selProto.saveRanges = function() {
            return {
                backward: this.isBackward(),
                ranges: this.callMethodOnEachRange("cloneRange")
            };
        };

        selProto.restoreRanges = function(selRanges) {
            this.removeAllRanges();
            for (var i = 0, range; range = selRanges.ranges[i]; ++i) {
                this.addRange(range, (selRanges.backward && i == 0));
            }
        };

        selProto.toHtml = function() {
            var rangeHtmls = [];
            this.eachRange(function(range) {
                rangeHtmls.push( DomRange.toHtml(range) );
            });
            return rangeHtmls.join("");
        };

        if (features.implementsTextRange) {
            selProto.getNativeTextRange = function() {
                var sel;
                if ( (sel = this.docSelection) ) {
                    var range = sel.createRange();
                    if (isTextRange(range)) {
                        return range;
                    } else {
                        throw module.createError("getNativeTextRange: selection is a control selection");
                    }
                } else if (this.rangeCount > 0) {
                    return api.WrappedTextRange.rangeToTextRange( this.getRangeAt(0) );
                } else {
                    throw module.createError("getNativeTextRange: selection contains no range");
                }
            };
        }

        function inspect(sel) {
            var rangeInspects = [];
            var anchor = new DomPosition(sel.anchorNode, sel.anchorOffset);
            var focus = new DomPosition(sel.focusNode, sel.focusOffset);
            var name = (typeof sel.getName == "function") ? sel.getName() : "Selection";

            if (typeof sel.rangeCount != "undefined") {
                for (var i = 0, len = sel.rangeCount; i < len; ++i) {
                    rangeInspects[i] = DomRange.inspect(sel.getRangeAt(i));
                }
            }
            return "[" + name + "(Ranges: " + rangeInspects.join(", ") +
                    ")(anchor: " + anchor.inspect() + ", focus: " + focus.inspect() + "]";
        }

        selProto.getName = function() {
            return "WrappedSelection";
        };

        selProto.inspect = function() {
            return inspect(this);
        };

        selProto.detach = function() {
            actOnCachedSelection(this.win, "delete");
            deleteProperties(this);
        };

        WrappedSelection.detachAll = function() {
            actOnCachedSelection(null, "deleteAll");
        };

        WrappedSelection.inspect = inspect;
        WrappedSelection.isDirectionBackward = isDirectionBackward;

        api.Selection = WrappedSelection;

        api.selectionPrototype = selProto;

        api.addShimListener(function(win) {
            if (typeof win.getSelection == "undefined") {
                win.getSelection = function() {
                    return getSelection(win);
                };
            }
            win = null;
        });
    });
    

    /*----------------------------------------------------------------------------------------------------------------*/

    // Wait for document to load before initializing
    var docReady = false;

    var loadHandler = function(e) {
        if (!docReady) {
            docReady = true;
            if (!api.initialized && api.config.autoInitialize) {
                init();
            }
        }
    };

    if (isBrowser) {
        // Test whether the document has already been loaded and initialize immediately if so
        if (document.readyState == "complete") {
            loadHandler();
        } else {
            if (isHostMethod(document, "addEventListener")) {
                document.addEventListener("DOMContentLoaded", loadHandler, false);
            }

            // Add a fallback in case the DOMContentLoaded event isn't supported
            addListener(window, "load", loadHandler);
        }
    }

    return api;
});
}(rangyCore));

var rangy = rangyCore.exports;

/*
import font_arial from './fonts/arial.png';
import font_courier from './fonts/courier.png';
import font_georgia from './fonts/georgia.png';
import font_monospace from './fonts/monospace.png';
import font_sans_serif from './fonts/sans_serif.png';
import font_serif from './fonts/serif.png';
import font_abel from './fonts/abel.png';
import font_abril_fatface from './fonts/abril_fatface.png';
import font_advent_pro from './fonts/advent_pro.png';
import font_aladin from './fonts/aladin.png';
import font_alegreya from './fonts/alegreya.png';
import font_alegreya_sans_sc from './fonts/alegreya_sans_sc.png';
import font_alegreya_sc from './fonts/alegreya_sc.png';
import font_alice from './fonts/alice.png';
import font_allerta_stencil from './fonts/allerta_stencil.png';
import font_allura from './fonts/allura.png';
import font_almendra_display from './fonts/almendra_display.png';
import font_amatic_sc from './fonts/amatic_sc.png';
import font_andika from './fonts/andika.png';
import font_anonymous_pro from './fonts/anonymous_pro.png';
import font_architects_daughter from './fonts/architects_daughter.png';
import font_arimo from './fonts/arimo.png';
import font_arsenal from './fonts/arsenal.png';
import font_assistant from './fonts/assistant.png';
import font_aubrey from './fonts/aubrey.png';
import font_anton from './fonts/anton.png';
import font_archivo_narrow from './fonts/archivo_narrow.png';
import font_bad_script from './fonts/bad_script.png';
import font_benchNine from './fonts/benchNine.png';
import font_bevan from './fonts/bevan.png';
import font_bigelow_rules from './fonts/bigelow_rules.png';
import font_bilbo from './fonts/bilbo.png';
import font_bonbon from './fonts/bonbon.png';
import font_bowlby_one_sc from './fonts/bowlby_one_sc.png';
import font_cabin_condensed from './fonts/cabin_condensed.png';
import font_carrois_gothic_sc from './fonts/carrois_gothic_sc.png';
import font_caveat from './fonts/caveat.png';
import font_chewy from './fonts/chewy.png';
import font_cinzel from './fonts/cinzel.png';
import font_comfortaa from './fonts/comfortaa.png';
import font_concert_one from './fonts/concert_one.png';
import font_cormorant from './fonts/cormorant.png';
import font_cormorant_garamond from './fonts/cormorant_garamond.png';
import font_cormorant_infant from './fonts/cormorant_infant.png';
import font_cormorant_sc from './fonts/cormorant_sc.png';
import font_cormorant_unicase from './fonts/cormorant_unicase.png';
import font_cousine from './fonts/cousine.png';
import font_crafty_girls from './fonts/crafty_girls.png';
import font_cuprum from './fonts/cuprum.png';
import font_cutive_mono from './fonts/cutive_mono.png';
import font_devonshire from './fonts/devonshire.png';
import font_didact_gothic from './fonts/didact_gothic.png';
import font_diplomata_sc from './fonts/diplomata_sc.png';
import font_dosis from './fonts/dosis.png';
import font_eb_garamond from './fonts/eb_garamond.png';
import font_el_messiri from './fonts/el_messiri.png';
import font_elsie from './fonts/elsie.png';
import font_encode_sans from './fonts/encode_sans.png';
import font_exo from './fonts/exo.png';
import font_exo_2 from './fonts/exo_2.png';
import font_felipa from './fonts/felipa.png';
import font_fira_code from './fonts/fira_code.png';
import font_fira_mono from './fonts/fira_mono.png';
import font_fira_sans from './fonts/fira_sans.png';
import font_fira_sans_condensed from './fonts/fira_sans_condensed.png';
import font_fira_sans_extra_condensed from './fonts/fira_sans_extra_condensed.png';
import font_fjalla_one from './fonts/fjalla_one.png';
import font_forum from './fonts/forum.png';
import font_frank_ruhl_libre from './fonts/frank_ruhl_libre.png';
import font_fredericka_the_great from './fonts/fredericka_the_great.png';
import font_gabriela from './fonts/gabriela.png';
import font_gilda_display from './fonts/gilda_display.png';
import font_give_you_glory from './fonts/give_you_glory.png';
import font_gruppo from './fonts/gruppo.png';
import font_handlee from './fonts/handlee.png';
import font_happy_monkey from './fonts/happy_monkey.png';
import font_hind from './fonts/hind.png';
import font_ibm_plex_mono from './fonts/ibm_plex_mono.png';
import font_ibm_plex_sans from './fonts/ibm_plex_sans.png';
import font_ibm_plex_serif from './fonts/ibm_plex_serif.png';
import font_iceland from './fonts/iceland.png';
import font_inconsolata from './fonts/inconsolata.png';
import font_josefin_sans from './fonts/josefin_sans.png';
import font_istok_web from './fonts/istok_web.png';
import font_julee from './fonts/julee.png';
import font_julius_sans_one from './fonts/julius_sans_one.png';
import font_junge from './fonts/junge.png';
import font_jura from './fonts/jura.png';
import font_just_me_again_down_here from './fonts/just_me_again_down_here.png';
import font_kaushan_script from './fonts/kaushan_script.png';
import font_kelly_slab from './fonts/kelly_slab.png';
import font_kite_one from './fonts/kite_one.png';
import font_kosugi from './fonts/kosugi.png';
import font_kosugi_maru from './fonts/kosugi_maru.png';
import font_kurale from './fonts/kurale.png';
import font_lato from './fonts/lato.png';
import font_ledger from './fonts/ledger.png';
import font_lekton from './fonts/lekton.png';
import font_life_savers from './fonts/life_savers.png';
import font_literata from './fonts/literata.png';
import font_lobster from './fonts/lobster.png';
import font_lobster_two from './fonts/lobster_two.png';
import font_londrina_shadow from './fonts/londrina_shadow.png';
import font_lora from './fonts/lora.png';
import font_lovers_quarrel from './fonts/lovers_quarrel.png';
import font_m_plus_1p from './fonts/m_plus_1p.png';
import font_m_plus_rounded_1c from './fonts/m_plus_rounded_1c.png';
import font_macondo from './fonts/macondo.png';
import font_marcellus_sc from './fonts/marcellus_sc.png';
import font_marck_script from './fonts/marck_script.png';
import font_martel from './fonts/martel.png';
import font_maven_pro from './fonts/maven_pro.png';
import font_merriweather from './fonts/merriweather.png';
import font_merriweather_sans from './fonts/merriweather_sans.png';
import font_mogra from './fonts/mogra.png';
import font_monoton from './fonts/monoton.png';
import font_montez from './fonts/montez.png';
import font_montserrat from './fonts/montserrat.png';
import font_montserrat_alternates from './fonts/montserrat_alternates.png';
import font_montserrat_subrayada from './fonts/montserrat_subrayada.png';
import font_neucha from './fonts/neucha.png';
import font_neuton from './fonts/neuton.png';
import font_nixie_one from './fonts/nixie_one.png';
import font_nothing_you_could_do from './fonts/nothing_you_could_do.png';
import font_noto_sans from './fonts/noto_sans.png';
import font_noto_sans_sc from './fonts/noto_sans_sc.png';
import font_noto_serif from './fonts/noto_serif.png';
import font_noto_serif_tc from './fonts/noto_serif_tc.png';
import font_nunito from './fonts/nunito.png';
import font_old_standard_tt from './fonts/old_standard_tt.png';
import font_open_sans from './fonts/open_sans.png';
import font_open_sans_condensed from './fonts/open_sans_condensed.png';
import font_oranienbaum from './fonts/oranienbaum.png';
import font_oswald from './fonts/oswald.png';
import font_oxygen from './fonts/oxygen.png';
import font_pacifico from './fonts/pacifico.png';
import font_pangolin from './fonts/pangolin.png';
import font_passion_one from './fonts/passion_one.png';
import font_pathway_gothic_one from './fonts/pathway_gothic_one.png';
import font_pattaya from './fonts/pattaya.png';
import font_petit_formal_script from './fonts/petit_formal_script.png';
import font_philosopher from './fonts/philosopher.png';
import font_play from './fonts/play.png';
import font_playfair_display from './fonts/playfair_display.png';
import font_playfair_display_sc from './fonts/playfair_display_sc.png';
import font_podkova from './fonts/podkova.png';
import font_poiret_one from './fonts/poiret_one.png';
import font_pompiere from './fonts/pompiere.png';
import font_poppins from './fonts/poppins.png';
import font_prata from './fonts/prata.png';
import font_press_start_2p from './fonts/press_start_2p.png';
import font_prosto_one from './fonts/prosto_one.png';
import font_pt_mono from './fonts/pt_mono.png';
import font_pt_sans from './fonts/pt_sans.png';
import font_pt_sans_caption from './fonts/pt_sans_caption.png';
import font_pt_sans_narrow from './fonts/pt_sans_narrow.png';
import font_pt_serif from './fonts/pt_serif.png';
import font_pt_serif_caption from './fonts/pt_serif_caption.png';
import font_quattrocento_sans from './fonts/quattrocento_sans.png';
import font_quattrocento from './fonts/quattrocento.png';
import font_quicksand from './fonts/quicksand.png';
import font_qwigley from './fonts/qwigley.png';
import font_raleway from './fonts/raleway.png';
import font_raleway_dots from './fonts/raleway_dots.png';
import font_redressed from './fonts/redressed.png';
import font_ribeye_marrow from './fonts/ribeye_marrow.png';
import font_righteous from './fonts/righteous.png';
import font_roboto from './fonts/roboto.png';
import font_roboto_condensed from './fonts/roboto_condensed.png';
import font_roboto_mono from './fonts/roboto_mono.png';
import font_roboto_slab from './fonts/roboto_slab.png';
import font_rochester from './fonts/rochester.png';
import font_rouge_script from './fonts/rouge_script.png';
import font_rubik from './fonts/rubik.png';
import font_rubik_mono_one from './fonts/rubik_mono_one.png';
import font_ruslan_display from './fonts/ruslan_display.png';
import font_russo_one from './fonts/russo_one.png';
import font_sacramento from './fonts/sacramento.png';
import font_sanchez from './fonts/sanchez.png';
import font_satisfy from './fonts/satisfy.png';
import font_sawarabi_gothic from './fonts/sawarabi_gothic.png';
import font_scada from './fonts/scada.png';
import font_seaweed_script from './fonts/seaweed_script.png';
import font_seymour_one from './fonts/seymour_one.png';
import font_shadows_into_light_two from './fonts/shadows_into_light_two.png';
import font_six_caps from './fonts/six_caps.png';
import font_snowburst_one from './fonts/snowburst_one.png';
import font_source_code_pro from './fonts/source_code_pro.png';
import font_source_sans_pro from './fonts/source_sans_pro.png';
import font_special_elite from './fonts/special_elite.png';
import font_spectral from './fonts/spectral.png';
import font_spectral_sc from './fonts/spectral_sc.png';
import font_squada_one from './fonts/squada_one.png';
import font_stalinist_one from './fonts/stalinist_one.png';
import font_stint_ultra_expanded from './fonts/stint_ultra_expanded.png';
import font_syncopate from './fonts/syncopate.png';
import font_tangerine from './fonts/tangerine.png';
import font_tenor_sans from './fonts/tenor_sans.png';
import font_tinos from './fonts/tinos.png';
import font_ubuntu from './fonts/ubuntu.png';
import font_ubuntu_condensed from './fonts/ubuntu_condensed.png';
import font_ubuntu_mono from './fonts/ubuntu_mono.png';
import font_underdog from './fonts/underdog.png';
import font_unifrakturmaguntia from './fonts/unifrakturmaguntia.png';
import font_vast_shadow from './fonts/vast_shadow.png';
import font_viga from './fonts/viga.png';
import font_vollkorn from './fonts/vollkorn.png';
import font_vollkorn_sc from './fonts/vollkorn_sc.png';
import font_voltaire from './fonts/voltaire.png';
import font_wire_one from './fonts/wire_one.png';
import font_yanone_kaffeesatz from './fonts/yanone_kaffeesatz.png';
import font_yeseva_one from './fonts/yeseva_one.png';
*/
class Util {
  constructor(builder) {
    this.builder = builder;
    this.dom = this.builder.dom;
  }

  cellSelected() {
    return this.builder.doc.querySelector('.cell-active');
  }

  rowSelected() {
    return this.builder.doc.querySelector('.row-active');
  }

  builderStuff() {
    return document.querySelector('#_cbhtml');
  }

  cellNext(cell) {
    const dom = this.dom;
    let c = cell.nextElementSibling;

    if (c) {
      if (!dom.hasClass(c, 'is-row-tool') && !dom.hasClass(c, 'is-col-tool') && !dom.hasClass(c, 'is-rowadd-tool') && !dom.hasClass(c, 'is-row-overlay')) {
        return c;
      } else {
        return null;
      }
    }

    return null;
  }

  out(s) {
    if (this.builder) {
      let val = this.builder.opts.lang[s];
      if (val) return val;else {
        if (this.builder.checkLang) console.log(s);
        return s;
      }
    } else {
      return s;
    }
  }

  confirm(message, callback, yesno, yestext) {
    const dom = this.dom;
    let html = '';
    html = `<div class="is-modal is-confirm" tabindex="-1" role="dialog" aria-modal="true" aria-hidden="true">
            <div class="is-modal-content" style="padding-bottom:20px;">
                <div style="margin: 20px 0 30px;font-size: 14px;">${message}</div>
                <button title="${this.out('Delete')}" class="input-ok classic focus-warning">${this.out('Delete')}</button>
            </div>
        </div>`;

    if (yesno) {
      html = `<div class="is-modal is-confirm" tabindex="-1" role="dialog" aria-modal="true" aria-hidden="true">
                <div class="is-modal-content" style="padding-bottom:20px;">
                    <div style="margin: 20px 0 30px;font-size: 14px;">${message}</div>
                    <button title="${this.out('Cancel')}" class="input-cancel classic-secondary">${this.out('Cancel')}</button>
                    <button title="${yestext}" class="input-ok classic-primary">${yestext}</button>
                </div>
            </div>`;
    }

    const builderStuff = this.builder.builderStuff;
    let confirmmodal = builderStuff.querySelector('.is-confirm');

    if (!confirmmodal) {
      dom.appendHtml(builderStuff, html);
      confirmmodal = builderStuff.querySelector('.is-confirm');
    }

    this.showModal(confirmmodal, false, () => {
      //this function runs when overlay is clicked. Remove modal.
      confirmmodal.parentNode.removeChild(confirmmodal); //do task

      callback(false);
    }, true);
    let buttonok = confirmmodal.querySelector('.is-confirm .input-ok');
    dom.addEventListener(buttonok, 'click', () => {
      this.hideModal(confirmmodal);
      confirmmodal.parentNode.removeChild(confirmmodal); //remove modal
      //do task

      callback(true);
    });
    let buttoncancel = confirmmodal.querySelector('.is-confirm .input-cancel');
    dom.addEventListener(buttoncancel, 'click', e => {
      this.hideModal(confirmmodal);
      confirmmodal.parentNode.removeChild(confirmmodal); //remove modal
      //do task

      callback(false);
      e.preventDefault();
      e.stopImmediatePropagation();
    });
  }

  showRtePop(pop, onShow, btn) {
    const dom = this.dom;
    pop.style.display = 'flex';
    if (onShow) onShow();
    dom.removeClass(pop, 'deactive');
    dom.addClass(pop, 'active');

    const handleClickOut = e => {
      if (!pop.contains(e.target) && !btn.contains(e.target) && !e.target.closest('.pickcolormore')) {
        // click outside
        this.hideRtePop(pop); // hide

        pop.removeEventListener('keydown', handleKeyDown);
        document.removeEventListener('click', handleClickOut);

        if (this.builder.iframeDocument) {
          this.builder.doc.removeEventListener('click', handleClickOut);
        }
      }
    };

    const handleKeyDown = e => {
      if (e.keyCode === 27) {
        // escape key
        this.hideRtePop(pop); // hide

        pop.removeEventListener('keydown', handleKeyDown);
        document.removeEventListener('click', handleClickOut);

        if (this.builder.iframeDocument) {
          this.builder.doc.removeEventListener('click', handleClickOut);
        }
      }
    };

    pop.addEventListener('keydown', handleKeyDown);
    document.addEventListener('click', handleClickOut);

    if (this.builder.iframeDocument) {
      this.builder.doc.addEventListener('click', handleClickOut);
    }
  }

  hideRtePop(pop) {
    const dom = this.dom;
    pop.style.display = '';
    dom.removeClass(pop, 'active');
    dom.addClass(pop, 'deactive');
  }

  clearAllEventListener(pop) {
    for (let i = 0; i < 10; i++) {
      pop.removeEventListener('keydown', this.handlePopKeyDown);
      document.removeEventListener('click', this.handlePopClickOut);

      if (this.builder.iframeDocument) {
        this.builder.doc.removeEventListener('click', this.handlePopClickOut);
      }
    }
  }

  showPop(pop, cancelCallback, btn) {
    const dom = this.dom; // Hide other pops

    let elms = document.querySelectorAll('.is-pop.active');
    Array.prototype.forEach.call(elms, elm => {
      if (elm !== pop) {
        elm.style.display = '';
        dom.removeClass(elm, 'active');
        elm.setAttribute('aria-hidden', true);
      }
    }); // if(pop.style.display === 'flex') return; 

    this.clearAllEventListener(pop); // for safety of uncleared events as a result of closing without hidePop()

    pop.style.display = 'flex';
    dom.addClass(pop, 'active');
    pop.setAttribute('aria-hidden', false);
    this.setupTabKeys(pop);
    pop.focus({
      preventScroll: true
    });

    this.handlePopClickOut = e => {
      if (!pop.contains(e.target) && !btn.contains(e.target)) {
        // click outside
        // hide
        this.hidePop(pop); // pop.removeEventListener('keydown', this.handlePopKeyDown);
        // document.removeEventListener('click', this.handlePopClickOut);

        if (cancelCallback) cancelCallback();
      }
    };

    this.handlePopKeyDown = e => {
      if (e.keyCode === 27) {
        // escape key
        // hide
        this.hidePop(pop); // pop.removeEventListener('keydown', this.handlePopKeyDown);
        // document.removeEventListener('click', this.handlePopClickOut);

        if (cancelCallback) cancelCallback();
      }
    };

    pop.addEventListener('keydown', this.handlePopKeyDown);
    document.addEventListener('click', this.handlePopClickOut);

    if (this.builder.iframeDocument) {
      this.builder.doc.addEventListener('click', this.handlePopClickOut);
    }
  }

  hidePop(pop) {
    const dom = this.dom;
    pop.style.display = '';
    dom.removeClass(pop, 'active');
    pop.setAttribute('aria-hidden', true); // pop.removeEventListener('keydown', this.handlePopKeyDown);
    // document.removeEventListener('click', this.handlePopClickOut);

    this.clearAllEventListener(pop);
  }

  setupTabKeys(div) {
    let inputs = div.querySelectorAll('a[href], input:not([disabled]):not([type="hidden"]), select:not([disabled]), textarea:not([disabled]), button:not([disabled]):not([tabindex="-1"]), iframe, object, embed, *[tabindex]:not([tabindex="-1"])');
    if (inputs.length === 0) return;
    let firstInput = inputs[0];
    let lastInput = inputs[inputs.length - 1]; // firstInput.focus();
    // Redirect last tab to first input

    lastInput.addEventListener('keydown', e => {
      if (e.which === 9 && !e.shiftKey) {
        e.preventDefault(); // re-select (because controls can enabled/disapbled)

        inputs = div.querySelectorAll('a[href], input:not([disabled]):not([type="hidden"]), select:not([disabled]), textarea:not([disabled]), button:not([disabled]):not([tabindex="-1"]), iframe, object, embed, *[tabindex]:not([tabindex="-1"])');
        firstInput = inputs[0];
        firstInput.focus();
      }
    }); // Redirect first shift+tab to last input

    firstInput.addEventListener('keydown', e => {
      if (e.which === 9 && e.shiftKey) {
        e.preventDefault();
        lastInput.focus();
      }
    });
    const iframes = div.querySelectorAll('iframe');
    iframes.forEach(iframe => {
      iframe.addEventListener('focus', () => {
        let doc = iframe.contentWindow.document;
        const ifrInputs = doc.querySelectorAll('a[href], input:not([disabled]):not([type="hidden"]), select:not([disabled]), textarea:not([disabled]), button:not([disabled]):not([tabindex="-1"]), iframe, object, embed, *[tabindex]:not([tabindex="-1"])');
        if (ifrInputs.length === 0) return;
        let firstIfrInput = ifrInputs[0];
        if (firstIfrInput) firstIfrInput.focus();
      });
    });
  }
  /*
  Note:
  - hideModal will remove the modal element, so calling show modal multiple times won't attach multiple events (safe).
  */


  showModal(modal, overlayStay, cancelCallback, animated) {
    const dom = this.dom;
    dom.addClass(modal, 'active');
    modal.setAttribute('aria-hidden', false);

    const handleKeyDown = e => {
      if (e.keyCode === 27) {
        // escape key
        if (dom.hasClass(modal, 'active')) this.hideModal(modal); // check first in case programmatically ESC key that already call hideModal is triggered (see _hideModal)

        modal.removeEventListener('keydown', handleKeyDown);
        const btnClose = modal.querySelector('.is-modal-close');
        if (btnClose) btnClose.removeEventListener('click', close);
        if (cancelCallback) cancelCallback();
      }
    };

    modal.addEventListener('keydown', handleKeyDown);
    let content = modal.querySelector('.is-tab-content.active');

    if (content) {
      content.focus();
      this.builder.tabs.setupTabKey(content);
    } else {
      this.setupTabKeys(modal);
      modal.focus();
    }
    /* Disable Modal Animation */


    let animate = false;

    if (this.builder) {
      if (this.builder.opts.animateModal) {
        animate = true;

        if (!animated) {
          // if not set or false
          animate = false; // overide   
        }
      }
    } else {
      if (animated) {
        // if set true
        animate = true; // overide
      }
    }

    if (animate) {
      if (this.builder) {
        const buildercontainers = this.builder.doc.querySelectorAll(this.builder.opts.container);
        Array.prototype.forEach.call(buildercontainers, buildercontainer => {
          // buildercontainer.style.transform = 'scale(0.98)';
          // buildercontainer.style.WebkitTransform= 'scale(0.98)';
          // buildercontainer.style.MozTransform= 'scale(0.98)';
          buildercontainer.style.transform = `scale(${this.builder.opts.zoom - 0.02})`;
          buildercontainer.style.WebkitTransform = `scale(${this.builder.opts.zoom - 0.02})`;
          buildercontainer.style.MozTransform = `scale(${this.builder.opts.zoom - 0.02})`;
          buildercontainer.setAttribute('scaled-down', '1');
        });
      }
    }

    const ovl = modal.querySelector('.is-modal-overlay');
    if (ovl) ovl.parentNode.removeChild(ovl); // new

    if (!modal.querySelector('.is-modal-overlay')) {
      let html;

      if (overlayStay) {
        html = '<div class="is-modal-overlay overlay-stay"></div>';
      } else {
        html = '<div class="is-modal-overlay"></div>';
      }

      modal.insertAdjacentHTML('afterbegin', html);

      if (!overlayStay) {
        let overlay = modal.querySelector('.is-modal-overlay');
        dom.addEventListener(overlay, 'click', () => {
          this.hideModal(modal);
          modal.removeEventListener('keydown', handleKeyDown);
          const btnClose = modal.querySelector('.is-modal-close');
          if (btnClose) btnClose.removeEventListener('click', close);
          if (cancelCallback) cancelCallback();
        });
      }
    }

    const close = () => {
      if (dom.hasClass(modal, 'active')) this.hideModal(modal); // check first in case programmatically ESC key that already call hideModal is triggered (see _hideModal)

      modal.removeEventListener('keydown', handleKeyDown);
      const btnClose = modal.querySelector('.is-modal-close');
      if (btnClose) btnClose.removeEventListener('click', close);
      if (cancelCallback) cancelCallback();
    };

    const btnClose = modal.querySelector('.is-modal-close');
    if (btnClose) dom.addEventListener(btnClose, 'click', close);
  }

  _hideModal(modal) {
    let overlay = modal.querySelector('.is-modal-overlay');

    if (overlay) {
      if (overlay.classList.contains('overlay-stay')) {
        this.hideModal(modal); // just to make sure

        modal.dispatchEvent(new KeyboardEvent('keydown', {
          key: 'e',
          keyCode: 27,
          code: 'KeyE',
          which: 27,
          shiftKey: false,
          ctrlKey: false,
          metaKey: false
        })); // Programmatically call ESC so that cancelCallback can be triggered
      } else {
        overlay.click(); // so that cancelCallback can be triggered
      }
    } else {
      this.hideModal(modal); // just to make sure

      modal.dispatchEvent(new KeyboardEvent('keydown', {
        key: 'e',
        keyCode: 27,
        code: 'KeyE',
        which: 27,
        shiftKey: false,
        ctrlKey: false,
        metaKey: false
      })); // Programmatically call ESC so that cancelCallback can be triggered
    }
  }

  hideModal(modal) {
    modal.setAttribute('aria-hidden', true);
    /* Disable Modal Animation */

    if (this.builder) {
      const buildercontainers = this.builder.doc.querySelectorAll(this.builder.opts.container);
      Array.prototype.forEach.call(buildercontainers, buildercontainer => {
        // buildercontainer.style.transform = '';
        // buildercontainer.style.WebkitTransform= '';
        // buildercontainer.style.MozTransform= '';
        if (buildercontainer.getAttribute('scaled-down')) {
          buildercontainer.style.transform = `scale(${this.builder.opts.zoom})`;
          buildercontainer.style.WebkitTransform = `scale(${this.builder.opts.zoom})`;
          buildercontainer.style.MozTransform = `scale(${this.builder.opts.zoom})`;
          buildercontainer.removeAttribute('scaled-down');
        }
      });
    }

    const dom = this.dom;
    dom.removeClass(modal, 'active');
  } // attachInputTextClear(inp) {
  //     const dom = this.dom;
  //     inp.addEventListener('click', ()=>{
  //         dom.addClass(inp.parentNode, 'focus'); 
  //     });
  //     inp.addEventListener('blur', (e)=>{
  //         setTimeout(()=>{
  //             dom.removeClass(inp.parentNode, 'focus');
  //         },400);
  //     });
  //     const btnClear = inp.parentNode.querySelector('.input-clear');
  //     if(btnClear) btnClear.addEventListener('click', ()=> {
  //         inp.value = '';
  //         dom.removeClass(inp.parentNode, 'focus');
  //         inp.focus();
  //     });
  // }


  refreshModuleLayout(col) {
    let html = decodeURIComponent(col.getAttribute('data-html'));
    html = html.replace(/{id}/g, this.builder.util.makeId());
    col.innerHTML = '';
    let range = document.createRange();
    range.setStart(col, 0);
    col.appendChild(range.createContextualFragment(html));
    let subblocks = col.querySelectorAll('[data-subblock]');
    var i = 1;
    Array.prototype.forEach.call(subblocks, subblock => {
      if (col.getAttribute('data-html-' + i)) {
        subblock.innerHTML = decodeURIComponent(col.getAttribute('data-html-' + i));
        subblock.contentEditable = true;
      }

      i++;
    });
  }

  fixLayout(row) {
    const dom = this.dom; // if(this.builder.opts.enableDragResize) {
    //     Array.from(row.children).map((item) => {
    //         if(item.classList.contains('is-row-tool')) return;
    //         if(item.classList.contains('is-rowadd-tool')) return;
    //         // if(this.opts.filterSibling) if(item.classList.contains(this.opts.filterSibling)) return;
    //         item.style.width = '100%';
    //         item.style.flex = '';
    //     });
    //     return;
    // }

    let smHidden = false;
    let mdHidden = false;
    Array.from(row.children).map(item => {
      if (item.classList.contains('is-row-tool')) return;
      if (item.classList.contains('is-col-tool')) return;
      if (item.classList.contains('is-rowadd-tool')) return;
      item.style.width = '';
      item.style.flex = '';

      if (item.classList.contains('sm-hidden')) {
        smHidden = true;
      }

      if (item.classList.contains('md-hidden')) {
        mdHidden = true;
      }
    });

    if (smHidden) {
      row.classList.add('sm-autofit');
    } else {
      row.classList.remove('sm-autofit');
    }

    if (mdHidden) {
      row.classList.add('md-autofit');
    } else {
      row.classList.remove('md-autofit');
    }

    let num = 3; //is-row-tool, is-col-tool & is-rowadd-tool

    if (row.querySelector('.is-row-overlay')) {
      num = 4; //is-row-tool, is-col-tool, is-rowadd-tool & is-row-overlay
    }

    const cellCount = row.childElementCount - num;
    const rowClass = this.builder.opts.row;
    let colClass = this.builder.opts.cols;
    const colEqual = this.builder.opts.colequal;

    if (colEqual.length > 0) {
      const cols = dom.elementChildren(row);
      cols.forEach(col => {
        if (dom.hasClass(col, 'is-row-tool') || dom.hasClass(col, 'is-col-tool') || dom.hasClass(col, 'is-rowadd-tool') || dom.hasClass(col, 'is-row-overlay')) return;

        for (let i = 0; i <= colClass.length - 1; i++) {
          dom.removeClass(col, colClass[i]);
        }

        for (let i = 0; i <= colEqual.length - 1; i++) {
          if (colEqual[i].length === cellCount) {
            dom.addClass(col, colEqual[i][0]);
            break;
          }
        }

        if (cellCount === 1) {
          dom.addClass(col, colClass[colClass.length - 1]);
        }
      }); // Refresh Module

      Array.from(row.children).map(item => {
        if (item.classList.contains('is-row-tool')) return;
        if (item.classList.contains('is-rowadd-tool')) return;
        if (item.classList.contains('is-col-tool')) return;

        if (item.getAttribute('data-module')) {
          this.refreshModuleLayout(item);
        }
      });
      return;
    } //others (12 columns grid)


    if (rowClass !== '' && colClass.length > 0) {
      let n = 0;
      const cols = dom.elementChildren(row);
      cols.forEach(col => {
        if (dom.hasClass(col, 'is-row-tool') || dom.hasClass(col, 'is-col-tool') || dom.hasClass(col, 'is-rowadd-tool') || dom.hasClass(col, 'is-row-overlay')) return; // Bootstrap stuff

        if (col.className.indexOf('col-md-') !== -1) ; else if (col.className.indexOf('col-sm-') !== -1) {
          colClass = ['col-sm-1', 'col-sm-2', 'col-sm-3', 'col-sm-4', 'col-sm-5', 'col-sm-6', 'col-sm-7', 'col-sm-8', 'col-sm-9', 'col-sm-10', 'col-sm-11', 'col-sm-12'];
        } else if (col.className.indexOf('col-xs-') !== -1) {
          colClass = ['col-xs-1', 'col-xs-2', 'col-xs-3', 'col-xs-4', 'col-xs-5', 'col-xs-6', 'col-xs-7', 'col-xs-8', 'col-xs-9', 'col-xs-10', 'col-xs-11', 'col-xs-12'];
        } else if (col.className.indexOf('col-lg-') !== -1) {
          colClass = ['col-lg-1', 'col-lg-2', 'col-lg-3', 'col-lg-4', 'col-lg-5', 'col-lg-6', 'col-lg-7', 'col-lg-8', 'col-lg-9', 'col-lg-10', 'col-lg-11', 'col-lg-12'];
        } else if (col.className.indexOf('col-xl-') !== -1) {
          colClass = ['col-xl-1', 'col-xl-2', 'col-xl-3', 'col-xl-4', 'col-xl-5', 'col-xl-6', 'col-xl-7', 'col-xl-8', 'col-xl-9', 'col-xl-10', 'col-xl-11', 'col-xl-12'];
        } else if (col.className.indexOf('col-xxl-') !== -1) {
          colClass = ['col-xxl-1', 'col-xxl-2', 'col-xxl-3', 'col-xxl-4', 'col-xxl-5', 'col-xxl-6', 'col-xxl-7', 'col-xxl-8', 'col-xxl-9', 'col-xxl-10', 'col-xxl-11', 'col-xxl-12'];
        }

        n++;

        for (var i = 0; i <= colClass.length - 1; i++) {
          dom.removeClass(col, colClass[i]);
        }

        if (cellCount === 1) dom.addClass(col, colClass[11]);
        if (cellCount === 2) dom.addClass(col, colClass[5]);
        if (cellCount === 3) dom.addClass(col, colClass[3]);

        if (cellCount === 4) {
          dom.addClass(col, colClass[2]);
        }

        if (cellCount === 5) {
          // 2, 2, 2, 2, 4
          if (n === 5) dom.addClass(col, colClass[3]);else dom.addClass(col, colClass[1]);
        }

        if (cellCount === 6) dom.addClass(col, colClass[1]); // 2, 2, 2, 2, 2, 2

        if (cellCount === 7) {
          // 2, 2, 2, 2, 2, 1, 1
          if (n >= 6) dom.addClass(col, colClass[0]);else dom.addClass(col, colClass[1]);
        }

        if (cellCount === 8) {
          // 2, 2, 2, 2, 1, 1, 1, 1
          if (n >= 5) dom.addClass(col, colClass[0]);else dom.addClass(col, colClass[1]);
        }

        if (cellCount === 9) {
          // 2, 2, 2, 1, 1, 1, 1, 1, 1
          if (n >= 4) dom.addClass(col, colClass[0]);else dom.addClass(col, colClass[1]);
        }

        if (cellCount === 10) {
          // 2, 2, 1, 1, 1, 1, 1, 1, 1, 1
          if (n >= 3) dom.addClass(col, colClass[0]);else dom.addClass(col, colClass[1]);
        }

        if (cellCount === 11) {
          // 2, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1
          if (n >= 2) dom.addClass(col, colClass[0]);else dom.addClass(col, colClass[1]);
        }

        if (cellCount === 12) dom.addClass(col, colClass[0]); // 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1
      });
    } // Refresh Module


    Array.from(row.children).map(item => {
      if (item.classList.contains('is-row-tool')) return;
      if (item.classList.contains('is-rowadd-tool')) return;
      if (item.classList.contains('is-col-tool')) return;

      if (item.getAttribute('data-module')) {
        this.refreshModuleLayout(item);
      }
    });
  }

  addContent(html, mode, attr) {
    const dom = this.dom;

    if (this.builder.opts.onAdd) {
      html = this.builder.opts.onAdd(html);
    }

    const cell = this.cellSelected();
    let row;

    if (!cell) {
      // If no active cell, check if it is from .row-add-initial (empty info)
      row = this.builder.doc.querySelector('.row-active');
      if (!row) return;else {
        // Empty content will always use 'row' mode to insert block.
        mode = 'row';
      }
    } else {
      row = cell.parentNode;
    }

    if (mode === 'cell' || mode === 'cell-left' || mode === 'cell-right') {
      let maxCols = 4;

      if (this.builder.maxColumns) {
        maxCols = this.builder.maxColumns;
      } //Limit up to 4 cells in a row


      let num = 3; //is-row-tool, is-col-tool & is-rowadd-tool

      if (row.querySelector('.is-row-overlay')) {
        num = 4; //is-row-tool, is-col-tool, is-rowadd-tool & is-row-overlay
      }

      if (row.childElementCount >= maxCols + num) {
        //+3 => includes is-row-tool, is-col-tool, is-rowadd-tool & is-row-overlay
        alert(this.out('You have reached the maximum number of columns'));
        return false;
      }

      this.builder.uo.saveForUndo();
      let cellElement;

      if (this.builder.opts.row === '') {
        // TODO: Test using in old Insite
        let s = this.builder.opts.cellFormat;
        let pos = s.indexOf('</');
        html = s.substring(0, pos) + html + s.substring(pos);
        cellElement = this.createElementFromHTML(html);
      } else {
        cellElement = cell.cloneNode(true); // Cleanup from module related clone

        cellElement.removeAttribute('data-noedit');
        cellElement.removeAttribute('data-protected');
        cellElement.removeAttribute('data-module');
        cellElement.removeAttribute('data-module-desc');
        cellElement.removeAttribute('data-dialog-width');
        cellElement.removeAttribute('data-html');
        cellElement.removeAttribute('data-settings');

        for (let i = 1; i <= 20; i++) {
          cellElement.removeAttribute('data-html-' + i);
        }

        cellElement.removeAttribute('data-noedit');
        dom.removeClass(cellElement, 'cell-active');
        dom.removeClass(cellElement, 'is-light-text');
        dom.removeClass(cellElement, 'is-dark-text');
        dom.removeClass(cellElement, 'padding-0');
        dom.removeClass(cellElement, 'p-0');
        cellElement.removeAttribute('data-click');
        cellElement.removeAttribute('style');

        if (attr) {
          cellElement.setAttribute(attr, '');
        }

        cellElement.innerHTML = html;
      }

      row.insertBefore(cellElement, cell);

      if (mode === 'cell' || mode === 'cell-right') {
        dom.moveAfter(cellElement, cell);
      }

      let builderActive = this.builder.doc.querySelector('.builder-active');
      if (builderActive) this.builder.applyBehaviorOn(builderActive);
      this.fixLayout(row);
      cellElement.click(); //change active block to the newly created
    }

    if (mode === 'row') {
      this.builder.uo.saveForUndo();
      let rowElement, cellElement;

      if (this.builder.opts.row === '') {
        rowElement = this.htmlToElement(this.builder.opts.rowFormat);
        let s = this.builder.opts.cellFormat;
        let pos = s.indexOf('</');
        html = s.substring(0, pos) + html + s.substring(pos); // go to last deeper level

        let targetrow = dom.elementChildren(rowElement);

        while (targetrow.length > 0) {
          targetrow = targetrow[0];

          if (dom.elementChildren(targetrow).length > 0) {
            targetrow = dom.elementChildren(targetrow);
          } else {
            break;
          }
        }

        targetrow.innerHTML = html;
        cellElement = targetrow.firstChild;

        if (attr) {
          cellElement.setAttribute(attr, '');
        }
      } else {
        cellElement = dom.createElement('div');
        dom.addClass(cellElement, this.builder.opts.cols[this.builder.opts.cols.length - 1]);
        cellElement.innerHTML = html;

        if (attr) {
          cellElement.setAttribute(attr, '');
        }

        rowElement = dom.createElement('div');
        dom.addClass(rowElement, this.builder.opts.row);
        dom.appendChild(rowElement, cellElement);
      }

      row.parentNode.insertBefore(rowElement, row);
      dom.moveAfter(rowElement, row);
      let builderActive = this.builder.doc.querySelector('.builder-active');
      if (builderActive) this.builder.applyBehaviorOn(builderActive);
      cellElement.click(); //change active block to the newly created
    }

    if (mode === 'elm') {
      let elm = this.builder.activeElement; // See elementtool.js line 195-196. // document.querySelector('.elm-active');

      if (!elm) return;
      this.builder.uo.saveForUndo();
      let element = elm; // while(!dom.hasClass(element.parentNode, 'cell-active')) {
      //     element = element.parentNode;
      // }

      element.insertAdjacentHTML('afterend', html);
      let builderActive = this.builder.doc.querySelector('.builder-active');
      if (builderActive) this.builder.applyBehaviorOn(builderActive);
      let newelement = element.nextElementSibling;

      if (newelement.tagName.toLowerCase() === 'img') {
        let checkLoad = setInterval(() => {
          if (newelement.complete) {
            newelement.click();
            clearInterval(checkLoad);
          }
        }, 30);
      } else {
        newelement.click();
      }

      const builderStuff = this.builder.builderStuff;
      let quickadd = builderStuff.querySelector('.quickadd');
      this.hidePop(quickadd); // LATER: auto scroll
      // LATER: If image, then it needs time to load (resulting incorrect position), so hide element tool. 
    }

    if (this.builder.useCssClasses) {
      let builderActive = this.builder.doc.querySelector('.builder-active');
      if (builderActive) dom.contentReformat(builderActive, this.builder.cssClasses);
    } //Trigger Change event


    this.builder.opts.onChange(); //Trigger Render event

    this.builder.opts.onRender();
  } // https://stackoverflow.com/questions/494143/creating-a-new-dom-element-from-an-html-string-using-built-in-dom-methods-or-pro


  htmlToElement(html) {
    var template = document.createElement('template');
    html = html.trim(); // Never return a text node of whitespace as the result

    template.innerHTML = html;
    return template.content.firstChild;
  }

  addSnippet(html, bSnippet, noedit) {
    this.builder.uo.saveForUndo();
    const dom = this.dom;
    let rowElement;
    let bAddLast = false;
    let cell;
    let cellElement;
    const builderStuff = this.builder.builderStuff;
    let quickadd = builderStuff.querySelector('.quickadd');
    const mode = quickadd.getAttribute('data-mode');

    if (bSnippet && (mode === 'cell' || mode === 'cell-left' || mode === 'cell-right')) {
      if (noedit) {
        this.addContent(html, mode, 'data-noedit');
      } else {
        this.addContent(html, mode);
      }

      return;
    } else if (bSnippet && mode === 'row') {
      /*
      Buttons, line, social, video, map (Grid layout not included).
      Can be inserted after current row, cell, element, or last row.
      */
      // NEW: See contentbuilder.js line 328
      // OLD: See contentbuilder-jquery.js addSnippet() line 16529
      // Just snippet (without row/column grid), ex. buttons, line, social, video, map.
      // Can be inserted after current row, column (cell), element, or last row.
      // html = `<div class="${this.builder.opts.row}"><div class="${this.builder.opts.cols[this.builder.opts.cols.length-1]}"${(noedit? ' data-noedit': '')}>${html}</div></div>`;
      // OR like addContent() in util.js line 245)
      cellElement = document.createElement('div');
      cellElement.className = this.builder.opts.cols[this.builder.opts.cols.length - 1];
      cellElement.innerHTML = html;

      if (noedit) {
        cellElement.setAttribute('data-noedit', '');
      }

      rowElement = document.createElement('div');
      rowElement.className = this.builder.opts.row;
      rowElement.appendChild(cellElement); // Add after selected row

      cell = this.builder.cellSelected();
      let row;

      if (cell) {
        row = cell.parentNode;
      } else {
        // If no active cell, check if it is from .row-add-initial (empty info)
        row = this.builder.doc.querySelector('.row-active');

        if (!row) {
          bAddLast = true;
        }
      } // Add after last row


      if (bAddLast) {
        const nodes = this.builder.doc.querySelectorAll('.is-builder');
        const last = nodes[nodes.length - 1];
        const rows = dom.elementChildren(last);
        const lastrow = rows[rows.length - 1];
        row = lastrow;
      }

      row.parentNode.insertBefore(rowElement, row);
      dom.moveAfter(rowElement, row);
      let builderActive = this.builder.doc.querySelector('.builder-active');
      if (builderActive) this.builder.applyBehaviorOn(builderActive);
      cellElement.click(); //change active block to the newly created
      // Change to row selection

      rowElement.className = rowElement.className.replace('row-outline', ''); //Hide Column tool (new!)

      this.builder.util.hideColumnTool();
    } else if (bSnippet) {
      if (noedit) {
        this.addContent(html, mode, 'data-noedit');
      } else {
        this.addContent(html, mode);
      }

      return;
    } else {
      /*
      Complete with grid layout. Also may containes custom script(data-html)
      Can be inserted after current row or last row.
      */
      // NEW: See contentbuilder.js line 341 AND contentbuilder-jquery.js (addContentMore) line 11526
      // OLD: See contentbuilder-jquery.js (addContentMore) line 11526
      // Snippet is wrapped in row/colum (may contain custom code or has [data-html] attribute)
      // Can only be inserted after current row or last row (not on column or element).
      var snippet = document.createElement('div');
      snippet.innerHTML = html;
      var blocks = snippet.querySelectorAll('[data-html]');
      Array.prototype.forEach.call(blocks, block => {
        // Render custom code block
        html = decodeURIComponent(block.getAttribute('data-html'));
        html = html.replace(/{id}/g, this.makeId());

        for (var i = 1; i <= 20; i++) {
          html = html.replace('[%HTML' + i + '%]', block.getAttribute('data-html-' + i) === undefined ? '' : decodeURIComponent(block.getAttribute('data-html-' + i))); //render editable area
        }

        block.innerHTML = html;
      }); //html = snippet.innerHTML; 
      // NEW: This allows snippets with complete row & column format (should be single row/col) can be added as a column

      if (snippet.childNodes.length === 1) if (snippet.childNodes[0].childNodes.length === 1 && (mode === 'cell' || mode === 'cell-left' || mode === 'cell-right')) {
        // this.addContent(html, mode, 'data-noedit');
        const cell = this.cellSelected();
        let row = cell.parentNode;
        let maxCols = 4;

        if (this.builder.maxColumns) {
          maxCols = this.builder.maxColumns;
        } //Limit up to 4 cells in a row


        if (row.querySelector('.is-row-overlay')) {
          if (row.childElementCount >= maxCols + 4) {
            //+4 => includes is-row-tool, is-col-tool, is-rowadd-tool & is-row-overlay
            alert(this.out('You have reached the maximum number of columns'));
            return false;
          }
        } else {
          if (row.childElementCount >= maxCols + 3) {
            //+3 => includes is-row-tool, is-col-tool & is-rowadd-tool
            alert(this.out('You have reached the maximum number of columns'));
            return false;
          }
        }

        this.builder.uo.saveForUndo();
        let cellElement = snippet.childNodes[0].childNodes[0]; // row.insertBefore(cellElement, cell);

        var range = document.createRange();
        cell.parentNode.insertBefore(range.createContextualFragment(cellElement.outerHTML), cell);

        if (mode === 'cell' || mode === 'cell-right') {
          // dom.moveAfter(cellElement, cell);
          cell.parentNode.insertBefore(cell.previousElementSibling, cell);
          cell.parentNode.insertBefore(cell, cell.previousElementSibling);
        }

        let builderActive = this.builder.doc.querySelector('.builder-active');
        if (builderActive) this.builder.applyBehaviorOn(builderActive);else {
          builderActive = this.builder.doc.querySelector('.is-builder');
          this.builder.applyBehaviorOn(builderActive);
        }
        this.fixLayout(row); // cellElement.click(); //change active block to the newly created

        if (mode === 'cell' || mode === 'cell-right') {
          cell.nextElementSibling.click();
        } else {
          cell.previousElementSibling.click();
        } //Trigger Change event


        this.builder.opts.onChange(); //Trigger Render event

        this.builder.opts.onRender();
        return;
      } // Add after selected row

      cell = this.builder.activeCol;
      let row;

      if (cell) {
        row = cell.parentNode; // in email mode, cell active is also under row active (incorrect, but cell active is not needed in email mode. So this line works!)
      } else {
        // If no active cell, check if it is from .row-add-initial (empty info)
        row = this.builder.doc.querySelector('.row-active');

        if (!row) {
          bAddLast = true;
        }
      } // Add after last row


      if (bAddLast) {
        const nodes = this.builder.doc.querySelectorAll('.is-builder');
        const last = nodes[nodes.length - 1];
        const rows = dom.elementChildren(last);
        const lastrow = rows[rows.length - 1];
        row = lastrow;
      } // Use createContextualFragment() to make embedded script executable
      // https://ghinda.net/article/script-tags/


      range = document.createRange();
      row.parentNode.insertBefore(range.createContextualFragment(snippet.innerHTML), row.nextSibling);
      rowElement = snippet.childNodes[0]; // Auto scroll
      // const y = row.getBoundingClientRect().top +  row.offsetHeight + window.pageYOffset - 120;
      // window.scroll({
      //     top: y,
      //     behavior: 'smooth'
      // });
      // window.scrollTo(0, y);
      // setTimeout(()=>{
      //     row.scrollIntoView({ behavior: 'smooth', block: 'center' });
      // }, 600);

      rowElement = row.nextElementSibling; // a must. Must be before applyBehavior to prevent element delete during fixLayout
      // checkEmpty & onRender called here

      let builderActive = this.builder.doc.querySelector('.builder-active');
      if (builderActive) this.builder.applyBehaviorOn(builderActive);else {
        builderActive = this.builder.doc.querySelector('.is-builder');
        this.builder.applyBehaviorOn(builderActive);
      }
      cellElement = rowElement.querySelector('div');
      if (cellElement) cellElement.click(); //change active block to the newly created
      // Change to row selection

      rowElement.className = rowElement.className.replace('row-outline', ''); //Hide Column tool (new!)

      this.builder.util.hideColumnTool();
    }

    if (this.builder.useCssClasses) {
      let builderActive = this.builder.doc.querySelector('.builder-active');
      if (builderActive) dom.contentReformat(builderActive, this.builder.cssClasses);
    } //Trigger Change event


    this.builder.opts.onChange(); //Trigger Render event

    this.builder.opts.onRender();
  }

  clearActiveCell() {
    // this.builder.lastActiveCol = this.cellSelected(); // get active cell before cleared (will be used by snippets dialog)
    const builderStuff = this.builder.builderStuff;
    if (!builderStuff) return; // in case the builder is destroyed

    if (builderStuff.getAttribute('preventDevault')) {
      setTimeout(() => {
        builderStuff.removeAttribute('preventDevault');
      }, 30);
      return;
    }

    let divs = this.builder.doc.getElementsByClassName('cell-active');

    while (divs.length) divs[0].classList.remove('cell-active');

    divs = this.builder.doc.getElementsByClassName('row-outline');

    while (divs.length) divs[0].classList.remove('row-outline');

    divs = this.builder.doc.getElementsByClassName('row-active');

    while (divs.length) divs[0].classList.remove('row-active');

    divs = this.builder.doc.getElementsByClassName('builder-active');

    while (divs.length) divs[0].classList.remove('builder-active'); // let columnTool = builderStuff.querySelector('.is-column-tool');
    // columnTool.style.display = '';
    // dom.removeClass(columnTool,'active');
    // let elmTool = builderStuff.querySelector('.is-element-tool');
    // elmTool.style.display = '';
    // if(this.builder.iframe) {
    //     let columnTool = this.builder.contentStuff.querySelector('.is-column-tool');
    //     columnTool.style.display = '';
    //     dom.removeClass(columnTool,'active');
    //     let elmTool = this.builder.contentStuff.querySelector('.is-element-tool');
    //     elmTool.style.display = '';
    // }


    this.builder.activeCol = null;
    this.builder.dom.removeClass(this.builder.doc.body, 'content-edit');
  }

  clearAfterUndoRedo() {
    const dom = this.dom;
    const builderStuff = this.builder.builderStuff;
    let tools = builderStuff.querySelectorAll('.is-tool');
    Array.prototype.forEach.call(tools, tool => {
      tool.style.display = '';
      dom.removeClass(tool, 'active');
    });

    if (this.builder.iframe) {
      tools = this.builder.contentStuff.querySelectorAll('.is-tool');
      Array.prototype.forEach.call(tools, tool => {
        tool.style.display = '';
        dom.removeClass(tool, 'active');
      });
    }

    this.builder.moveable.updateRect();
    document.querySelector('.moveable-control-box').style.display = 'none';
    this.builder.activeSpacer = null;
    this.builder.activeCodeBlock = null;
    this.builder.activeLink = null;
    this.builder.activeLinkButton = null;
    this.builder.activeButton = null;
    this.builder.activeIframe = null;
    this.builder.activeTd = null;
    this.builder.activeTable = null;
    this.builder.activeModule = null;
    const icons = this.builder.doc.querySelectorAll('.icon-active');
    Array.prototype.forEach.call(icons, icon => {
      dom.removeClass(icon, 'icon-active');
    });
    this.builder.activeIcon = null; // RTE

    let rteTool = builderStuff.querySelector('.is-rte-tool'); // rteTool.style.display = 'none';

    let rteButtons = rteTool.querySelectorAll('button');
    Array.prototype.forEach.call(rteButtons, rteButton => {
      dom.removeClass(rteButton, 'on');
    });
    let elementRteTool = builderStuff.querySelector('.is-elementrte-tool'); // rteTool.style.display = 'none';

    rteButtons = elementRteTool.querySelectorAll('button');
    Array.prototype.forEach.call(rteButtons, rteButton => {
      dom.removeClass(rteButton, 'on');
    });
    let pops = builderStuff.querySelectorAll('.is-pop');
    Array.prototype.forEach.call(pops, pop => {
      pop.style.display = '';
    });
  }

  hidePops() {
    const dom = this.dom;
    const builderStuff = this.builder.builderStuff;
    let tools = builderStuff.querySelectorAll('.is-tool');
    Array.prototype.forEach.call(tools, tool => {
      // if(tool.classList.contains('is-row-tool')||
      // tool.classList.contains('is-column-tool')||
      // tool.classList.contains('is-element-tool')) return;
      tool.style.display = '';
      dom.removeClass(tool, 'active');
    });
    let pops = builderStuff.querySelectorAll('.is-pop.active');
    Array.prototype.forEach.call(pops, pop => {
      pop.style.transition = 'all 200ms ease';
      setTimeout(() => {
        pop.style.opacity = '0';
      }, 10);
      setTimeout(() => {
        pop.style.display = '';
        dom.removeClass(pop, 'active');
        pop.style.opacity = '';
        pop.style.transition = '';
      }, 200);
    });
    this.builder.moveable.updateRect();
    document.querySelector('.moveable-control-box').style.display = 'none';
  }

  hideControls() {
    const dom = this.dom;
    const builderStuff = this.builder.builderStuff;
    let tools = builderStuff.querySelectorAll('.is-tool');
    Array.prototype.forEach.call(tools, tool => {
      tool.style.display = '';
      dom.removeClass(tool, 'active');
    });

    if (this.builder.iframe) {
      tools = this.builder.contentStuff.querySelectorAll('.is-tool');
      Array.prototype.forEach.call(tools, tool => {
        tool.style.display = '';
        dom.removeClass(tool, 'active');
      });
    }

    this.builder.moveable.updateRect();
    document.querySelector('.moveable-control-box').style.display = 'none';
  }

  clearActiveElement(all) {
    const dom = this.dom;
    const builderStuff = this.builder.builderStuff;
    const icons = this.builder.doc.querySelectorAll('.icon-active');
    Array.prototype.forEach.call(icons, icon => {
      dom.removeClass(icon, 'icon-active');
    });
    let elms = this.builder.doc.querySelectorAll('.elm-inspected');
    Array.prototype.forEach.call(elms, elm => {
      dom.removeClass(elm, 'elm-inspected');
    });
    elms = this.builder.doc.querySelectorAll('.elm-active');
    Array.prototype.forEach.call(elms, elm => {
      dom.removeClass(elm, 'elm-active');
    });
    let elmTool = builderStuff.querySelector('.is-element-tool');
    elmTool.style.display = '';
    let linkTool = builderStuff.querySelector('#divLinkTool');
    linkTool.style.display = '';

    if (all) {
      this.builder.activeIcon = null;
      this.builder.inspectedElement = null;
      this.builder.activeElement = null; // RTE

      let rtetool = builderStuff.querySelector('.is-rte-tool');
      if (rtetool) rtetool.style.display = 'none';
      let elementRtetool = builderStuff.querySelector('.is-elementrte-tool');
      if (elementRtetool) elementRtetool.style.display = 'flex'; // Click ok on code view should hide these as well

      let rtetoolmore = builderStuff.querySelector('.rte-more-options');
      if (rtetoolmore) rtetoolmore.style.display = '';
      let elementRtetoolmore = builderStuff.querySelector('.elementrte-more-options');
      if (elementRtetoolmore) elementRtetoolmore.style.display = '';
      let btns = elementRtetool.querySelectorAll('button[data-align]');
      Array.prototype.forEach.call(btns, btn => {
        btn.style.display = 'none';
      });
      this.builder.rte.positionToolbar();
    }
  }

  clearControls() {
    const dom = this.dom;
    const builderStuff = this.builder.builderStuff;
    if (!builderStuff) return; // in case the builder is destroyed

    if (builderStuff.getAttribute('preventDevault')) {
      setTimeout(() => {
        builderStuff.removeAttribute('preventDevault');
      }, 30);
      return;
    }

    let tools = builderStuff.querySelectorAll('.is-tool');
    Array.prototype.forEach.call(tools, tool => {
      tool.style.display = '';
      dom.removeClass(tool, 'active');
    });

    if (this.builder.iframe) {
      tools = this.builder.contentStuff.querySelectorAll('.is-tool');
      Array.prototype.forEach.call(tools, tool => {
        tool.style.display = '';
        dom.removeClass(tool, 'active');
      });
    }

    this.builder.moveable.updateRect();
    document.querySelector('.moveable-control-box').style.display = 'none';
    this.builder.activeSpacer = null;
    this.builder.activeCodeBlock = null;
    this.builder.activeLink = null;
    this.builder.activeLinkButton = null;
    this.builder.activeButton = null;
    this.builder.activeIframe = null;
    this.builder.activeTd = null;
    this.builder.activeTable = null;
    this.builder.activeModule = null;
    this.builder.activeImage = null;
    const icons = this.builder.doc.querySelectorAll('.icon-active');
    Array.prototype.forEach.call(icons, icon => {
      dom.removeClass(icon, 'icon-active');
    });
    this.builder.activeIcon = null; // show iframe overlay to make it clickable

    let ovls = this.builder.doc.querySelectorAll('.ovl');
    Array.prototype.forEach.call(ovls, ovl => {
      ovl.style.display = 'block';
    }); // Element Panel & Snippets sidebar

    var panels = builderStuff.querySelectorAll('.is-side.elementstyles');
    Array.prototype.forEach.call(panels, panel => {
      dom.removeClass(panel, 'active');
    }); // Element Panel things

    let elms = this.builder.doc.querySelectorAll('[data-saveforundo]');
    Array.prototype.forEach.call(elms, elm => {
      elm.removeAttribute('data-saveforundo');
    });
    elms = this.builder.doc.querySelectorAll('.elm-inspected');
    Array.prototype.forEach.call(elms, elm => {
      dom.removeClass(elm, 'elm-inspected');
    }); // RTE

    if (this.builder.toolbarDisplay === 'auto') {
      let rtetool = builderStuff.querySelector('.is-rte-tool');
      if (rtetool) rtetool.style.display = 'none';
      let elementRtetool = builderStuff.querySelector('.is-elementrte-tool');
      if (elementRtetool) elementRtetool.style.display = 'none'; // Click ok on code view should hide these as well

      let rtetoolmore = builderStuff.querySelector('.rte-more-options');
      if (rtetoolmore) rtetoolmore.style.display = '';
      let elementRtetoolmore = builderStuff.querySelector('.elementrte-more-options');
      if (elementRtetoolmore) elementRtetoolmore.style.display = '';
    } // Element


    elms = this.builder.doc.querySelectorAll('.elm-active');
    Array.prototype.forEach.call(elms, elm => {
      dom.removeClass(elm, 'elm-active');
    });
    let rtepops = builderStuff.querySelectorAll('.is-rte-pop');
    Array.prototype.forEach.call(rtepops, rtepop => {
      rtepop.style.display = '';
      dom.removeClass(rtepop, 'active');
      dom.removeClass(rtepop, 'deactive'); // dom.addClass(rtepop, 'deactive');
    });
    let pops = builderStuff.querySelectorAll('.is-pop');
    Array.prototype.forEach.call(pops, pop => {
      pop.style.display = '';
      dom.removeClass(pop, 'active');
      pop.setAttribute('aria-hidden', true);
    });
    this.builder.colTool.lockIndicator.style.display = ''; // Clear resizable images
    // this.builder.element.image.clearImageResizer();
  } // source: http://stackoverflow.com/questions/1349404/generate-a-string-of-5-random-characters-in-javascript


  makeId() {
    let text = '';
    let possible = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';

    for (let i = 0; i < 2; i++) text += possible.charAt(Math.floor(Math.random() * possible.length));

    let text2 = '';
    let possible2 = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';

    for (let i = 0; i < 5; i++) text2 += possible2.charAt(Math.floor(Math.random() * possible2.length));

    return text + text2;
  }

  getSelection() {
    // more precise than saveSelection() => used for hyperlink
    const dom = this.dom;
    const selection = dom.getSelection();
    if (!selection) return;
    const anchorNode = selection.anchorNode;

    if (anchorNode) {
      const container = anchorNode.nodeType !== Node.TEXT_NODE && anchorNode.nodeType !== Node.COMMENT_NODE ? anchorNode : anchorNode.parentElement;
      const sameSelection = container && container.innerText === selection.toString();
      let multi = false;
      var selectedNodes = [];
      var sel = rangy.getSelection();

      for (var i = 0; i < sel.rangeCount; ++i) {
        selectedNodes = selectedNodes.concat(sel.getRangeAt(i).getNodes());
      }

      const blockElms = ['p', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'div', 'pre'];
      selectedNodes.forEach(item => {
        if (item.tagName) {
          const tagName = item.tagName.toLowerCase();

          if (blockElms.indexOf(tagName) !== -1) {
            multi = true;
          }
        }
      });

      if (multi) {
        this.builder.selectionMulti = true;
      } else {
        this.builder.selectionMulti = false;
      }

      if (sameSelection || selection.toString().trim() === '') {
        this.builder.selectionElm = container; // extra

        let nodeName = container.nodeName.toLowerCase();

        if (nodeName === 'a') {
          this.builder.activeLink = container;
        } else {
          this.builder.activeLink = null;
        }
      } else {
        this.builder.selectionText = selection; // extra

        this.builder.activeLink = null;
      }
    }
  } // source: http://stackoverflow.com/questions/5605401/insert-link-in-contenteditable-element 


  saveSelection() {
    if (this.builder.win.getSelection) {
      let sel = this.builder.win.getSelection();

      if (sel.getRangeAt && sel.rangeCount) {
        let ranges = [];

        for (let i = 0, len = sel.rangeCount; i < len; ++i) {
          ranges.push(sel.getRangeAt(i));
        }

        this.builder.selection = ranges;
        return ranges;
      }
    } else if (this.builder.doc.selection && this.builder.doc.selection.createRange) {
      this.builder.selection = this.builder.doc.selection.createRange();
      return this.builder.doc.selection.createRange();
    }

    this.builder.selection = null;
    return null;
  }

  restoreSelection() {
    let savedSel = this.builder.selection;

    if (savedSel) {
      if (this.builder.win.getSelection) {
        let sel = this.builder.win.getSelection(); // sel.removeAllRanges();

        if (this.builder.doc.body.createTextRange) {
          // All IE but Edge
          var range = this.builder.doc.body.createTextRange();
          range.collapse();
          range.select();
        } else if (this.builder.win.getSelection) {
          if (this.builder.win.getSelection().empty) {
            this.builder.win.getSelection().empty();
          } else if (this.builder.win.getSelection().removeAllRanges) {
            this.builder.win.getSelection().removeAllRanges();
          }
        } else if (this.builder.doc.selection) {
          this.builder.doc.selection.empty();
        }

        for (var i = 0, len = savedSel.length; i < len; ++i) {
          sel.addRange(savedSel[i]);
        }
      } else if (this.builder.doc.selection && savedSel.select) {
        savedSel.select();
      }
    }
  } // Clean Word. Source: 
  // http://patisserie.keensoftware.com/en/pages/remove-word-formatting-from-rich-text-editor-with-javascript
  // http://community.sitepoint.com/t/strip-unwanted-formatting-from-pasted-content/16848/3
  // http://www.1stclassmedia.co.uk/developers/clean-ms-word-formatting.php


  cleanHTML(input, cleanstyle) {
    let stringStripper = /(\n|\r| class=(")?Mso[a-zA-Z]+(")?)/g;
    let output = input.replace(stringStripper, ' ');
    let commentSripper = new RegExp('<!--(.*?)-->', 'g');
    output = output.replace(commentSripper, '');
    let tagStripper;

    if (cleanstyle) {
      tagStripper = new RegExp('<(/)*(meta|link|span|\\?xml:|st1:|o:|font)(.*?)>', 'gi');
    } else {
      tagStripper = new RegExp('<(/)*(meta|link|\\?xml:|st1:|o:|font)(.*?)>', 'gi');
    }

    output = output.replace(tagStripper, '');
    let badTags = ['style', 'script', 'applet', 'embed', 'noframes', 'noscript'];

    for (let i = 0; i < badTags.length; i++) {
      tagStripper = new RegExp('<' + badTags[i] + '.*?' + badTags[i] + '(.*?)>', 'gi');
      output = output.replace(tagStripper, '');
    }

    let badAttributes;

    if (cleanstyle) {
      badAttributes = ['style', 'start'];
    } else {
      badAttributes = ['start'];
    }

    for (let i = 0; i < badAttributes.length; i++) {
      let attributeStripper = new RegExp(' ' + badAttributes[i] + '="(.*?)"', 'gi');
      output = output.replace(attributeStripper, '');
    } // https://gist.github.com/sbrin/6801034
    //output = output.replace(/<!--[\s\S]+?-->/gi, ''); //done (see above)
    //output = output.replace(/<(!|script[^>]*>.*?<\/script(?=[>\s])|\/?(\?xml(:\w+)?|img|meta|link|style|\w:\w+)(?=[\s\/>]))[^>]*>/gi, '');


    output = output.replace(/<(!|script[^>]*>.*?<\/script(?=[>\s])|\/?(\?xml(:\w+)?|img|meta|link|style|\w:\w+)(?=[\s/>]))[^>]*>/gi, '');
    output = output.replace(/<(\/?)s>/gi, '<$1strike>');
    output = output.replace(/&nbsp;/gi, ' '); //output = output.replace(/<span\s+style\s*=\s*"\s*mso-spacerun\s*:\s*yes\s*;?\s*"\s*>([\s\u00a0]*)<\/span>/gi, function(str, spaces) {
    //    return (spaces.length > 0) ? spaces.replace(/./, " ").slice(Math.floor(spaces.length/2)).split("").join("\u00a0") : '';
    //});
    //clean copied elm-active background-color (LATER: improve)

    output = output.replace(/background-color: rgba\(200, 200, 201, 0.11\);/gi, '');
    output = output.replace(/background-color: rgba\(200, 200, 201, 0.11\)/gi, '');
    return output;
  }

  checkEmpty() {
    // Get all builder areas
    const builders = this.builder.doc.querySelectorAll(this.builder.opts.container);
    Array.prototype.forEach.call(builders, builder => {
      this.checkEmptyOn(builder);
    });
  }

  checkEmptyOn(builder) {
    const dom = this.dom;
    const rows = dom.elementChildren(builder);
    let empty = true;
    rows.forEach(row => {
      if (dom.hasClass(row, 'row-add-initial')) return;
      if (dom.hasClass(row, 'dummy-space')) return;
      empty = false;
    });

    if (empty) {
      let emptyinfo = builder.querySelector('.row-add-initial');

      if (!emptyinfo) {
        builder.innerHTML = `<button type="button" class="row-add-initial">${this.out('Empty')}<br><span class="block">${this.out('+ Click to add content')}</span></div>`;
        emptyinfo = builder.querySelector('.row-add-initial');
      }

      emptyinfo.addEventListener('click', () => {
        this.clearActiveCell();
        const builders = this.builder.doc.querySelectorAll(this.builder.opts.container);
        Array.prototype.forEach.call(builders, item => {
          dom.removeClass(item, 'builder-active');
        });
        dom.addClass(builder, 'builder-active');
        dom.addClass(emptyinfo, 'row-active'); // Needed for addContent(). Directly apply class in Util is fine.

        const builderStuff = this.builder.builderStuff;
        let quickadd = builderStuff.querySelector('.quickadd'); // see quickadd.js. Directly select by class in Util is fine.

        let tabs = quickadd.querySelector('.is-pop-tabs');
        tabs.style.display = 'none';
        const viewportHeight = window.innerHeight; // let top = emptyinfo.getBoundingClientRect().top;
        // const left = emptyinfo.getBoundingClientRect().left + (emptyinfo.offsetWidth * this.builder.opts.zoom)/2 - 11;

        let top, left;

        if (!this.builder.iframe) {
          top = emptyinfo.getBoundingClientRect().top;
          left = emptyinfo.getBoundingClientRect().left + emptyinfo.offsetWidth * this.builder.opts.zoom / 2 - 11;
        } else {
          let adjY = this.builder.iframe.getBoundingClientRect().top;
          let adjX = this.builder.iframe.getBoundingClientRect().left;
          top = emptyinfo.getBoundingClientRect().top + adjY;
          left = emptyinfo.getBoundingClientRect().left + (emptyinfo.offsetWidth * this.builder.opts.zoom / 2 - 11) + adjX;
        }

        quickadd.style.display = 'flex';
        this.showPop(quickadd, false, emptyinfo);
        const w = quickadd.offsetWidth; //to get value, element must not hidden (display:none). So set display:flex before this.

        const h = quickadd.offsetHeight;

        if (viewportHeight - top > h) {
          top = top + emptyinfo.offsetHeight * this.builder.opts.zoom - 19;
          quickadd.style.top = top + window.pageYOffset + 27 + 'px';
          quickadd.style.left = left - w / 2 + 7 + 'px';
          dom.removeClass(quickadd, 'arrow-bottom');
          dom.removeClass(quickadd, 'arrow-right');
          dom.removeClass(quickadd, 'arrow-left');
          dom.removeClass(quickadd, 'center');
          dom.addClass(quickadd, 'arrow-top');
          dom.addClass(quickadd, 'center');
        } else {
          quickadd.style.top = top + window.pageYOffset - h - 8 + 'px';
          quickadd.style.left = left - w / 2 + 7 + 'px';
          dom.removeClass(quickadd, 'arrow-top');
          dom.removeClass(quickadd, 'arrow-right');
          dom.removeClass(quickadd, 'arrow-left');
          dom.removeClass(quickadd, 'center');
          dom.addClass(quickadd, 'arrow-bottom');
          dom.addClass(quickadd, 'center');
        }

        quickadd.setAttribute('data-mode', 'row');
      });
    } else {
      let emptyinfo = builder.querySelector('.row-add-initial');
      if (emptyinfo) emptyinfo.parentNode.removeChild(emptyinfo);
    }
  }

  clearPreferences() {
    localStorage.removeItem('_theme'); //zoom

    localStorage.removeItem('_zoom'); //zoom

    localStorage.removeItem('_buildermode'); //builderMode

    localStorage.removeItem('_editingtoolbar'); //toolbar

    localStorage.removeItem('_editingtoolbardisplay'); //toolbarDisplay

    localStorage.removeItem('_hidecelltool'); //columnTool

    localStorage.removeItem('_rowtool'); //rowTool

    localStorage.removeItem('_hideelementtool'); //elementTool

    localStorage.removeItem('_hidesnippetaddtool'); //snippetAddTool

    localStorage.removeItem('_outlinemode'); //outlineMode

    localStorage.removeItem('_hiderowcoloutline'); //rowcolOutline

    localStorage.removeItem('_outlinestyle'); //outlineStyle

    localStorage.removeItem('_hideelementhighlight'); //elementHighlight

    localStorage.removeItem('_opensnippets'); //snippetOpen

    localStorage.removeItem('_toolstyle'); //toolStyle

    localStorage.removeItem('_snippetssidebardisplay'); //snippetsSidebarDisplay

    localStorage.removeItem('_htmlview');
    localStorage.removeItem('_pasteresult'); //DON'T HAVE PROP
    //NOT USED

    localStorage.removeItem('_scrollableeditor');
    localStorage.removeItem('_animatedsorting');
    localStorage.removeItem('_addbuttonplace');
    localStorage.removeItem('_hiderowtool');
    localStorage.removeItem('_dragwithouthandle');
    localStorage.removeItem('_advancedhtmleditor');
    localStorage.removeItem('_hidecolhtmleditor');
    localStorage.removeItem('_hiderowhtmleditor');
  } // source: http://stackoverflow.com/questions/6690752/insert-html-at-caret-in-a-contenteditable-div */


  pasteHtmlAtCaret(html, selectPastedContent) {
    this.restoreSelection();
    var sel, range;

    if (this.builder.win.getSelection) {
      if (!this.builder.activeCol) return;
      sel = this.builder.win.getSelection();

      if (sel.getRangeAt && sel.rangeCount) {
        range = sel.getRangeAt(0);
        range.deleteContents();
        var el = this.builder.doc.createElement('div');
        el.innerHTML = html;
        var frag = this.builder.doc.createDocumentFragment(),
            node,
            lastNode;

        while (node = el.firstChild) {
          lastNode = frag.appendChild(node);
        }

        var firstNode = frag.firstChild;
        range.insertNode(frag);

        if (lastNode) {
          range = range.cloneRange();
          range.setStartAfter(lastNode);

          if (selectPastedContent) {
            range.setStartBefore(firstNode);
          } else {
            range.collapse(true);
          }

          sel.removeAllRanges();
          if (!this.builder.isTouchSupport) sel.addRange(range);
        }
      }
    } else if ((sel = this.builder.doc.selection) && sel.type !== 'Control') {
      if (!this.builder.activeCol) return;
      var originalRange = sel.createRange();
      originalRange.collapse(true);
      sel.createRange().pasteHTML(html);

      if (selectPastedContent) {
        range = sel.createRange();
        range.setEndPoint('StartToStart', originalRange);
        if (!this.builder.isTouchSupport) range.select();
      }
    }
  }

  refreshModule() {
    let module = this.builder.activeModule;
    if (!module) return;
    let index = 1;
    let subblocks = module.querySelectorAll('[data-subblock]');
    Array.prototype.forEach.call(subblocks, subblock => {
      let builderhtml = subblock.innerHTML;
      module.setAttribute('data-html-' + index, encodeURIComponent(builderhtml));
      index++;
    });
    let html = decodeURIComponent(module.getAttribute('data-html'));
    html = html.replace(/{id}/g, this.makeId());
    module.innerHTML = '';
    var range = document.createRange();
    range.setStart(module, 0);
    module.appendChild(range.createContextualFragment(html));
    subblocks = module.querySelectorAll('[data-subblock]');
    var i = 1;
    Array.prototype.forEach.call(subblocks, subblock => {
      if (module.getAttribute('data-html-' + i)) {
        subblock.innerHTML = decodeURIComponent(module.getAttribute('data-html-' + i));
      }

      i++;
    });
    this.repositionColumnTool(true);
    if (this.builder.onRefreshTool) this.builder.onRefreshTool();
  } // Quick reposition column tool


  repositionColumnTool() {
    let col = this.builder.activeCol;
    if (!col) return;
    let row = col.parentNode;

    if (!col.parentNode) {
      // when column has just been deleted
      row = this.rowSelected();
    }

    this.colTool = row.querySelector('.is-col-tool');

    if (!col.parentNode) {
      // when column has just been deleted
      this.colTool.style.display = 'none';
    }

    this.colTool.style.left = col.offsetLeft + 'px'; // if columnMore is opened

    const columnMore = this.builder.builderStuff.querySelector('.columnmore.active');

    if (columnMore) {
      const btnCellMore = this.colTool.querySelector('.cell-more');
      columnMore.classList.add('transition1');
      setTimeout(() => {
        let top, left;

        if (!this.builder.iframe) {
          top = btnCellMore.getBoundingClientRect().top + window.pageYOffset;
          left = btnCellMore.getBoundingClientRect().left + window.pageXOffset;
        } else {
          let adjY = this.builder.iframe.getBoundingClientRect().top + window.pageYOffset;
          let adjX = this.builder.iframe.getBoundingClientRect().left + window.pageXOffset;
          top = btnCellMore.getBoundingClientRect().top + adjY;
          left = btnCellMore.getBoundingClientRect().left + adjX;
        }

        columnMore.style.top = top + 35 + 'px';
        columnMore.style.left = left - 7 + 'px';
        setTimeout(() => {
          columnMore.classList.remove('transition1');
        }, 300);
      }, 30);
    }
  }

  hideColumnTool() {
    let tools = this.builder.doc.querySelectorAll('.is-col-tool');
    tools.forEach(tool => {
      tool.style.display = 'none';
    });
  }

  isTouchSupport() {
    /*
    if(('ontouchstart' in window) || (navigator.MaxTouchPoints > 0) || (navigator.msMaxTouchPoints > 0)) {
        return true;
    }else {
        return false;
    }
    */
    // https://stackoverflow.com/questions/4817029/whats-the-best-way-to-detect-a-touch-screen-device-using-javascript
    if (window.matchMedia('(pointer: coarse)').matches) {
      return true;
    } else {
      return false;
    }
  } // https://stackoverflow.com/questions/31757852/how-can-i-detect-internet-explorer-ie-and-microsoft-edge-using-javascript


  detectIE() {
    /*
    var ua = window.navigator.userAgent;
    var msie = ua.indexOf('MSIE ');
    if (msie > 0) {
        return parseInt(ua.substring(msie + 5, ua.indexOf('.', msie)), 10);
    }
         var trident = ua.indexOf('Trident/');
    if (trident > 0) {
        var rv = ua.indexOf('rv:');
        return parseInt(ua.substring(rv + 3, ua.indexOf('.', rv)), 10);
    }
         var edge = ua.indexOf('Edge/');
    if (edge > 0) {
        return parseInt(ua.substring(edge + 5, ua.indexOf('.', edge)), 10);
    }
    
    return false;
    */
    // https://stackoverflow.com/questions/49986720/how-to-detect-internet-explorer-11-and-below-versions/49986758
    if (document.documentMode) {
      return true;
    }

    return false;
  } // Source: https://css-tricks.com/snippets/javascript/lighten-darken-color/


  LightenDarkenColor(col, amt) {
    var usePound = false;

    if (col[0] === '#') {
      col = col.slice(1);
      usePound = true;
    }

    var num = parseInt(col, 16);
    var r = (num >> 16) + amt;
    if (r > 255) r = 255;else if (r < 0) r = 0;
    var b = (num >> 8 & 0x00FF) + amt;
    if (b > 255) b = 255;else if (b < 0) b = 0;
    var g = (num & 0x0000FF) + amt;
    if (g > 255) g = 255;else if (g < 0) g = 0; //return (usePound ? '#' : '') + (g | (b << 8) | (r << 16)).toString(16);

    return (usePound ? '#' : '') + String('000000' + (g | b << 8 | r << 16).toString(16)).slice(-6);
  }

  getUIStyles() {
    const dom = this.dom;
    const html = `
        <input type="text" class="style-helper-input" style="display:none;">
        <label class="style-helper-label" style="display:none;"></label>
        <button class="style-helper-button-classic classic" style="display:none;"><svg><use xlink:href="#ion-code-working"></use></svg></button>
        <select class="style-helper-select" style="display:none;"><option value=""></option></select>

        <div class="style-helper modal-color"></div>
        <div class="style-helper modal-background"></div>
        <div class="style-helper button-pickcolor-border"></div>
        <div class="style-helper button-pickcolor-background"></div>

        <button class="style-helper base"><svg><use xlink:href="#ion-code-working"></use></svg></button>
        <div class="style-helper base on"></div>
        <div class="style-helper base hover"></div>

        <div class="style-helper snippet-color"></div>
        <div class="style-helper snippet-background"></div>
        <div class="style-helper snippet-tabs-background"></div>
        <div class="style-helper snippet-tab-item-background"></div>
        <div class="style-helper snippet-tab-item-background-active"></div>
        <div class="style-helper snippet-tab-item-background-hover"></div>
        <div class="style-helper snippet-tab-item-color"></div>

        <div class="style-helper snippet-more-item-background"></div>
        <div class="style-helper snippet-more-item-background-active"></div>
        <div class="style-helper snippet-more-item-background-hover"></div>
        <div class="style-helper snippet-more-item-color"></div>

        <div class="style-helper tabs-background"></div>
        <div class="style-helper tab-item-active-border-bottom"></div>
        <div class="style-helper tab-item-color"></div>
        <div class="style-helper tabs-more-background"></div>
        <div class="style-helper tabs-more-border"></div>
        <div class="style-helper tabs-more-item-color"></div>
        <div class="style-helper tabs-more-item-background-hover"></div>
        <div class="style-helper separator-color"></div>
        <div class="style-helper outline-color"></div>
        `;
    dom.appendHtml(this.builder.builderStuff, html); // new method

    const getVal = (selector, rule) => {
      const stuff = this.builder.builderStuff.querySelector('.style-helper' + selector);
      return window.getComputedStyle(stuff, null).getPropertyValue(rule);
    };

    const getSvgFill = () => {
      const btn = this.builder.builderStuff.querySelector('.style-helper.base');
      return window.getComputedStyle(btn.querySelector('svg'), null).getPropertyValue('fill');
    };

    const inp = this.builder.builderStuff.querySelector('.style-helper-input');
    const lbl = this.builder.builderStuff.querySelector('.style-helper-label');
    const sel = this.builder.builderStuff.querySelector('.style-helper-select');
    const btnClassic = this.builder.builderStuff.querySelector('.style-helper-button-classic');
    this.builder.styleModalColor = getVal('.modal-color', 'background-color');
    this.builder.styleModalBackground = getVal('.modal-background', 'background-color');
    this.builder.styleButtonPickColorBorder = getVal('.button-pickcolor-border', 'border');
    this.builder.styleButtonPickColorBackground = getVal('.button-pickcolor-background', 'background-color');
    this.builder.styleToolBackground = getVal('.base', 'background-color');
    this.builder.styleButtonColor = getVal('.base', 'color');
    this.builder.styleButtonSvgFill = getSvgFill(); // this.builder.styleButtonBackgroundOn = getVal('.base.on', 'background-color');

    this.builder.styleButtonBackgroundHover = getVal('.base.hover', 'background-color');
    this.builder.styleSnippetColor = getVal('.snippet-color', 'background-color');
    this.builder.styleSnippetBackground = getVal('.snippet-background', 'background-color');
    this.builder.styleSnippetTabsBackground = getVal('.snippet-tabs-background', 'background-color');
    this.builder.styleSnippetTabItemBackground = getVal('.snippet-tab-item-background', 'background-color');
    this.builder.styleSnippetTabItemBackgroundActive = getVal('.snippet-tab-item-background-active', 'background-color');
    this.builder.styleSnippetTabItemBackgroundHover = getVal('.snippet-tab-item-background-hover', 'background-color');
    this.builder.styleSnippetTabItemColor = getVal('.snippet-tab-item-color', 'background-color');
    this.builder.styleSnippetMoreItemBackground = getVal('.snippet-more-item-background', 'background-color');
    this.builder.styleSnippetMoreItemBackgroundActive = getVal('.snippet-more-item-background-active', 'background-color');
    this.builder.styleSnippetMoreItemBackgroundHover = getVal('.snippet-more-item-background-hover', 'background-color');
    this.builder.styleSnippetMoreItemColor = getVal('.snippet-more-item-color', 'background-color'); // Normal Tabs (ex. used in 'Symbol' plugin)

    this.builder.styleTabsBackground = getVal('.tabs-background', 'background-color');
    this.builder.styleTabItemBorderBottomActive = getVal('.tab-item-active-border-bottom', 'border');
    this.builder.styleTabItemColor = getVal('.tab-item-color', 'background-color');
    this.builder.styleTabsMoreBackground = getVal('.tabs-more-background', 'background-color');
    this.builder.styleTabsMoreBorder = getVal('.tabs-more-border', 'border');
    this.builder.styleTabsMoreItemColor = getVal('.tabs-more-item-color', 'background-color');
    this.builder.styleTabsMoreBackgroundHover = getVal('.tabs-more-item-background-hover', 'background-color');
    this.builder.styleSeparatorColor = getVal('.separator-color', 'background-color');
    this.builder.styleOutlineColor = getVal('.outline-color', 'background-color'); // Select (ex. used in 'Button Editor' plugin, 'Slider' plugin, 'Slider Content' plugin)

    this.builder.styleSelectBackground = window.getComputedStyle(sel, null).getPropertyValue('background-color');
    this.builder.styleSelectColor = window.getComputedStyle(sel, null).getPropertyValue('color');
    this.builder.styleSelectOptionBackground = window.getComputedStyle(sel.querySelector('option'), null).getPropertyValue('background-color'); // Input (ex. used in 'Search & Replace' plugin)

    this.builder.styleInputBackground = window.getComputedStyle(inp, null).getPropertyValue('background-color');
    this.builder.styleInputBorderBottom = window.getComputedStyle(inp, null).getPropertyValue('border-bottom');
    this.builder.styleInputColor = window.getComputedStyle(inp, null).getPropertyValue('color'); // Label (ex. used in 'Search & Replace' plugin)

    this.builder.styleLabelColor = window.getComputedStyle(lbl, null).getPropertyValue('color'); // Button Classic (ex. used in 'Search & Replace' plugin)

    this.builder.styleButtonClassicBackground = window.getComputedStyle(btnClassic, null).getPropertyValue('background-color');
    this.builder.styleButtonClassicColor = window.getComputedStyle(btnClassic, null).getPropertyValue('color');
    this.builder.styleButtonClassicBackgroundHover = this.getUIStyleValue(btnClassic, 'hover', 'background-color');
    this.builder.styleDark = false;
    this.builder.styleColored = false;
    this.builder.styleColoredDark = false;
    this.builder.styleLight = false;

    if (document.body.getAttribute('class')) {
      if (document.body.getAttribute('class').indexOf('colored-dark') !== -1) {
        this.builder.styleColoredDark = true;
      } else if (document.body.getAttribute('class').indexOf('dark') !== -1) {
        this.builder.styleDark = true;
      } else if (document.body.getAttribute('class').indexOf('colored') !== -1) {
        this.builder.styleColored = true;
      } else if (document.body.getAttribute('class').indexOf('light') !== -1) {
        this.builder.styleLight = true;
      }
    }
  }

  getUIStyles_OLD() {
    const dom = this.dom;
    const html = `<button class="style-helper"><svg><use xlink:href="#ion-code-working"></use></svg></button>
        <input type="text" class="style-helper-input" style="display:none;">
        <label class="style-helper-label" style="display:none;"></label>
        <input class="style-helper-checkbox" type="checkbox" style="display:none;">
        <button class="style-helper-button-classic classic" style="display:none;"><svg><use xlink:href="#ion-code-working"></use></svg></button>
        <select class="style-helper-select" style="display:none;"><option value=""></option></select>
        `;
    dom.appendHtml(this.builder.builderStuff, html); // Get some styles

    const btn = this.builder.builderStuff.querySelector('.style-helper');
    const inp = this.builder.builderStuff.querySelector('.style-helper-input');
    const lbl = this.builder.builderStuff.querySelector('.style-helper-label'); // const chk = this.builder.builderStuff.querySelector('.style-helper-checkbox');

    const sel = this.builder.builderStuff.querySelector('.style-helper-select');
    const btnClassic = this.builder.builderStuff.querySelector('.style-helper-button-classic');
    this.builder.styleModalColor = this.getUIStyleValue(btn, 'modal-color', 'background-color');
    this.builder.styleModalBackground = this.getUIStyleValue(btn, 'modal-background', 'background-color');
    this.builder.styleButtonPickColorBorder = this.getUIStyleValue(btn, 'button-pickcolor-border', 'border');
    this.builder.styleButtonPickColorBackground = this.getUIStyleValue(btn, 'button-pickcolor-background', 'background-color');
    this.builder.styleToolBackground = window.getComputedStyle(btn, null).getPropertyValue('background-color');
    this.builder.styleButtonColor = window.getComputedStyle(btn, null).getPropertyValue('color');
    this.builder.styleButtonSvgFill = window.getComputedStyle(btn.querySelector('svg'), null).getPropertyValue('fill'); // this.builder.styleButtonBackgroundOn = this.getUIStyleValue(btn, 'on', 'background-color');

    this.builder.styleButtonBackgroundHover = this.getUIStyleValue(btn, 'hover', 'background-color');
    this.builder.styleSnippetColor = this.getUIStyleValue(btn, 'snippet-color', 'background-color');
    this.builder.styleSnippetBackground = this.getUIStyleValue(btn, 'snippet-background', 'background-color');
    this.builder.styleSnippetTabsBackground = this.getUIStyleValue(btn, 'snippet-tabs-background', 'background-color');
    this.builder.styleSnippetTabItemBackground = this.getUIStyleValue(btn, 'snippet-tab-item-background', 'background-color');
    this.builder.styleSnippetTabItemBackgroundActive = this.getUIStyleValue(btn, 'snippet-tab-item-background-active', 'background-color');
    this.builder.styleSnippetTabItemBackgroundHover = this.getUIStyleValue(btn, 'snippet-tab-item-background-hover', 'background-color');
    this.builder.styleSnippetTabItemColor = this.getUIStyleValue(btn, 'snippet-tab-item-color', 'background-color');
    this.builder.styleSnippetMoreItemBackground = this.getUIStyleValue(btn, 'snippet-more-item-background', 'background-color');
    this.builder.styleSnippetMoreItemBackgroundActive = this.getUIStyleValue(btn, 'snippet-more-item-background-active', 'background-color');
    this.builder.styleSnippetMoreItemBackgroundHover = this.getUIStyleValue(btn, 'snippet-more-item-background-hover', 'background-color');
    this.builder.styleSnippetMoreItemColor = this.getUIStyleValue(btn, 'snippet-more-item-color', 'background-color'); // Normal Tabs (ex. used in 'Symbol' plugin)

    this.builder.styleTabsBackground = this.getUIStyleValue(btn, 'tabs-background', 'background-color');
    this.builder.styleTabItemBorderBottomActive = this.getUIStyleValue(btn, 'tab-item-active-border-bottom', 'border');
    this.builder.styleTabItemColor = this.getUIStyleValue(btn, 'tab-item-color', 'background-color');
    this.builder.styleTabsMoreBackground = this.getUIStyleValue(btn, 'tabs-more-background', 'background-color');
    this.builder.styleTabsMoreBorder = this.getUIStyleValue(btn, 'tabs-more-border', 'border');
    this.builder.styleTabsMoreItemColor = this.getUIStyleValue(btn, 'tabs-more-item-color', 'background-color');
    this.builder.styleTabsMoreBackgroundHover = this.getUIStyleValue(btn, 'tabs-more-item-background-hover', 'background-color');
    this.builder.styleSeparatorColor = this.getUIStyleValue(btn, 'separator-color', 'background-color'); // Preview (ex. used in 'Preview' plugin)
    // this.builder.styleModalPreviewColor = this.getUIStyleValue(btn, 'modal-preview-color', 'background-color');
    // this.builder.styleModalPreviewBackground = this.getUIStyleValue(btn, 'modal-preview-background', 'background-color');
    // this.builder.styleModalPreviewSizeControlBackground = this.getUIStyleValue(btn, 'modal-preview-sizecontrol-background', 'background-color');
    // this.builder.styleModalPreviewSizeControlBackgroundHover = this.getUIStyleValue(btn, 'modal-preview-sizecontrol-background-hover', 'background-color');
    // this.builder.styleModalPreviewSizeControlSeparatorColor = this.getUIStyleValue(btn, 'modal-preview-sizecontrol-separator-color', 'background-color');
    // UI/Modal (ex. used in 'Button Editor' plugin)
    // this.builder.styleTextColor = this.getUIStyleValue(btn, 'text-color', 'background-color');
    // Select (ex. used in 'Button Editor' plugin, 'Slider' plugin, 'Slider Content' plugin)

    this.builder.styleSelectBackground = window.getComputedStyle(sel, null).getPropertyValue('background-color');
    this.builder.styleSelectColor = window.getComputedStyle(sel, null).getPropertyValue('color');
    this.builder.styleSelectOptionBackground = window.getComputedStyle(sel.querySelector('option'), null).getPropertyValue('background-color'); // Input (ex. used in 'Search & Replace' plugin)

    this.builder.styleInputBackground = window.getComputedStyle(inp, null).getPropertyValue('background-color');
    this.builder.styleInputBorderBottom = window.getComputedStyle(inp, null).getPropertyValue('border-bottom');
    this.builder.styleInputColor = window.getComputedStyle(inp, null).getPropertyValue('color'); // Label (ex. used in 'Search & Replace' plugin)

    this.builder.styleLabelColor = window.getComputedStyle(lbl, null).getPropertyValue('color'); // Button Classic (ex. used in 'Search & Replace' plugin)

    this.builder.styleButtonClassicBackground = window.getComputedStyle(btnClassic, null).getPropertyValue('background-color');
    this.builder.styleButtonClassicColor = window.getComputedStyle(btnClassic, null).getPropertyValue('color');
    this.builder.styleButtonClassicBackgroundHover = this.getUIStyleValue(btnClassic, 'hover', 'background-color');
    this.builder.styleOutlineColor = this.getUIStyleValue(btn, 'outline-color', 'background-color');
    this.builder.styleDark = false;
    this.builder.styleColored = false;
    this.builder.styleColoredDark = false;
    this.builder.styleLight = false;

    if (document.body.getAttribute('class')) {
      if (document.body.getAttribute('class').indexOf('colored-dark') !== -1) {
        this.builder.styleColoredDark = true;
      } else if (document.body.getAttribute('class').indexOf('dark') !== -1) {
        this.builder.styleDark = true;
      } else if (document.body.getAttribute('class').indexOf('colored') !== -1) {
        this.builder.styleColored = true;
      } else if (document.body.getAttribute('class').indexOf('light') !== -1) {
        this.builder.styleLight = true;
      }
    }
  }

  getUIStyleValue(elm, classname, prop) {
    const dom = this.dom;
    dom.addClass(elm, classname);
    let val = window.getComputedStyle(elm, null).getPropertyValue(prop);
    dom.removeClass(elm, classname);
    return val;
  }

  getFontFamilyStyle(showInModal) {
    let css = `
        html, body {height:100%}
        body {overflow:hidden;margin:0;
            font-family: "Helvetica Neue", Helvetica, Arial, sans-serif;
            font-size:100%; 
            line-height:1.7;
        }
        #divFontList {margin:0;padding:0 0 9px 9px;height:100%;overflow-y:scroll !important;box-sizing:border-box;
            list-style: none;
            padding: 0;
            margin: 0;
        }
        #divFontList > li {width:100%;cursor:pointer;overflow:hidden;text-align:center;position:relative;
            display: flex;
            align-items: center;
            justify-content: center;
            text-overflow: ellipsis;
            white-space: nowrap;
            outline:none;
        }
        #divFontList > li img {margin:7px 5px 7px 5px;max-width: 230px;max-height: 27px;pointer-events: none;}
        #divFontList > li div {position:absolute;top:0;left:0;width:100%;height:100%;}

        #divFontList > li > div {
            position:absolute;left:0px;top:0px;width:100%;height:100%;
            // z-index: 1;
            pointer-events: none;}

        #divFontList > li {
            color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
            background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
        }
        #divFontList > li > div:after {
            // background: rgba(0, 0, 0, 0.04); 
            background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
            position: absolute;
            content: "";
            display: block;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            opacity: 0;
        }
        #divFontList > li:hover > div:after,
        #divFontList > li:focus > div:after {
            opacity: 1;
        }
        #divFontList > li.on > div:after {
            opacity: 1;
        }

        ${!this.builder.styleDark && !this.builder.styleColoredDark && !this.builder.styleColore ? `
        #divFontList > li img {
            mix-blend-mode: multiply;
        }
        ` : ''}

        .dark #divFontList > li img {
            ${showInModal ? '' : 'mix-blend-mode: screen;'};
            filter: invert(1);
        }
        .dark #divFontList > li {
            color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
            background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
        }
        .dark #divFontList > li > div:after {
            background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
        }
        
        .colored-dark #divFontList > li img {
            ${showInModal ? '' : 'mix-blend-mode: screen;'};
            ${showInModal ? '' : 'filter: invert(1);'};
        }
        .colored-dark #divFontList > li {
            color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
            background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
        }
        .colored-dark #divFontList > li > div:after {
            background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
        }


        .colored #divFontList > li img {
            ${showInModal ? '' : 'mix-blend-mode: screen;'};
            ${showInModal ? '' : 'filter: invert(1);'};
        }
        .colored #divFontList > li {
            color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
            background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
        }
        .colored #divFontList > li > div:after {
            background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
        }


        /* Scrollbar for toolbar/RTE and modal */

        .dark * {
            scrollbar-width: thin;
            scrollbar-color: rgba(255, 255, 255, 0.3) auto;
        }
        .dark *::-webkit-scrollbar {
            width: 12px;
        }
        .dark *::-webkit-scrollbar-track {
            background: transparent;
        }
        .dark *::-webkit-scrollbar-thumb {
            background-color:rgba(255, 255, 255, 0.3);
        } 

        .colored-dark * {
            scrollbar-width: thin;
            scrollbar-color: rgb(100, 100, 100) auto;
        }
        .colored-dark *::-webkit-scrollbar {
            width: 12px;
        }
        .colored-dark *::-webkit-scrollbar-track {
            background: transparent;
        }
        .colored-dark *::-webkit-scrollbar-thumb {
            background-color:rgb(100, 100, 100);
        } 

        .colored * {
            scrollbar-width: thin;
            scrollbar-color: rgba(0, 0, 0, 0.4) auto;
        }
        .colored *::-webkit-scrollbar {
            width: 12px;
        }
        .colored *::-webkit-scrollbar-track {
            background: transparent;
        }
        .colored *::-webkit-scrollbar-thumb {
            background-color: rgba(0, 0, 0, 0.4);
        } 

        .light * {
            scrollbar-width: thin;
            scrollbar-color: rgba(0, 0, 0, 0.4) auto;
        }
        .light *::-webkit-scrollbar {
            width: 12px;
        }
        .light *::-webkit-scrollbar-track {
            background: transparent;
        }
        .light *::-webkit-scrollbar-thumb {
            background-color: rgba(0, 0, 0, 0.4);
        } 
        `;
    return css;
  }

  refreshFontFamilyStyle1() {
    let iframeRte = this.builder.rte.rteFontFamilyOptions.querySelector('iframe');
    let doc1 = iframeRte.contentWindow.document;
    const divMainStyle1 = doc1.querySelector('#mainstyle');
    divMainStyle1.innerHTML = this.getFontFamilyStyle();

    if (this.builder.styleDark) {
      doc1.body.setAttribute('class', 'dark');
    } else if (this.builder.styleColored) {
      doc1.body.className = 'colored';
    } else if (this.builder.styleColoredDark) {
      doc1.body.className = 'colored-dark';
    } else if (this.builder.styleLight) {
      doc1.body.className = 'light';
    } else {
      doc1.body.className = '';
    }
  }

  refreshFontFamilyStyle2() {
    const fontModal = this.builder.builderStuff.querySelector('.is-modal.pickfontfamily');
    let iframePanel = fontModal.querySelector('iframe');
    let doc2 = iframePanel.contentWindow.document;
    const divMainStyle2 = doc2.querySelector('#mainstyle');
    divMainStyle2.innerHTML = this.getFontFamilyStyle(true);

    if (this.builder.styleDark) {
      doc2.body.setAttribute('class', 'dark');
    } else if (this.builder.styleColored) {
      doc2.body.className = 'colored';
    } else if (this.builder.styleColoredDark) {
      doc2.body.className = 'colored-dark';
    } else if (this.builder.styleLight) {
      doc2.body.className = 'light';
    } else {
      doc2.body.className = '';
    }
  }

  getFontPreview() {
    // let path = this.builder.scriptPath + 'fonts/';
    let path = this.builder.fontAssetPath;
    let html = '';
    /*
    const base64 = true;
    if(base64) {
        html = `
        <li role="option" tabindex="0" data-provider="" data-font-family="" style="font-size:12px;padding:10px 7px;box-sizing:border-box;"><div></div>
            <span style="z-index:1;position: relative;pointer-events: none;">${this.out('None')}</span>
        </div>
        <li role="option" tabindex="0" data-provider="" data-font-family="Arial, sans-serif"><div></div><img src="${font_arial}"></div>
        <li role="option" tabindex="0" data-provider="" data-font-family="courier"><div></div><img src="${font_courier}"></div>
        <li role="option" tabindex="0" data-provider="" data-font-family="Georgia, serif"><div></div><img src="${font_georgia}"></div>
        <li role="option" tabindex="0" data-provider="" data-font-family="monospace"><div></div><img src="${font_monospace}"></div>
        <li role="option" tabindex="0" data-provider="" data-font-family="sans-serif"><div></div><img src="${font_sans_serif}"></div>
        <li role="option" tabindex="0" data-provider="" data-font-family="serif"><div></div><img src="${font_serif}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Abel, sans-serif"><div></div><img src="${font_abel}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Abril Fatface"><div></div><img src="${font_abril_fatface}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Advent Pro, sans-serif" data-font-style="300"><div></div><img src="${font_advent_pro}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Aladin, cursive"><div></div><img src="${font_aladin}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Alegreya, serif" data-font-style="400,400i,500,500i,700,700i,800,800i,900,900i"><div></div><img src="${font_alegreya}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Alegreya Sans SC, sans-serif" data-font-style="300,700"><div></div><img src="${font_alegreya_sans_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Alegreya SC, serif" data-font-style="400,400i,500,500i,700,700i,800,800i,900,900i"><div></div><img src="${font_alegreya_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Alice, serif"><div></div><img src="${font_alice}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Allerta Stencil, sans-serif"><div></div><img src="${font_allerta_stencil}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Allura, cursive"><div></div><img src="${font_allura}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Almendra Display, cursive"><div></div><img src="${font_almendra_display}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Amatic SC, cursive" data-font-style="400,700"><div></div><img src="${font_amatic_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Andika, sans-serif"><div></div><img src="${font_andika}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Anonymous Pro, monospace" data-font-style="400,400i,700,700i"><div></div><img src="${font_anonymous_pro}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Architects Daughter, cursive"><div></div><img style="transform:scale(1.1);" src="${font_architects_daughter}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Arimo, sans-serif" data-font-style="400,400i,700,700i"><div></div><img src="${font_arimo}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Arsenal, sans-serif" data-font-style="400,400i,700,700i"><div></div><img src="${font_arsenal}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Assistant" data-font-style="300,700"><div></div><img src="${font_assistant}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Aubrey, cursive"><div></div><img src="${font_aubrey}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Anton, sans-serif"><div></div><img src="${font_anton}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Archivo Narrow, sans-serif"><div></div><img src="${font_archivo_narrow}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bad Script, cursive"><div></div><img src="${font_bad_script}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="BenchNine, sans-serif"><div></div><img src="${font_benchNine}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bevan, cursive"><div></div><img src="${font_bevan}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bigelow Rules, cursive"><div></div><img src="${font_bigelow_rules}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bilbo, cursive"><div></div><img src="${font_bilbo}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bonbon, cursive"><div></div><img src="${font_bonbon}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bowlby One SC, cursive"><div></div><img src="${font_bowlby_one_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cabin Condensed, sans-serif"><div></div><img src="${font_cabin_condensed}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Carrois Gothic SC, sans-serif"><div></div><img src="${font_carrois_gothic_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Caveat, cursive" data-font-style="400,700"><div></div><img src="${font_caveat}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Chewy, cursive"><div></div><img src="${font_chewy}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cinzel, serif"><div></div><img src="${font_cinzel}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Comfortaa, cursive" data-font-style="300"><div></div><img src="${font_comfortaa}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Concert One, cursive"><div></div><img src="${font_concert_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cormorant, serif" data-font-style="300,300i,600,600i,700,700i"><div></div><img src="${font_cormorant}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cormorant Garamond, serif" data-font-style="300,300i,600,600i,700,700i"><div></div><img src="${font_cormorant_garamond}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cormorant Infant, serif" data-font-style="300,300i,600,600i,700,700i"><div></div><img src="${font_cormorant_infant}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cormorant SC, serif" data-font-style="300,600,700"><div></div><img src="${font_cormorant_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cormorant Unicase, serif" data-font-style="300,600,700"><div></div><img src="${font_cormorant_unicase}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cousine" data-font-style="400,700"><div></div><img src="${font_cousine}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Crafty Girls, cursive"><div></div><img src="${font_crafty_girls}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cuprum, sans-serif" data-font-style="400,400i,700,700i"><div></div><img src="${font_cuprum}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cutive Mono, monospace"><div></div><img src="${font_cutive_mono}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Devonshire, cursive"><div></div><img src="${font_devonshire}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Didact Gothic, sans-serif"><div></div><img src="${font_didact_gothic}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Diplomata SC, cursive"><div></div><img src="${font_diplomata_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Dosis, sans-serif" data-font-style="200"><div></div><img src="${font_dosis}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="EB Garamond, serif" data-font-style="400,400i,600,600i,700,700i,800,800i"><div></div><img src="${font_eb_garamond}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="El Messiri, sans-serif" data-font-style="400,600,700"><div></div><img src="${font_el_messiri}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Elsie, cursive" data-font-style="400,900"><div></div><img src="${font_elsie}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Encode Sans, sans-serif" data-font-style="300,700"><div></div><img src="${font_encode_sans}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Exo, sans-serif" data-font-style="100"><div></div><img src="${font_exo}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="'Exo 2', sans-serif" data-font-style="200,200i,600,600i,700,700i,800,800i,900,900i" data-font-display="swap"><div></div><img src="${font_exo_2}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Felipa, cursive"><div></div><img src="${font_felipa}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fira Code, monospace" data-font-style="300,500,600,700"><div></div><img src="${font_fira_code}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fira Mono, monospace" data-font-style="400,500,700"><div></div><img src="${font_fira_mono}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fira Sans, sans-serif" data-font-style="200,200i,500,500i,700,700i,800,800i,900,900i"><div></div><img src="${font_fira_sans}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fira Sans Condensed, sans-serif" data-font-style="200,200i,500,500i,700,700i,800,800i,900,900i"><div></div><img src="${font_fira_sans_condensed}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fira Sans Extra Condensed, sans-serif" data-font-style="200,200i,500,500i,700,700i,800,800i,900,900i"><div></div><img src="${font_fira_sans_extra_condensed}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fjalla One, sans-serif"><div></div><img src="${font_fjalla_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Forum, cursive"><div></div><img src="${font_forum}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Frank Ruhl Libre" data-font-style="300,700"><div></div><img src="${font_frank_ruhl_libre}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fredericka the Great, cursive"><div></div><img src="${font_fredericka_the_great}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Gabriela, serif"><div></div><img src="${font_gabriela}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Gilda Display, serif"><div></div><img src="${font_gilda_display}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Give You Glory, cursive"><div></div><img style="transform:scale(1.3);" src="${font_give_you_glory}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Gruppo, cursive"><div></div><img src="${font_gruppo}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Handlee, cursive"><div></div><img src="${font_handlee}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Happy Monkey, cursive"><div></div><img src="${font_happy_monkey}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Hind" data-font-style="300,700"><div></div><img src="${font_hind}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="IBM Plex Mono, monospace" data-font-style="300,300i,500,500i,600,600i,700,700i"><div></div><img src="${font_ibm_plex_mono}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="IBM Plex Sans, sans-serif" data-font-style="300,300i,500,500i,600,600i,700,700i"><div></div><img src="${font_ibm_plex_sans}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="IBM Plex Serif, serif" data-font-style="300,300i,500,500i,600,600i,700,700i"><div></div><img src="${font_ibm_plex_serif}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Iceland, cursive"><div></div><img src="${font_iceland}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Inconsolata, monospace" data-font-style="400,700"><div></div><img src="${font_inconsolata}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Josefin Sans, sans-serif" data-font-style="300,700"><div></div><img src="${font_josefin_sans}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Istok Web, sans-serif" data-font-style="400,400i,700,700i"><div></div><img src="${font_istok_web}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Julee, cursive"><div></div><img src="${font_julee}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Julius Sans One, sans-serif"><div></div><img src="${font_julius_sans_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Junge, serif"><div></div><img src="${font_junge}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Jura, sans-serif" data-font-style="300,600,700"><div></div><img src="${font_jura}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Just Me Again Down Here, cursive"><div></div><img style="transform:scale(1.1);" src="${font_just_me_again_down_here}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kaushan Script, cursive"><div></div><img src="${font_kaushan_script}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kelly Slab, cursive"><div></div><img src="${font_kelly_slab}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kite One, sans-serif"><div></div><img src="${font_kite_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kosugi, sans-serif"><div></div><img src="${font_kosugi}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kosugi Maru, sans-serif"><div></div><img src="${font_kosugi_maru}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kurale, serif"><div></div><img src="${font_kurale}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lato, sans-serif" data-font-style="300,700"><div></div><img src="${font_lato}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ledger, serif" data-font-style="300,700" data-font-display="swap"><div></div><img src="${font_ledger}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lekton, sans-serif" data-font-style="400,700"><div></div><img src="${font_lekton}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Life Savers, cursive"><div></div><img src="${font_life_savers}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Literata, serif" data-font-style="400,400i,600,600i,700,700i"><div></div><img src="${font_literata}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lobster, cursive"><div></div><img src="${font_lobster}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lobster Two, cursive"><div></div><img src="${font_lobster_two}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Londrina Shadow, cursive"><div></div><img src="${font_londrina_shadow}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lora, serif" data-font-style="400,700"><div></div><img src="${font_lora}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lovers Quarrel, cursive"><div></div><img style="transform:scale(1.1);" src="${font_lovers_quarrel}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="'M PLUS 1p', sans-serif" data-font-style="300,500,700,800,900" data-font-display="swap"><div></div><img src="${font_m_plus_1p}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="'M PLUS Rounded 1c', sans-serif" data-font-style="300,500,700,800,900" data-font-display="swap"><div></div><img src="${font_m_plus_rounded_1c}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Macondo, cursive"><div></div><img src="${font_macondo}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Marcellus SC, serif"><div></div><img src="${font_marcellus_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Marck Script, cursive"><div></div><img src="${font_marck_script}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Martel, serif" data-font-style="300,700"><div></div><img src="${font_martel}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Maven Pro, sans-serif"><div></div><img src="${font_maven_pro}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Merriweather, serif" data-font-style="300,700"><div></div><img src="${font_merriweather}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Merriweather Sans" data-font-style="300,700"><div></div><img src="${font_merriweather_sans}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Mogra, cursive"><div></div><img src="${font_mogra}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Monoton, cursive"><div></div><img src="${font_monoton}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Montez, cursive"><div></div><img src="${font_montez}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Montserrat, sans-serif" data-font-style="300,400,700"><div></div><img src="${font_montserrat}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Montserrat Alternates, sans-serif" data-font-style="300,300i,500,500i,700,700i,800,800i,900,900i"><div></div><img src="${font_montserrat_alternates}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Montserrat Subrayada, sans-serif"><div></div><img src="${font_montserrat_subrayada}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Neucha, cursive"><div></div><img src="${font_neucha}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Neuton, serif" data-font-style="200,700"><div></div><img src="${font_neuton}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Nixie One, cursive"><div></div><img src="${font_nixie_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Nothing You Could Do, cursive"><div></div><img style="transform:scale(1.1);" src="${font_nothing_you_could_do}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Noto Sans, sans-serif" data-font-style="400,400i,700,700i"><div></div><img src="${font_noto_sans}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Noto Sans SC, sans-serif" data-font-style="300,500,700,900"><div></div><img src="${font_noto_sans_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Noto Serif, serif" data-font-style="400,400i,700,700i"><div></div><img src="${font_noto_serif}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Noto Serif TC, serif" data-font-style="300,600,700,900"><div></div><img src="${font_noto_serif_tc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Nunito, sans-serif" data-font-style="200,200i,600,600i,700,700i,800,800i,900,900i"><div></div><img src="${font_nunito}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Old Standard TT, serif" data-font-style="400,400i,700"><div></div><img src="${font_old_standard_tt}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Open Sans, sans-serif" data-font-style="300,400,600,800"><div></div><img src="${font_open_sans}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Oranienbaum, serif"><div></div><img src="${font_oranienbaum}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Oswald, sans-serif" data-font-style="300,400,700"><div></div><img src="${font_oswald}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Oxygen, sans-serif" data-font-style="300,700"><div></div><img src="${font_oxygen}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Pacifico, cursive"><div></div><img src="${font_pacifico}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Pangolin, cursive"><div></div><img src="${font_pangolin}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Passion One, cursive"><div></div><img src="${font_passion_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Pathway Gothic One, sans-serif"><div></div><img src="${font_pathway_gothic_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Pattaya, sans-serif"><div></div><img src="${font_pattaya}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Petit Formal Script, cursive"><div></div><img style="transform:scale(1.1);" src="${font_petit_formal_script}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Philosopher, sans-serif"><div></div><img src="${font_philosopher}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Play, sans-serif" data-font-style="400,700"><div></div><img src="${font_play}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Playfair Display, serif" data-font-style="400,400i,700,700i,900,900i"><div></div><img src="${font_playfair_display}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Playfair Display SC, serif" data-font-style="400,400i,700,700i,900,900i"><div></div><img src="${font_playfair_display_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Podkova, serif" data-font-style="400,600,700,800"><div></div><img src="${font_podkova}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Poiret One, cursive"><div></div><img src="${font_poiret_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Pompiere, cursive"><div></div><img style="transform:scale(1.1);" src="${font_pompiere}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Poppins, sans-serif" data-font-style="400,600"><div></div><img src="${font_poppins}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Prata, serif"><div></div><img src="${font_prata}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="'Press Start 2P', cursive" data-font-display="swap"><div></div><img src="${font_press_start_2p}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Prosto One, cursive"><div></div><img src="${font_prosto_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Mono, monospace"><div></div><img src="${font_pt_mono}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Sans, sans-serif" data-font-style="400,400i,700,700i"><div></div><img src="${font_pt_sans}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Sans Caption, sans-serif" data-font-style="400,700"><div></div><img src="${font_pt_sans_caption}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Sans Narrow, sans-serif" data-font-style="400,700"><div></div><img src="${font_pt_sans_narrow}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Serif, serif" data-font-style="400,700"><div></div><img src="${font_pt_serif}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Serif Caption, serif" data-font-style="400,700"><div></div><img src="${font_pt_serif_caption}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Quattrocento Sans, sans-serif"><div></div><img src="${font_quattrocento_sans}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Quattrocento, serif"><div></div><img src="${font_quattrocento}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Quicksand, sans-serif"><div></div><img src="${font_quicksand}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Qwigley, cursive"><div></div><img src="${font_qwigley}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Raleway, sans-serif" data-font-style="100"><div></div><img src="${font_raleway}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Raleway Dots, sans-serif"><div></div><img src="${font_raleway_dots}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Redressed, cursive"><div></div><img src="${font_redressed}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ribeye Marrow, cursive"><div></div><img src="${font_ribeye_marrow}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Righteous, cursive"><div></div><img src="${font_righteous}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Roboto, sans-serif" data-font-style="300"><div></div><img src="${font_roboto}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Roboto Condensed, sans-serif" data-font-style="300,300i,700,700i"><div></div><img src="${font_roboto_condensed}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Roboto Mono, monospace" data-font-style="300,700"><div></div><img src="${font_roboto_mono}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Roboto Slab, serif" data-font-style="200,600,700,800,900"><div></div><img src="${font_roboto_slab}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Rochester, cursive"><div></div><img src="${font_rochester}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Rouge Script, cursive"><div></div><img src="${font_rouge_script}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Rubik, sans-serif" data-font-style="300,300i,500,500i,700,700i,900,900i"><div></div><img src="${font_rubik}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Rubik Mono One, sans-serif"><div></div><img src="${font_rubik_mono_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ruslan Display, cursive"><div></div><img src="${font_ruslan_display}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Russo One, sans-serif"><div></div><img src="${font_russo_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Sacramento, cursive"><div></div><img src="${font_sacramento}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Sanchez, serif"><div></div><img src="${font_sanchez}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Satisfy, cursive"><div></div><img src="${font_satisfy}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Sawarabi Gothic, sans-serif"><div></div><img src="${font_sawarabi_gothic}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Scada, sans-serif" data-font-style="400,400i,700,700i"><div></div><img src="${font_scada}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Seaweed Script, cursive"><div></div><img src="${font_seaweed_script}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Seymour One, sans-serif"><div></div><img src="${font_seymour_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Shadows Into Light Two, cursive"><div></div><img src="${font_shadows_into_light_two}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Six Caps, sans-serif"><div></div><img src="${font_six_caps}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Snowburst One, cursive"><div></div><img src="${font_snowburst_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Source Code Pro, monospace" data-font-style="300,700"><div></div><img src="${font_source_code_pro}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Source Sans Pro, sans-serif" data-font-style="200"><div></div><img src="${font_source_sans_pro}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Special Elite, cursive"><div></div><img src="${font_special_elite}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Spectral, serif" data-font-style="200,200i,600,600i,700,700i,800,800i"><div></div><img src="${font_spectral}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Spectral SC, serif" data-font-style="300,300i,600,600i,700,700i,800,800i"><div></div><img src="${font_spectral_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Squada One, cursive"><div></div><img src="${font_squada_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Stalinist One, cursive"><div></div><img src="${font_stalinist_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Stint Ultra Expanded, cursive"><div></div><img src="${font_stint_ultra_expanded}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Syncopate, sans-serif"><div></div><img src="${font_syncopate}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Tangerine, cursive"><div></div><img src="${font_tangerine}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Tenor Sans, sans-serif"><div></div><img src="${font_tenor_sans}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Tinos, serif" data-font-style="400,400i,700,700i"><div></div><img src="${font_tinos}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ubuntu, sans-serif" data-font-style="300,300i,500,500i,700,700i"><div></div><img src="${font_ubuntu}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ubuntu Condensed, sans-serif"><div></div><img src="${font_ubuntu_condensed}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ubuntu Mono, monospace" data-font-style="400,700"><div></div><img src="${font_ubuntu_mono}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Underdog, cursive"><div></div><img src="${font_underdog}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="UnifrakturMaguntia, cursive"><div></div><img src="${font_unifrakturmaguntia}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Vast Shadow, cursive"><div></div><img src="${font_vast_shadow}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Viga, sans-serif"><div></div><img src="${font_viga}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Vollkorn, serif" data-font-style="400,400i,600,600i,700,700i,900,900i"><div></div><img src="${font_vollkorn}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Vollkorn SC, serif" data-font-style="400,600,700,900"><div></div><img src="${font_vollkorn_sc}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Voltaire, sans-serif"><div></div><img src="${font_voltaire}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Wire One, sans-serif"><div></div><img src="${font_wire_one}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Yanone Kaffeesatz, sans-serif" data-font-style="300,700"><div></div><img src="${font_yanone_kaffeesatz}"></div>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Yeseva One, cursive"><div></div><img src="${font_yeseva_one}"></div>
        `;
        return;
    } 
    */

    html = `
        <li role="option" tabindex="0" data-provider="" data-font-family="" style="font-size:12px;padding:10px 7px;box-sizing:border-box;"><div></div>
            <span style="z-index:1;position: relative;pointer-events: none;">${this.out('None')}</span>
        </li>
        <li role="option" tabindex="0" data-provider="" data-font-family="Arial, sans-serif"><div></div><img src="${path}arial.png"></li>
        <li role="option" tabindex="0" data-provider="" data-font-family="courier"><div></div><img src="${path}courier.png"></li>
        <li role="option" tabindex="0" data-provider="" data-font-family="Georgia, serif"><div></div><img src="${path}georgia.png"></li>
        <li role="option" tabindex="0" data-provider="" data-font-family="monospace"><div></div><img src="${path}monospace.png"></li>
        <li role="option" tabindex="0" data-provider="" data-font-family="sans-serif"><div></div><img src="${path}sans_serif.png"></li>
        <li role="option" tabindex="0" data-provider="" data-font-family="serif"><div></div><img src="${path}serif.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Abel, sans-serif"><div></div><img src="${path}abel.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Abril Fatface"><div></div><img src="${path}abril_fatface.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Advent Pro, sans-serif" data-font-style="wght@100;200;300;400;500;600;700"><div></div><img src="${path}advent_pro.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Aladin, cursive"><div></div><img src="${path}aladin.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Alegreya, serif" data-font-style="ital,wght@0,400;0,500;0,600;0,700;0,800;0,900;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}alegreya.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Alegreya Sans SC, sans-serif" data-font-style="ital,wght@0,100;0,300;0,400;0,500;0,700;0,800;0,900;1,100;1,300;1,400;1,500;1,700;1,800;1,900"><div></div><img src="${path}alegreya_sans_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Alegreya SC, serif" data-font-style="ital,wght@0,400;0,500;0,700;0,800;0,900;1,400;1,500;1,700;1,800;1,900"><div></div><img src="${path}alegreya_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Alice, serif"><div></div><img src="${path}alice.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Allerta Stencil, sans-serif"><div></div><img src="${path}allerta_stencil.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Allura, cursive"><div></div><img src="${path}allura.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Almendra Display, cursive"><div></div><img src="${path}almendra_display.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Amatic SC, cursive" data-font-style="wght@400;700"><div></div><img src="${path}amatic_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Andika, sans-serif" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}andika.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Anonymous Pro, monospace" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}anonymous_pro.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Architects Daughter, cursive"><div></div><img style="transform:scale(1.1);" src="${path}architects_daughter.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Arimo, sans-serif" data-font-style="ital,wght@0,400;0,500;0,600;0,700;1,400;1,500;1,600;1,700"><div></div><img src="${path}arimo.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Arsenal, sans-serif" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}arsenal.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Assistant" data-font-style="wght@200;300;400;500;600;700;800"><div></div><img src="${path}assistant.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Aubrey, cursive"><div></div><img src="${path}aubrey.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Anton, sans-serif"><div></div><img src="${path}anton.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Archivo Narrow, sans-serif" data-font-style="ital,wght@0,400;0,500;0,600;0,700;1,400;1,500;1,600;1,700"><div></div><img src="${path}archivo_narrow.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bad Script, cursive"><div></div><img src="${path}bad_script.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="BenchNine, sans-serif" data-font-style="wght@300;400;700"><div></div><img src="${path}benchNine.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bevan, cursive"><div></div><img src="${path}bevan.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bigelow Rules, cursive"><div></div><img src="${path}bigelow_rules.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bilbo, cursive"><div></div><img src="${path}bilbo.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bonbon, cursive"><div></div><img src="${path}bonbon.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Bowlby One SC, cursive"><div></div><img src="${path}bowlby_one_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cabin Condensed, sans-serif" data-font-style="wght@400;500;600;700"><div></div><img src="${path}cabin_condensed.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Carrois Gothic SC, sans-serif"><div></div><img src="${path}carrois_gothic_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Caveat, cursive" data-font-style="wght@400;500;600;700"><div></div><img src="${path}caveat.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Chewy, cursive"><div></div><img src="${path}chewy.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cinzel, serif" data-font-style="wght@400;500;600;700;800;900"><div></div><img src="${path}cinzel.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Comfortaa, cursive" data-font-style="wght@300;400;500;600;700"><div></div><img src="${path}comfortaa.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Concert One, cursive"><div></div><img src="${path}concert_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cormorant, serif" data-font-style="ital,wght@0,300;0,400;0,500;0,600;0,700;1,300;1,400;1,500;1,600;1,700"><div></div><img src="${path}cormorant.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cormorant Garamond, serif" data-font-style="ital,wght@0,300;0,400;0,500;0,600;0,700;1,300;1,400;1,500;1,600;1,700"><div></div><img src="${path}cormorant_garamond.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cormorant Infant, serif" data-font-style="ital,wght@0,300;0,400;0,500;0,600;0,700;1,300;1,400;1,500;1,600;1,700"><div></div><img src="${path}cormorant_infant.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cormorant SC, serif" data-font-style="wght@300;400;500;600;700"><div></div><img src="${path}cormorant_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cormorant Unicase, serif" data-font-style="wght@300;400;500;600;700"><div></div><img src="${path}cormorant_unicase.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cousine" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}cousine.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Crafty Girls, cursive"><div></div><img src="${path}crafty_girls.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cuprum, sans-serif" data-font-style="ital,wght@0,400;0,500;0,600;0,700;1,400;1,500;1,600;1,700"><div></div><img src="${path}cuprum.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Cutive Mono, monospace"><div></div><img src="${path}cutive_mono.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Devonshire, cursive"><div></div><img src="${path}devonshire.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Didact Gothic, sans-serif"><div></div><img src="${path}didact_gothic.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Diplomata SC, cursive"><div></div><img src="${path}diplomata_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Dosis, sans-serif" data-font-style="wght@200;300;400;500;600;700;800"><div></div><img src="${path}dosis.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="EB Garamond, serif" data-font-style="ital,wght@0,400;0,500;0,600;0,700;0,800;1,400;1,500;1,600;1,700;1,800"><div></div><img src="${path}eb_garamond.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="El Messiri, sans-serif" data-font-style="wght@400;500;600;700"><div></div><img src="${path}el_messiri.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Elsie, cursive" data-font-style="wght@400;900"><div></div><img src="${path}elsie.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Encode Sans, sans-serif" data-font-style="wght@100;200;300;400;500;600;700;800;900"><div></div><img src="${path}encode_sans.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Exo, sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}exo.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="'Exo 2', sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900" data-font-display="swap"><div></div><img src="${path}exo_2.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Felipa, cursive"><div></div><img src="${path}felipa.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fira Code, monospace" data-font-style="wght@300;400;500;600;700"><div></div><img src="${path}fira_code.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fira Mono, monospace" data-font-style="wght@400;500;700"><div></div><img src="${path}fira_mono.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fira Sans, sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}fira_sans.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fira Sans Condensed, sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}fira_sans_condensed.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fira Sans Extra Condensed, sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}fira_sans_extra_condensed.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fjalla One, sans-serif"><div></div><img src="${path}fjalla_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Forum, cursive"><div></div><img src="${path}forum.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Frank Ruhl Libre" data-font-style="wght@300;400;500;700;900"><div></div><img src="${path}frank_ruhl_libre.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Fredericka the Great, cursive"><div></div><img src="${path}fredericka_the_great.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Gabriela, serif"><div></div><img src="${path}gabriela.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Gilda Display, serif"><div></div><img src="${path}gilda_display.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Give You Glory, cursive"><div></div><img style="transform:scale(1.3);" src="${path}give_you_glory.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Gruppo, cursive"><div></div><img src="${path}gruppo.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Handlee, cursive"><div></div><img src="${path}handlee.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Happy Monkey, cursive"><div></div><img src="${path}happy_monkey.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Hind" data-font-style="wght@300;400;500;600;700"><div></div><img src="${path}hind.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="IBM Plex Mono, monospace" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;1,100;1,200;1,300;1,400;1,500;1,600;1,700"><div></div><img src="${path}ibm_plex_mono.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="IBM Plex Sans, sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;1,100;1,200;1,300;1,400;1,500;1,600;1,700"><div></div><img src="${path}ibm_plex_sans.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="IBM Plex Serif, serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;1,100;1,200;1,300;1,400;1,500;1,600;1,700"><div></div><img src="${path}ibm_plex_serif.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Iceland, cursive"><div></div><img src="${path}iceland.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Inconsolata, monospace" data-font-style="wght@200;300;400;500;600;700;800;900"><div></div><img src="${path}inconsolata.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Josefin Sans, sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;1,100;1,200;1,300;1,400;1,500;1,600;1,700"><div></div><img src="${path}josefin_sans.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Istok Web, sans-serif" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}istok_web.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Julee, cursive"><div></div><img src="${path}julee.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Julius Sans One, sans-serif"><div></div><img src="${path}julius_sans_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Junge, serif"><div></div><img src="${path}junge.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Jura, sans-serif" data-font-style="wght@300;400;500;600;700"><div></div><img src="${path}jura.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Just Me Again Down Here, cursive"><div></div><img style="transform:scale(1.1);" src="${path}just_me_again_down_here.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kaushan Script, cursive"><div></div><img src="${path}kaushan_script.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kelly Slab, cursive"><div></div><img src="${path}kelly_slab.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kite One, sans-serif"><div></div><img src="${path}kite_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kosugi, sans-serif"><div></div><img src="${path}kosugi.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kosugi Maru, sans-serif"><div></div><img src="${path}kosugi_maru.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Kurale, serif"><div></div><img src="${path}kurale.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lato, sans-serif" data-font-style="ital,wght@0,100;0,300;0,400;0,700;0,900;1,100;1,300;1,400;1,700;1,900"><div></div><img src="${path}lato.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ledger, serif" data-font-display="swap"><div></div><img src="${path}ledger.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lekton, sans-serif" data-font-style="ital,wght@0,400;0,700;1,400"><div></div><img src="${path}lekton.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Life Savers, cursive" data-font-style="wght@400;700;800"><div></div><img src="${path}life_savers.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Literata, serif" data-font-style="ital,wght@0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}literata.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lobster, cursive"><div></div><img src="${path}lobster.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lobster Two, cursive" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}lobster_two.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Londrina Shadow, cursive"><div></div><img src="${path}londrina_shadow.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lora, serif" data-font-style="ital,wght@0,400;0,500;0,600;0,700;1,400;1,500;1,600;1,700"><div></div><img src="${path}lora.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Lovers Quarrel, cursive"><div></div><img style="transform:scale(1.1);" src="${path}lovers_quarrel.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="'M PLUS 1p', sans-serif" data-font-style="wght@100;300;400;500;700;800;900" data-font-display="swap"><div></div><img src="${path}m_plus_1p.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="'M PLUS Rounded 1c', sans-serif" data-font-style="wght@100;300;400;500;700;800;900" data-font-display="swap"><div></div><img src="${path}m_plus_rounded_1c.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Macondo, cursive"><div></div><img src="${path}macondo.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Marcellus SC, serif"><div></div><img src="${path}marcellus_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Marck Script, cursive"><div></div><img src="${path}marck_script.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Martel, serif" data-font-style="wght@200;300;400;600;700;800;900"><div></div><img src="${path}martel.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Maven Pro, sans-serif"><div></div><img src="${path}maven_pro.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Merriweather, serif" data-font-style="ital,wght@0,300;0,400;0,700;0,900;1,300;1,400;1,700;1,900"><div></div><img src="${path}merriweather.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Merriweather Sans" data-font-style="ital,wght@0,300;0,400;0,500;0,600;0,700;0,800;1,300;1,400;1,500;1,600;1,700;1,800"><div></div><img src="${path}merriweather_sans.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Mogra, cursive"><div></div><img src="${path}mogra.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Monoton, cursive"><div></div><img src="${path}monoton.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Montez, cursive"><div></div><img src="${path}montez.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Montserrat, sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}montserrat.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Montserrat Alternates, sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}montserrat_alternates.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Montserrat Subrayada, sans-serif" data-font-style="wght@400;700"><div></div><img src="${path}montserrat_subrayada.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Neucha, cursive"><div></div><img src="${path}neucha.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Neuton, serif" data-font-style="ital,wght@0,200;0,300;0,400;0,700;0,800;1,400"><div></div><img src="${path}neuton.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Nixie One, cursive"><div></div><img src="${path}nixie_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Nothing You Could Do, cursive"><div></div><img style="transform:scale(1.1);" src="${path}nothing_you_could_do.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Noto Sans, sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}noto_sans.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Noto Sans SC, sans-serif" data-font-style="wght@100;300;400;500;700;900"><div></div><img src="${path}noto_sans_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Noto Serif, serif" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}noto_serif.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Noto Serif TC, serif" data-font-style="wght@200;300;400;500;600;700;900"><div></div><img src="${path}noto_serif_tc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Nunito, sans-serif" data-font-style="ital,wght@0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;0,1000;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900;1,1000"><div></div><img src="${path}nunito.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Old Standard TT, serif" data-font-style="ital,wght@0,400;0,700;1,400"><div></div><img src="${path}old_standard_tt.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Open Sans, sans-serif" data-font-style="ital,wght@0,300;0,400;0,500;0,600;0,700;0,800;1,300;1,400;1,500;1,600;1,700;1,800"><div></div><img src="${path}open_sans.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Oranienbaum, serif"><div></div><img src="${path}oranienbaum.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Oswald, sans-serif" data-font-style="wght@200;300;400;500;600;700"><div></div><img src="${path}oswald.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Oxygen, sans-serif" data-font-style="wght@300;400;700"><div></div><img src="${path}oxygen.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Pacifico, cursive"><div></div><img src="${path}pacifico.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Pangolin, cursive"><div></div><img src="${path}pangolin.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Passion One, cursive" data-font-style="wght@400;700;900"><div></div><img src="${path}passion_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Pathway Gothic One, sans-serif"><div></div><img src="${path}pathway_gothic_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Pattaya, sans-serif"><div></div><img src="${path}pattaya.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Petit Formal Script, cursive"><div></div><img style="transform:scale(1.1);" src="${path}petit_formal_script.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Philosopher, sans-serif" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}philosopher.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Play, sans-serif" data-font-style="wght@400;700"><div></div><img src="${path}play.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Playfair Display, serif" data-font-style="ital,wght@0,400;0,500;0,600;0,700;0,800;0,900;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}playfair_display.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Playfair Display SC, serif" data-font-style="ital,wght@0,400;0,700;0,900;1,400;1,700;1,900"><div></div><img src="${path}playfair_display_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Podkova, serif" data-font-style="wght@400;500;600;700;800"><div></div><img src="${path}podkova.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Poiret One, cursive"><div></div><img src="${path}poiret_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Pompiere, cursive"><div></div><img style="transform:scale(1.1);" src="${path}pompiere.png"></li>
        `;
    html += `<li role="option" tabindex="0" data-provider="google" data-font-family="Poppins, sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}poppins.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Prata, serif"><div></div><img src="${path}prata.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="'Press Start 2P', cursive"><div></div><img src="${path}press_start_2p.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Prosto One, cursive"><div></div><img src="${path}prosto_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Mono, monospace"><div></div><img src="${path}pt_mono.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Sans, sans-serif" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}pt_sans.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Sans Caption, sans-serif" data-font-style="wght@400;700"><div></div><img src="${path}pt_sans_caption.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Sans Narrow, sans-serif" data-font-style="wght@400;700"><div></div><img src="${path}pt_sans_narrow.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Serif, serif" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}pt_serif.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="PT Serif Caption, serif" data-font-style="ital@0;1"><div></div><img src="${path}pt_serif_caption.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Quattrocento Sans, sans-serif" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}quattrocento_sans.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Quattrocento, serif" data-font-style="wght@400;700"><div></div><img src="${path}quattrocento.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Quicksand, sans-serif" data-font-style="wght@300;400;500;600;700"><div></div><img src="${path}quicksand.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Qwigley, cursive"><div></div><img src="${path}qwigley.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Raleway, sans-serif" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}raleway.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Raleway Dots, sans-serif"><div></div><img src="${path}raleway_dots.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Redressed, cursive"><div></div><img src="${path}redressed.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ribeye Marrow, cursive"><div></div><img src="${path}ribeye_marrow.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Righteous, cursive"><div></div><img src="${path}righteous.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Roboto, sans-serif" data-font-style="ital,wght@0,100;0,300;0,400;0,500;0,700;0,900;1,100;1,300;1,400;1,500;1,700;1,900"><div></div><img src="${path}roboto.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Roboto Condensed, sans-serif" data-font-style="ital,wght@0,300;0,400;0,700;1,300;1,400;1,700"><div></div><img src="${path}roboto_condensed.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Roboto Mono, monospace" data-font-style="ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;1,100;1,200;1,300;1,400;1,500;1,600;1,700"><div></div><img src="${path}roboto_mono.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Roboto Slab, serif" data-font-style="wght@100;200;300;400;500;600;700;800;900"><div></div><img src="${path}roboto_slab.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Rochester, cursive"><div></div><img src="${path}rochester.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Rouge Script, cursive"><div></div><img src="${path}rouge_script.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Rubik, sans-serif" data-font-style="ital,wght@0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}rubik.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Rubik Mono One, sans-serif"><div></div><img src="${path}rubik_mono_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ruslan Display, cursive"><div></div><img src="${path}ruslan_display.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Russo One, sans-serif"><div></div><img src="${path}russo_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Sacramento, cursive"><div></div><img src="${path}sacramento.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Sanchez, serif" data-font-style="ital@0;1"><div></div><img src="${path}sanchez.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Satisfy, cursive"><div></div><img src="${path}satisfy.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Sawarabi Gothic, sans-serif"><div></div><img src="${path}sawarabi_gothic.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Scada, sans-serif" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}scada.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Seaweed Script, cursive"><div></div><img src="${path}seaweed_script.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Seymour One, sans-serif"><div></div><img src="${path}seymour_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Shadows Into Light Two, cursive"><div></div><img src="${path}shadows_into_light_two.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Six Caps, sans-serif"><div></div><img src="${path}six_caps.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Snowburst One, cursive"><div></div><img src="${path}snowburst_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Source Code Pro, monospace" data-font-style="ital,wght@0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}source_code_pro.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Source Sans Pro, sans-serif" data-font-style="ital,wght@0,200;0,300;0,400;0,600;0,700;0,900;1,200;1,300;1,400;1,600;1,700;1,900"><div></div><img src="${path}source_sans_pro.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Special Elite, cursive"><div></div><img src="${path}special_elite.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Spectral, serif" data-font-style="ital,wght@0,200;0,300;0,400;0,500;0,600;0,700;0,800;1,200;1,300;1,400;1,500;1,600;1,700;1,800"><div></div><img src="${path}spectral.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Spectral SC, serif" data-font-style="ital,wght@0,200;0,300;0,400;0,500;0,600;0,700;0,800;1,200;1,300;1,400;1,500;1,600;1,700;1,800"><div></div><img src="${path}spectral_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Squada One, cursive"><div></div><img src="${path}squada_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Stalinist One, cursive"><div></div><img src="${path}stalinist_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Stint Ultra Expanded, cursive"><div></div><img src="${path}stint_ultra_expanded.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Syncopate, sans-serif" data-font-style="wght@400;700"><div></div><img src="${path}syncopate.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Tangerine, cursive" data-font-style="wght@400;700"><div></div><img src="${path}tangerine.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Tenor Sans, sans-serif"><div></div><img src="${path}tenor_sans.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Tinos, serif" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}tinos.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ubuntu, sans-serif" data-font-style="ital,wght@0,300;0,400;0,500;0,700;1,300;1,400;1,500;1,700"><div></div><img src="${path}ubuntu.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ubuntu Condensed, sans-serif"><div></div><img src="${path}ubuntu_condensed.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Ubuntu Mono, monospace" data-font-style="ital,wght@0,400;0,700;1,400;1,700"><div></div><img src="${path}ubuntu_mono.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Underdog, cursive"><div></div><img src="${path}underdog.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="UnifrakturMaguntia, cursive"><div></div><img src="${path}unifrakturmaguntia.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Vast Shadow, cursive"><div></div><img src="${path}vast_shadow.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Viga, sans-serif"><div></div><img src="${path}viga.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Vollkorn, serif" data-font-style="ital,wght@0,400;0,500;0,600;0,700;0,800;0,900;1,400;1,500;1,600;1,700;1,800;1,900"><div></div><img src="${path}vollkorn.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Vollkorn SC, serif" data-font-style="wght@400;600;700;900"><div></div><img src="${path}vollkorn_sc.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Voltaire, sans-serif"><div></div><img src="${path}voltaire.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Wire One, sans-serif"><div></div><img src="${path}wire_one.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Yanone Kaffeesatz, sans-serif" data-font-style="wght@200;300;400;500;600;700"><div></div><img src="${path}yanone_kaffeesatz.png"></li>
        <li role="option" tabindex="0" data-provider="google" data-font-family="Yeseva One, cursive"><div></div><img src="${path}yeseva_one.png"></li>
        `;
    return html;
  }

  getFontFamilyHTML(showInModal) {
    const html = `
        <!DOCTYPE HTML>
        <html>
        <head>
            <meta charset="utf-8">
            <title>Fonts</title>
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <meta name="description" content="">  
            <style id="mainstyle">
                html, body {height:100%}
                body {overflow:hidden;margin:0;
                    font-family: "Helvetica Neue", Helvetica, Arial, sans-serif;
                    font-size:100%; 
                    line-height:1.7;
                }
                #divFontList {margin:0;padding:0 0 9px 9px;height:100%;overflow-y:scroll !important;box-sizing:border-box;
                    list-style: none;
                    padding: 0;
                    margin: 0;
                }
                #divFontList > li {width:100%;cursor:pointer;overflow:hidden;text-align:center;position:relative;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    text-overflow: ellipsis;
                    white-space: nowrap;
                    outline:none;
                }
                #divFontList > li img {margin:7px 5px 7px 5px;max-width: 230px;max-height: 27px;pointer-events: none;}
                #divFontList > li div {position:absolute;top:0;left:0;width:100%;height:100%;}

                #divFontList > li > div {
                    position:absolute;left:0px;top:0px;width:100%;height:100%;
                    // z-index: 1;
                    pointer-events: none;}

                #divFontList > li {
                    color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
                    background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
                }
                #divFontList > li > div:after {
                    // background: rgba(0, 0, 0, 0.04); 
                    background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
                    position: absolute;
                    content: "";
                    display: block;
                    top: 0;
                    left: 0;
                    right: 0;
                    bottom: 0;
                    opacity: 0;
                }
                #divFontList > li:hover > div:after,
                #divFontList > li:focus > div:after {
                    opacity: 1;
                }
                #divFontList > li.on > div:after {
                    opacity: 1;
                }

                ${!this.builder.styleDark && !this.builder.styleColoredDark && !this.builder.styleColore ? `
                #divFontList > li img {
                    mix-blend-mode: multiply;
                }
                ` : ''}

                .dark #divFontList > li img {
                    ${showInModal ? '' : 'mix-blend-mode: screen;'};
                    filter: invert(1);
                }
                .dark #divFontList > li {
                    color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
                    background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
                }
                .dark #divFontList > li > div:after {
                    background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
                }
                
                .colored-dark #divFontList > li img {
                    ${showInModal ? '' : 'mix-blend-mode: screen;'};
                    ${showInModal ? '' : 'filter: invert(1);'};
                }
                .colored-dark #divFontList > li {
                    color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
                    background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
                }
                .colored-dark #divFontList > li > div:after {
                    background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
                }


                .colored #divFontList > li img {
                    ${showInModal ? '' : 'mix-blend-mode: screen;'};
                    ${showInModal ? '' : 'filter: invert(1);'};
                }
                .colored #divFontList > li {
                    color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
                    background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
                }
                .colored #divFontList > li > div:after {
                    background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
                }


                /* Scrollbar for toolbar/RTE and modal */

                .dark * {
                    scrollbar-width: thin;
                    scrollbar-color: rgba(255, 255, 255, 0.3) auto;
                }
                .dark *::-webkit-scrollbar {
                    width: 12px;
                }
                .dark *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .dark *::-webkit-scrollbar-thumb {
                    background-color:rgba(255, 255, 255, 0.3);
                } 

                .colored-dark * {
                    scrollbar-width: thin;
                    scrollbar-color: rgb(100, 100, 100) auto;
                }
                .colored-dark *::-webkit-scrollbar {
                    width: 12px;
                }
                .colored-dark *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .colored-dark *::-webkit-scrollbar-thumb {
                    background-color:rgb(100, 100, 100);
                } 

                .colored * {
                    scrollbar-width: thin;
                    scrollbar-color: rgba(0, 0, 0, 0.4) auto;
                }
                .colored *::-webkit-scrollbar {
                    width: 12px;
                }
                .colored *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .colored *::-webkit-scrollbar-thumb {
                    background-color: rgba(0, 0, 0, 0.4);
                } 

                .light * {
                    scrollbar-width: thin;
                    scrollbar-color: rgba(0, 0, 0, 0.4) auto;
                }
                .light *::-webkit-scrollbar {
                    width: 12px;
                }
                .light *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .light *::-webkit-scrollbar-thumb {
                    background-color: rgba(0, 0, 0, 0.4);
                } 
            </style>

        </head>
        <body${this.builder.styleDark ? ' class="dark"' : ''}${this.builder.styleColored ? ' class="colored"' : ''}${this.builder.styleColoredDark ? ' class="colored-dark"' : ''}${this.builder.styleLight ? ' class="light"' : ''}>

        <ul id="divFontList">
        ${this.getFontPreview()}
        </ul>

        <script type="text/javascript">

            const close = () => {
                parent.focus();

                let pop = parent.document.querySelector('.rte-fontfamily-options.active, .pickfontfamily.active');
                if(pop)  {
                    pop.classList.remove('active');
                    pop.classList.add('deactive');
                }
            }

            const setValue = (item) => {
                let provider = item.getAttribute('data-provider');
                let fontfamily = item.getAttribute('data-font-family');
                let fontstyle = item.getAttribute('data-font-style');
                let fontdisplay = item.getAttribute('data-font-display');
                parent._cb.setFont(fontfamily, fontstyle, fontdisplay, provider);

                let elms = document.querySelectorAll('#divFontList > li');
                elms.forEach(elm=>{
                    elm.classList.remove('on');
                });
                item.classList.add('on');
            }

            const handleItemClick = (e) => {
                e.preventDefault();

                let item = e.target;
                setValue(item);
            }

            const handleItemKeyDown = (e) => {
                e.preventDefault();

                if(e.keyCode === 38 && e.target.previousElementSibling) { // up
                    e.target.previousElementSibling.focus();
                } else if(e.keyCode === 40 && e.target.nextElementSibling) { // down
                    e.target.nextElementSibling.focus();
                } else if(e.keyCode === 27) { // escape key
                    close();
                } else if(e.keyCode === 13 || e.keyCode === 32) { // enter or spacebar key
                    setValue(e.target);
                }
            }
            
            let elms = document.querySelectorAll('#divFontList > li');
            elms.forEach(elm=>{
                elm.addEventListener('keydown', handleItemKeyDown);
                elm.addEventListener('click', handleItemClick);
            });
        </script>
        
        </body>
        </html>
                
        
        `;
    return html;
  }

  getFontFamilyEmail(showInModal) {
    // let path = this.builder.scriptPath + 'fonts/';
    let path = this.builder.fontAssetPath;
    const html = `
        <!DOCTYPE HTML>
        <html>
        <head>
            <meta charset="utf-8">
            <title>Fonts</title>
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <meta name="description" content="">  
            <style>
                html, body {height:100%}
                body {overflow:hidden;margin:0;
                    font-family: "Helvetica Neue", Helvetica, Arial, sans-serif;
                    font-size:100%; 
                    line-height:1.7;
                }
                #divFontList {margin:0;padding:0 0 9px 9px;height:100%;overflow-y:scroll !important;box-sizing:border-box;
                    list-style: none;
                    padding: 0;
                    margin: 0;
                }
                #divFontList > li {width:100%;cursor:pointer;overflow:hidden;text-align:center;position:relative;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    text-overflow: ellipsis;
                    white-space: nowrap;
                    outline:none;
                }
                #divFontList > li img {margin:7px 5px 7px 5px;max-width: 230px;max-height: 27px;pointer-events: none;}
                #divFontList > li div {position:absolute;top:0;left:0;width:100%;height:100%;}

                #divFontList > li > div {
                    position:absolute;left:0px;top:0px;width:100%;height:100%;
                    // z-index: 1;
                    pointer-events: none;}

                #divFontList > li {
                    color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
                    background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
                }
                #divFontList > li > div:after {
                    // background: rgba(0, 0, 0, 0.04); 
                    background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
                    position: absolute;
                    content: "";
                    display: block;
                    top: 0;
                    left: 0;
                    right: 0;
                    bottom: 0;
                    opacity: 0;
                }
                #divFontList > li:hover > div:after,
                #divFontList > li:focus > div:after {
                    opacity: 1;
                }
                #divFontList > li.on > div:after {
                    opacity: 1;
                }

                ${!this.builder.styleDark && !this.builder.styleColoredDark && !this.builder.styleColore ? `
                #divFontList > li img {
                    mix-blend-mode: multiply;
                }
                ` : ''}

                .dark #divFontList > li img {
                    ${showInModal ? '' : 'mix-blend-mode: screen;'};
                    filter: invert(1);
                }
                .dark #divFontList > li {
                    color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
                    background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
                }
                .dark #divFontList > li > div:after {
                    background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
                }
                
                .colored-dark #divFontList > li img {
                    ${showInModal ? '' : 'mix-blend-mode: screen;'};
                    ${showInModal ? '' : 'filter: invert(1);'};
                }
                .colored-dark #divFontList > li {
                    color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
                    background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
                }
                .colored-dark #divFontList > li > div:after {
                    background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
                }


                .colored #divFontList > li img {
                    ${showInModal ? '' : 'mix-blend-mode: screen;'};
                    ${showInModal ? '' : 'filter: invert(1);'};
                }
                .colored #divFontList > li {
                    color: ${showInModal ? this.builder.styleModalColor : this.builder.styleButtonColor};
                    background: ${showInModal ? this.builder.styleModalBackground : this.builder.styleToolBackground};
                }
                .colored #divFontList > li > div:after {
                    background: ${showInModal ? this.builder.styleButtonClassicBackgroundHover : this.builder.styleButtonBackgroundHover};
                }


                /* Scrollbar for toolbar/RTE and modal */

                .dark * {
                    scrollbar-width: thin;
                    scrollbar-color: rgba(255, 255, 255, 0.3) auto;
                }
                .dark *::-webkit-scrollbar {
                    width: 12px;
                }
                .dark *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .dark *::-webkit-scrollbar-thumb {
                    background-color:rgba(255, 255, 255, 0.3);
                } 

                .colored-dark * {
                    scrollbar-width: thin;
                    scrollbar-color: rgb(100, 100, 100) auto;
                }
                .colored-dark *::-webkit-scrollbar {
                    width: 12px;
                }
                .colored-dark *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .colored-dark *::-webkit-scrollbar-thumb {
                    background-color:rgb(100, 100, 100);
                } 

                .colored * {
                    scrollbar-width: thin;
                    scrollbar-color: rgba(0, 0, 0, 0.4) auto;
                }
                .colored *::-webkit-scrollbar {
                    width: 12px;
                }
                .colored *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .colored *::-webkit-scrollbar-thumb {
                    background-color: rgba(0, 0, 0, 0.4);
                } 

                .light * {
                    scrollbar-width: thin;
                    scrollbar-color: rgba(0, 0, 0, 0.4) auto;
                }
                .light *::-webkit-scrollbar {
                    width: 12px;
                }
                .light *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .light *::-webkit-scrollbar-thumb {
                    background-color: rgba(0, 0, 0, 0.4);
                } 
            </style>

        </head>
        <body${this.builder.styleDark ? ' class="dark"' : ''}${this.builder.styleColored ? ' class="colored"' : ''}${this.builder.styleColoredDark ? ' class="colored-dark"' : ''}${this.builder.styleLight ? ' class="light"' : ''}>

        <ul id="divFontList">
            <li role="option" tabindex="0" data-provider="" data-font-family="" style="font-size:12px;padding:10px 7px;box-sizing:border-box;"><div></div>
                <span style="z-index:1;position: relative;pointer-events: none;">${this.out('None')}</span>
            </li>
            <li role="option" tabindex="0" data-provider="" data-font-family="Arial, sans-serif"><div></div><img src="${path}arial.png"></li>
            <li role="option" tabindex="0" data-provider="" data-font-family="courier"><div></div><img src="${path}courier.png"></li>
            <li role="option" tabindex="0" data-provider="" data-font-family="Georgia, serif"><div></div><img src="${path}georgia.png"></li>
            <li role="option" tabindex="0" data-provider="" data-font-family="monospace"><div></div><img src="${path}monospace.png"></li>
            <li role="option" tabindex="0" data-provider="" data-font-family="sans-serif"><div></div><img src="${path}sans_serif.png"></li>
            <li role="option" tabindex="0" data-provider="" data-font-family="serif"><div></div><img src="${path}serif.png"></li>
        </ul>

        <script type="text/javascript">

            const close = () => {
                parent.focus();

                let pop = parent.document.querySelector('.is-rte-pop.active');
                if(pop)  {
                    pop.classList.remove('active');
                    pop.classList.add('deactive');
                }
            }

            const setValue = (item) => {
                let provider = item.getAttribute('data-provider');
                let fontfamily = item.getAttribute('data-font-family');
                let fontstyle = item.getAttribute('data-font-style');
                let fontdisplay = item.getAttribute('data-font-display');
                parent._cb.setFont(fontfamily, fontstyle, fontdisplay, provider);

                let elms = document.querySelectorAll('#divFontList > li');
                elms.forEach(elm=>{
                    elm.classList.remove('on');
                });
                item.classList.add('on');
            }

            const handleItemClick = (e) => {
                e.preventDefault();

                let item = e.target;
                setValue(item);
            }

            const handleItemKeyDown = (e) => {
                e.preventDefault();

                if(e.keyCode === 38 && e.target.previousElementSibling) { // up
                    e.target.previousElementSibling.focus();
                } else if(e.keyCode === 40 && e.target.nextElementSibling) { // down
                    e.target.nextElementSibling.focus();
                } else if(e.keyCode === 27) { // escape key
                    close();
                } else if(e.keyCode === 13 || e.keyCode === 32) { // enter or spacebar key
                    setValue(e.target);
                }
            }
            
            let elms = document.querySelectorAll('#divFontList > li');
            elms.forEach(elm=>{
                elm.addEventListener('keydown', handleItemKeyDown);
                elm.addEventListener('click', handleItemClick);
            });
        </script>
        
        </body>
        </html>
                
        
        `;
    return html;
  }

  getPageTemplate(framework) {
    let pageTemplate = '';

    if (framework === '') {
      pageTemplate = `
            <!DOCTYPE html>
            <html lang="en">
            <head>
                <meta charset="utf-8">
                <title>Page</title>
                <meta name="viewport" content="width=device-width, initial-scale=1">
                <meta name="description" content="">
                <link rel="shortcut icon" href="#" />
                
                <link href="[%PATH%]/assets/minimalist-blocks/content.css" rel="stylesheet" type="text/css" />
                
                <link href="[%PATH%]/assets/scripts/glide/css/glide.core.css" rel="stylesheet" type="text/css" />
                <link href="[%PATH%]/assets/scripts/glide/css/glide.theme.css" rel="stylesheet" type="text/css" />
                <script src="[%PATH%]/assets/scripts/glide/glide.js" type="text/javascript"></script>
                
                <style>
                    .container {
                        margin: 150px auto 0;
                        max-width: 800px;
                        width: 100%;
                        padding: 0 20px;
                        box-sizing: border-box;
                    }
                </style>
            </head>
            <body>
            
            <div class="container">
            [%CONTENT%]
            </div>
            
            </body>
            </html>
            `;
    } else if (framework === 'tailwind') {
      pageTemplate = `
            <!DOCTYPE html>
            <html lang="en">
            <head>
                <meta charset="utf-8">
                <title>Page</title>
                <meta name="viewport" content="width=device-width, initial-scale=1">
                <meta name="description" content="">
                <link rel="shortcut icon" href="#" />
                
                <link href="[%PATH%]/assets/minimalist-blocks/content-tailwind.css" rel="stylesheet" type="text/css" />
                
                <link href="[%PATH%]/assets/scripts/glide/css/glide.core.css" rel="stylesheet" type="text/css" />
                <link href="[%PATH%]/assets/scripts/glide/css/glide.theme.css" rel="stylesheet" type="text/css" />
                <script src="[%PATH%]/assets/scripts/glide/glide.js" type="text/javascript"></script>
                
                <link rel="stylesheet" href="[%PATH%]/assets/frameworks/tailwindcss/styles.css">
                <!-- To build your own, please see: https://tailwindcss.com/docs/installation -->
                
                <style>
                    .container {
                        margin: 150px auto 0;
                        max-width: 800px;
                        width: 100%;
                        padding: 0 20px;
                        box-sizing: border-box;
                    }
                </style>
            </head>
            <body>
            
            <div class="container">
            [%CONTENT%]
            </div>
            
            </body>
            </html>
            `;
    } else if (framework === 'bootstrap') {
      pageTemplate = `
            <!DOCTYPE html>
            <html lang="en">
            <head>
                <meta charset="utf-8">
                <title>Page</title>
                <meta name="viewport" content="width=device-width, initial-scale=1">
                <meta name="description" content="">
                <link rel="shortcut icon" href="#" />

                <link rel="stylesheet" href="https://stackpath.bootstrapcdn.com/bootstrap/4.1.3/css/bootstrap.min.css" integrity="sha384-MCw98/SFnGE8fJT3GXwEOngsV7Zt27NXFoaoApmYm81iuXoPkFOJwJ8ERdknLPMO" crossorigin="anonymous">
    
                <link href="[%PATH%]/assets/minimalist-blocks/content.css" rel="stylesheet" type="text/css" />
                
                <link href="[%PATH%]/assets/scripts/glide/css/glide.core.css" rel="stylesheet" type="text/css" />
                <link href="[%PATH%]/assets/scripts/glide/css/glide.theme.css" rel="stylesheet" type="text/css" />
                <script src="[%PATH%]/assets/scripts/glide/glide.js" type="text/javascript"></script>
                
                <style>
                    .container {margin: 140px auto; max-width: 800px; width:100%; padding:0 35px; box-sizing: border-box;}
                    /* Bootstrap css adjustment to enable column drag to resize */
                    .row > *,
                    .column {
                        max-width: unset !important;
                    }
                </style>
            </head>
            <body>
            
            <div class="container">
            [%CONTENT%]
            </div>
            
            </body>
            </html>
            `;
    } else if (framework === 'foundation') {
      pageTemplate = `
            <!DOCTYPE html>
            <html lang="en">
            <head>
                <meta charset="utf-8">
                <title>Page</title>
                <meta name="viewport" content="width=device-width, initial-scale=1">
                <meta name="description" content="">
                <link rel="shortcut icon" href="#" />

                <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/foundation/6.5.3/css/foundation.min.css">
                <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/foundation/6.5.3/css/foundation-float.min.css">
            
                <link href="[%PATH%]/assets/minimalist-blocks/content.css" rel="stylesheet" type="text/css" />
                
                <link href="[%PATH%]/assets/scripts/glide/css/glide.core.css" rel="stylesheet" type="text/css" />
                <link href="[%PATH%]/assets/scripts/glide/css/glide.theme.css" rel="stylesheet" type="text/css" />
                <script src="[%PATH%]/assets/scripts/glide/glide.js" type="text/javascript"></script>
                
                <style>
                    .container {
                        margin: 150px auto 0;
                        max-width: 800px;
                        width: 100%;
                        padding: 0 20px;
                        box-sizing: border-box;
                    }
                </style>
            </head>
            <body>
            
            <div class="container">
            [%CONTENT%]
            </div>
            
            </body>
            </html>
            `;
    } else {
      pageTemplate = '[%CONTENT%]';
    }

    return pageTemplate;
  }

}
class Dom {
  constructor(builder) {
    this.builder = builder;
  }

  getScale(container) {
    let matrix = window.getComputedStyle(container).transform;
    let values = matrix.split('(')[1];
    values = values.split(')')[0];
    values = values.split(',');
    let a = values[0];
    let b = values[1];
    let scale = Math.sqrt(a * a + b * b);
    return scale;
  }

  createElement(tag) {
    return document.createElement(tag);
  }

  appendChild(parent, child) {
    if (parent) parent.appendChild(child);
  }

  appendHtml(parent, html) {
    if (parent) parent.insertAdjacentHTML('beforeend', html);
  }

  addEventListener(parent, type, listener) {
    if (parent) parent.addEventListener(type, listener);
  }

  addClass(element, classname) {
    if (!element) return;
    if (this.hasClass(element, classname)) return;
    if (element.classList.length === 0) element.className = classname;else element.className = element.className + ' ' + classname;
    element.className = element.className.replace(/  +/g, ' '); //else element.classList.add(classname); //error if there is -
  }

  removeClass(element, classname) {
    if (!element) return;

    if (element.classList.length > 0) {
      // element.className = element.className.replace(new RegExp('\\b'+ classname+'\\b', 'g'), '');
      // element.className = element.className.replace(/  +/g, ' ');
      let i, j, imax, jmax;
      let classesToDel = classname.split(' ');

      for (i = 0, imax = classesToDel.length; i < imax; ++i) {
        if (!classesToDel[i]) continue;
        let classtoDel = classesToDel[i]; // https://jsperf.com/removeclass-methods 

        let sClassName = '';
        let currentClasses = element.className.split(' ');

        for (j = 0, jmax = currentClasses.length; j < jmax; ++j) {
          if (!currentClasses[j]) continue;
          if (currentClasses[j] !== classtoDel) sClassName += currentClasses[j] + ' ';
        }

        element.className = sClassName.trim();
      }

      if (element.className === '') element.removeAttribute('class');
    }
  }

  removeClassesByPrefix(element, prefix) {
    if (element.classList) {
      element.classList.forEach(item => {
        if (item.indexOf(prefix) === 0) {
          this.removeClass(element, item);
        }
      });
    }
  } // https://plainjs.com/javascript/attributes/adding-removing-and-testing-for-classes-9/
  // addClass(element, classname) {
  //     if (element.classList) element.classList.add(classname);
  //     else if (!this.hasClass(element, classname)) element.className += ' ' + classname;
  // }
  // removeClass(element, classname) {
  //     if (element.classList) element.classList.remove(classname);
  //     else element.className = element.className.replace(new RegExp('\\b'+ classname+'\\b', 'g'), '');
  // }

  /*
  hasClass(element, classname) {
      if(!element) return false;
      try{
          let s = element.getAttribute('class');
          return new RegExp('\\b'+ classname+'\\b').test(s);
      } catch(e) {
          // Do Nothing
      }
      //return element.classList ? element.classList.contains(classname) : new RegExp('\\b'+ classname+'\\b').test(element.className);
  }
  */


  hasClass(element, className) {
    if (!element) return false;
    if (!element.className) return false;
    return (' ' + element.className + ' ').indexOf(' ' + className + ' ') > -1;
  }

  moveAfter(element, targetElement) {
    targetElement.parentNode.insertBefore(element, targetElement);
    targetElement.parentNode.insertBefore(targetElement, targetElement.previousElementSibling);
  } // https://stackoverflow.com/questions/10381296/best-way-to-get-child-nodes


  elementChildren(element) {
    const childNodes = element.childNodes;
    let children = [];
    let i = childNodes.length;

    while (i--) {
      if (childNodes[i].nodeType === 1
      /*&& childNodes[i].tagName === 'DIV'*/
      ) {
        children.unshift(childNodes[i]);
      }
    }

    return children;
  }

  getElementOffset(el) {
    const rect = el.getBoundingClientRect();
    return {
      top: rect.top + window.pageYOffset,
      left: rect.left + window.pageXOffset,
      width: rect.width,
      height: rect.height
    };
  }

  parentsHasClass(element, classname) {
    while (element) {
      // if(classname==='is-side') console.log(element.nodeName); // NOTE: click on svg can still returns undefined in IE11
      if (!element.tagName) return false;
      if (element.tagName === 'BODY' || element.tagName === 'HTML') return false; // if(!element.classList) {
      //     return false;
      // }

      if (this.hasClass(element, classname)) {
        return true;
      } // TODO: if(element.nodeName.toLowerCase() === 'svg') console.log(element);


      element = element.parentNode;
    }
  }

  parentsHasId(element, id) {
    while (element) {
      if (!element.tagName) return false;
      if (element.tagName === 'BODY' || element.tagName === 'HTML') return false;

      if (element.id === id) {
        return true;
      }

      element = element.parentNode;
    }
  }

  parentsHasTag(element, tagname) {
    while (element) {
      if (!element.tagName) return false;
      if (element.tagName === 'BODY' || element.tagName === 'HTML') return false;

      if (element.tagName.toLowerCase() === tagname.toLowerCase()) {
        return true;
      }

      element = element.parentNode;
    }
  }

  parentsHasAttribute(element, attrname) {
    while (element) {
      if (!element.tagName) return false;
      if (element.tagName === 'BODY' || element.tagName === 'HTML') return false;

      try {
        if (element.hasAttribute(attrname)) {
          // error on svg element
          return true;
        }
      } catch (e) {// Do Nothing
        // return false;
      }

      element = element.parentNode;
    }
  }

  getParentByTag(element, tagname) {
    while (element) {
      if (!element.tagName) return false;
      if (element.tagName === 'BODY' || element.tagName === 'HTML') return false;

      if (element.tagName.toLowerCase() === tagname.toLowerCase()) {
        return element;
      }

      element = element.parentNode;
    }
  }

  getParentByAttribute(element, attrname) {
    while (element) {
      if (!element.tagName) return false;
      if (element.tagName === 'BODY' || element.tagName === 'HTML') return false;

      try {
        if (element.hasAttribute(attrname)) {
          // error on svg element
          return element;
        }
      } catch (e) {// Do Nothing
        // return false;
      }

      element = element.parentNode;
    }
  }

  parentsHasElement(element, tagname) {
    while (element) {
      if (!element.tagName) return false;
      if (element.tagName === 'BODY' || element.tagName === 'HTML') return false;
      element = element.parentNode;
      if (!element) return false;
      if (!element.tagName) return false;

      if (element.tagName.toLowerCase() === tagname) {
        return true;
      }
    }
  }

  getParentElement(element, tagname) {
    while (element) {
      if (!element.tagName) return false;
      if (element.tagName === 'BODY' || element.tagName === 'HTML') return false;
      element = element.parentNode;
      if (!element) return false;
      if (!element.tagName) return false;

      if (element.tagName.toLowerCase() === tagname) {
        return element;
      }
    }
  }

  removeClasses(elms, classname) {
    for (let i = 0; i < elms.length; i++) {
      elms[i].classList.remove(classname);
    }
  }

  removeAttributes(elms, attrname) {
    for (let i = 0; i < elms.length; i++) {
      elms[i].removeAttribute(attrname);
    }
  }

  removeElements(elms) {
    Array.prototype.forEach.call(elms, el => {
      el.parentNode.removeChild(el);
    });
  } // source: https://stackoverflow.com/questions/2871081/jquery-setting-cursor-position-in-contenteditable-div


  moveCursorToElement(element) {
    var sel, range;

    if (window.getSelection && this.builder.doc.createRange) {
      range = this.builder.doc.createRange();
      range.selectNodeContents(element);
      range.collapse(false);
      sel = window.getSelection();
      sel.removeAllRanges();
      sel.addRange(range);
    } else if (this.builder.doc.body.createTextRange) {
      range = this.builder.doc.body.createTextRange();
      range.moveToElementText(element);
      range.collapse(false);
      range.select();
    }
  } // source: https://stackoverflow.com/questions/6139107/programmatically-select-text-in-a-contenteditable-html-element


  selectElementContents(el) {
    var range = this.builder.doc.createRange();
    range.selectNodeContents(el);
    var sel = this.builder.win.getSelection();
    sel.removeAllRanges();
    sel.addRange(range);
  } // Get selected text


  getSelected() {
    if (this.builder.win.getSelection) {
      return this.builder.win.getSelection().toString();
    } else if (this.builder.doc.getSelection) {
      return this.builder.doc.getSelection().toString();
    } else {
      var selection = this.builder.doc.selection && this.builder.doc.selection.createRange();

      if (selection.text) {
        return selection.text;
      }

      return false;
    }
  }

  checkEditable() {
    try {
      var el;
      var curr;

      if (this.builder.win.getSelection) {
        curr = this.builder.win.getSelection().getRangeAt(0).commonAncestorContainer;

        if (curr.nodeType === 3) {
          //ini text node
          el = curr.parentNode;
        } else {
          el = curr;
        }
      } else if (this.builder.doc.selection) {
        curr = this.builder.doc.selection.createRange();
        el = this.builder.doc.selection.createRange().parentElement();
      }

      if (this.parentsHasAttribute(el, 'contenteditable')) return true;else return false;
    } catch (e) {
      return false;
    }
  }

  textSelection() {
    /*
    try {
        var elm;
        var curr = window.getSelection().getRangeAt(0).commonAncestorContainer;
        if (curr.nodeType === 3) {  // text node
            elm = curr.parentNode;
            if(this.parentsHasClass(elm, 'is-builder')) {
                return elm;
            }
            else {
                return false;  
            } 
        } else {
            elm = curr;
            var nodeName = elm.nodeName.toLowerCase();
            if(nodeName === 'i' && elm.innerHTML === '') { //icon
                if(this.parentsHasClass(elm, 'is-builder')) {
                    return elm;
                }
            }
            
            // Check if a block (because when placing cursor using arrow keys on empty block, nodeType=1 not 3)
            if(nodeName === 'p' || nodeName === 'h1' || nodeName === 'h2'
            || nodeName === 'h3' || nodeName === 'h4' || nodeName === 'h5'
            || nodeName === 'h6' || nodeName === 'li' || nodeName === 'pre'
            || nodeName === 'blockquote') {
                return elm;
            }
            return false;
        }
    } catch (e) {
        return false;
    }
    */
    const selection = this.getSelection();
    const anchorNode = selection.anchorNode;
    if (!anchorNode) return false;
    const container = anchorNode.nodeType !== Node.TEXT_NODE && anchorNode.nodeType !== Node.COMMENT_NODE ? anchorNode : anchorNode.parentElement;
    return container;
  }

  getStyle(element, property) {
    return window.getComputedStyle ? window.getComputedStyle(element, null).getPropertyValue(property) : element.style[property.replace(/-([a-z])/g, function (g) {
      return g[1].toUpperCase();
    })];
  } //https://stackoverflow.com/questions/32383349/detect-input-value-change-with-mutationobserver/61975440#61975440


  observeElement(element, property, callback, delay = 0) {
    let elementPrototype = Object.getPrototypeOf(element);

    if (Object.prototype.hasOwnProperty.call(elementPrototype, property)) {
      let descriptor = Object.getOwnPropertyDescriptor(elementPrototype, property);
      Object.defineProperty(element, property, {
        get: function () {
          return descriptor.get.apply(this, arguments);
        },
        set: function () {
          let oldValue = this[property];
          descriptor.set.apply(this, arguments);
          let newValue = this[property];

          if (typeof callback == 'function') {
            setTimeout(callback.bind(this, oldValue, newValue), delay);
          } // return newValue;

        }
      });
    }
  }

  baseName(str) {
    if (typeof str !== 'string') return;
    var frags = str.split('.');
    return frags.splice(0, frags.length - 1).join('.');
  }
  /** added by Jack */


  doFunction(elm, fn, incChilds) {
    fn(elm);

    if (incChilds) {
      let elmns = Array.prototype.slice.call(elm.getElementsByTagName('*'));

      for (var i = 0; i < elmns.length; i++) {
        fn(elmns[i]);
      }
    }
  } // ---
  // execCommand replacement


  execCommand(action, value, callback) {
    const selection = this.getSelection();
    if (!selection) return;
    const anchorNode = selection.anchorNode;

    if (anchorNode) {
      const container = anchorNode.nodeType !== Node.TEXT_NODE && anchorNode.nodeType !== Node.COMMENT_NODE ? anchorNode : anchorNode.parentElement;
      const sameSelection = container && container.innerText === selection.toString();
      let newElement;
      let multi = false;
      var selectedNodes = [];
      var sel = rangy.getSelection();

      for (var i = 0; i < sel.rangeCount; ++i) {
        selectedNodes = selectedNodes.concat(sel.getRangeAt(i).getNodes());
      }

      const blockElms = ['p', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'div', 'pre'];
      selectedNodes.forEach(item => {
        if (item.tagName) {
          const tagName = item.tagName.toLowerCase();

          if (blockElms.indexOf(tagName) !== -1) {
            multi = true;
            callback(true, item);
          }
        }
      });
      if (multi) return;

      if (sameSelection || selection.toString().trim() === '') {
        newElement = this.updateSelection(action, value, container);
      } else {
        newElement = this.replaceSelection(action, value, selection, container);
      }

      if (callback) callback(true, newElement);
    } else {
      if (callback) callback(false);
    }
  }

  updateSelection(action, value, container) {
    // container.style[action] = value; //APPLY => Use getStyleValue to check if toggle is needed.
    container.style[action] = this.getStyleValue(container, action, value); // Optional, better if used: check if the style is the same as its parent. If so, clear it.

    this.checkStyleIfSameAsParent(container, action);
    this.cleanChildren(action, container);
    this.cleanUnusedSpan(container); // REVIEW

    return container;
  }

  replaceSelection(action, value, selection, container) {
    const range = selection.getRangeAt(0);
    const fragment = range.extractContents();
    const span = this.createSpan(container, action, value);
    span.appendChild(fragment);
    this.cleanChildren(action, span);
    this.cleanUnusedSpan(span); // REVIEW

    range.insertNode(span);
    selection.selectAllChildren(span); // Optional, better if used: check if the style is the same as its parent. If so, clear it.

    this.checkStyleIfSameAsParent(span, action);
    return span;
  }

  checkStyleIfSameAsParent(element, action) {
    if (element.parentNode) {
      if (element.parentNode.style[action] === element.style[action] || window.getComputedStyle(element.parentNode, null).getPropertyValue(action) === element.style[action]) {
        element.style[action] = '';

        if (element.getAttribute('style') === '' || element.style === null) {
          element.removeAttribute('style');
        } // REVIEW


        let unused = element.attributes.length === 0 || element.attributes.length === 1 && element.hasAttribute('data-keep');

        if (element.nodeName.toLowerCase() === 'span' && unused) {
          const text = document.createTextNode(element.textContent);
          element.parentElement.replaceChild(text, element);
        }
      }
    }
  }

  createSpan(container, action, value) {
    const span = document.createElement('span'); // span.style[action] = value; //APPLY => Use getStyleValue to check if toggle is needed.

    span.style[action] = this.getStyleValue(container, action, value);
    return span;
  }

  findParentStyle(node, action) {
    if (node.nodeName.toUpperCase() === 'HTML' || node.nodeName.toUpperCase() === 'BODY') {
      return null;
    }

    if (!node.parentNode) {
      return null;
    }

    const hasStyle = node.style[action] !== null && node.style[action] !== undefined && node.style[action] !== '';

    if (hasStyle) {
      return node;
    }

    this.findParentStyle(node.parentNode, action);
  }

  getStyleValue(container, action, value) {
    if (action === 'font-weight' || action === 'font-style' || action === 'text-decoration' || action === 'text-transform') {
      // Toggle
      // let currentValue = window.getComputedStyle(container, null).getPropertyValue(action);
      let parent = this.findParentStyle(container, action);
      let currentValue;

      if (parent) {
        currentValue = parent.style[action];
      }

      if (value === currentValue) {
        //toggle here
        // value = 'initial'; // the problem with initial is that it is always refer to font-weight: 400 (won;t work for custom 300 for normal paragraph)
        // so we'll do the test first, to find the actuat initial value
        container.style[action] = ''; // test

        let initialValue = window.getComputedStyle(container, null).getPropertyValue(action); // get the actual initial value
        // console.log('initial:' +initialValue);

        container.style[action] = currentValue; //return back from test

        value = initialValue;
      }
    }

    return value;
  }

  getParentBlock(element) {
    while (element) {
      if (!element.tagName) return false;
      if (element.tagName === 'BODY' || element.tagName === 'HTML') return false;

      if (this.getStyle(element, 'display') !== 'inline') {
        return element;
      }

      element = element.parentNode;
    }
  } // execCommand replacement => for general, apply class to current element


  execCommandClass(newClassName, classList, currentElement = true, mode) {
    const selection = this.getSelection();
    if (!selection) return;
    const anchorNode = selection.anchorNode;

    if (anchorNode) {
      let container = anchorNode.nodeType !== Node.TEXT_NODE && anchorNode.nodeType !== Node.COMMENT_NODE ? anchorNode : anchorNode.parentElement;
      const sameSelection = container && container.innerText === selection.toString();
      let multi = false;
      var selectedNodes = [];
      var sel = rangy.getSelection();

      for (var i = 0; i < sel.rangeCount; ++i) {
        selectedNodes = selectedNodes.concat(sel.getRangeAt(i).getNodes());
      }

      const blockElms = ['p', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'div', 'pre'];
      selectedNodes.forEach(item => {
        if (item.tagName) {
          const tagName = item.tagName.toLowerCase();

          if (blockElms.indexOf(tagName) !== -1) {
            multi = true;

            for (let i = 0; i < Object.keys(classList).length; i++) {
              let className = Object.values(classList)[i];
              this.removeClass(item, className);
            }

            if (newClassName !== '') this.addClass(item, newClassName);
            this.cleanChildrenClass(item, classList);
            this.cleanUnusedSpan(item); // REVIEW
          }
        }
      });
      if (multi) return;

      if (currentElement) {
        if (mode === 'block') if (this.getStyle(container, 'display') === 'inline') {
          container = this.getParentBlock(container);
        }
        this.updateSelectionClass(newClassName, classList, container);
      } else {
        if (sameSelection || selection.toString().trim() === '') {
          if (mode === 'block') if (this.getStyle(container, 'display') === 'inline') {
            container = this.getParentBlock(container);
          }
          this.updateSelectionClass(newClassName, classList, container);
        }
      }
    }
  }

  updateSelectionClass(newClassName, classList, container) {
    for (let i = 0; i < Object.keys(classList).length; i++) {
      let className = Object.values(classList)[i];
      this.removeClass(container, className);
    }

    if (newClassName !== '') this.addClass(container, newClassName);
    this.cleanChildrenClass(container, classList);
    this.cleanUnusedSpan(container); // REVIEW

    return container;
  }

  cleanChildrenClass(span, classList) {
    if (!span.hasChildNodes()) {
      return;
    }

    Array.from(span.children).map(element => {
      for (let i = 0; i < Object.keys(classList).length; i++) {
        let className = Object.values(classList)[i];
        this.removeClass(element, className);
      }
    }); // Deeper childrens

    const cleanChildrenChildren = Array.from(span.children).map(element => {
      return this.cleanChildrenClass(element, classList);
    });

    if (!cleanChildrenChildren || cleanChildrenChildren.length <= 0) {
      return;
    }
  } // execCommand replacement => for CLASS


  execCommandToggle(action, value, config) {
    const selection = this.getSelection();
    if (!selection) return;
    const anchorNode = selection.anchorNode;

    if (anchorNode) {
      const container = anchorNode.nodeType !== Node.TEXT_NODE && anchorNode.nodeType !== Node.COMMENT_NODE ? anchorNode : anchorNode.parentElement;
      const sameSelection = container && container.innerText === selection.toString();
      let multi = false;
      var selectedNodes = [];
      var sel = rangy.getSelection();

      for (var i = 0; i < sel.rangeCount; ++i) {
        selectedNodes = selectedNodes.concat(sel.getRangeAt(i).getNodes());
      }

      const blockElms = ['p', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'div', 'pre'];
      selectedNodes.forEach(item => {
        if (item.tagName) {
          const tagName = item.tagName.toLowerCase();

          if (blockElms.indexOf(tagName) !== -1) {
            multi = true;

            if (action === 'fontWeight') {
              let bold = false;

              if (this.hasClass(item, config.fontWeight.defaultBold)) {
                bold = true;
              } // const num = parseInt(window.getComputedStyle(item, null).getPropertyValue('font-weight'));
              // if(num>=500) bold = true;


              this.removeClass(item, config.fontWeight.thin);
              this.removeClass(item, config.fontWeight.extralight);
              this.removeClass(item, config.fontWeight.light);
              this.removeClass(item, config.fontWeight.normal);
              this.removeClass(item, config.fontWeight.medium);
              this.removeClass(item, config.fontWeight.semibold);
              this.removeClass(item, config.fontWeight.bold);
              this.removeClass(item, config.fontWeight.extrabold);
              this.removeClass(item, config.fontWeight.black);

              if (bold) ; else {
                this.addClass(item, config.fontWeight.defaultBold);
              }
            }

            if (action === 'fontStyle') {
              let italic = false;

              if (this.hasClass(item, config.fontStyle.italic)) {
                italic = true;
              }

              this.removeClass(item, config.fontStyle.italic);
              this.removeClass(item, config.fontStyle.normal);

              if (italic) ; else {
                this.addClass(item, config.fontStyle.italic);
              }
            }

            if (action === 'textUnderline') {
              let underline = false;

              if (this.hasClass(item, config.textDecoration.underline)) {
                underline = true;
              }

              this.removeClass(item, config.textDecoration.linethrough);
              this.removeClass(item, config.textDecoration.underline);
              this.removeClass(item, config.textDecoration.normal);

              if (underline) ; else {
                this.addClass(item, config.textDecoration.underline);
              }
            }

            if (action === 'textLinethrough') {
              let linethrough = false;

              if (this.hasClass(item, config.textDecoration.linethrough)) {
                linethrough = true;
              }

              this.removeClass(item, config.textDecoration.underline);
              this.removeClass(item, config.textDecoration.linethrough);
              this.removeClass(item, config.textDecoration.normal);

              if (linethrough) ; else {
                this.addClass(item, config.textDecoration.linethrough);
              }
            }

            if (action === 'textTransform') {
              let val = '';

              if (this.hasClass(item, config.textTransform.uppercase)) {
                val = 'uppercase';
              } else if (this.hasClass(item, config.textTransform.lowercase)) {
                val = 'lowercase';
              } else if (this.hasClass(item, config.textTransform.capitalize)) {
                val = 'capitalize';
              } else if (this.hasClass(item, config.textTransform.normal)) {
                val = 'normal';
              }

              this.removeClass(item, config.textTransform.uppercase);
              this.removeClass(item, config.textTransform.lowercase);
              this.removeClass(item, config.textTransform.capitalize);
              this.removeClass(item, config.textTransform.normal);

              if (val === '') {
                this.addClass(item, config.textTransform.uppercase);
              } else if (val === 'uppercase') {
                this.addClass(item, config.textTransform.lowercase);
              } else if (val === 'lowercase') {
                this.addClass(item, config.textTransform.capitalize);
              } else if (val === 'capitalize') ; else if (val === 'normal') {
                this.addClass(item, config.textTransform.uppercase);
              }
            }

            if (action === 'clean') {
              item.className = '';
              item.style.cssText = '';
            }

            if (action === 'extend') {
              let toggle = false;

              if (this.hasClass(item, config.extend[value])) {
                toggle = true;
              }

              if (toggle) {
                this.removeClass(item, config.extend[value]);
              } else {
                this.addClass(item, config.extend[value]);
              }
            }

            this.cleanChildrenToggle(action, value, item, config);
            this.cleanUnusedSpan(item); // REVIEW
          }
        }
      });
      if (multi) return;

      if (sameSelection || selection.toString().trim() === '') {
        this.updateSelectionToggle(action, value, config, container);
      } else {
        this.replaceSelectionToggle(action, value, config, selection, container);
      }
    }
  }

  updateSelectionToggle(action, value, config, container) {
    // container.setAttribute('data-dummy','1');
    if (action === 'fontWeight') {
      let bold = false;

      if (this.hasClass(container, config.fontWeight.defaultBold)) {
        bold = true;
      }

      const num = parseInt(window.getComputedStyle(container, null).getPropertyValue('font-weight'));
      if (num >= 500) bold = true;
      this.removeClass(container, config.fontWeight.thin);
      this.removeClass(container, config.fontWeight.extralight);
      this.removeClass(container, config.fontWeight.light);
      this.removeClass(container, config.fontWeight.normal);
      this.removeClass(container, config.fontWeight.medium);
      this.removeClass(container, config.fontWeight.semibold);
      this.removeClass(container, config.fontWeight.bold);
      this.removeClass(container, config.fontWeight.extrabold);
      this.removeClass(container, config.fontWeight.black);

      if (bold) {
        // A must to keep selection no change, otherwise, removing class & then span makes the selection changes (to check, click B/bold 3 times to toggle. The third will not work)
        this.addClass(container, config.fontWeight.defaultNormal);
      } else {
        this.addClass(container, config.fontWeight.defaultBold);
      }
    }

    if (action === 'fontStyle') {
      let italic = false;

      if (this.hasClass(container, config.fontStyle.italic)) {
        italic = true;
      }

      this.removeClass(container, config.fontStyle.italic);
      this.removeClass(container, config.fontStyle.normal);

      if (italic) {
        this.addClass(container, config.fontStyle.normal);
      } else {
        this.addClass(container, config.fontStyle.italic);
      }
    }

    if (action === 'textUnderline') {
      let underline = false;

      if (this.hasClass(container, config.textDecoration.underline)) {
        underline = true;
      }

      this.removeClass(container, config.textDecoration.underline);
      this.removeClass(container, config.textDecoration.linethrough);
      this.removeClass(container, config.textDecoration.normal);

      if (underline) {
        this.addClass(container, config.textDecoration.normal);
      } else {
        this.addClass(container, config.textDecoration.underline);
      }
    }

    if (action === 'textLinethrough') {
      let linethrough = false;

      if (this.hasClass(container, config.textDecoration.linethrough)) {
        linethrough = true;
      }

      this.removeClass(container, config.textDecoration.underline);
      this.removeClass(container, config.textDecoration.linethrough);
      this.removeClass(container, config.textDecoration.normal);

      if (linethrough) {
        this.addClass(container, config.textDecoration.normal);
      } else {
        this.addClass(container, config.textDecoration.linethrough);
      }
    }

    if (action === 'textTransform') {
      let val = '';

      if (this.hasClass(container, config.textTransform.uppercase)) {
        val = 'uppercase';
      } else if (this.hasClass(container, config.textTransform.lowercase)) {
        val = 'lowercase';
      } else if (this.hasClass(container, config.textTransform.capitalize)) {
        val = 'capitalize';
      } else if (this.hasClass(container, config.textTransform.normal)) {
        val = 'normal';
      }

      this.removeClass(container, config.textTransform.uppercase);
      this.removeClass(container, config.textTransform.lowercase);
      this.removeClass(container, config.textTransform.capitalize);
      this.removeClass(container, config.textTransform.normal);

      if (val === '') {
        this.addClass(container, config.textTransform.uppercase);
      } else if (val === 'uppercase') {
        this.addClass(container, config.textTransform.lowercase);
      } else if (val === 'lowercase') {
        this.addClass(container, config.textTransform.capitalize);
      } else if (val === 'capitalize') {
        this.addClass(container, config.textTransform.normal);
      } else if (val === 'normal') {
        this.addClass(container, config.textTransform.uppercase);
      }
    }

    if (action === 'clean') {
      container.className = '';
      container.style.cssText = '';
    }

    if (action === 'extend') {
      let toggle = false;

      if (this.hasClass(container, config.extend[value])) {
        toggle = true;
      }

      if (toggle) {
        this.removeClass(container, config.extend[value]);
      } else {
        this.addClass(container, config.extend[value]);
      }
    }

    this.cleanChildrenToggle(action, value, container, config);
    this.cleanUnusedSpan(container); // REVIEW

    return container;
  }

  replaceSelectionToggle(action, value, config, selection, container) {
    const range = selection.getRangeAt(0);
    const fragment = range.extractContents();
    const span = document.createElement('span');

    if (action === 'fontWeight') {
      let bold = false;

      if (container.closest(`.${config.fontWeight.defaultBold}`)) {
        bold = true;

        if (this.hasClass(container, config.fontWeight.defaultNormal)) {
          bold = false;
        }
      }

      const num = parseInt(window.getComputedStyle(container, null).getPropertyValue('font-weight'));
      if (num >= 500) bold = true;

      if (bold) {
        this.addClass(span, config.fontWeight.defaultNormal);
      } else {
        this.addClass(span, config.fontWeight.defaultBold);
      }
    }

    if (action === 'fontStyle') {
      let italic = false;

      if (container.closest(`.${config.fontStyle.italic}`)) {
        italic = true;

        if (this.hasClass(container, config.fontStyle.normal)) {
          italic = false;
        }
      }

      if (italic) {
        this.addClass(span, config.fontStyle.normal);
      } else {
        this.addClass(span, config.fontStyle.italic);
      }
    }

    if (action === 'textUnderline') {
      let underline = false;

      if (container.closest(`.${config.textDecoration.underline}`)) {
        underline = true;
      }

      if (underline) ; else {
        this.addClass(span, config.textDecoration.underline);
      }
    }

    if (action === 'textLinethrough') {
      let linethrough = false;

      if (container.closest(`.${config.textDecoration.linethrough}`)) {
        linethrough = true;
      }

      if (linethrough) ; else {
        this.addClass(span, config.textDecoration.linethrough);
      }
    }

    if (action === 'textTransform') {
      let val = '';

      if (container.closest(`.${config.textTransform.uppercase}`)) {
        val = 'uppercase';
      } else if (container.closest(`.${config.textTransform.lowercase}`)) {
        val = 'lowercase';
      } else if (container.closest(`.${config.textTransform.capitalize}`)) {
        val = 'capitalize';
      } else if (container.closest(`.${config.textTransform.normal}`)) {
        val = 'normal';
      }

      if (val === '') {
        this.addClass(span, config.textTransform.uppercase);
      } else if (val === 'uppercase') {
        this.addClass(span, config.textTransform.lowercase);
      } else if (val === 'lowercase') {
        this.addClass(span, config.textTransform.capitalize);
      } else if (val === 'capitalize') {
        this.addClass(span, config.textTransform.normal);
      } else if (val === 'normal') {
        this.addClass(span, config.textTransform.uppercase);
      }
    }

    if (action === 'extend') {
      let exist = false;

      if (container.closest(`.${config.extend[value]}`)) {
        exist = true;
      }

      if (exist) ; else {
        this.addClass(span, config.extend[value]);
      }
    }

    span.appendChild(fragment);
    this.cleanChildrenToggle(action, value, span, config);
    this.cleanUnusedSpan(span); // REVIEW

    range.insertNode(span);
    selection.selectAllChildren(span);
    return span;
  }

  cleanChildrenToggle(action, value, span, config) {
    if (!span.hasChildNodes()) {
      return;
    }

    if (action === 'fontWeight') {
      Array.from(span.children).map(element => {
        this.removeClass(element, config.fontWeight.thin);
        this.removeClass(element, config.fontWeight.extralight);
        this.removeClass(element, config.fontWeight.light);
        this.removeClass(element, config.fontWeight.normal);
        this.removeClass(element, config.fontWeight.medium);
        this.removeClass(element, config.fontWeight.semibold);
        this.removeClass(element, config.fontWeight.bold);
        this.removeClass(element, config.fontWeight.extrabold);
        this.removeClass(element, config.fontWeight.black);
      });
    }

    if (action === 'fontStyle') {
      Array.from(span.children).map(element => {
        this.removeClass(element, config.fontStyle.italic);
        this.removeClass(element, config.fontStyle.normal);
      });
    }

    if (action === 'textUnderline') {
      Array.from(span.children).map(element => {
        this.removeClass(element, config.textDecoration.underline);
        this.removeClass(element, config.textDecoration.linethrough);
        this.removeClass(element, config.textDecoration.normal);
      });
    }

    if (action === 'textLinethrough') {
      Array.from(span.children).map(element => {
        this.removeClass(element, config.textDecoration.underline);
        this.removeClass(element, config.textDecoration.linethrough);
        this.removeClass(element, config.textDecoration.normal);
      });
    }

    if (action === 'textTransform') {
      Array.from(span.children).map(element => {
        this.removeClass(element, config.textTransform.uppercase);
        this.removeClass(element, config.textTransform.lowercase);
        this.removeClass(element, config.textTransform.capitalize);
        this.removeClass(element, config.textTransform.normal);
      });
    }

    if (action === 'clean') {
      Array.from(span.children).map(element => {
        element.className = '';
        element.style.cssText = '';
      });
    }

    if (action === 'extend') {
      Array.from(span.children).map(element => {
        this.removeClass(element, config.extend[value]);
      });
    } // Deeper childrens


    const cleanChildrenChildren = Array.from(span.children).map(element => {
      return this.cleanChildrenToggle(action, value, element, config);
    });

    if (!cleanChildrenChildren || cleanChildrenChildren.length <= 0) {
      return;
    }
  }

  getSelection() {
    if (this.builder.win && this.builder.win.getSelection) {
      return this.builder.win.getSelection();
    } else if (this.builder.doc && this.builder.doc.getSelection) {
      return this.builder.doc.getSelection();
    } else if (this.builder.doc && this.builder.doc.selection) {
      return this.builder.doc.selection.createRange().text;
    }

    return null;
  }

  cleanChildren(action, span) {
    if (!span.hasChildNodes()) {
      return;
    } // Direct children with same style rule


    const children = Array.from(span.children).filter(element => {
      return element.style[action] !== undefined && element.style[action.style] !== '';
    });

    if (children && children.length > 0) {
      children.forEach(element => {
        element.style[action] = '';

        if (element.getAttribute('style') === '' || element.style === null) {
          element.removeAttribute('style');
        }
      });
    } // Deeper childrens


    const cleanChildrenChildren = Array.from(span.children).map(element => {
      return this.cleanChildren(action, element);
    });

    if (!cleanChildrenChildren || cleanChildrenChildren.length <= 0) {
      return;
    }
  }

  addCssClass(elm, className, classes) {
    this.removeCssClasses(elm, classes);
    this.addClass(elm, className);
  }

  removeCssClasses(elm, classes) {
    for (let i = 0; i < Object.keys(classes).length; i++) {
      const item = Object.values(classes)[i];
      this.removeClass(elm, item);
    }
  } // Clean HTML stuff


  removeEmptyStyle(area) {
    if (!area.hasChildNodes()) {
      return;
    }

    const children = Array.from(area.children).filter(element => {
      return element.getAttribute('style');
    });

    if (children && children.length > 0) {
      children.forEach(element => {
        if (element.getAttribute('style').trim() === '' || element.style === null) {
          element.removeAttribute('style');
        }
      });
    } // Deeper childrens


    const removeEmptyStyleChildren = Array.from(area.children).map(element => {
      return this.removeEmptyStyle(element);
    });

    if (!removeEmptyStyleChildren || removeEmptyStyleChildren.length <= 0) {
      return;
    }
  } // Font Size stuff


  cleanClassSize(span, className) {
    // used in rte.js 2991
    if (!span.hasChildNodes()) {
      return;
    } // Direct children with same style rule


    const children = Array.from(span.children).filter(element => {
      return this.hasClass(element, className);
    });

    if (children && children.length > 0) {
      children.forEach(element => {
        this.removeClass(element, className);
      });
    } // Deeper childrens


    const cleanClassSizeChildren = Array.from(span.children).map(element => {
      return this.cleanClassSize(element, className);
    });

    if (!cleanClassSizeChildren || cleanClassSizeChildren.length <= 0) {
      return;
    }
  } // Remove Format stuff


  removeFormat() {
    const selection = this.getSelection();
    if (!selection) return;
    const anchorNode = selection.anchorNode;
    if (!anchorNode) return;
    const container = anchorNode.nodeType !== Node.TEXT_NODE && anchorNode.nodeType !== Node.COMMENT_NODE ? anchorNode : anchorNode.parentElement;
    const sameSelection = container && container.innerText === selection.toString();

    if (sameSelection || selection.toString().trim() === '') {
      this.cleanElement(container);
    } else {
      this.cleanSelection(selection);
    }
  }

  cleanElement(elm) {
    elm.style.fontWeight = '';
    elm.style.fontStyle = '';
    elm.style.textDecoration = '';
    elm.style.textTransform = '';
    elm.style.fontFamily = '';
    elm.style.fontSize = '';
    elm.style.color = '';
    elm.style.backgroundColor = '';
    elm.classList.forEach(item => {
      if (item.indexOf('size-') !== -1) {
        this.removeClass(elm, item);
      }
    });
    let elms = elm.querySelectorAll('*');
    elms.forEach(elm => {
      elm.style.fontWeight = '';
      elm.style.fontStyle = '';
      elm.style.textDecoration = '';
      elm.style.textTransform = '';
      elm.style.fontFamily = '';
      elm.style.fontSize = '';
      elm.style.color = '';
      elm.style.backgroundColor = '';
      elm.classList.forEach(item => {
        if (item.indexOf('size-') !== -1) {
          this.removeClass(elm, item);
        }
      });
    });
    this.cleanUnusedSpan(elm); // REVIEW (causes lost selection)
  }

  cleanSelection(selection) {
    const range = selection.getRangeAt(0);
    const fragment = range.extractContents();
    const span = document.createElement('span');
    span.appendChild(fragment);
    range.insertNode(span);
    selection.selectAllChildren(span);
    this.cleanElement(span);
  }
  /*
  cleanUnusedSpan(span) {
       if (!span.hasChildNodes()) {
          return;
      }
       // Direct children with no style
      const children = Array.from(span.children).filter((element) => {
          if (element.getAttribute('style') === '' || element.style === null) {
              element.removeAttribute('style');
          }
          let unused = ( element.attributes.length === 0 || (element.attributes.length === 1 && element.hasAttribute('data-keep')));
          return (element.nodeName.toLowerCase() === 'span' && unused);
      });
       if (children && children.length > 0) {
          children.forEach((element) => { 
              const text = document.createTextNode(element.textContent);
              element.parentElement.replaceChild(text, element);
          });
          return;
      }
       // Deeper childrens
      const cleanUnusedSpanChildren = Array.from(span.children).map((element) => {
          return this.cleanUnusedSpan(element);
      });
       if (!cleanUnusedSpanChildren || cleanUnusedSpanChildren.length <= 0) {
          return;
      }
  }
  */


  cleanUnusedSpan(area) {
    // for specific element only (during execCommand), 
    // not for the entire ContentBuilder container, 
    // because there can be custom blocks
    let spans = area.querySelectorAll('span');
    const filter = Array.prototype.filter;
    let children = filter.call(spans, element => {
      return element.attributes.length === 0;
    }); // Remove empty spans

    if (children && children.length > 0) {
      children.forEach(element => {
        element.outerHTML = element.innerHTML;
      });
    } // Remove spans which have empty content


    spans = area.querySelectorAll('span');
    filter.call(spans, element => {
      if (element.innerHTML === '') {
        element.parentNode.removeChild(element);
      }
    }); // Recheck

    spans = area.querySelectorAll('span');
    children = filter.call(spans, element => {
      return element.attributes.length === 0;
    });

    if (children && children.length > 0) {
      this.cleanUnusedSpan(area);
    } else {
      return;
    }
  }

  contentReformat(area, cssClasses) {
    this.replaceTag(area, 'b', cssClasses.fontWeight.defaultBold);
    this.replaceTag(area, 'i', cssClasses.fontStyle.italic);
    this.replaceTag(area, 'u', cssClasses.textDecoration.underline);
    this.replaceTag(area, 'strike', cssClasses.textDecoration.linethrough);
    this.replaceInline(area, 'font-weight', 'normal', cssClasses.fontWeight.defaultBold);
    this.replaceInline(area, 'font-weight', 'bold', cssClasses.fontWeight.defaultBold);
    this.replaceInline(area, 'font-weight', '600', cssClasses.fontWeight.semibold);
    this.replaceInline(area, 'text-align', 'left', cssClasses.textAlign.left);
    this.replaceInline(area, 'text-align', 'right', cssClasses.textAlign.right);
    this.replaceInline(area, 'text-align', 'center', cssClasses.textAlign.center);
    this.replaceInline(area, 'text-align', 'justify', cssClasses.textAlign.justify);
    this.replaceInline(area, 'display', 'flex', cssClasses.display.flex);
    this.replaceInline(area, 'justiify-content', 'flex-start', cssClasses.justifyContent.start);
    this.replaceInline(area, 'justiify-content', 'flex-end', cssClasses.justifyContent.end);
    this.replaceInline(area, 'justiify-content', 'center', cssClasses.justifyContent.center);
    this.replaceInline(area, 'justiify-content', 'space-between', cssClasses.justifyContent.between);
    this.replaceInline(area, 'flex-direction', 'column', cssClasses.flexDirection.column);
    this.replaceInline(area, 'flex-direction', 'row', cssClasses.flexDirection.row);
    this.replaceInline(area, 'align-items', 'flex-start', cssClasses.alignItems.start);
    this.replaceInline(area, 'align-items', 'flex-end', cssClasses.alignItems.end);
    this.replaceInline(area, 'align-items', 'center', cssClasses.alignItems.center);
    this.replaceInline(area, 'text-transform', 'uppercase', cssClasses.textTransform.uppercase);
    this.replaceInline(area, 'text-transform', 'lowercase', cssClasses.textTransform.lowercase);
    this.replaceInline(area, 'text-transform', 'none', cssClasses.textTransform.normal);
    this.replaceInline(area, 'line-height', '1', cssClasses.leading.leading_10);
    this.replaceInline(area, 'line-height', '1.1', cssClasses.leading.leading_11);
    this.replaceInline(area, 'line-height', '1.2', cssClasses.leading.leading_12);
    this.replaceInline(area, 'line-height', '1.25', cssClasses.leading.leading_125);
    this.replaceInline(area, 'line-height', '1.3', cssClasses.leading.leading_13);
    this.replaceInline(area, 'line-height', '1.375', cssClasses.leading.leading_1375);
    this.replaceInline(area, 'line-height', '1.4', cssClasses.leading.leading_14);
    this.replaceInline(area, 'line-height', '1.5', cssClasses.leading.leading_15);
    this.replaceInline(area, 'line-height', '1.6', cssClasses.leading.leading_16);
    this.replaceInline(area, 'line-height', '1.625', cssClasses.leading.leading_1625);
    this.replaceInline(area, 'line-height', '1.7', cssClasses.leading.leading_17);
    this.replaceInline(area, 'line-height', '1.8', cssClasses.leading.leading_18);
    this.replaceInline(area, 'line-height', '1.9', cssClasses.leading.leading_19);
    this.replaceInline(area, 'line-height', '2', cssClasses.leading.leading_20);
    this.replaceInline(area, 'line-height', '2.1', cssClasses.leading.leading_21);
    this.replaceInline(area, 'line-height', '2.2', cssClasses.leading.leading_22);
  }

  replaceTag(area, tag, className) {
    let elms = area.querySelectorAll(tag);
    const filter = Array.prototype.filter;
    let children = filter.call(elms, element => {
      return element.attributes.length === 0;
    });

    if (children && children.length > 0) {
      children.forEach(element => {
        const span = document.createElement('span');
        span.setAttribute('class', className);
        span.innerHTML = element.innerHTML;
        element.outerHTML = span.outerHTML;
      });
    }
  }

  replaceInline(builder, css, value, className) {
    let elms = builder.querySelectorAll('*');
    elms.forEach(elm => {
      if (css === 'justiify-content') {
        /*
        For some reason, cannot use elm.style[css] for 'justify-content'
        */
        if (elm.style.justifyContent === `${value}`) {
          this.addClass(elm, className);
          elm.style.justifyContent = '';
        }
      } else if (elm.style[css] === `${value}`) {
        this.addClass(elm, className);
        elm.style[css] = '';
      }
    });
  }

}

var js$1 = {exports: {}};

var src = {};

var javascript = {exports: {}};

var beautifier$2 = {};

var output = {};

/*jshint node:true */

function OutputLine(parent) {
  this.__parent = parent;
  this.__character_count = 0;
  // use indent_count as a marker for this.__lines that have preserved indentation
  this.__indent_count = -1;
  this.__alignment_count = 0;
  this.__wrap_point_index = 0;
  this.__wrap_point_character_count = 0;
  this.__wrap_point_indent_count = -1;
  this.__wrap_point_alignment_count = 0;

  this.__items = [];
}

OutputLine.prototype.clone_empty = function() {
  var line = new OutputLine(this.__parent);
  line.set_indent(this.__indent_count, this.__alignment_count);
  return line;
};

OutputLine.prototype.item = function(index) {
  if (index < 0) {
    return this.__items[this.__items.length + index];
  } else {
    return this.__items[index];
  }
};

OutputLine.prototype.has_match = function(pattern) {
  for (var lastCheckedOutput = this.__items.length - 1; lastCheckedOutput >= 0; lastCheckedOutput--) {
    if (this.__items[lastCheckedOutput].match(pattern)) {
      return true;
    }
  }
  return false;
};

OutputLine.prototype.set_indent = function(indent, alignment) {
  if (this.is_empty()) {
    this.__indent_count = indent || 0;
    this.__alignment_count = alignment || 0;
    this.__character_count = this.__parent.get_indent_size(this.__indent_count, this.__alignment_count);
  }
};

OutputLine.prototype._set_wrap_point = function() {
  if (this.__parent.wrap_line_length) {
    this.__wrap_point_index = this.__items.length;
    this.__wrap_point_character_count = this.__character_count;
    this.__wrap_point_indent_count = this.__parent.next_line.__indent_count;
    this.__wrap_point_alignment_count = this.__parent.next_line.__alignment_count;
  }
};

OutputLine.prototype._should_wrap = function() {
  return this.__wrap_point_index &&
    this.__character_count > this.__parent.wrap_line_length &&
    this.__wrap_point_character_count > this.__parent.next_line.__character_count;
};

OutputLine.prototype._allow_wrap = function() {
  if (this._should_wrap()) {
    this.__parent.add_new_line();
    var next = this.__parent.current_line;
    next.set_indent(this.__wrap_point_indent_count, this.__wrap_point_alignment_count);
    next.__items = this.__items.slice(this.__wrap_point_index);
    this.__items = this.__items.slice(0, this.__wrap_point_index);

    next.__character_count += this.__character_count - this.__wrap_point_character_count;
    this.__character_count = this.__wrap_point_character_count;

    if (next.__items[0] === " ") {
      next.__items.splice(0, 1);
      next.__character_count -= 1;
    }
    return true;
  }
  return false;
};

OutputLine.prototype.is_empty = function() {
  return this.__items.length === 0;
};

OutputLine.prototype.last = function() {
  if (!this.is_empty()) {
    return this.__items[this.__items.length - 1];
  } else {
    return null;
  }
};

OutputLine.prototype.push = function(item) {
  this.__items.push(item);
  var last_newline_index = item.lastIndexOf('\n');
  if (last_newline_index !== -1) {
    this.__character_count = item.length - last_newline_index;
  } else {
    this.__character_count += item.length;
  }
};

OutputLine.prototype.pop = function() {
  var item = null;
  if (!this.is_empty()) {
    item = this.__items.pop();
    this.__character_count -= item.length;
  }
  return item;
};


OutputLine.prototype._remove_indent = function() {
  if (this.__indent_count > 0) {
    this.__indent_count -= 1;
    this.__character_count -= this.__parent.indent_size;
  }
};

OutputLine.prototype._remove_wrap_indent = function() {
  if (this.__wrap_point_indent_count > 0) {
    this.__wrap_point_indent_count -= 1;
  }
};
OutputLine.prototype.trim = function() {
  while (this.last() === ' ') {
    this.__items.pop();
    this.__character_count -= 1;
  }
};

OutputLine.prototype.toString = function() {
  var result = '';
  if (this.is_empty()) {
    if (this.__parent.indent_empty_lines) {
      result = this.__parent.get_indent_string(this.__indent_count);
    }
  } else {
    result = this.__parent.get_indent_string(this.__indent_count, this.__alignment_count);
    result += this.__items.join('');
  }
  return result;
};

function IndentStringCache(options, baseIndentString) {
  this.__cache = [''];
  this.__indent_size = options.indent_size;
  this.__indent_string = options.indent_char;
  if (!options.indent_with_tabs) {
    this.__indent_string = new Array(options.indent_size + 1).join(options.indent_char);
  }

  // Set to null to continue support for auto detection of base indent
  baseIndentString = baseIndentString || '';
  if (options.indent_level > 0) {
    baseIndentString = new Array(options.indent_level + 1).join(this.__indent_string);
  }

  this.__base_string = baseIndentString;
  this.__base_string_length = baseIndentString.length;
}

IndentStringCache.prototype.get_indent_size = function(indent, column) {
  var result = this.__base_string_length;
  column = column || 0;
  if (indent < 0) {
    result = 0;
  }
  result += indent * this.__indent_size;
  result += column;
  return result;
};

IndentStringCache.prototype.get_indent_string = function(indent_level, column) {
  var result = this.__base_string;
  column = column || 0;
  if (indent_level < 0) {
    indent_level = 0;
    result = '';
  }
  column += indent_level * this.__indent_size;
  this.__ensure_cache(column);
  result += this.__cache[column];
  return result;
};

IndentStringCache.prototype.__ensure_cache = function(column) {
  while (column >= this.__cache.length) {
    this.__add_column();
  }
};

IndentStringCache.prototype.__add_column = function() {
  var column = this.__cache.length;
  var indent = 0;
  var result = '';
  if (this.__indent_size && column >= this.__indent_size) {
    indent = Math.floor(column / this.__indent_size);
    column -= indent * this.__indent_size;
    result = new Array(indent + 1).join(this.__indent_string);
  }
  if (column) {
    result += new Array(column + 1).join(' ');
  }

  this.__cache.push(result);
};

function Output$3(options, baseIndentString) {
  this.__indent_cache = new IndentStringCache(options, baseIndentString);
  this.raw = false;
  this._end_with_newline = options.end_with_newline;
  this.indent_size = options.indent_size;
  this.wrap_line_length = options.wrap_line_length;
  this.indent_empty_lines = options.indent_empty_lines;
  this.__lines = [];
  this.previous_line = null;
  this.current_line = null;
  this.next_line = new OutputLine(this);
  this.space_before_token = false;
  this.non_breaking_space = false;
  this.previous_token_wrapped = false;
  // initialize
  this.__add_outputline();
}

Output$3.prototype.__add_outputline = function() {
  this.previous_line = this.current_line;
  this.current_line = this.next_line.clone_empty();
  this.__lines.push(this.current_line);
};

Output$3.prototype.get_line_number = function() {
  return this.__lines.length;
};

Output$3.prototype.get_indent_string = function(indent, column) {
  return this.__indent_cache.get_indent_string(indent, column);
};

Output$3.prototype.get_indent_size = function(indent, column) {
  return this.__indent_cache.get_indent_size(indent, column);
};

Output$3.prototype.is_empty = function() {
  return !this.previous_line && this.current_line.is_empty();
};

Output$3.prototype.add_new_line = function(force_newline) {
  // never newline at the start of file
  // otherwise, newline only if we didn't just add one or we're forced
  if (this.is_empty() ||
    (!force_newline && this.just_added_newline())) {
    return false;
  }

  // if raw output is enabled, don't print additional newlines,
  // but still return True as though you had
  if (!this.raw) {
    this.__add_outputline();
  }
  return true;
};

Output$3.prototype.get_code = function(eol) {
  this.trim(true);

  // handle some edge cases where the last tokens
  // has text that ends with newline(s)
  var last_item = this.current_line.pop();
  if (last_item) {
    if (last_item[last_item.length - 1] === '\n') {
      last_item = last_item.replace(/\n+$/g, '');
    }
    this.current_line.push(last_item);
  }

  if (this._end_with_newline) {
    this.__add_outputline();
  }

  var sweet_code = this.__lines.join('\n');

  if (eol !== '\n') {
    sweet_code = sweet_code.replace(/[\n]/g, eol);
  }
  return sweet_code;
};

Output$3.prototype.set_wrap_point = function() {
  this.current_line._set_wrap_point();
};

Output$3.prototype.set_indent = function(indent, alignment) {
  indent = indent || 0;
  alignment = alignment || 0;

  // Next line stores alignment values
  this.next_line.set_indent(indent, alignment);

  // Never indent your first output indent at the start of the file
  if (this.__lines.length > 1) {
    this.current_line.set_indent(indent, alignment);
    return true;
  }

  this.current_line.set_indent();
  return false;
};

Output$3.prototype.add_raw_token = function(token) {
  for (var x = 0; x < token.newlines; x++) {
    this.__add_outputline();
  }
  this.current_line.set_indent(-1);
  this.current_line.push(token.whitespace_before);
  this.current_line.push(token.text);
  this.space_before_token = false;
  this.non_breaking_space = false;
  this.previous_token_wrapped = false;
};

Output$3.prototype.add_token = function(printable_token) {
  this.__add_space_before_token();
  this.current_line.push(printable_token);
  this.space_before_token = false;
  this.non_breaking_space = false;
  this.previous_token_wrapped = this.current_line._allow_wrap();
};

Output$3.prototype.__add_space_before_token = function() {
  if (this.space_before_token && !this.just_added_newline()) {
    if (!this.non_breaking_space) {
      this.set_wrap_point();
    }
    this.current_line.push(' ');
  }
};

Output$3.prototype.remove_indent = function(index) {
  var output_length = this.__lines.length;
  while (index < output_length) {
    this.__lines[index]._remove_indent();
    index++;
  }
  this.current_line._remove_wrap_indent();
};

Output$3.prototype.trim = function(eat_newlines) {
  eat_newlines = (eat_newlines === undefined) ? false : eat_newlines;

  this.current_line.trim();

  while (eat_newlines && this.__lines.length > 1 &&
    this.current_line.is_empty()) {
    this.__lines.pop();
    this.current_line = this.__lines[this.__lines.length - 1];
    this.current_line.trim();
  }

  this.previous_line = this.__lines.length > 1 ?
    this.__lines[this.__lines.length - 2] : null;
};

Output$3.prototype.just_added_newline = function() {
  return this.current_line.is_empty();
};

Output$3.prototype.just_added_blankline = function() {
  return this.is_empty() ||
    (this.current_line.is_empty() && this.previous_line.is_empty());
};

Output$3.prototype.ensure_empty_line_above = function(starts_with, ends_with) {
  var index = this.__lines.length - 2;
  while (index >= 0) {
    var potentialEmptyLine = this.__lines[index];
    if (potentialEmptyLine.is_empty()) {
      break;
    } else if (potentialEmptyLine.item(0).indexOf(starts_with) !== 0 &&
      potentialEmptyLine.item(-1) !== ends_with) {
      this.__lines.splice(index + 1, 0, new OutputLine(this));
      this.previous_line = this.__lines[this.__lines.length - 2];
      break;
    }
    index--;
  }
};

output.Output = Output$3;

var token = {};

/*jshint node:true */

function Token$2(type, text, newlines, whitespace_before) {
  this.type = type;
  this.text = text;

  // comments_before are
  // comments that have a new line before them
  // and may or may not have a newline after
  // this is a set of comments before
  this.comments_before = null; /* inline comment*/


  // this.comments_after =  new TokenStream(); // no new line before and newline after
  this.newlines = newlines || 0;
  this.whitespace_before = whitespace_before || '';
  this.parent = null;
  this.next = null;
  this.previous = null;
  this.opened = null;
  this.closed = null;
  this.directives = null;
}


token.Token = Token$2;

var acorn$2 = {};

/* jshint node: true, curly: false */

(function (exports) {

// acorn used char codes to squeeze the last bit of performance out
// Beautifier is okay without that, so we're using regex
// permit # (23), $ (36), and @ (64). @ is used in ES7 decorators.
// 65 through 91 are uppercase letters.
// permit _ (95).
// 97 through 123 are lowercase letters.
var baseASCIIidentifierStartChars = "\\x23\\x24\\x40\\x41-\\x5a\\x5f\\x61-\\x7a";

// inside an identifier @ is not allowed but 0-9 are.
var baseASCIIidentifierChars = "\\x24\\x30-\\x39\\x41-\\x5a\\x5f\\x61-\\x7a";

// Big ugly regular expressions that match characters in the
// whitespace, identifier, and identifier-start categories. These
// are only applied when a character is found to actually have a
// code point above 128.
var nonASCIIidentifierStartChars = "\\xaa\\xb5\\xba\\xc0-\\xd6\\xd8-\\xf6\\xf8-\\u02c1\\u02c6-\\u02d1\\u02e0-\\u02e4\\u02ec\\u02ee\\u0370-\\u0374\\u0376\\u0377\\u037a-\\u037d\\u0386\\u0388-\\u038a\\u038c\\u038e-\\u03a1\\u03a3-\\u03f5\\u03f7-\\u0481\\u048a-\\u0527\\u0531-\\u0556\\u0559\\u0561-\\u0587\\u05d0-\\u05ea\\u05f0-\\u05f2\\u0620-\\u064a\\u066e\\u066f\\u0671-\\u06d3\\u06d5\\u06e5\\u06e6\\u06ee\\u06ef\\u06fa-\\u06fc\\u06ff\\u0710\\u0712-\\u072f\\u074d-\\u07a5\\u07b1\\u07ca-\\u07ea\\u07f4\\u07f5\\u07fa\\u0800-\\u0815\\u081a\\u0824\\u0828\\u0840-\\u0858\\u08a0\\u08a2-\\u08ac\\u0904-\\u0939\\u093d\\u0950\\u0958-\\u0961\\u0971-\\u0977\\u0979-\\u097f\\u0985-\\u098c\\u098f\\u0990\\u0993-\\u09a8\\u09aa-\\u09b0\\u09b2\\u09b6-\\u09b9\\u09bd\\u09ce\\u09dc\\u09dd\\u09df-\\u09e1\\u09f0\\u09f1\\u0a05-\\u0a0a\\u0a0f\\u0a10\\u0a13-\\u0a28\\u0a2a-\\u0a30\\u0a32\\u0a33\\u0a35\\u0a36\\u0a38\\u0a39\\u0a59-\\u0a5c\\u0a5e\\u0a72-\\u0a74\\u0a85-\\u0a8d\\u0a8f-\\u0a91\\u0a93-\\u0aa8\\u0aaa-\\u0ab0\\u0ab2\\u0ab3\\u0ab5-\\u0ab9\\u0abd\\u0ad0\\u0ae0\\u0ae1\\u0b05-\\u0b0c\\u0b0f\\u0b10\\u0b13-\\u0b28\\u0b2a-\\u0b30\\u0b32\\u0b33\\u0b35-\\u0b39\\u0b3d\\u0b5c\\u0b5d\\u0b5f-\\u0b61\\u0b71\\u0b83\\u0b85-\\u0b8a\\u0b8e-\\u0b90\\u0b92-\\u0b95\\u0b99\\u0b9a\\u0b9c\\u0b9e\\u0b9f\\u0ba3\\u0ba4\\u0ba8-\\u0baa\\u0bae-\\u0bb9\\u0bd0\\u0c05-\\u0c0c\\u0c0e-\\u0c10\\u0c12-\\u0c28\\u0c2a-\\u0c33\\u0c35-\\u0c39\\u0c3d\\u0c58\\u0c59\\u0c60\\u0c61\\u0c85-\\u0c8c\\u0c8e-\\u0c90\\u0c92-\\u0ca8\\u0caa-\\u0cb3\\u0cb5-\\u0cb9\\u0cbd\\u0cde\\u0ce0\\u0ce1\\u0cf1\\u0cf2\\u0d05-\\u0d0c\\u0d0e-\\u0d10\\u0d12-\\u0d3a\\u0d3d\\u0d4e\\u0d60\\u0d61\\u0d7a-\\u0d7f\\u0d85-\\u0d96\\u0d9a-\\u0db1\\u0db3-\\u0dbb\\u0dbd\\u0dc0-\\u0dc6\\u0e01-\\u0e30\\u0e32\\u0e33\\u0e40-\\u0e46\\u0e81\\u0e82\\u0e84\\u0e87\\u0e88\\u0e8a\\u0e8d\\u0e94-\\u0e97\\u0e99-\\u0e9f\\u0ea1-\\u0ea3\\u0ea5\\u0ea7\\u0eaa\\u0eab\\u0ead-\\u0eb0\\u0eb2\\u0eb3\\u0ebd\\u0ec0-\\u0ec4\\u0ec6\\u0edc-\\u0edf\\u0f00\\u0f40-\\u0f47\\u0f49-\\u0f6c\\u0f88-\\u0f8c\\u1000-\\u102a\\u103f\\u1050-\\u1055\\u105a-\\u105d\\u1061\\u1065\\u1066\\u106e-\\u1070\\u1075-\\u1081\\u108e\\u10a0-\\u10c5\\u10c7\\u10cd\\u10d0-\\u10fa\\u10fc-\\u1248\\u124a-\\u124d\\u1250-\\u1256\\u1258\\u125a-\\u125d\\u1260-\\u1288\\u128a-\\u128d\\u1290-\\u12b0\\u12b2-\\u12b5\\u12b8-\\u12be\\u12c0\\u12c2-\\u12c5\\u12c8-\\u12d6\\u12d8-\\u1310\\u1312-\\u1315\\u1318-\\u135a\\u1380-\\u138f\\u13a0-\\u13f4\\u1401-\\u166c\\u166f-\\u167f\\u1681-\\u169a\\u16a0-\\u16ea\\u16ee-\\u16f0\\u1700-\\u170c\\u170e-\\u1711\\u1720-\\u1731\\u1740-\\u1751\\u1760-\\u176c\\u176e-\\u1770\\u1780-\\u17b3\\u17d7\\u17dc\\u1820-\\u1877\\u1880-\\u18a8\\u18aa\\u18b0-\\u18f5\\u1900-\\u191c\\u1950-\\u196d\\u1970-\\u1974\\u1980-\\u19ab\\u19c1-\\u19c7\\u1a00-\\u1a16\\u1a20-\\u1a54\\u1aa7\\u1b05-\\u1b33\\u1b45-\\u1b4b\\u1b83-\\u1ba0\\u1bae\\u1baf\\u1bba-\\u1be5\\u1c00-\\u1c23\\u1c4d-\\u1c4f\\u1c5a-\\u1c7d\\u1ce9-\\u1cec\\u1cee-\\u1cf1\\u1cf5\\u1cf6\\u1d00-\\u1dbf\\u1e00-\\u1f15\\u1f18-\\u1f1d\\u1f20-\\u1f45\\u1f48-\\u1f4d\\u1f50-\\u1f57\\u1f59\\u1f5b\\u1f5d\\u1f5f-\\u1f7d\\u1f80-\\u1fb4\\u1fb6-\\u1fbc\\u1fbe\\u1fc2-\\u1fc4\\u1fc6-\\u1fcc\\u1fd0-\\u1fd3\\u1fd6-\\u1fdb\\u1fe0-\\u1fec\\u1ff2-\\u1ff4\\u1ff6-\\u1ffc\\u2071\\u207f\\u2090-\\u209c\\u2102\\u2107\\u210a-\\u2113\\u2115\\u2119-\\u211d\\u2124\\u2126\\u2128\\u212a-\\u212d\\u212f-\\u2139\\u213c-\\u213f\\u2145-\\u2149\\u214e\\u2160-\\u2188\\u2c00-\\u2c2e\\u2c30-\\u2c5e\\u2c60-\\u2ce4\\u2ceb-\\u2cee\\u2cf2\\u2cf3\\u2d00-\\u2d25\\u2d27\\u2d2d\\u2d30-\\u2d67\\u2d6f\\u2d80-\\u2d96\\u2da0-\\u2da6\\u2da8-\\u2dae\\u2db0-\\u2db6\\u2db8-\\u2dbe\\u2dc0-\\u2dc6\\u2dc8-\\u2dce\\u2dd0-\\u2dd6\\u2dd8-\\u2dde\\u2e2f\\u3005-\\u3007\\u3021-\\u3029\\u3031-\\u3035\\u3038-\\u303c\\u3041-\\u3096\\u309d-\\u309f\\u30a1-\\u30fa\\u30fc-\\u30ff\\u3105-\\u312d\\u3131-\\u318e\\u31a0-\\u31ba\\u31f0-\\u31ff\\u3400-\\u4db5\\u4e00-\\u9fcc\\ua000-\\ua48c\\ua4d0-\\ua4fd\\ua500-\\ua60c\\ua610-\\ua61f\\ua62a\\ua62b\\ua640-\\ua66e\\ua67f-\\ua697\\ua6a0-\\ua6ef\\ua717-\\ua71f\\ua722-\\ua788\\ua78b-\\ua78e\\ua790-\\ua793\\ua7a0-\\ua7aa\\ua7f8-\\ua801\\ua803-\\ua805\\ua807-\\ua80a\\ua80c-\\ua822\\ua840-\\ua873\\ua882-\\ua8b3\\ua8f2-\\ua8f7\\ua8fb\\ua90a-\\ua925\\ua930-\\ua946\\ua960-\\ua97c\\ua984-\\ua9b2\\ua9cf\\uaa00-\\uaa28\\uaa40-\\uaa42\\uaa44-\\uaa4b\\uaa60-\\uaa76\\uaa7a\\uaa80-\\uaaaf\\uaab1\\uaab5\\uaab6\\uaab9-\\uaabd\\uaac0\\uaac2\\uaadb-\\uaadd\\uaae0-\\uaaea\\uaaf2-\\uaaf4\\uab01-\\uab06\\uab09-\\uab0e\\uab11-\\uab16\\uab20-\\uab26\\uab28-\\uab2e\\uabc0-\\uabe2\\uac00-\\ud7a3\\ud7b0-\\ud7c6\\ud7cb-\\ud7fb\\uf900-\\ufa6d\\ufa70-\\ufad9\\ufb00-\\ufb06\\ufb13-\\ufb17\\ufb1d\\ufb1f-\\ufb28\\ufb2a-\\ufb36\\ufb38-\\ufb3c\\ufb3e\\ufb40\\ufb41\\ufb43\\ufb44\\ufb46-\\ufbb1\\ufbd3-\\ufd3d\\ufd50-\\ufd8f\\ufd92-\\ufdc7\\ufdf0-\\ufdfb\\ufe70-\\ufe74\\ufe76-\\ufefc\\uff21-\\uff3a\\uff41-\\uff5a\\uff66-\\uffbe\\uffc2-\\uffc7\\uffca-\\uffcf\\uffd2-\\uffd7\\uffda-\\uffdc";
var nonASCIIidentifierChars = "\\u0300-\\u036f\\u0483-\\u0487\\u0591-\\u05bd\\u05bf\\u05c1\\u05c2\\u05c4\\u05c5\\u05c7\\u0610-\\u061a\\u0620-\\u0649\\u0672-\\u06d3\\u06e7-\\u06e8\\u06fb-\\u06fc\\u0730-\\u074a\\u0800-\\u0814\\u081b-\\u0823\\u0825-\\u0827\\u0829-\\u082d\\u0840-\\u0857\\u08e4-\\u08fe\\u0900-\\u0903\\u093a-\\u093c\\u093e-\\u094f\\u0951-\\u0957\\u0962-\\u0963\\u0966-\\u096f\\u0981-\\u0983\\u09bc\\u09be-\\u09c4\\u09c7\\u09c8\\u09d7\\u09df-\\u09e0\\u0a01-\\u0a03\\u0a3c\\u0a3e-\\u0a42\\u0a47\\u0a48\\u0a4b-\\u0a4d\\u0a51\\u0a66-\\u0a71\\u0a75\\u0a81-\\u0a83\\u0abc\\u0abe-\\u0ac5\\u0ac7-\\u0ac9\\u0acb-\\u0acd\\u0ae2-\\u0ae3\\u0ae6-\\u0aef\\u0b01-\\u0b03\\u0b3c\\u0b3e-\\u0b44\\u0b47\\u0b48\\u0b4b-\\u0b4d\\u0b56\\u0b57\\u0b5f-\\u0b60\\u0b66-\\u0b6f\\u0b82\\u0bbe-\\u0bc2\\u0bc6-\\u0bc8\\u0bca-\\u0bcd\\u0bd7\\u0be6-\\u0bef\\u0c01-\\u0c03\\u0c46-\\u0c48\\u0c4a-\\u0c4d\\u0c55\\u0c56\\u0c62-\\u0c63\\u0c66-\\u0c6f\\u0c82\\u0c83\\u0cbc\\u0cbe-\\u0cc4\\u0cc6-\\u0cc8\\u0cca-\\u0ccd\\u0cd5\\u0cd6\\u0ce2-\\u0ce3\\u0ce6-\\u0cef\\u0d02\\u0d03\\u0d46-\\u0d48\\u0d57\\u0d62-\\u0d63\\u0d66-\\u0d6f\\u0d82\\u0d83\\u0dca\\u0dcf-\\u0dd4\\u0dd6\\u0dd8-\\u0ddf\\u0df2\\u0df3\\u0e34-\\u0e3a\\u0e40-\\u0e45\\u0e50-\\u0e59\\u0eb4-\\u0eb9\\u0ec8-\\u0ecd\\u0ed0-\\u0ed9\\u0f18\\u0f19\\u0f20-\\u0f29\\u0f35\\u0f37\\u0f39\\u0f41-\\u0f47\\u0f71-\\u0f84\\u0f86-\\u0f87\\u0f8d-\\u0f97\\u0f99-\\u0fbc\\u0fc6\\u1000-\\u1029\\u1040-\\u1049\\u1067-\\u106d\\u1071-\\u1074\\u1082-\\u108d\\u108f-\\u109d\\u135d-\\u135f\\u170e-\\u1710\\u1720-\\u1730\\u1740-\\u1750\\u1772\\u1773\\u1780-\\u17b2\\u17dd\\u17e0-\\u17e9\\u180b-\\u180d\\u1810-\\u1819\\u1920-\\u192b\\u1930-\\u193b\\u1951-\\u196d\\u19b0-\\u19c0\\u19c8-\\u19c9\\u19d0-\\u19d9\\u1a00-\\u1a15\\u1a20-\\u1a53\\u1a60-\\u1a7c\\u1a7f-\\u1a89\\u1a90-\\u1a99\\u1b46-\\u1b4b\\u1b50-\\u1b59\\u1b6b-\\u1b73\\u1bb0-\\u1bb9\\u1be6-\\u1bf3\\u1c00-\\u1c22\\u1c40-\\u1c49\\u1c5b-\\u1c7d\\u1cd0-\\u1cd2\\u1d00-\\u1dbe\\u1e01-\\u1f15\\u200c\\u200d\\u203f\\u2040\\u2054\\u20d0-\\u20dc\\u20e1\\u20e5-\\u20f0\\u2d81-\\u2d96\\u2de0-\\u2dff\\u3021-\\u3028\\u3099\\u309a\\ua640-\\ua66d\\ua674-\\ua67d\\ua69f\\ua6f0-\\ua6f1\\ua7f8-\\ua800\\ua806\\ua80b\\ua823-\\ua827\\ua880-\\ua881\\ua8b4-\\ua8c4\\ua8d0-\\ua8d9\\ua8f3-\\ua8f7\\ua900-\\ua909\\ua926-\\ua92d\\ua930-\\ua945\\ua980-\\ua983\\ua9b3-\\ua9c0\\uaa00-\\uaa27\\uaa40-\\uaa41\\uaa4c-\\uaa4d\\uaa50-\\uaa59\\uaa7b\\uaae0-\\uaae9\\uaaf2-\\uaaf3\\uabc0-\\uabe1\\uabec\\uabed\\uabf0-\\uabf9\\ufb20-\\ufb28\\ufe00-\\ufe0f\\ufe20-\\ufe26\\ufe33\\ufe34\\ufe4d-\\ufe4f\\uff10-\\uff19\\uff3f";
//var nonASCIIidentifierStart = new RegExp("[" + nonASCIIidentifierStartChars + "]");
//var nonASCIIidentifier = new RegExp("[" + nonASCIIidentifierStartChars + nonASCIIidentifierChars + "]");

var identifierStart = "(?:\\\\u[0-9a-fA-F]{4}|[" + baseASCIIidentifierStartChars + nonASCIIidentifierStartChars + "])";
var identifierChars = "(?:\\\\u[0-9a-fA-F]{4}|[" + baseASCIIidentifierChars + nonASCIIidentifierStartChars + nonASCIIidentifierChars + "])*";

exports.identifier = new RegExp(identifierStart + identifierChars, 'g');
exports.identifierStart = new RegExp(identifierStart);
exports.identifierMatch = new RegExp("(?:\\\\u[0-9a-fA-F]{4}|[" + baseASCIIidentifierChars + nonASCIIidentifierStartChars + nonASCIIidentifierChars + "])+");

// Whether a single character denotes a newline.

exports.newline = /[\n\r\u2028\u2029]/;

// Matches a whole line break (where CRLF is considered a single
// line break). Used to count lines.

// in javascript, these two differ
// in python they are the same, different methods are called on them
exports.lineBreak = new RegExp('\r\n|' + exports.newline.source);
exports.allLineBreaks = new RegExp(exports.lineBreak.source, 'g');
}(acorn$2));

var options$3 = {};

var options$2 = {};

/*jshint node:true */

function Options$9(options, merge_child_field) {
  this.raw_options = _mergeOpts(options, merge_child_field);

  // Support passing the source text back with no change
  this.disabled = this._get_boolean('disabled');

  this.eol = this._get_characters('eol', 'auto');
  this.end_with_newline = this._get_boolean('end_with_newline');
  this.indent_size = this._get_number('indent_size', 4);
  this.indent_char = this._get_characters('indent_char', ' ');
  this.indent_level = this._get_number('indent_level');

  this.preserve_newlines = this._get_boolean('preserve_newlines', true);
  this.max_preserve_newlines = this._get_number('max_preserve_newlines', 32786);
  if (!this.preserve_newlines) {
    this.max_preserve_newlines = 0;
  }

  this.indent_with_tabs = this._get_boolean('indent_with_tabs', this.indent_char === '\t');
  if (this.indent_with_tabs) {
    this.indent_char = '\t';

    // indent_size behavior changed after 1.8.6
    // It used to be that indent_size would be
    // set to 1 for indent_with_tabs. That is no longer needed and
    // actually doesn't make sense - why not use spaces? Further,
    // that might produce unexpected behavior - tabs being used
    // for single-column alignment. So, when indent_with_tabs is true
    // and indent_size is 1, reset indent_size to 4.
    if (this.indent_size === 1) {
      this.indent_size = 4;
    }
  }

  // Backwards compat with 1.3.x
  this.wrap_line_length = this._get_number('wrap_line_length', this._get_number('max_char'));

  this.indent_empty_lines = this._get_boolean('indent_empty_lines');

  // valid templating languages ['django', 'erb', 'handlebars', 'php', 'smarty']
  // For now, 'auto' = all off for javascript, all on for html (and inline javascript).
  // other values ignored
  this.templating = this._get_selection_list('templating', ['auto', 'none', 'django', 'erb', 'handlebars', 'php', 'smarty'], ['auto']);
}

Options$9.prototype._get_array = function(name, default_value) {
  var option_value = this.raw_options[name];
  var result = default_value || [];
  if (typeof option_value === 'object') {
    if (option_value !== null && typeof option_value.concat === 'function') {
      result = option_value.concat();
    }
  } else if (typeof option_value === 'string') {
    result = option_value.split(/[^a-zA-Z0-9_\/\-]+/);
  }
  return result;
};

Options$9.prototype._get_boolean = function(name, default_value) {
  var option_value = this.raw_options[name];
  var result = option_value === undefined ? !!default_value : !!option_value;
  return result;
};

Options$9.prototype._get_characters = function(name, default_value) {
  var option_value = this.raw_options[name];
  var result = default_value || '';
  if (typeof option_value === 'string') {
    result = option_value.replace(/\\r/, '\r').replace(/\\n/, '\n').replace(/\\t/, '\t');
  }
  return result;
};

Options$9.prototype._get_number = function(name, default_value) {
  var option_value = this.raw_options[name];
  default_value = parseInt(default_value, 10);
  if (isNaN(default_value)) {
    default_value = 0;
  }
  var result = parseInt(option_value, 10);
  if (isNaN(result)) {
    result = default_value;
  }
  return result;
};

Options$9.prototype._get_selection = function(name, selection_list, default_value) {
  var result = this._get_selection_list(name, selection_list, default_value);
  if (result.length !== 1) {
    throw new Error(
      "Invalid Option Value: The option '" + name + "' can only be one of the following values:\n" +
      selection_list + "\nYou passed in: '" + this.raw_options[name] + "'");
  }

  return result[0];
};


Options$9.prototype._get_selection_list = function(name, selection_list, default_value) {
  if (!selection_list || selection_list.length === 0) {
    throw new Error("Selection list cannot be empty.");
  }

  default_value = default_value || [selection_list[0]];
  if (!this._is_valid_selection(default_value, selection_list)) {
    throw new Error("Invalid Default Value!");
  }

  var result = this._get_array(name, default_value);
  if (!this._is_valid_selection(result, selection_list)) {
    throw new Error(
      "Invalid Option Value: The option '" + name + "' can contain only the following values:\n" +
      selection_list + "\nYou passed in: '" + this.raw_options[name] + "'");
  }

  return result;
};

Options$9.prototype._is_valid_selection = function(result, selection_list) {
  return result.length && selection_list.length &&
    !result.some(function(item) { return selection_list.indexOf(item) === -1; });
};


// merges child options up with the parent options object
// Example: obj = {a: 1, b: {a: 2}}
//          mergeOpts(obj, 'b')
//
//          Returns: {a: 2}
function _mergeOpts(allOptions, childFieldName) {
  var finalOpts = {};
  allOptions = _normalizeOpts(allOptions);
  var name;

  for (name in allOptions) {
    if (name !== childFieldName) {
      finalOpts[name] = allOptions[name];
    }
  }

  //merge in the per type settings for the childFieldName
  if (childFieldName && allOptions[childFieldName]) {
    for (name in allOptions[childFieldName]) {
      finalOpts[name] = allOptions[childFieldName][name];
    }
  }
  return finalOpts;
}

function _normalizeOpts(options) {
  var convertedOpts = {};
  var key;

  for (key in options) {
    var newKey = key.replace(/-/g, "_");
    convertedOpts[newKey] = options[key];
  }
  return convertedOpts;
}

options$2.Options = Options$9;
options$2.normalizeOpts = _normalizeOpts;
options$2.mergeOpts = _mergeOpts;

/*jshint node:true */

var BaseOptions$2 = options$2.Options;

var validPositionValues$1 = ['before-newline', 'after-newline', 'preserve-newline'];

function Options$8(options) {
  BaseOptions$2.call(this, options, 'js');

  // compatibility, re
  var raw_brace_style = this.raw_options.brace_style || null;
  if (raw_brace_style === "expand-strict") { //graceful handling of deprecated option
    this.raw_options.brace_style = "expand";
  } else if (raw_brace_style === "collapse-preserve-inline") { //graceful handling of deprecated option
    this.raw_options.brace_style = "collapse,preserve-inline";
  } else if (this.raw_options.braces_on_own_line !== undefined) { //graceful handling of deprecated option
    this.raw_options.brace_style = this.raw_options.braces_on_own_line ? "expand" : "collapse";
    // } else if (!raw_brace_style) { //Nothing exists to set it
    //   raw_brace_style = "collapse";
  }

  //preserve-inline in delimited string will trigger brace_preserve_inline, everything
  //else is considered a brace_style and the last one only will have an effect

  var brace_style_split = this._get_selection_list('brace_style', ['collapse', 'expand', 'end-expand', 'none', 'preserve-inline']);

  this.brace_preserve_inline = false; //Defaults in case one or other was not specified in meta-option
  this.brace_style = "collapse";

  for (var bs = 0; bs < brace_style_split.length; bs++) {
    if (brace_style_split[bs] === "preserve-inline") {
      this.brace_preserve_inline = true;
    } else {
      this.brace_style = brace_style_split[bs];
    }
  }

  this.unindent_chained_methods = this._get_boolean('unindent_chained_methods');
  this.break_chained_methods = this._get_boolean('break_chained_methods');
  this.space_in_paren = this._get_boolean('space_in_paren');
  this.space_in_empty_paren = this._get_boolean('space_in_empty_paren');
  this.jslint_happy = this._get_boolean('jslint_happy');
  this.space_after_anon_function = this._get_boolean('space_after_anon_function');
  this.space_after_named_function = this._get_boolean('space_after_named_function');
  this.keep_array_indentation = this._get_boolean('keep_array_indentation');
  this.space_before_conditional = this._get_boolean('space_before_conditional', true);
  this.unescape_strings = this._get_boolean('unescape_strings');
  this.e4x = this._get_boolean('e4x');
  this.comma_first = this._get_boolean('comma_first');
  this.operator_position = this._get_selection('operator_position', validPositionValues$1);

  // For testing of beautify preserve:start directive
  this.test_output_raw = this._get_boolean('test_output_raw');

  // force this._options.space_after_anon_function to true if this._options.jslint_happy
  if (this.jslint_happy) {
    this.space_after_anon_function = true;
  }

}
Options$8.prototype = new BaseOptions$2();



options$3.Options = Options$8;

var tokenizer$2 = {};

var inputscanner = {};

/*jshint node:true */

var regexp_has_sticky = RegExp.prototype.hasOwnProperty('sticky');

function InputScanner$3(input_string) {
  this.__input = input_string || '';
  this.__input_length = this.__input.length;
  this.__position = 0;
}

InputScanner$3.prototype.restart = function() {
  this.__position = 0;
};

InputScanner$3.prototype.back = function() {
  if (this.__position > 0) {
    this.__position -= 1;
  }
};

InputScanner$3.prototype.hasNext = function() {
  return this.__position < this.__input_length;
};

InputScanner$3.prototype.next = function() {
  var val = null;
  if (this.hasNext()) {
    val = this.__input.charAt(this.__position);
    this.__position += 1;
  }
  return val;
};

InputScanner$3.prototype.peek = function(index) {
  var val = null;
  index = index || 0;
  index += this.__position;
  if (index >= 0 && index < this.__input_length) {
    val = this.__input.charAt(index);
  }
  return val;
};

// This is a JavaScript only helper function (not in python)
// Javascript doesn't have a match method
// and not all implementation support "sticky" flag.
// If they do not support sticky then both this.match() and this.test() method
// must get the match and check the index of the match.
// If sticky is supported and set, this method will use it.
// Otherwise it will check that global is set, and fall back to the slower method.
InputScanner$3.prototype.__match = function(pattern, index) {
  pattern.lastIndex = index;
  var pattern_match = pattern.exec(this.__input);

  if (pattern_match && !(regexp_has_sticky && pattern.sticky)) {
    if (pattern_match.index !== index) {
      pattern_match = null;
    }
  }

  return pattern_match;
};

InputScanner$3.prototype.test = function(pattern, index) {
  index = index || 0;
  index += this.__position;

  if (index >= 0 && index < this.__input_length) {
    return !!this.__match(pattern, index);
  } else {
    return false;
  }
};

InputScanner$3.prototype.testChar = function(pattern, index) {
  // test one character regex match
  var val = this.peek(index);
  pattern.lastIndex = 0;
  return val !== null && pattern.test(val);
};

InputScanner$3.prototype.match = function(pattern) {
  var pattern_match = this.__match(pattern, this.__position);
  if (pattern_match) {
    this.__position += pattern_match[0].length;
  } else {
    pattern_match = null;
  }
  return pattern_match;
};

InputScanner$3.prototype.read = function(starting_pattern, until_pattern, until_after) {
  var val = '';
  var match;
  if (starting_pattern) {
    match = this.match(starting_pattern);
    if (match) {
      val += match[0];
    }
  }
  if (until_pattern && (match || !starting_pattern)) {
    val += this.readUntil(until_pattern, until_after);
  }
  return val;
};

InputScanner$3.prototype.readUntil = function(pattern, until_after) {
  var val = '';
  var match_index = this.__position;
  pattern.lastIndex = this.__position;
  var pattern_match = pattern.exec(this.__input);
  if (pattern_match) {
    match_index = pattern_match.index;
    if (until_after) {
      match_index += pattern_match[0].length;
    }
  } else {
    match_index = this.__input_length;
  }

  val = this.__input.substring(this.__position, match_index);
  this.__position = match_index;
  return val;
};

InputScanner$3.prototype.readUntilAfter = function(pattern) {
  return this.readUntil(pattern, true);
};

InputScanner$3.prototype.get_regexp = function(pattern, match_from) {
  var result = null;
  var flags = 'g';
  if (match_from && regexp_has_sticky) {
    flags = 'y';
  }
  // strings are converted to regexp
  if (typeof pattern === "string" && pattern !== '') {
    // result = new RegExp(pattern.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&'), flags);
    result = new RegExp(pattern, flags);
  } else if (pattern) {
    result = new RegExp(pattern.source, flags);
  }
  return result;
};

InputScanner$3.prototype.get_literal_regexp = function(literal_string) {
  return RegExp(literal_string.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&'));
};

/* css beautifier legacy helpers */
InputScanner$3.prototype.peekUntilAfter = function(pattern) {
  var start = this.__position;
  var val = this.readUntilAfter(pattern);
  this.__position = start;
  return val;
};

InputScanner$3.prototype.lookBack = function(testVal) {
  var start = this.__position - 1;
  return start >= testVal.length && this.__input.substring(start - testVal.length, start)
    .toLowerCase() === testVal;
};

inputscanner.InputScanner = InputScanner$3;

var tokenizer$1 = {};

var tokenstream = {};

/*jshint node:true */

function TokenStream$1(parent_token) {
  // private
  this.__tokens = [];
  this.__tokens_length = this.__tokens.length;
  this.__position = 0;
  this.__parent_token = parent_token;
}

TokenStream$1.prototype.restart = function() {
  this.__position = 0;
};

TokenStream$1.prototype.isEmpty = function() {
  return this.__tokens_length === 0;
};

TokenStream$1.prototype.hasNext = function() {
  return this.__position < this.__tokens_length;
};

TokenStream$1.prototype.next = function() {
  var val = null;
  if (this.hasNext()) {
    val = this.__tokens[this.__position];
    this.__position += 1;
  }
  return val;
};

TokenStream$1.prototype.peek = function(index) {
  var val = null;
  index = index || 0;
  index += this.__position;
  if (index >= 0 && index < this.__tokens_length) {
    val = this.__tokens[index];
  }
  return val;
};

TokenStream$1.prototype.add = function(token) {
  if (this.__parent_token) {
    token.parent = this.__parent_token;
  }
  this.__tokens.push(token);
  this.__tokens_length += 1;
};

tokenstream.TokenStream = TokenStream$1;

var whitespacepattern = {};

var pattern = {};

/*jshint node:true */

function Pattern$4(input_scanner, parent) {
  this._input = input_scanner;
  this._starting_pattern = null;
  this._match_pattern = null;
  this._until_pattern = null;
  this._until_after = false;

  if (parent) {
    this._starting_pattern = this._input.get_regexp(parent._starting_pattern, true);
    this._match_pattern = this._input.get_regexp(parent._match_pattern, true);
    this._until_pattern = this._input.get_regexp(parent._until_pattern);
    this._until_after = parent._until_after;
  }
}

Pattern$4.prototype.read = function() {
  var result = this._input.read(this._starting_pattern);
  if (!this._starting_pattern || result) {
    result += this._input.read(this._match_pattern, this._until_pattern, this._until_after);
  }
  return result;
};

Pattern$4.prototype.read_match = function() {
  return this._input.match(this._match_pattern);
};

Pattern$4.prototype.until_after = function(pattern) {
  var result = this._create();
  result._until_after = true;
  result._until_pattern = this._input.get_regexp(pattern);
  result._update();
  return result;
};

Pattern$4.prototype.until = function(pattern) {
  var result = this._create();
  result._until_after = false;
  result._until_pattern = this._input.get_regexp(pattern);
  result._update();
  return result;
};

Pattern$4.prototype.starting_with = function(pattern) {
  var result = this._create();
  result._starting_pattern = this._input.get_regexp(pattern, true);
  result._update();
  return result;
};

Pattern$4.prototype.matching = function(pattern) {
  var result = this._create();
  result._match_pattern = this._input.get_regexp(pattern, true);
  result._update();
  return result;
};

Pattern$4.prototype._create = function() {
  return new Pattern$4(this._input, this);
};

Pattern$4.prototype._update = function() {};

pattern.Pattern = Pattern$4;

/*jshint node:true */

var Pattern$3 = pattern.Pattern;

function WhitespacePattern$1(input_scanner, parent) {
  Pattern$3.call(this, input_scanner, parent);
  if (parent) {
    this._line_regexp = this._input.get_regexp(parent._line_regexp);
  } else {
    this.__set_whitespace_patterns('', '');
  }

  this.newline_count = 0;
  this.whitespace_before_token = '';
}
WhitespacePattern$1.prototype = new Pattern$3();

WhitespacePattern$1.prototype.__set_whitespace_patterns = function(whitespace_chars, newline_chars) {
  whitespace_chars += '\\t ';
  newline_chars += '\\n\\r';

  this._match_pattern = this._input.get_regexp(
    '[' + whitespace_chars + newline_chars + ']+', true);
  this._newline_regexp = this._input.get_regexp(
    '\\r\\n|[' + newline_chars + ']');
};

WhitespacePattern$1.prototype.read = function() {
  this.newline_count = 0;
  this.whitespace_before_token = '';

  var resulting_string = this._input.read(this._match_pattern);
  if (resulting_string === ' ') {
    this.whitespace_before_token = ' ';
  } else if (resulting_string) {
    var matches = this.__split(this._newline_regexp, resulting_string);
    this.newline_count = matches.length - 1;
    this.whitespace_before_token = matches[this.newline_count];
  }

  return resulting_string;
};

WhitespacePattern$1.prototype.matching = function(whitespace_chars, newline_chars) {
  var result = this._create();
  result.__set_whitespace_patterns(whitespace_chars, newline_chars);
  result._update();
  return result;
};

WhitespacePattern$1.prototype._create = function() {
  return new WhitespacePattern$1(this._input, this);
};

WhitespacePattern$1.prototype.__split = function(regexp, input_string) {
  regexp.lastIndex = 0;
  var start_index = 0;
  var result = [];
  var next_match = regexp.exec(input_string);
  while (next_match) {
    result.push(input_string.substring(start_index, next_match.index));
    start_index = next_match.index + next_match[0].length;
    next_match = regexp.exec(input_string);
  }

  if (start_index < input_string.length) {
    result.push(input_string.substring(start_index, input_string.length));
  } else {
    result.push('');
  }

  return result;
};



whitespacepattern.WhitespacePattern = WhitespacePattern$1;

/*jshint node:true */

var InputScanner$2 = inputscanner.InputScanner;
var Token$1 = token.Token;
var TokenStream = tokenstream.TokenStream;
var WhitespacePattern = whitespacepattern.WhitespacePattern;

var TOKEN$4 = {
  START: 'TK_START',
  RAW: 'TK_RAW',
  EOF: 'TK_EOF'
};

var Tokenizer$4 = function(input_string, options) {
  this._input = new InputScanner$2(input_string);
  this._options = options || {};
  this.__tokens = null;

  this._patterns = {};
  this._patterns.whitespace = new WhitespacePattern(this._input);
};

Tokenizer$4.prototype.tokenize = function() {
  this._input.restart();
  this.__tokens = new TokenStream();

  this._reset();

  var current;
  var previous = new Token$1(TOKEN$4.START, '');
  var open_token = null;
  var open_stack = [];
  var comments = new TokenStream();

  while (previous.type !== TOKEN$4.EOF) {
    current = this._get_next_token(previous, open_token);
    while (this._is_comment(current)) {
      comments.add(current);
      current = this._get_next_token(previous, open_token);
    }

    if (!comments.isEmpty()) {
      current.comments_before = comments;
      comments = new TokenStream();
    }

    current.parent = open_token;

    if (this._is_opening(current)) {
      open_stack.push(open_token);
      open_token = current;
    } else if (open_token && this._is_closing(current, open_token)) {
      current.opened = open_token;
      open_token.closed = current;
      open_token = open_stack.pop();
      current.parent = open_token;
    }

    current.previous = previous;
    previous.next = current;

    this.__tokens.add(current);
    previous = current;
  }

  return this.__tokens;
};


Tokenizer$4.prototype._is_first_token = function() {
  return this.__tokens.isEmpty();
};

Tokenizer$4.prototype._reset = function() {};

Tokenizer$4.prototype._get_next_token = function(previous_token, open_token) { // jshint unused:false
  this._readWhitespace();
  var resulting_string = this._input.read(/.+/g);
  if (resulting_string) {
    return this._create_token(TOKEN$4.RAW, resulting_string);
  } else {
    return this._create_token(TOKEN$4.EOF, '');
  }
};

Tokenizer$4.prototype._is_comment = function(current_token) { // jshint unused:false
  return false;
};

Tokenizer$4.prototype._is_opening = function(current_token) { // jshint unused:false
  return false;
};

Tokenizer$4.prototype._is_closing = function(current_token, open_token) { // jshint unused:false
  return false;
};

Tokenizer$4.prototype._create_token = function(type, text) {
  var token = new Token$1(type, text,
    this._patterns.whitespace.newline_count,
    this._patterns.whitespace.whitespace_before_token);
  return token;
};

Tokenizer$4.prototype._readWhitespace = function() {
  return this._patterns.whitespace.read();
};



tokenizer$1.Tokenizer = Tokenizer$4;
tokenizer$1.TOKEN = TOKEN$4;

var directives = {};

/*jshint node:true */

function Directives$3(start_block_pattern, end_block_pattern) {
  start_block_pattern = typeof start_block_pattern === 'string' ? start_block_pattern : start_block_pattern.source;
  end_block_pattern = typeof end_block_pattern === 'string' ? end_block_pattern : end_block_pattern.source;
  this.__directives_block_pattern = new RegExp(start_block_pattern + / beautify( \w+[:]\w+)+ /.source + end_block_pattern, 'g');
  this.__directive_pattern = / (\w+)[:](\w+)/g;

  this.__directives_end_ignore_pattern = new RegExp(start_block_pattern + /\sbeautify\signore:end\s/.source + end_block_pattern, 'g');
}

Directives$3.prototype.get_directives = function(text) {
  if (!text.match(this.__directives_block_pattern)) {
    return null;
  }

  var directives = {};
  this.__directive_pattern.lastIndex = 0;
  var directive_match = this.__directive_pattern.exec(text);

  while (directive_match) {
    directives[directive_match[1]] = directive_match[2];
    directive_match = this.__directive_pattern.exec(text);
  }

  return directives;
};

Directives$3.prototype.readIgnored = function(input) {
  return input.readUntilAfter(this.__directives_end_ignore_pattern);
};


directives.Directives = Directives$3;

var templatablepattern = {};

/*jshint node:true */

var Pattern$2 = pattern.Pattern;


var template_names = {
  django: false,
  erb: false,
  handlebars: false,
  php: false,
  smarty: false
};

// This lets templates appear anywhere we would do a readUntil
// The cost is higher but it is pay to play.
function TemplatablePattern$2(input_scanner, parent) {
  Pattern$2.call(this, input_scanner, parent);
  this.__template_pattern = null;
  this._disabled = Object.assign({}, template_names);
  this._excluded = Object.assign({}, template_names);

  if (parent) {
    this.__template_pattern = this._input.get_regexp(parent.__template_pattern);
    this._excluded = Object.assign(this._excluded, parent._excluded);
    this._disabled = Object.assign(this._disabled, parent._disabled);
  }
  var pattern = new Pattern$2(input_scanner);
  this.__patterns = {
    handlebars_comment: pattern.starting_with(/{{!--/).until_after(/--}}/),
    handlebars_unescaped: pattern.starting_with(/{{{/).until_after(/}}}/),
    handlebars: pattern.starting_with(/{{/).until_after(/}}/),
    php: pattern.starting_with(/<\?(?:[= ]|php)/).until_after(/\?>/),
    erb: pattern.starting_with(/<%[^%]/).until_after(/[^%]%>/),
    // django coflicts with handlebars a bit.
    django: pattern.starting_with(/{%/).until_after(/%}/),
    django_value: pattern.starting_with(/{{/).until_after(/}}/),
    django_comment: pattern.starting_with(/{#/).until_after(/#}/),
    smarty: pattern.starting_with(/{(?=[^}{\s\n])/).until_after(/[^\s\n]}/),
    smarty_comment: pattern.starting_with(/{\*/).until_after(/\*}/),
    smarty_literal: pattern.starting_with(/{literal}/).until_after(/{\/literal}/)
  };
}
TemplatablePattern$2.prototype = new Pattern$2();

TemplatablePattern$2.prototype._create = function() {
  return new TemplatablePattern$2(this._input, this);
};

TemplatablePattern$2.prototype._update = function() {
  this.__set_templated_pattern();
};

TemplatablePattern$2.prototype.disable = function(language) {
  var result = this._create();
  result._disabled[language] = true;
  result._update();
  return result;
};

TemplatablePattern$2.prototype.read_options = function(options) {
  var result = this._create();
  for (var language in template_names) {
    result._disabled[language] = options.templating.indexOf(language) === -1;
  }
  result._update();
  return result;
};

TemplatablePattern$2.prototype.exclude = function(language) {
  var result = this._create();
  result._excluded[language] = true;
  result._update();
  return result;
};

TemplatablePattern$2.prototype.read = function() {
  var result = '';
  if (this._match_pattern) {
    result = this._input.read(this._starting_pattern);
  } else {
    result = this._input.read(this._starting_pattern, this.__template_pattern);
  }
  var next = this._read_template();
  while (next) {
    if (this._match_pattern) {
      next += this._input.read(this._match_pattern);
    } else {
      next += this._input.readUntil(this.__template_pattern);
    }
    result += next;
    next = this._read_template();
  }

  if (this._until_after) {
    result += this._input.readUntilAfter(this._until_pattern);
  }
  return result;
};

TemplatablePattern$2.prototype.__set_templated_pattern = function() {
  var items = [];

  if (!this._disabled.php) {
    items.push(this.__patterns.php._starting_pattern.source);
  }
  if (!this._disabled.handlebars) {
    items.push(this.__patterns.handlebars._starting_pattern.source);
  }
  if (!this._disabled.erb) {
    items.push(this.__patterns.erb._starting_pattern.source);
  }
  if (!this._disabled.django) {
    items.push(this.__patterns.django._starting_pattern.source);
    // The starting pattern for django is more complex because it has different
    // patterns for value, comment, and other sections
    items.push(this.__patterns.django_value._starting_pattern.source);
    items.push(this.__patterns.django_comment._starting_pattern.source);
  }
  if (!this._disabled.smarty) {
    items.push(this.__patterns.smarty._starting_pattern.source);
  }

  if (this._until_pattern) {
    items.push(this._until_pattern.source);
  }
  this.__template_pattern = this._input.get_regexp('(?:' + items.join('|') + ')');
};

TemplatablePattern$2.prototype._read_template = function() {
  var resulting_string = '';
  var c = this._input.peek();
  if (c === '<') {
    var peek1 = this._input.peek(1);
    //if we're in a comment, do something special
    // We treat all comments as literals, even more than preformatted tags
    // we just look for the appropriate close tag
    if (!this._disabled.php && !this._excluded.php && peek1 === '?') {
      resulting_string = resulting_string ||
        this.__patterns.php.read();
    }
    if (!this._disabled.erb && !this._excluded.erb && peek1 === '%') {
      resulting_string = resulting_string ||
        this.__patterns.erb.read();
    }
  } else if (c === '{') {
    if (!this._disabled.handlebars && !this._excluded.handlebars) {
      resulting_string = resulting_string ||
        this.__patterns.handlebars_comment.read();
      resulting_string = resulting_string ||
        this.__patterns.handlebars_unescaped.read();
      resulting_string = resulting_string ||
        this.__patterns.handlebars.read();
    }
    if (!this._disabled.django) {
      // django coflicts with handlebars a bit.
      if (!this._excluded.django && !this._excluded.handlebars) {
        resulting_string = resulting_string ||
          this.__patterns.django_value.read();
      }
      if (!this._excluded.django) {
        resulting_string = resulting_string ||
          this.__patterns.django_comment.read();
        resulting_string = resulting_string ||
          this.__patterns.django.read();
      }
    }
    if (!this._disabled.smarty) {
      // smarty cannot be enabled with django or handlebars enabled
      if (this._disabled.django && this._disabled.handlebars) {
        resulting_string = resulting_string ||
          this.__patterns.smarty_comment.read();
        resulting_string = resulting_string ||
          this.__patterns.smarty_literal.read();
        resulting_string = resulting_string ||
          this.__patterns.smarty.read();
      }
    }
  }
  return resulting_string;
};


templatablepattern.TemplatablePattern = TemplatablePattern$2;

/*jshint node:true */

var InputScanner$1 = inputscanner.InputScanner;
var BaseTokenizer$1 = tokenizer$1.Tokenizer;
var BASETOKEN$1 = tokenizer$1.TOKEN;
var Directives$2 = directives.Directives;
var acorn$1 = acorn$2;
var Pattern$1 = pattern.Pattern;
var TemplatablePattern$1 = templatablepattern.TemplatablePattern;


function in_array$2(what, arr) {
  return arr.indexOf(what) !== -1;
}


var TOKEN$3 = {
  START_EXPR: 'TK_START_EXPR',
  END_EXPR: 'TK_END_EXPR',
  START_BLOCK: 'TK_START_BLOCK',
  END_BLOCK: 'TK_END_BLOCK',
  WORD: 'TK_WORD',
  RESERVED: 'TK_RESERVED',
  SEMICOLON: 'TK_SEMICOLON',
  STRING: 'TK_STRING',
  EQUALS: 'TK_EQUALS',
  OPERATOR: 'TK_OPERATOR',
  COMMA: 'TK_COMMA',
  BLOCK_COMMENT: 'TK_BLOCK_COMMENT',
  COMMENT: 'TK_COMMENT',
  DOT: 'TK_DOT',
  UNKNOWN: 'TK_UNKNOWN',
  START: BASETOKEN$1.START,
  RAW: BASETOKEN$1.RAW,
  EOF: BASETOKEN$1.EOF
};


var directives_core$2 = new Directives$2(/\/\*/, /\*\//);

var number_pattern = /0[xX][0123456789abcdefABCDEF_]*n?|0[oO][01234567_]*n?|0[bB][01_]*n?|\d[\d_]*n|(?:\.\d[\d_]*|\d[\d_]*\.?[\d_]*)(?:[eE][+-]?[\d_]+)?/;

var digit = /[0-9]/;

// Dot "." must be distinguished from "..." and decimal
var dot_pattern = /[^\d\.]/;

var positionable_operators$1 = (
  ">>> === !== " +
  "<< && >= ** != == <= >> || ?? |> " +
  "< / - + > : & % ? ^ | *").split(' ');

// IMPORTANT: this must be sorted longest to shortest or tokenizing many not work.
// Also, you must update possitionable operators separately from punct
var punct =
  ">>>= " +
  "... >>= <<= === >>> !== **= " +
  "=> ^= :: /= << <= == && -= >= >> != -- += ** || ?? ++ %= &= *= |= |> " +
  "= ! ? > < : / ^ - + * & % ~ |";

punct = punct.replace(/[-[\]{}()*+?.,\\^$|#]/g, "\\$&");
// ?. but not if followed by a number 
punct = '\\?\\.(?!\\d) ' + punct;
punct = punct.replace(/ /g, '|');

var punct_pattern = new RegExp(punct);

// words which should always start on new line.
var line_starters$1 = 'continue,try,throw,return,var,let,const,if,switch,case,default,for,while,break,function,import,export'.split(',');
var reserved_words = line_starters$1.concat(['do', 'in', 'of', 'else', 'get', 'set', 'new', 'catch', 'finally', 'typeof', 'yield', 'async', 'await', 'from', 'as']);
var reserved_word_pattern = new RegExp('^(?:' + reserved_words.join('|') + ')$');

// var template_pattern = /(?:(?:<\?php|<\?=)[\s\S]*?\?>)|(?:<%[\s\S]*?%>)/g;

var in_html_comment;

var Tokenizer$3 = function(input_string, options) {
  BaseTokenizer$1.call(this, input_string, options);

  this._patterns.whitespace = this._patterns.whitespace.matching(
    /\u00A0\u1680\u180e\u2000-\u200a\u202f\u205f\u3000\ufeff/.source,
    /\u2028\u2029/.source);

  var pattern_reader = new Pattern$1(this._input);
  var templatable = new TemplatablePattern$1(this._input)
    .read_options(this._options);

  this.__patterns = {
    template: templatable,
    identifier: templatable.starting_with(acorn$1.identifier).matching(acorn$1.identifierMatch),
    number: pattern_reader.matching(number_pattern),
    punct: pattern_reader.matching(punct_pattern),
    // comment ends just before nearest linefeed or end of file
    comment: pattern_reader.starting_with(/\/\//).until(/[\n\r\u2028\u2029]/),
    //  /* ... */ comment ends with nearest */ or end of file
    block_comment: pattern_reader.starting_with(/\/\*/).until_after(/\*\//),
    html_comment_start: pattern_reader.matching(/<!--/),
    html_comment_end: pattern_reader.matching(/-->/),
    include: pattern_reader.starting_with(/#include/).until_after(acorn$1.lineBreak),
    shebang: pattern_reader.starting_with(/#!/).until_after(acorn$1.lineBreak),
    xml: pattern_reader.matching(/[\s\S]*?<(\/?)([-a-zA-Z:0-9_.]+|{[\s\S]+?}|!\[CDATA\[[\s\S]*?\]\]|)(\s+{[\s\S]+?}|\s+[-a-zA-Z:0-9_.]+|\s+[-a-zA-Z:0-9_.]+\s*=\s*('[^']*'|"[^"]*"|{[\s\S]+?}))*\s*(\/?)\s*>/),
    single_quote: templatable.until(/['\\\n\r\u2028\u2029]/),
    double_quote: templatable.until(/["\\\n\r\u2028\u2029]/),
    template_text: templatable.until(/[`\\$]/),
    template_expression: templatable.until(/[`}\\]/)
  };

};
Tokenizer$3.prototype = new BaseTokenizer$1();

Tokenizer$3.prototype._is_comment = function(current_token) {
  return current_token.type === TOKEN$3.COMMENT || current_token.type === TOKEN$3.BLOCK_COMMENT || current_token.type === TOKEN$3.UNKNOWN;
};

Tokenizer$3.prototype._is_opening = function(current_token) {
  return current_token.type === TOKEN$3.START_BLOCK || current_token.type === TOKEN$3.START_EXPR;
};

Tokenizer$3.prototype._is_closing = function(current_token, open_token) {
  return (current_token.type === TOKEN$3.END_BLOCK || current_token.type === TOKEN$3.END_EXPR) &&
    (open_token && (
      (current_token.text === ']' && open_token.text === '[') ||
      (current_token.text === ')' && open_token.text === '(') ||
      (current_token.text === '}' && open_token.text === '{')));
};

Tokenizer$3.prototype._reset = function() {
  in_html_comment = false;
};

Tokenizer$3.prototype._get_next_token = function(previous_token, open_token) { // jshint unused:false
  var token = null;
  this._readWhitespace();
  var c = this._input.peek();

  if (c === null) {
    return this._create_token(TOKEN$3.EOF, '');
  }

  token = token || this._read_non_javascript(c);
  token = token || this._read_string(c);
  token = token || this._read_word(previous_token);
  token = token || this._read_singles(c);
  token = token || this._read_comment(c);
  token = token || this._read_regexp(c, previous_token);
  token = token || this._read_xml(c, previous_token);
  token = token || this._read_punctuation();
  token = token || this._create_token(TOKEN$3.UNKNOWN, this._input.next());

  return token;
};

Tokenizer$3.prototype._read_word = function(previous_token) {
  var resulting_string;
  resulting_string = this.__patterns.identifier.read();
  if (resulting_string !== '') {
    resulting_string = resulting_string.replace(acorn$1.allLineBreaks, '\n');
    if (!(previous_token.type === TOKEN$3.DOT ||
        (previous_token.type === TOKEN$3.RESERVED && (previous_token.text === 'set' || previous_token.text === 'get'))) &&
      reserved_word_pattern.test(resulting_string)) {
      if (resulting_string === 'in' || resulting_string === 'of') { // hack for 'in' and 'of' operators
        return this._create_token(TOKEN$3.OPERATOR, resulting_string);
      }
      return this._create_token(TOKEN$3.RESERVED, resulting_string);
    }
    return this._create_token(TOKEN$3.WORD, resulting_string);
  }

  resulting_string = this.__patterns.number.read();
  if (resulting_string !== '') {
    return this._create_token(TOKEN$3.WORD, resulting_string);
  }
};

Tokenizer$3.prototype._read_singles = function(c) {
  var token = null;
  if (c === '(' || c === '[') {
    token = this._create_token(TOKEN$3.START_EXPR, c);
  } else if (c === ')' || c === ']') {
    token = this._create_token(TOKEN$3.END_EXPR, c);
  } else if (c === '{') {
    token = this._create_token(TOKEN$3.START_BLOCK, c);
  } else if (c === '}') {
    token = this._create_token(TOKEN$3.END_BLOCK, c);
  } else if (c === ';') {
    token = this._create_token(TOKEN$3.SEMICOLON, c);
  } else if (c === '.' && dot_pattern.test(this._input.peek(1))) {
    token = this._create_token(TOKEN$3.DOT, c);
  } else if (c === ',') {
    token = this._create_token(TOKEN$3.COMMA, c);
  }

  if (token) {
    this._input.next();
  }
  return token;
};

Tokenizer$3.prototype._read_punctuation = function() {
  var resulting_string = this.__patterns.punct.read();

  if (resulting_string !== '') {
    if (resulting_string === '=') {
      return this._create_token(TOKEN$3.EQUALS, resulting_string);
    } else if (resulting_string === '?.') {
      return this._create_token(TOKEN$3.DOT, resulting_string);
    } else {
      return this._create_token(TOKEN$3.OPERATOR, resulting_string);
    }
  }
};

Tokenizer$3.prototype._read_non_javascript = function(c) {
  var resulting_string = '';

  if (c === '#') {
    if (this._is_first_token()) {
      resulting_string = this.__patterns.shebang.read();

      if (resulting_string) {
        return this._create_token(TOKEN$3.UNKNOWN, resulting_string.trim() + '\n');
      }
    }

    // handles extendscript #includes
    resulting_string = this.__patterns.include.read();

    if (resulting_string) {
      return this._create_token(TOKEN$3.UNKNOWN, resulting_string.trim() + '\n');
    }

    c = this._input.next();

    // Spidermonkey-specific sharp variables for circular references. Considered obsolete.
    var sharp = '#';
    if (this._input.hasNext() && this._input.testChar(digit)) {
      do {
        c = this._input.next();
        sharp += c;
      } while (this._input.hasNext() && c !== '#' && c !== '=');
      if (c === '#') ; else if (this._input.peek() === '[' && this._input.peek(1) === ']') {
        sharp += '[]';
        this._input.next();
        this._input.next();
      } else if (this._input.peek() === '{' && this._input.peek(1) === '}') {
        sharp += '{}';
        this._input.next();
        this._input.next();
      }
      return this._create_token(TOKEN$3.WORD, sharp);
    }

    this._input.back();

  } else if (c === '<' && this._is_first_token()) {
    resulting_string = this.__patterns.html_comment_start.read();
    if (resulting_string) {
      while (this._input.hasNext() && !this._input.testChar(acorn$1.newline)) {
        resulting_string += this._input.next();
      }
      in_html_comment = true;
      return this._create_token(TOKEN$3.COMMENT, resulting_string);
    }
  } else if (in_html_comment && c === '-') {
    resulting_string = this.__patterns.html_comment_end.read();
    if (resulting_string) {
      in_html_comment = false;
      return this._create_token(TOKEN$3.COMMENT, resulting_string);
    }
  }

  return null;
};

Tokenizer$3.prototype._read_comment = function(c) {
  var token = null;
  if (c === '/') {
    var comment = '';
    if (this._input.peek(1) === '*') {
      // peek for comment /* ... */
      comment = this.__patterns.block_comment.read();
      var directives = directives_core$2.get_directives(comment);
      if (directives && directives.ignore === 'start') {
        comment += directives_core$2.readIgnored(this._input);
      }
      comment = comment.replace(acorn$1.allLineBreaks, '\n');
      token = this._create_token(TOKEN$3.BLOCK_COMMENT, comment);
      token.directives = directives;
    } else if (this._input.peek(1) === '/') {
      // peek for comment // ...
      comment = this.__patterns.comment.read();
      token = this._create_token(TOKEN$3.COMMENT, comment);
    }
  }
  return token;
};

Tokenizer$3.prototype._read_string = function(c) {
  if (c === '`' || c === "'" || c === '"') {
    var resulting_string = this._input.next();
    this.has_char_escapes = false;

    if (c === '`') {
      resulting_string += this._read_string_recursive('`', true, '${');
    } else {
      resulting_string += this._read_string_recursive(c);
    }

    if (this.has_char_escapes && this._options.unescape_strings) {
      resulting_string = unescape_string(resulting_string);
    }

    if (this._input.peek() === c) {
      resulting_string += this._input.next();
    }

    resulting_string = resulting_string.replace(acorn$1.allLineBreaks, '\n');

    return this._create_token(TOKEN$3.STRING, resulting_string);
  }

  return null;
};

Tokenizer$3.prototype._allow_regexp_or_xml = function(previous_token) {
  // regex and xml can only appear in specific locations during parsing
  return (previous_token.type === TOKEN$3.RESERVED && in_array$2(previous_token.text, ['return', 'case', 'throw', 'else', 'do', 'typeof', 'yield'])) ||
    (previous_token.type === TOKEN$3.END_EXPR && previous_token.text === ')' &&
      previous_token.opened.previous.type === TOKEN$3.RESERVED && in_array$2(previous_token.opened.previous.text, ['if', 'while', 'for'])) ||
    (in_array$2(previous_token.type, [TOKEN$3.COMMENT, TOKEN$3.START_EXPR, TOKEN$3.START_BLOCK, TOKEN$3.START,
      TOKEN$3.END_BLOCK, TOKEN$3.OPERATOR, TOKEN$3.EQUALS, TOKEN$3.EOF, TOKEN$3.SEMICOLON, TOKEN$3.COMMA
    ]));
};

Tokenizer$3.prototype._read_regexp = function(c, previous_token) {

  if (c === '/' && this._allow_regexp_or_xml(previous_token)) {
    // handle regexp
    //
    var resulting_string = this._input.next();
    var esc = false;

    var in_char_class = false;
    while (this._input.hasNext() &&
      ((esc || in_char_class || this._input.peek() !== c) &&
        !this._input.testChar(acorn$1.newline))) {
      resulting_string += this._input.peek();
      if (!esc) {
        esc = this._input.peek() === '\\';
        if (this._input.peek() === '[') {
          in_char_class = true;
        } else if (this._input.peek() === ']') {
          in_char_class = false;
        }
      } else {
        esc = false;
      }
      this._input.next();
    }

    if (this._input.peek() === c) {
      resulting_string += this._input.next();

      // regexps may have modifiers /regexp/MOD , so fetch those, too
      // Only [gim] are valid, but if the user puts in garbage, do what we can to take it.
      resulting_string += this._input.read(acorn$1.identifier);
    }
    return this._create_token(TOKEN$3.STRING, resulting_string);
  }
  return null;
};

Tokenizer$3.prototype._read_xml = function(c, previous_token) {

  if (this._options.e4x && c === "<" && this._allow_regexp_or_xml(previous_token)) {
    var xmlStr = '';
    var match = this.__patterns.xml.read_match();
    // handle e4x xml literals
    //
    if (match) {
      // Trim root tag to attempt to
      var rootTag = match[2].replace(/^{\s+/, '{').replace(/\s+}$/, '}');
      var isCurlyRoot = rootTag.indexOf('{') === 0;
      var depth = 0;
      while (match) {
        var isEndTag = !!match[1];
        var tagName = match[2];
        var isSingletonTag = (!!match[match.length - 1]) || (tagName.slice(0, 8) === "![CDATA[");
        if (!isSingletonTag &&
          (tagName === rootTag || (isCurlyRoot && tagName.replace(/^{\s+/, '{').replace(/\s+}$/, '}')))) {
          if (isEndTag) {
            --depth;
          } else {
            ++depth;
          }
        }
        xmlStr += match[0];
        if (depth <= 0) {
          break;
        }
        match = this.__patterns.xml.read_match();
      }
      // if we didn't close correctly, keep unformatted.
      if (!match) {
        xmlStr += this._input.match(/[\s\S]*/g)[0];
      }
      xmlStr = xmlStr.replace(acorn$1.allLineBreaks, '\n');
      return this._create_token(TOKEN$3.STRING, xmlStr);
    }
  }

  return null;
};

function unescape_string(s) {
  // You think that a regex would work for this
  // return s.replace(/\\x([0-9a-f]{2})/gi, function(match, val) {
  //         return String.fromCharCode(parseInt(val, 16));
  //     })
  // However, dealing with '\xff', '\\xff', '\\\xff' makes this more fun.
  var out = '',
    escaped = 0;

  var input_scan = new InputScanner$1(s);
  var matched = null;

  while (input_scan.hasNext()) {
    // Keep any whitespace, non-slash characters
    // also keep slash pairs.
    matched = input_scan.match(/([\s]|[^\\]|\\\\)+/g);

    if (matched) {
      out += matched[0];
    }

    if (input_scan.peek() === '\\') {
      input_scan.next();
      if (input_scan.peek() === 'x') {
        matched = input_scan.match(/x([0-9A-Fa-f]{2})/g);
      } else if (input_scan.peek() === 'u') {
        matched = input_scan.match(/u([0-9A-Fa-f]{4})/g);
      } else {
        out += '\\';
        if (input_scan.hasNext()) {
          out += input_scan.next();
        }
        continue;
      }

      // If there's some error decoding, return the original string
      if (!matched) {
        return s;
      }

      escaped = parseInt(matched[1], 16);

      if (escaped > 0x7e && escaped <= 0xff && matched[0].indexOf('x') === 0) {
        // we bail out on \x7f..\xff,
        // leaving whole string escaped,
        // as it's probably completely binary
        return s;
      } else if (escaped >= 0x00 && escaped < 0x20) {
        // leave 0x00...0x1f escaped
        out += '\\' + matched[0];
        continue;
      } else if (escaped === 0x22 || escaped === 0x27 || escaped === 0x5c) {
        // single-quote, apostrophe, backslash - escape these
        out += '\\' + String.fromCharCode(escaped);
      } else {
        out += String.fromCharCode(escaped);
      }
    }
  }

  return out;
}

// handle string
//
Tokenizer$3.prototype._read_string_recursive = function(delimiter, allow_unescaped_newlines, start_sub) {
  var current_char;
  var pattern;
  if (delimiter === '\'') {
    pattern = this.__patterns.single_quote;
  } else if (delimiter === '"') {
    pattern = this.__patterns.double_quote;
  } else if (delimiter === '`') {
    pattern = this.__patterns.template_text;
  } else if (delimiter === '}') {
    pattern = this.__patterns.template_expression;
  }

  var resulting_string = pattern.read();
  var next = '';
  while (this._input.hasNext()) {
    next = this._input.next();
    if (next === delimiter ||
      (!allow_unescaped_newlines && acorn$1.newline.test(next))) {
      this._input.back();
      break;
    } else if (next === '\\' && this._input.hasNext()) {
      current_char = this._input.peek();

      if (current_char === 'x' || current_char === 'u') {
        this.has_char_escapes = true;
      } else if (current_char === '\r' && this._input.peek(1) === '\n') {
        this._input.next();
      }
      next += this._input.next();
    } else if (start_sub) {
      if (start_sub === '${' && next === '$' && this._input.peek() === '{') {
        next += this._input.next();
      }

      if (start_sub === next) {
        if (delimiter === '`') {
          next += this._read_string_recursive('}', allow_unescaped_newlines, '`');
        } else {
          next += this._read_string_recursive('`', allow_unescaped_newlines, '${');
        }
        if (this._input.hasNext()) {
          next += this._input.next();
        }
      }
    }
    next += pattern.read();
    resulting_string += next;
  }

  return resulting_string;
};

tokenizer$2.Tokenizer = Tokenizer$3;
tokenizer$2.TOKEN = TOKEN$3;
tokenizer$2.positionable_operators = positionable_operators$1.slice();
tokenizer$2.line_starters = line_starters$1.slice();

/*jshint node:true */

var Output$2 = output.Output;
var Token = token.Token;
var acorn = acorn$2;
var Options$7 = options$3.Options;
var Tokenizer$2 = tokenizer$2.Tokenizer;
var line_starters = tokenizer$2.line_starters;
var positionable_operators = tokenizer$2.positionable_operators;
var TOKEN$2 = tokenizer$2.TOKEN;


function in_array$1(what, arr) {
  return arr.indexOf(what) !== -1;
}

function ltrim(s) {
  return s.replace(/^\s+/g, '');
}

function generateMapFromStrings(list) {
  var result = {};
  for (var x = 0; x < list.length; x++) {
    // make the mapped names underscored instead of dash
    result[list[x].replace(/-/g, '_')] = list[x];
  }
  return result;
}

function reserved_word(token, word) {
  return token && token.type === TOKEN$2.RESERVED && token.text === word;
}

function reserved_array(token, words) {
  return token && token.type === TOKEN$2.RESERVED && in_array$1(token.text, words);
}
// Unsure of what they mean, but they work. Worth cleaning up in future.
var special_words = ['case', 'return', 'do', 'if', 'throw', 'else', 'await', 'break', 'continue', 'async'];

var validPositionValues = ['before-newline', 'after-newline', 'preserve-newline'];

// Generate map from array
var OPERATOR_POSITION = generateMapFromStrings(validPositionValues);

var OPERATOR_POSITION_BEFORE_OR_PRESERVE = [OPERATOR_POSITION.before_newline, OPERATOR_POSITION.preserve_newline];

var MODE = {
  BlockStatement: 'BlockStatement', // 'BLOCK'
  Statement: 'Statement', // 'STATEMENT'
  ObjectLiteral: 'ObjectLiteral', // 'OBJECT',
  ArrayLiteral: 'ArrayLiteral', //'[EXPRESSION]',
  ForInitializer: 'ForInitializer', //'(FOR-EXPRESSION)',
  Conditional: 'Conditional', //'(COND-EXPRESSION)',
  Expression: 'Expression' //'(EXPRESSION)'
};

function remove_redundant_indentation(output, frame) {
  // This implementation is effective but has some issues:
  //     - can cause line wrap to happen too soon due to indent removal
  //           after wrap points are calculated
  // These issues are minor compared to ugly indentation.

  if (frame.multiline_frame ||
    frame.mode === MODE.ForInitializer ||
    frame.mode === MODE.Conditional) {
    return;
  }

  // remove one indent from each line inside this section
  output.remove_indent(frame.start_line_index);
}

// we could use just string.split, but
// IE doesn't like returning empty strings
function split_linebreaks(s) {
  //return s.split(/\x0d\x0a|\x0a/);

  s = s.replace(acorn.allLineBreaks, '\n');
  var out = [],
    idx = s.indexOf("\n");
  while (idx !== -1) {
    out.push(s.substring(0, idx));
    s = s.substring(idx + 1);
    idx = s.indexOf("\n");
  }
  if (s.length) {
    out.push(s);
  }
  return out;
}

function is_array(mode) {
  return mode === MODE.ArrayLiteral;
}

function is_expression(mode) {
  return in_array$1(mode, [MODE.Expression, MODE.ForInitializer, MODE.Conditional]);
}

function all_lines_start_with(lines, c) {
  for (var i = 0; i < lines.length; i++) {
    var line = lines[i].trim();
    if (line.charAt(0) !== c) {
      return false;
    }
  }
  return true;
}

function each_line_matches_indent(lines, indent) {
  var i = 0,
    len = lines.length,
    line;
  for (; i < len; i++) {
    line = lines[i];
    // allow empty lines to pass through
    if (line && line.indexOf(indent) !== 0) {
      return false;
    }
  }
  return true;
}


function Beautifier$5(source_text, options) {
  options = options || {};
  this._source_text = source_text || '';

  this._output = null;
  this._tokens = null;
  this._last_last_text = null;
  this._flags = null;
  this._previous_flags = null;

  this._flag_store = null;
  this._options = new Options$7(options);
}

Beautifier$5.prototype.create_flags = function(flags_base, mode) {
  var next_indent_level = 0;
  if (flags_base) {
    next_indent_level = flags_base.indentation_level;
    if (!this._output.just_added_newline() &&
      flags_base.line_indent_level > next_indent_level) {
      next_indent_level = flags_base.line_indent_level;
    }
  }

  var next_flags = {
    mode: mode,
    parent: flags_base,
    last_token: flags_base ? flags_base.last_token : new Token(TOKEN$2.START_BLOCK, ''), // last token text
    last_word: flags_base ? flags_base.last_word : '', // last TOKEN.WORD passed
    declaration_statement: false,
    declaration_assignment: false,
    multiline_frame: false,
    inline_frame: false,
    if_block: false,
    else_block: false,
    do_block: false,
    do_while: false,
    import_block: false,
    in_case_statement: false, // switch(..){ INSIDE HERE }
    in_case: false, // we're on the exact line with "case 0:"
    case_body: false, // the indented case-action block
    indentation_level: next_indent_level,
    alignment: 0,
    line_indent_level: flags_base ? flags_base.line_indent_level : next_indent_level,
    start_line_index: this._output.get_line_number(),
    ternary_depth: 0
  };
  return next_flags;
};

Beautifier$5.prototype._reset = function(source_text) {
  var baseIndentString = source_text.match(/^[\t ]*/)[0];

  this._last_last_text = ''; // pre-last token text
  this._output = new Output$2(this._options, baseIndentString);

  // If testing the ignore directive, start with output disable set to true
  this._output.raw = this._options.test_output_raw;


  // Stack of parsing/formatting states, including MODE.
  // We tokenize, parse, and output in an almost purely a forward-only stream of token input
  // and formatted output.  This makes the beautifier less accurate than full parsers
  // but also far more tolerant of syntax errors.
  //
  // For example, the default mode is MODE.BlockStatement. If we see a '{' we push a new frame of type
  // MODE.BlockStatement on the the stack, even though it could be object literal.  If we later
  // encounter a ":", we'll switch to to MODE.ObjectLiteral.  If we then see a ";",
  // most full parsers would die, but the beautifier gracefully falls back to
  // MODE.BlockStatement and continues on.
  this._flag_store = [];
  this.set_mode(MODE.BlockStatement);
  var tokenizer = new Tokenizer$2(source_text, this._options);
  this._tokens = tokenizer.tokenize();
  return source_text;
};

Beautifier$5.prototype.beautify = function() {
  // if disabled, return the input unchanged.
  if (this._options.disabled) {
    return this._source_text;
  }

  var sweet_code;
  var source_text = this._reset(this._source_text);

  var eol = this._options.eol;
  if (this._options.eol === 'auto') {
    eol = '\n';
    if (source_text && acorn.lineBreak.test(source_text || '')) {
      eol = source_text.match(acorn.lineBreak)[0];
    }
  }

  var current_token = this._tokens.next();
  while (current_token) {
    this.handle_token(current_token);

    this._last_last_text = this._flags.last_token.text;
    this._flags.last_token = current_token;

    current_token = this._tokens.next();
  }

  sweet_code = this._output.get_code(eol);

  return sweet_code;
};

Beautifier$5.prototype.handle_token = function(current_token, preserve_statement_flags) {
  if (current_token.type === TOKEN$2.START_EXPR) {
    this.handle_start_expr(current_token);
  } else if (current_token.type === TOKEN$2.END_EXPR) {
    this.handle_end_expr(current_token);
  } else if (current_token.type === TOKEN$2.START_BLOCK) {
    this.handle_start_block(current_token);
  } else if (current_token.type === TOKEN$2.END_BLOCK) {
    this.handle_end_block(current_token);
  } else if (current_token.type === TOKEN$2.WORD) {
    this.handle_word(current_token);
  } else if (current_token.type === TOKEN$2.RESERVED) {
    this.handle_word(current_token);
  } else if (current_token.type === TOKEN$2.SEMICOLON) {
    this.handle_semicolon(current_token);
  } else if (current_token.type === TOKEN$2.STRING) {
    this.handle_string(current_token);
  } else if (current_token.type === TOKEN$2.EQUALS) {
    this.handle_equals(current_token);
  } else if (current_token.type === TOKEN$2.OPERATOR) {
    this.handle_operator(current_token);
  } else if (current_token.type === TOKEN$2.COMMA) {
    this.handle_comma(current_token);
  } else if (current_token.type === TOKEN$2.BLOCK_COMMENT) {
    this.handle_block_comment(current_token, preserve_statement_flags);
  } else if (current_token.type === TOKEN$2.COMMENT) {
    this.handle_comment(current_token, preserve_statement_flags);
  } else if (current_token.type === TOKEN$2.DOT) {
    this.handle_dot(current_token);
  } else if (current_token.type === TOKEN$2.EOF) {
    this.handle_eof(current_token);
  } else if (current_token.type === TOKEN$2.UNKNOWN) {
    this.handle_unknown(current_token, preserve_statement_flags);
  } else {
    this.handle_unknown(current_token, preserve_statement_flags);
  }
};

Beautifier$5.prototype.handle_whitespace_and_comments = function(current_token, preserve_statement_flags) {
  var newlines = current_token.newlines;
  var keep_whitespace = this._options.keep_array_indentation && is_array(this._flags.mode);

  if (current_token.comments_before) {
    var comment_token = current_token.comments_before.next();
    while (comment_token) {
      // The cleanest handling of inline comments is to treat them as though they aren't there.
      // Just continue formatting and the behavior should be logical.
      // Also ignore unknown tokens.  Again, this should result in better behavior.
      this.handle_whitespace_and_comments(comment_token, preserve_statement_flags);
      this.handle_token(comment_token, preserve_statement_flags);
      comment_token = current_token.comments_before.next();
    }
  }

  if (keep_whitespace) {
    for (var i = 0; i < newlines; i += 1) {
      this.print_newline(i > 0, preserve_statement_flags);
    }
  } else {
    if (this._options.max_preserve_newlines && newlines > this._options.max_preserve_newlines) {
      newlines = this._options.max_preserve_newlines;
    }

    if (this._options.preserve_newlines) {
      if (newlines > 1) {
        this.print_newline(false, preserve_statement_flags);
        for (var j = 1; j < newlines; j += 1) {
          this.print_newline(true, preserve_statement_flags);
        }
      }
    }
  }

};

var newline_restricted_tokens = ['async', 'break', 'continue', 'return', 'throw', 'yield'];

Beautifier$5.prototype.allow_wrap_or_preserved_newline = function(current_token, force_linewrap) {
  force_linewrap = (force_linewrap === undefined) ? false : force_linewrap;

  // Never wrap the first token on a line
  if (this._output.just_added_newline()) {
    return;
  }

  var shouldPreserveOrForce = (this._options.preserve_newlines && current_token.newlines) || force_linewrap;
  var operatorLogicApplies = in_array$1(this._flags.last_token.text, positionable_operators) ||
    in_array$1(current_token.text, positionable_operators);

  if (operatorLogicApplies) {
    var shouldPrintOperatorNewline = (
        in_array$1(this._flags.last_token.text, positionable_operators) &&
        in_array$1(this._options.operator_position, OPERATOR_POSITION_BEFORE_OR_PRESERVE)
      ) ||
      in_array$1(current_token.text, positionable_operators);
    shouldPreserveOrForce = shouldPreserveOrForce && shouldPrintOperatorNewline;
  }

  if (shouldPreserveOrForce) {
    this.print_newline(false, true);
  } else if (this._options.wrap_line_length) {
    if (reserved_array(this._flags.last_token, newline_restricted_tokens)) {
      // These tokens should never have a newline inserted
      // between them and the following expression.
      return;
    }
    this._output.set_wrap_point();
  }
};

Beautifier$5.prototype.print_newline = function(force_newline, preserve_statement_flags) {
  if (!preserve_statement_flags) {
    if (this._flags.last_token.text !== ';' && this._flags.last_token.text !== ',' && this._flags.last_token.text !== '=' && (this._flags.last_token.type !== TOKEN$2.OPERATOR || this._flags.last_token.text === '--' || this._flags.last_token.text === '++')) {
      var next_token = this._tokens.peek();
      while (this._flags.mode === MODE.Statement &&
        !(this._flags.if_block && reserved_word(next_token, 'else')) &&
        !this._flags.do_block) {
        this.restore_mode();
      }
    }
  }

  if (this._output.add_new_line(force_newline)) {
    this._flags.multiline_frame = true;
  }
};

Beautifier$5.prototype.print_token_line_indentation = function(current_token) {
  if (this._output.just_added_newline()) {
    if (this._options.keep_array_indentation &&
      current_token.newlines &&
      (current_token.text === '[' || is_array(this._flags.mode))) {
      this._output.current_line.set_indent(-1);
      this._output.current_line.push(current_token.whitespace_before);
      this._output.space_before_token = false;
    } else if (this._output.set_indent(this._flags.indentation_level, this._flags.alignment)) {
      this._flags.line_indent_level = this._flags.indentation_level;
    }
  }
};

Beautifier$5.prototype.print_token = function(current_token) {
  if (this._output.raw) {
    this._output.add_raw_token(current_token);
    return;
  }

  if (this._options.comma_first && current_token.previous && current_token.previous.type === TOKEN$2.COMMA &&
    this._output.just_added_newline()) {
    if (this._output.previous_line.last() === ',') {
      var popped = this._output.previous_line.pop();
      // if the comma was already at the start of the line,
      // pull back onto that line and reprint the indentation
      if (this._output.previous_line.is_empty()) {
        this._output.previous_line.push(popped);
        this._output.trim(true);
        this._output.current_line.pop();
        this._output.trim();
      }

      // add the comma in front of the next token
      this.print_token_line_indentation(current_token);
      this._output.add_token(',');
      this._output.space_before_token = true;
    }
  }

  this.print_token_line_indentation(current_token);
  this._output.non_breaking_space = true;
  this._output.add_token(current_token.text);
  if (this._output.previous_token_wrapped) {
    this._flags.multiline_frame = true;
  }
};

Beautifier$5.prototype.indent = function() {
  this._flags.indentation_level += 1;
  this._output.set_indent(this._flags.indentation_level, this._flags.alignment);
};

Beautifier$5.prototype.deindent = function() {
  if (this._flags.indentation_level > 0 &&
    ((!this._flags.parent) || this._flags.indentation_level > this._flags.parent.indentation_level)) {
    this._flags.indentation_level -= 1;
    this._output.set_indent(this._flags.indentation_level, this._flags.alignment);
  }
};

Beautifier$5.prototype.set_mode = function(mode) {
  if (this._flags) {
    this._flag_store.push(this._flags);
    this._previous_flags = this._flags;
  } else {
    this._previous_flags = this.create_flags(null, mode);
  }

  this._flags = this.create_flags(this._previous_flags, mode);
  this._output.set_indent(this._flags.indentation_level, this._flags.alignment);
};


Beautifier$5.prototype.restore_mode = function() {
  if (this._flag_store.length > 0) {
    this._previous_flags = this._flags;
    this._flags = this._flag_store.pop();
    if (this._previous_flags.mode === MODE.Statement) {
      remove_redundant_indentation(this._output, this._previous_flags);
    }
    this._output.set_indent(this._flags.indentation_level, this._flags.alignment);
  }
};

Beautifier$5.prototype.start_of_object_property = function() {
  return this._flags.parent.mode === MODE.ObjectLiteral && this._flags.mode === MODE.Statement && (
    (this._flags.last_token.text === ':' && this._flags.ternary_depth === 0) || (reserved_array(this._flags.last_token, ['get', 'set'])));
};

Beautifier$5.prototype.start_of_statement = function(current_token) {
  var start = false;
  start = start || reserved_array(this._flags.last_token, ['var', 'let', 'const']) && current_token.type === TOKEN$2.WORD;
  start = start || reserved_word(this._flags.last_token, 'do');
  start = start || (!(this._flags.parent.mode === MODE.ObjectLiteral && this._flags.mode === MODE.Statement)) && reserved_array(this._flags.last_token, newline_restricted_tokens) && !current_token.newlines;
  start = start || reserved_word(this._flags.last_token, 'else') &&
    !(reserved_word(current_token, 'if') && !current_token.comments_before);
  start = start || (this._flags.last_token.type === TOKEN$2.END_EXPR && (this._previous_flags.mode === MODE.ForInitializer || this._previous_flags.mode === MODE.Conditional));
  start = start || (this._flags.last_token.type === TOKEN$2.WORD && this._flags.mode === MODE.BlockStatement &&
    !this._flags.in_case &&
    !(current_token.text === '--' || current_token.text === '++') &&
    this._last_last_text !== 'function' &&
    current_token.type !== TOKEN$2.WORD && current_token.type !== TOKEN$2.RESERVED);
  start = start || (this._flags.mode === MODE.ObjectLiteral && (
    (this._flags.last_token.text === ':' && this._flags.ternary_depth === 0) || reserved_array(this._flags.last_token, ['get', 'set'])));

  if (start) {
    this.set_mode(MODE.Statement);
    this.indent();

    this.handle_whitespace_and_comments(current_token, true);

    // Issue #276:
    // If starting a new statement with [if, for, while, do], push to a new line.
    // if (a) if (b) if(c) d(); else e(); else f();
    if (!this.start_of_object_property()) {
      this.allow_wrap_or_preserved_newline(current_token,
        reserved_array(current_token, ['do', 'for', 'if', 'while']));
    }
    return true;
  }
  return false;
};

Beautifier$5.prototype.handle_start_expr = function(current_token) {
  // The conditional starts the statement if appropriate.
  if (!this.start_of_statement(current_token)) {
    this.handle_whitespace_and_comments(current_token);
  }

  var next_mode = MODE.Expression;
  if (current_token.text === '[') {

    if (this._flags.last_token.type === TOKEN$2.WORD || this._flags.last_token.text === ')') {
      // this is array index specifier, break immediately
      // a[x], fn()[x]
      if (reserved_array(this._flags.last_token, line_starters)) {
        this._output.space_before_token = true;
      }
      this.print_token(current_token);
      this.set_mode(next_mode);
      this.indent();
      if (this._options.space_in_paren) {
        this._output.space_before_token = true;
      }
      return;
    }

    next_mode = MODE.ArrayLiteral;
    if (is_array(this._flags.mode)) {
      if (this._flags.last_token.text === '[' ||
        (this._flags.last_token.text === ',' && (this._last_last_text === ']' || this._last_last_text === '}'))) {
        // ], [ goes to new line
        // }, [ goes to new line
        if (!this._options.keep_array_indentation) {
          this.print_newline();
        }
      }
    }

    if (!in_array$1(this._flags.last_token.type, [TOKEN$2.START_EXPR, TOKEN$2.END_EXPR, TOKEN$2.WORD, TOKEN$2.OPERATOR, TOKEN$2.DOT])) {
      this._output.space_before_token = true;
    }
  } else {
    if (this._flags.last_token.type === TOKEN$2.RESERVED) {
      if (this._flags.last_token.text === 'for') {
        this._output.space_before_token = this._options.space_before_conditional;
        next_mode = MODE.ForInitializer;
      } else if (in_array$1(this._flags.last_token.text, ['if', 'while', 'switch'])) {
        this._output.space_before_token = this._options.space_before_conditional;
        next_mode = MODE.Conditional;
      } else if (in_array$1(this._flags.last_word, ['await', 'async'])) {
        // Should be a space between await and an IIFE, or async and an arrow function
        this._output.space_before_token = true;
      } else if (this._flags.last_token.text === 'import' && current_token.whitespace_before === '') {
        this._output.space_before_token = false;
      } else if (in_array$1(this._flags.last_token.text, line_starters) || this._flags.last_token.text === 'catch') {
        this._output.space_before_token = true;
      }
    } else if (this._flags.last_token.type === TOKEN$2.EQUALS || this._flags.last_token.type === TOKEN$2.OPERATOR) {
      // Support of this kind of newline preservation.
      // a = (b &&
      //     (c || d));
      if (!this.start_of_object_property()) {
        this.allow_wrap_or_preserved_newline(current_token);
      }
    } else if (this._flags.last_token.type === TOKEN$2.WORD) {
      this._output.space_before_token = false;

      // function name() vs function name ()
      // function* name() vs function* name ()
      // async name() vs async name ()
      // In ES6, you can also define the method properties of an object
      // var obj = {a: function() {}}
      // It can be abbreviated
      // var obj = {a() {}}
      // var obj = { a() {}} vs var obj = { a () {}}
      // var obj = { * a() {}} vs var obj = { * a () {}}
      var peek_back_two = this._tokens.peek(-3);
      if (this._options.space_after_named_function && peek_back_two) {
        // peek starts at next character so -1 is current token
        var peek_back_three = this._tokens.peek(-4);
        if (reserved_array(peek_back_two, ['async', 'function']) ||
          (peek_back_two.text === '*' && reserved_array(peek_back_three, ['async', 'function']))) {
          this._output.space_before_token = true;
        } else if (this._flags.mode === MODE.ObjectLiteral) {
          if ((peek_back_two.text === '{' || peek_back_two.text === ',') ||
            (peek_back_two.text === '*' && (peek_back_three.text === '{' || peek_back_three.text === ','))) {
            this._output.space_before_token = true;
          }
        }
      }
    } else {
      // Support preserving wrapped arrow function expressions
      // a.b('c',
      //     () => d.e
      // )
      this.allow_wrap_or_preserved_newline(current_token);
    }

    // function() vs function ()
    // yield*() vs yield* ()
    // function*() vs function* ()
    if ((this._flags.last_token.type === TOKEN$2.RESERVED && (this._flags.last_word === 'function' || this._flags.last_word === 'typeof')) ||
      (this._flags.last_token.text === '*' &&
        (in_array$1(this._last_last_text, ['function', 'yield']) ||
          (this._flags.mode === MODE.ObjectLiteral && in_array$1(this._last_last_text, ['{', ',']))))) {
      this._output.space_before_token = this._options.space_after_anon_function;
    }
  }

  if (this._flags.last_token.text === ';' || this._flags.last_token.type === TOKEN$2.START_BLOCK) {
    this.print_newline();
  } else if (this._flags.last_token.type === TOKEN$2.END_EXPR || this._flags.last_token.type === TOKEN$2.START_EXPR || this._flags.last_token.type === TOKEN$2.END_BLOCK || this._flags.last_token.text === '.' || this._flags.last_token.type === TOKEN$2.COMMA) {
    // do nothing on (( and )( and ][ and ]( and .(
    // TODO: Consider whether forcing this is required.  Review failing tests when removed.
    this.allow_wrap_or_preserved_newline(current_token, current_token.newlines);
  }

  this.print_token(current_token);
  this.set_mode(next_mode);
  if (this._options.space_in_paren) {
    this._output.space_before_token = true;
  }

  // In all cases, if we newline while inside an expression it should be indented.
  this.indent();
};

Beautifier$5.prototype.handle_end_expr = function(current_token) {
  // statements inside expressions are not valid syntax, but...
  // statements must all be closed when their container closes
  while (this._flags.mode === MODE.Statement) {
    this.restore_mode();
  }

  this.handle_whitespace_and_comments(current_token);

  if (this._flags.multiline_frame) {
    this.allow_wrap_or_preserved_newline(current_token,
      current_token.text === ']' && is_array(this._flags.mode) && !this._options.keep_array_indentation);
  }

  if (this._options.space_in_paren) {
    if (this._flags.last_token.type === TOKEN$2.START_EXPR && !this._options.space_in_empty_paren) {
      // () [] no inner space in empty parens like these, ever, ref #320
      this._output.trim();
      this._output.space_before_token = false;
    } else {
      this._output.space_before_token = true;
    }
  }
  this.deindent();
  this.print_token(current_token);
  this.restore_mode();

  remove_redundant_indentation(this._output, this._previous_flags);

  // do {} while () // no statement required after
  if (this._flags.do_while && this._previous_flags.mode === MODE.Conditional) {
    this._previous_flags.mode = MODE.Expression;
    this._flags.do_block = false;
    this._flags.do_while = false;

  }
};

Beautifier$5.prototype.handle_start_block = function(current_token) {
  this.handle_whitespace_and_comments(current_token);

  // Check if this is should be treated as a ObjectLiteral
  var next_token = this._tokens.peek();
  var second_token = this._tokens.peek(1);
  if (this._flags.last_word === 'switch' && this._flags.last_token.type === TOKEN$2.END_EXPR) {
    this.set_mode(MODE.BlockStatement);
    this._flags.in_case_statement = true;
  } else if (this._flags.case_body) {
    this.set_mode(MODE.BlockStatement);
  } else if (second_token && (
      (in_array$1(second_token.text, [':', ',']) && in_array$1(next_token.type, [TOKEN$2.STRING, TOKEN$2.WORD, TOKEN$2.RESERVED])) ||
      (in_array$1(next_token.text, ['get', 'set', '...']) && in_array$1(second_token.type, [TOKEN$2.WORD, TOKEN$2.RESERVED]))
    )) {
    // We don't support TypeScript,but we didn't break it for a very long time.
    // We'll try to keep not breaking it.
    if (!in_array$1(this._last_last_text, ['class', 'interface'])) {
      this.set_mode(MODE.ObjectLiteral);
    } else {
      this.set_mode(MODE.BlockStatement);
    }
  } else if (this._flags.last_token.type === TOKEN$2.OPERATOR && this._flags.last_token.text === '=>') {
    // arrow function: (param1, paramN) => { statements }
    this.set_mode(MODE.BlockStatement);
  } else if (in_array$1(this._flags.last_token.type, [TOKEN$2.EQUALS, TOKEN$2.START_EXPR, TOKEN$2.COMMA, TOKEN$2.OPERATOR]) ||
    reserved_array(this._flags.last_token, ['return', 'throw', 'import', 'default'])
  ) {
    // Detecting shorthand function syntax is difficult by scanning forward,
    //     so check the surrounding context.
    // If the block is being returned, imported, export default, passed as arg,
    //     assigned with = or assigned in a nested object, treat as an ObjectLiteral.
    this.set_mode(MODE.ObjectLiteral);
  } else {
    this.set_mode(MODE.BlockStatement);
  }

  var empty_braces = !next_token.comments_before && next_token.text === '}';
  var empty_anonymous_function = empty_braces && this._flags.last_word === 'function' &&
    this._flags.last_token.type === TOKEN$2.END_EXPR;

  if (this._options.brace_preserve_inline) // check for inline, set inline_frame if so
  {
    // search forward for a newline wanted inside this block
    var index = 0;
    var check_token = null;
    this._flags.inline_frame = true;
    do {
      index += 1;
      check_token = this._tokens.peek(index - 1);
      if (check_token.newlines) {
        this._flags.inline_frame = false;
        break;
      }
    } while (check_token.type !== TOKEN$2.EOF &&
      !(check_token.type === TOKEN$2.END_BLOCK && check_token.opened === current_token));
  }

  if ((this._options.brace_style === "expand" ||
      (this._options.brace_style === "none" && current_token.newlines)) &&
    !this._flags.inline_frame) {
    if (this._flags.last_token.type !== TOKEN$2.OPERATOR &&
      (empty_anonymous_function ||
        this._flags.last_token.type === TOKEN$2.EQUALS ||
        (reserved_array(this._flags.last_token, special_words) && this._flags.last_token.text !== 'else'))) {
      this._output.space_before_token = true;
    } else {
      this.print_newline(false, true);
    }
  } else { // collapse || inline_frame
    if (is_array(this._previous_flags.mode) && (this._flags.last_token.type === TOKEN$2.START_EXPR || this._flags.last_token.type === TOKEN$2.COMMA)) {
      if (this._flags.last_token.type === TOKEN$2.COMMA || this._options.space_in_paren) {
        this._output.space_before_token = true;
      }

      if (this._flags.last_token.type === TOKEN$2.COMMA || (this._flags.last_token.type === TOKEN$2.START_EXPR && this._flags.inline_frame)) {
        this.allow_wrap_or_preserved_newline(current_token);
        this._previous_flags.multiline_frame = this._previous_flags.multiline_frame || this._flags.multiline_frame;
        this._flags.multiline_frame = false;
      }
    }
    if (this._flags.last_token.type !== TOKEN$2.OPERATOR && this._flags.last_token.type !== TOKEN$2.START_EXPR) {
      if (this._flags.last_token.type === TOKEN$2.START_BLOCK && !this._flags.inline_frame) {
        this.print_newline();
      } else {
        this._output.space_before_token = true;
      }
    }
  }
  this.print_token(current_token);
  this.indent();

  // Except for specific cases, open braces are followed by a new line.
  if (!empty_braces && !(this._options.brace_preserve_inline && this._flags.inline_frame)) {
    this.print_newline();
  }
};

Beautifier$5.prototype.handle_end_block = function(current_token) {
  // statements must all be closed when their container closes
  this.handle_whitespace_and_comments(current_token);

  while (this._flags.mode === MODE.Statement) {
    this.restore_mode();
  }

  var empty_braces = this._flags.last_token.type === TOKEN$2.START_BLOCK;

  if (this._flags.inline_frame && !empty_braces) { // try inline_frame (only set if this._options.braces-preserve-inline) first
    this._output.space_before_token = true;
  } else if (this._options.brace_style === "expand") {
    if (!empty_braces) {
      this.print_newline();
    }
  } else {
    // skip {}
    if (!empty_braces) {
      if (is_array(this._flags.mode) && this._options.keep_array_indentation) {
        // we REALLY need a newline here, but newliner would skip that
        this._options.keep_array_indentation = false;
        this.print_newline();
        this._options.keep_array_indentation = true;

      } else {
        this.print_newline();
      }
    }
  }
  this.restore_mode();
  this.print_token(current_token);
};

Beautifier$5.prototype.handle_word = function(current_token) {
  if (current_token.type === TOKEN$2.RESERVED) {
    if (in_array$1(current_token.text, ['set', 'get']) && this._flags.mode !== MODE.ObjectLiteral) {
      current_token.type = TOKEN$2.WORD;
    } else if (current_token.text === 'import' && this._tokens.peek().text === '(') {
      current_token.type = TOKEN$2.WORD;
    } else if (in_array$1(current_token.text, ['as', 'from']) && !this._flags.import_block) {
      current_token.type = TOKEN$2.WORD;
    } else if (this._flags.mode === MODE.ObjectLiteral) {
      var next_token = this._tokens.peek();
      if (next_token.text === ':') {
        current_token.type = TOKEN$2.WORD;
      }
    }
  }

  if (this.start_of_statement(current_token)) {
    // The conditional starts the statement if appropriate.
    if (reserved_array(this._flags.last_token, ['var', 'let', 'const']) && current_token.type === TOKEN$2.WORD) {
      this._flags.declaration_statement = true;
    }
  } else if (current_token.newlines && !is_expression(this._flags.mode) &&
    (this._flags.last_token.type !== TOKEN$2.OPERATOR || (this._flags.last_token.text === '--' || this._flags.last_token.text === '++')) &&
    this._flags.last_token.type !== TOKEN$2.EQUALS &&
    (this._options.preserve_newlines || !reserved_array(this._flags.last_token, ['var', 'let', 'const', 'set', 'get']))) {
    this.handle_whitespace_and_comments(current_token);
    this.print_newline();
  } else {
    this.handle_whitespace_and_comments(current_token);
  }

  if (this._flags.do_block && !this._flags.do_while) {
    if (reserved_word(current_token, 'while')) {
      // do {} ## while ()
      this._output.space_before_token = true;
      this.print_token(current_token);
      this._output.space_before_token = true;
      this._flags.do_while = true;
      return;
    } else {
      // do {} should always have while as the next word.
      // if we don't see the expected while, recover
      this.print_newline();
      this._flags.do_block = false;
    }
  }

  // if may be followed by else, or not
  // Bare/inline ifs are tricky
  // Need to unwind the modes correctly: if (a) if (b) c(); else d(); else e();
  if (this._flags.if_block) {
    if (!this._flags.else_block && reserved_word(current_token, 'else')) {
      this._flags.else_block = true;
    } else {
      while (this._flags.mode === MODE.Statement) {
        this.restore_mode();
      }
      this._flags.if_block = false;
      this._flags.else_block = false;
    }
  }

  if (this._flags.in_case_statement && reserved_array(current_token, ['case', 'default'])) {
    this.print_newline();
    if (this._flags.last_token.type !== TOKEN$2.END_BLOCK && (this._flags.case_body || this._options.jslint_happy)) {
      // switch cases following one another
      this.deindent();
    }
    this._flags.case_body = false;

    this.print_token(current_token);
    this._flags.in_case = true;
    return;
  }

  if (this._flags.last_token.type === TOKEN$2.COMMA || this._flags.last_token.type === TOKEN$2.START_EXPR || this._flags.last_token.type === TOKEN$2.EQUALS || this._flags.last_token.type === TOKEN$2.OPERATOR) {
    if (!this.start_of_object_property()) {
      this.allow_wrap_or_preserved_newline(current_token);
    }
  }

  if (reserved_word(current_token, 'function')) {
    if (in_array$1(this._flags.last_token.text, ['}', ';']) ||
      (this._output.just_added_newline() && !(in_array$1(this._flags.last_token.text, ['(', '[', '{', ':', '=', ',']) || this._flags.last_token.type === TOKEN$2.OPERATOR))) {
      // make sure there is a nice clean space of at least one blank line
      // before a new function definition
      if (!this._output.just_added_blankline() && !current_token.comments_before) {
        this.print_newline();
        this.print_newline(true);
      }
    }
    if (this._flags.last_token.type === TOKEN$2.RESERVED || this._flags.last_token.type === TOKEN$2.WORD) {
      if (reserved_array(this._flags.last_token, ['get', 'set', 'new', 'export']) ||
        reserved_array(this._flags.last_token, newline_restricted_tokens)) {
        this._output.space_before_token = true;
      } else if (reserved_word(this._flags.last_token, 'default') && this._last_last_text === 'export') {
        this._output.space_before_token = true;
      } else if (this._flags.last_token.text === 'declare') {
        // accomodates Typescript declare function formatting
        this._output.space_before_token = true;
      } else {
        this.print_newline();
      }
    } else if (this._flags.last_token.type === TOKEN$2.OPERATOR || this._flags.last_token.text === '=') {
      // foo = function
      this._output.space_before_token = true;
    } else if (!this._flags.multiline_frame && (is_expression(this._flags.mode) || is_array(this._flags.mode))) ; else {
      this.print_newline();
    }

    this.print_token(current_token);
    this._flags.last_word = current_token.text;
    return;
  }

  var prefix = 'NONE';

  if (this._flags.last_token.type === TOKEN$2.END_BLOCK) {

    if (this._previous_flags.inline_frame) {
      prefix = 'SPACE';
    } else if (!reserved_array(current_token, ['else', 'catch', 'finally', 'from'])) {
      prefix = 'NEWLINE';
    } else {
      if (this._options.brace_style === "expand" ||
        this._options.brace_style === "end-expand" ||
        (this._options.brace_style === "none" && current_token.newlines)) {
        prefix = 'NEWLINE';
      } else {
        prefix = 'SPACE';
        this._output.space_before_token = true;
      }
    }
  } else if (this._flags.last_token.type === TOKEN$2.SEMICOLON && this._flags.mode === MODE.BlockStatement) {
    // TODO: Should this be for STATEMENT as well?
    prefix = 'NEWLINE';
  } else if (this._flags.last_token.type === TOKEN$2.SEMICOLON && is_expression(this._flags.mode)) {
    prefix = 'SPACE';
  } else if (this._flags.last_token.type === TOKEN$2.STRING) {
    prefix = 'NEWLINE';
  } else if (this._flags.last_token.type === TOKEN$2.RESERVED || this._flags.last_token.type === TOKEN$2.WORD ||
    (this._flags.last_token.text === '*' &&
      (in_array$1(this._last_last_text, ['function', 'yield']) ||
        (this._flags.mode === MODE.ObjectLiteral && in_array$1(this._last_last_text, ['{', ',']))))) {
    prefix = 'SPACE';
  } else if (this._flags.last_token.type === TOKEN$2.START_BLOCK) {
    if (this._flags.inline_frame) {
      prefix = 'SPACE';
    } else {
      prefix = 'NEWLINE';
    }
  } else if (this._flags.last_token.type === TOKEN$2.END_EXPR) {
    this._output.space_before_token = true;
    prefix = 'NEWLINE';
  }

  if (reserved_array(current_token, line_starters) && this._flags.last_token.text !== ')') {
    if (this._flags.inline_frame || this._flags.last_token.text === 'else' || this._flags.last_token.text === 'export') {
      prefix = 'SPACE';
    } else {
      prefix = 'NEWLINE';
    }

  }

  if (reserved_array(current_token, ['else', 'catch', 'finally'])) {
    if ((!(this._flags.last_token.type === TOKEN$2.END_BLOCK && this._previous_flags.mode === MODE.BlockStatement) ||
        this._options.brace_style === "expand" ||
        this._options.brace_style === "end-expand" ||
        (this._options.brace_style === "none" && current_token.newlines)) &&
      !this._flags.inline_frame) {
      this.print_newline();
    } else {
      this._output.trim(true);
      var line = this._output.current_line;
      // If we trimmed and there's something other than a close block before us
      // put a newline back in.  Handles '} // comment' scenario.
      if (line.last() !== '}') {
        this.print_newline();
      }
      this._output.space_before_token = true;
    }
  } else if (prefix === 'NEWLINE') {
    if (reserved_array(this._flags.last_token, special_words)) {
      // no newline between 'return nnn'
      this._output.space_before_token = true;
    } else if (this._flags.last_token.text === 'declare' && reserved_array(current_token, ['var', 'let', 'const'])) {
      // accomodates Typescript declare formatting
      this._output.space_before_token = true;
    } else if (this._flags.last_token.type !== TOKEN$2.END_EXPR) {
      if ((this._flags.last_token.type !== TOKEN$2.START_EXPR || !reserved_array(current_token, ['var', 'let', 'const'])) && this._flags.last_token.text !== ':') {
        // no need to force newline on 'var': for (var x = 0...)
        if (reserved_word(current_token, 'if') && reserved_word(current_token.previous, 'else')) {
          // no newline for } else if {
          this._output.space_before_token = true;
        } else {
          this.print_newline();
        }
      }
    } else if (reserved_array(current_token, line_starters) && this._flags.last_token.text !== ')') {
      this.print_newline();
    }
  } else if (this._flags.multiline_frame && is_array(this._flags.mode) && this._flags.last_token.text === ',' && this._last_last_text === '}') {
    this.print_newline(); // }, in lists get a newline treatment
  } else if (prefix === 'SPACE') {
    this._output.space_before_token = true;
  }
  if (current_token.previous && (current_token.previous.type === TOKEN$2.WORD || current_token.previous.type === TOKEN$2.RESERVED)) {
    this._output.space_before_token = true;
  }
  this.print_token(current_token);
  this._flags.last_word = current_token.text;

  if (current_token.type === TOKEN$2.RESERVED) {
    if (current_token.text === 'do') {
      this._flags.do_block = true;
    } else if (current_token.text === 'if') {
      this._flags.if_block = true;
    } else if (current_token.text === 'import') {
      this._flags.import_block = true;
    } else if (this._flags.import_block && reserved_word(current_token, 'from')) {
      this._flags.import_block = false;
    }
  }
};

Beautifier$5.prototype.handle_semicolon = function(current_token) {
  if (this.start_of_statement(current_token)) {
    // The conditional starts the statement if appropriate.
    // Semicolon can be the start (and end) of a statement
    this._output.space_before_token = false;
  } else {
    this.handle_whitespace_and_comments(current_token);
  }

  var next_token = this._tokens.peek();
  while (this._flags.mode === MODE.Statement &&
    !(this._flags.if_block && reserved_word(next_token, 'else')) &&
    !this._flags.do_block) {
    this.restore_mode();
  }

  // hacky but effective for the moment
  if (this._flags.import_block) {
    this._flags.import_block = false;
  }
  this.print_token(current_token);
};

Beautifier$5.prototype.handle_string = function(current_token) {
  if (current_token.text.startsWith("`") && current_token.newlines === 0 && current_token.whitespace_before === '' && (current_token.previous.text === ')' || this._flags.last_token.type === TOKEN$2.WORD)) ; else if (this.start_of_statement(current_token)) {
    // The conditional starts the statement if appropriate.
    // One difference - strings want at least a space before
    this._output.space_before_token = true;
  } else {
    this.handle_whitespace_and_comments(current_token);
    if (this._flags.last_token.type === TOKEN$2.RESERVED || this._flags.last_token.type === TOKEN$2.WORD || this._flags.inline_frame) {
      this._output.space_before_token = true;
    } else if (this._flags.last_token.type === TOKEN$2.COMMA || this._flags.last_token.type === TOKEN$2.START_EXPR || this._flags.last_token.type === TOKEN$2.EQUALS || this._flags.last_token.type === TOKEN$2.OPERATOR) {
      if (!this.start_of_object_property()) {
        this.allow_wrap_or_preserved_newline(current_token);
      }
    } else if ((current_token.text.startsWith("`") && this._flags.last_token.type === TOKEN$2.END_EXPR && (current_token.previous.text === ']' || current_token.previous.text === ')') && current_token.newlines === 0)) {
      this._output.space_before_token = true;
    } else {
      this.print_newline();
    }
  }
  this.print_token(current_token);
};

Beautifier$5.prototype.handle_equals = function(current_token) {
  if (this.start_of_statement(current_token)) ; else {
    this.handle_whitespace_and_comments(current_token);
  }

  if (this._flags.declaration_statement) {
    // just got an '=' in a var-line, different formatting/line-breaking, etc will now be done
    this._flags.declaration_assignment = true;
  }
  this._output.space_before_token = true;
  this.print_token(current_token);
  this._output.space_before_token = true;
};

Beautifier$5.prototype.handle_comma = function(current_token) {
  this.handle_whitespace_and_comments(current_token, true);

  this.print_token(current_token);
  this._output.space_before_token = true;
  if (this._flags.declaration_statement) {
    if (is_expression(this._flags.parent.mode)) {
      // do not break on comma, for(var a = 1, b = 2)
      this._flags.declaration_assignment = false;
    }

    if (this._flags.declaration_assignment) {
      this._flags.declaration_assignment = false;
      this.print_newline(false, true);
    } else if (this._options.comma_first) {
      // for comma-first, we want to allow a newline before the comma
      // to turn into a newline after the comma, which we will fixup later
      this.allow_wrap_or_preserved_newline(current_token);
    }
  } else if (this._flags.mode === MODE.ObjectLiteral ||
    (this._flags.mode === MODE.Statement && this._flags.parent.mode === MODE.ObjectLiteral)) {
    if (this._flags.mode === MODE.Statement) {
      this.restore_mode();
    }

    if (!this._flags.inline_frame) {
      this.print_newline();
    }
  } else if (this._options.comma_first) {
    // EXPR or DO_BLOCK
    // for comma-first, we want to allow a newline before the comma
    // to turn into a newline after the comma, which we will fixup later
    this.allow_wrap_or_preserved_newline(current_token);
  }
};

Beautifier$5.prototype.handle_operator = function(current_token) {
  var isGeneratorAsterisk = current_token.text === '*' &&
    (reserved_array(this._flags.last_token, ['function', 'yield']) ||
      (in_array$1(this._flags.last_token.type, [TOKEN$2.START_BLOCK, TOKEN$2.COMMA, TOKEN$2.END_BLOCK, TOKEN$2.SEMICOLON]))
    );
  var isUnary = in_array$1(current_token.text, ['-', '+']) && (
    in_array$1(this._flags.last_token.type, [TOKEN$2.START_BLOCK, TOKEN$2.START_EXPR, TOKEN$2.EQUALS, TOKEN$2.OPERATOR]) ||
    in_array$1(this._flags.last_token.text, line_starters) ||
    this._flags.last_token.text === ','
  );

  if (this.start_of_statement(current_token)) ; else {
    var preserve_statement_flags = !isGeneratorAsterisk;
    this.handle_whitespace_and_comments(current_token, preserve_statement_flags);
  }

  if (reserved_array(this._flags.last_token, special_words)) {
    // "return" had a special handling in TK_WORD. Now we need to return the favor
    this._output.space_before_token = true;
    this.print_token(current_token);
    return;
  }

  // hack for actionscript's import .*;
  if (current_token.text === '*' && this._flags.last_token.type === TOKEN$2.DOT) {
    this.print_token(current_token);
    return;
  }

  if (current_token.text === '::') {
    // no spaces around exotic namespacing syntax operator
    this.print_token(current_token);
    return;
  }

  // Allow line wrapping between operators when operator_position is
  //   set to before or preserve
  if (this._flags.last_token.type === TOKEN$2.OPERATOR && in_array$1(this._options.operator_position, OPERATOR_POSITION_BEFORE_OR_PRESERVE)) {
    this.allow_wrap_or_preserved_newline(current_token);
  }

  if (current_token.text === ':' && this._flags.in_case) {
    this.print_token(current_token);

    this._flags.in_case = false;
    this._flags.case_body = true;
    if (this._tokens.peek().type !== TOKEN$2.START_BLOCK) {
      this.indent();
      this.print_newline();
    } else {
      this._output.space_before_token = true;
    }
    return;
  }

  var space_before = true;
  var space_after = true;
  var in_ternary = false;
  if (current_token.text === ':') {
    if (this._flags.ternary_depth === 0) {
      // Colon is invalid javascript outside of ternary and object, but do our best to guess what was meant.
      space_before = false;
    } else {
      this._flags.ternary_depth -= 1;
      in_ternary = true;
    }
  } else if (current_token.text === '?') {
    this._flags.ternary_depth += 1;
  }

  // let's handle the operator_position option prior to any conflicting logic
  if (!isUnary && !isGeneratorAsterisk && this._options.preserve_newlines && in_array$1(current_token.text, positionable_operators)) {
    var isColon = current_token.text === ':';
    var isTernaryColon = (isColon && in_ternary);
    var isOtherColon = (isColon && !in_ternary);

    switch (this._options.operator_position) {
      case OPERATOR_POSITION.before_newline:
        // if the current token is : and it's not a ternary statement then we set space_before to false
        this._output.space_before_token = !isOtherColon;

        this.print_token(current_token);

        if (!isColon || isTernaryColon) {
          this.allow_wrap_or_preserved_newline(current_token);
        }

        this._output.space_before_token = true;
        return;

      case OPERATOR_POSITION.after_newline:
        // if the current token is anything but colon, or (via deduction) it's a colon and in a ternary statement,
        //   then print a newline.

        this._output.space_before_token = true;

        if (!isColon || isTernaryColon) {
          if (this._tokens.peek().newlines) {
            this.print_newline(false, true);
          } else {
            this.allow_wrap_or_preserved_newline(current_token);
          }
        } else {
          this._output.space_before_token = false;
        }

        this.print_token(current_token);

        this._output.space_before_token = true;
        return;

      case OPERATOR_POSITION.preserve_newline:
        if (!isOtherColon) {
          this.allow_wrap_or_preserved_newline(current_token);
        }

        // if we just added a newline, or the current token is : and it's not a ternary statement,
        //   then we set space_before to false
        space_before = !(this._output.just_added_newline() || isOtherColon);

        this._output.space_before_token = space_before;
        this.print_token(current_token);
        this._output.space_before_token = true;
        return;
    }
  }

  if (isGeneratorAsterisk) {
    this.allow_wrap_or_preserved_newline(current_token);
    space_before = false;
    var next_token = this._tokens.peek();
    space_after = next_token && in_array$1(next_token.type, [TOKEN$2.WORD, TOKEN$2.RESERVED]);
  } else if (current_token.text === '...') {
    this.allow_wrap_or_preserved_newline(current_token);
    space_before = this._flags.last_token.type === TOKEN$2.START_BLOCK;
    space_after = false;
  } else if (in_array$1(current_token.text, ['--', '++', '!', '~']) || isUnary) {
    // unary operators (and binary +/- pretending to be unary) special cases
    if (this._flags.last_token.type === TOKEN$2.COMMA || this._flags.last_token.type === TOKEN$2.START_EXPR) {
      this.allow_wrap_or_preserved_newline(current_token);
    }

    space_before = false;
    space_after = false;

    // http://www.ecma-international.org/ecma-262/5.1/#sec-7.9.1
    // if there is a newline between -- or ++ and anything else we should preserve it.
    if (current_token.newlines && (current_token.text === '--' || current_token.text === '++')) {
      this.print_newline(false, true);
    }

    if (this._flags.last_token.text === ';' && is_expression(this._flags.mode)) {
      // for (;; ++i)
      //        ^^^
      space_before = true;
    }

    if (this._flags.last_token.type === TOKEN$2.RESERVED) {
      space_before = true;
    } else if (this._flags.last_token.type === TOKEN$2.END_EXPR) {
      space_before = !(this._flags.last_token.text === ']' && (current_token.text === '--' || current_token.text === '++'));
    } else if (this._flags.last_token.type === TOKEN$2.OPERATOR) {
      // a++ + ++b;
      // a - -b
      space_before = in_array$1(current_token.text, ['--', '-', '++', '+']) && in_array$1(this._flags.last_token.text, ['--', '-', '++', '+']);
      // + and - are not unary when preceeded by -- or ++ operator
      // a-- + b
      // a * +b
      // a - -b
      if (in_array$1(current_token.text, ['+', '-']) && in_array$1(this._flags.last_token.text, ['--', '++'])) {
        space_after = true;
      }
    }


    if (((this._flags.mode === MODE.BlockStatement && !this._flags.inline_frame) || this._flags.mode === MODE.Statement) &&
      (this._flags.last_token.text === '{' || this._flags.last_token.text === ';')) {
      // { foo; --i }
      // foo(); --bar;
      this.print_newline();
    }
  }

  this._output.space_before_token = this._output.space_before_token || space_before;
  this.print_token(current_token);
  this._output.space_before_token = space_after;
};

Beautifier$5.prototype.handle_block_comment = function(current_token, preserve_statement_flags) {
  if (this._output.raw) {
    this._output.add_raw_token(current_token);
    if (current_token.directives && current_token.directives.preserve === 'end') {
      // If we're testing the raw output behavior, do not allow a directive to turn it off.
      this._output.raw = this._options.test_output_raw;
    }
    return;
  }

  if (current_token.directives) {
    this.print_newline(false, preserve_statement_flags);
    this.print_token(current_token);
    if (current_token.directives.preserve === 'start') {
      this._output.raw = true;
    }
    this.print_newline(false, true);
    return;
  }

  // inline block
  if (!acorn.newline.test(current_token.text) && !current_token.newlines) {
    this._output.space_before_token = true;
    this.print_token(current_token);
    this._output.space_before_token = true;
    return;
  } else {
    this.print_block_commment(current_token, preserve_statement_flags);
  }
};

Beautifier$5.prototype.print_block_commment = function(current_token, preserve_statement_flags) {
  var lines = split_linebreaks(current_token.text);
  var j; // iterator for this case
  var javadoc = false;
  var starless = false;
  var lastIndent = current_token.whitespace_before;
  var lastIndentLength = lastIndent.length;

  // block comment starts with a new line
  this.print_newline(false, preserve_statement_flags);

  // first line always indented
  this.print_token_line_indentation(current_token);
  this._output.add_token(lines[0]);
  this.print_newline(false, preserve_statement_flags);


  if (lines.length > 1) {
    lines = lines.slice(1);
    javadoc = all_lines_start_with(lines, '*');
    starless = each_line_matches_indent(lines, lastIndent);

    if (javadoc) {
      this._flags.alignment = 1;
    }

    for (j = 0; j < lines.length; j++) {
      if (javadoc) {
        // javadoc: reformat and re-indent
        this.print_token_line_indentation(current_token);
        this._output.add_token(ltrim(lines[j]));
      } else if (starless && lines[j]) {
        // starless: re-indent non-empty content, avoiding trim
        this.print_token_line_indentation(current_token);
        this._output.add_token(lines[j].substring(lastIndentLength));
      } else {
        // normal comments output raw
        this._output.current_line.set_indent(-1);
        this._output.add_token(lines[j]);
      }

      // for comments on their own line or  more than one line, make sure there's a new line after
      this.print_newline(false, preserve_statement_flags);
    }

    this._flags.alignment = 0;
  }
};


Beautifier$5.prototype.handle_comment = function(current_token, preserve_statement_flags) {
  if (current_token.newlines) {
    this.print_newline(false, preserve_statement_flags);
  } else {
    this._output.trim(true);
  }

  this._output.space_before_token = true;
  this.print_token(current_token);
  this.print_newline(false, preserve_statement_flags);
};

Beautifier$5.prototype.handle_dot = function(current_token) {
  if (this.start_of_statement(current_token)) ; else {
    this.handle_whitespace_and_comments(current_token, true);
  }

  if (reserved_array(this._flags.last_token, special_words)) {
    this._output.space_before_token = false;
  } else {
    // allow preserved newlines before dots in general
    // force newlines on dots after close paren when break_chained - for bar().baz()
    this.allow_wrap_or_preserved_newline(current_token,
      this._flags.last_token.text === ')' && this._options.break_chained_methods);
  }

  // Only unindent chained method dot if this dot starts a new line.
  // Otherwise the automatic extra indentation removal will handle the over indent
  if (this._options.unindent_chained_methods && this._output.just_added_newline()) {
    this.deindent();
  }

  this.print_token(current_token);
};

Beautifier$5.prototype.handle_unknown = function(current_token, preserve_statement_flags) {
  this.print_token(current_token);

  if (current_token.text[current_token.text.length - 1] === '\n') {
    this.print_newline(false, preserve_statement_flags);
  }
};

Beautifier$5.prototype.handle_eof = function(current_token) {
  // Unwind any open statements
  while (this._flags.mode === MODE.Statement) {
    this.restore_mode();
  }
  this.handle_whitespace_and_comments(current_token);
};

beautifier$2.Beautifier = Beautifier$5;

/*jshint node:true */

var Beautifier$4 = beautifier$2.Beautifier,
  Options$6 = options$3.Options;

function js_beautify$1(js_source_text, options) {
  var beautifier = new Beautifier$4(js_source_text, options);
  return beautifier.beautify();
}

javascript.exports = js_beautify$1;
javascript.exports.defaultOptions = function() {
  return new Options$6();
};

var css$1 = {exports: {}};

var beautifier$1 = {};

var options$1 = {};

/*jshint node:true */

var BaseOptions$1 = options$2.Options;

function Options$5(options) {
  BaseOptions$1.call(this, options, 'css');

  this.selector_separator_newline = this._get_boolean('selector_separator_newline', true);
  this.newline_between_rules = this._get_boolean('newline_between_rules', true);
  var space_around_selector_separator = this._get_boolean('space_around_selector_separator');
  this.space_around_combinator = this._get_boolean('space_around_combinator') || space_around_selector_separator;

  var brace_style_split = this._get_selection_list('brace_style', ['collapse', 'expand', 'end-expand', 'none', 'preserve-inline']);
  this.brace_style = 'collapse';
  for (var bs = 0; bs < brace_style_split.length; bs++) {
    if (brace_style_split[bs] !== 'expand') {
      // default to collapse, as only collapse|expand is implemented for now
      this.brace_style = 'collapse';
    } else {
      this.brace_style = brace_style_split[bs];
    }
  }
}
Options$5.prototype = new BaseOptions$1();



options$1.Options = Options$5;

/*jshint node:true */

var Options$4 = options$1.Options;
var Output$1 = output.Output;
var InputScanner = inputscanner.InputScanner;
var Directives$1 = directives.Directives;

var directives_core$1 = new Directives$1(/\/\*/, /\*\//);

var lineBreak$1 = /\r\n|[\r\n]/;
var allLineBreaks$1 = /\r\n|[\r\n]/g;

// tokenizer
var whitespaceChar = /\s/;
var whitespacePattern = /(?:\s|\n)+/g;
var block_comment_pattern = /\/\*(?:[\s\S]*?)((?:\*\/)|$)/g;
var comment_pattern = /\/\/(?:[^\n\r\u2028\u2029]*)/g;

function Beautifier$3(source_text, options) {
  this._source_text = source_text || '';
  // Allow the setting of language/file-type specific options
  // with inheritance of overall settings
  this._options = new Options$4(options);
  this._ch = null;
  this._input = null;

  // https://developer.mozilla.org/en-US/docs/Web/CSS/At-rule
  this.NESTED_AT_RULE = {
    "@page": true,
    "@font-face": true,
    "@keyframes": true,
    // also in CONDITIONAL_GROUP_RULE below
    "@media": true,
    "@supports": true,
    "@document": true
  };
  this.CONDITIONAL_GROUP_RULE = {
    "@media": true,
    "@supports": true,
    "@document": true
  };

}

Beautifier$3.prototype.eatString = function(endChars) {
  var result = '';
  this._ch = this._input.next();
  while (this._ch) {
    result += this._ch;
    if (this._ch === "\\") {
      result += this._input.next();
    } else if (endChars.indexOf(this._ch) !== -1 || this._ch === "\n") {
      break;
    }
    this._ch = this._input.next();
  }
  return result;
};

// Skips any white space in the source text from the current position.
// When allowAtLeastOneNewLine is true, will output new lines for each
// newline character found; if the user has preserve_newlines off, only
// the first newline will be output
Beautifier$3.prototype.eatWhitespace = function(allowAtLeastOneNewLine) {
  var result = whitespaceChar.test(this._input.peek());
  var newline_count = 0;
  while (whitespaceChar.test(this._input.peek())) {
    this._ch = this._input.next();
    if (allowAtLeastOneNewLine && this._ch === '\n') {
      if (newline_count === 0 || newline_count < this._options.max_preserve_newlines) {
        newline_count++;
        this._output.add_new_line(true);
      }
    }
  }
  return result;
};

// Nested pseudo-class if we are insideRule
// and the next special character found opens
// a new block
Beautifier$3.prototype.foundNestedPseudoClass = function() {
  var openParen = 0;
  var i = 1;
  var ch = this._input.peek(i);
  while (ch) {
    if (ch === "{") {
      return true;
    } else if (ch === '(') {
      // pseudoclasses can contain ()
      openParen += 1;
    } else if (ch === ')') {
      if (openParen === 0) {
        return false;
      }
      openParen -= 1;
    } else if (ch === ";" || ch === "}") {
      return false;
    }
    i++;
    ch = this._input.peek(i);
  }
  return false;
};

Beautifier$3.prototype.print_string = function(output_string) {
  this._output.set_indent(this._indentLevel);
  this._output.non_breaking_space = true;
  this._output.add_token(output_string);
};

Beautifier$3.prototype.preserveSingleSpace = function(isAfterSpace) {
  if (isAfterSpace) {
    this._output.space_before_token = true;
  }
};

Beautifier$3.prototype.indent = function() {
  this._indentLevel++;
};

Beautifier$3.prototype.outdent = function() {
  if (this._indentLevel > 0) {
    this._indentLevel--;
  }
};

/*_____________________--------------------_____________________*/

Beautifier$3.prototype.beautify = function() {
  if (this._options.disabled) {
    return this._source_text;
  }

  var source_text = this._source_text;
  var eol = this._options.eol;
  if (eol === 'auto') {
    eol = '\n';
    if (source_text && lineBreak$1.test(source_text || '')) {
      eol = source_text.match(lineBreak$1)[0];
    }
  }


  // HACK: newline parsing inconsistent. This brute force normalizes the this._input.
  source_text = source_text.replace(allLineBreaks$1, '\n');

  // reset
  var baseIndentString = source_text.match(/^[\t ]*/)[0];

  this._output = new Output$1(this._options, baseIndentString);
  this._input = new InputScanner(source_text);
  this._indentLevel = 0;
  this._nestedLevel = 0;

  this._ch = null;
  var parenLevel = 0;

  var insideRule = false;
  // This is the value side of a property value pair (blue in the following ex)
  // label { content: blue }
  var insidePropertyValue = false;
  var enteringConditionalGroup = false;
  var insideAtExtend = false;
  var insideAtImport = false;
  var topCharacter = this._ch;
  var whitespace;
  var isAfterSpace;
  var previous_ch;

  while (true) {
    whitespace = this._input.read(whitespacePattern);
    isAfterSpace = whitespace !== '';
    previous_ch = topCharacter;
    this._ch = this._input.next();
    if (this._ch === '\\' && this._input.hasNext()) {
      this._ch += this._input.next();
    }
    topCharacter = this._ch;

    if (!this._ch) {
      break;
    } else if (this._ch === '/' && this._input.peek() === '*') {
      // /* css comment */
      // Always start block comments on a new line.
      // This handles scenarios where a block comment immediately
      // follows a property definition on the same line or where
      // minified code is being beautified.
      this._output.add_new_line();
      this._input.back();

      var comment = this._input.read(block_comment_pattern);

      // Handle ignore directive
      var directives = directives_core$1.get_directives(comment);
      if (directives && directives.ignore === 'start') {
        comment += directives_core$1.readIgnored(this._input);
      }

      this.print_string(comment);

      // Ensures any new lines following the comment are preserved
      this.eatWhitespace(true);

      // Block comments are followed by a new line so they don't
      // share a line with other properties
      this._output.add_new_line();
    } else if (this._ch === '/' && this._input.peek() === '/') {
      // // single line comment
      // Preserves the space before a comment
      // on the same line as a rule
      this._output.space_before_token = true;
      this._input.back();
      this.print_string(this._input.read(comment_pattern));

      // Ensures any new lines following the comment are preserved
      this.eatWhitespace(true);
    } else if (this._ch === '@') {
      this.preserveSingleSpace(isAfterSpace);

      // deal with less propery mixins @{...}
      if (this._input.peek() === '{') {
        this.print_string(this._ch + this.eatString('}'));
      } else {
        this.print_string(this._ch);

        // strip trailing space, if present, for hash property checks
        var variableOrRule = this._input.peekUntilAfter(/[: ,;{}()[\]\/='"]/g);

        if (variableOrRule.match(/[ :]$/)) {
          // we have a variable or pseudo-class, add it and insert one space before continuing
          variableOrRule = this.eatString(": ").replace(/\s$/, '');
          this.print_string(variableOrRule);
          this._output.space_before_token = true;
        }

        variableOrRule = variableOrRule.replace(/\s$/, '');

        if (variableOrRule === 'extend') {
          insideAtExtend = true;
        } else if (variableOrRule === 'import') {
          insideAtImport = true;
        }

        // might be a nesting at-rule
        if (variableOrRule in this.NESTED_AT_RULE) {
          this._nestedLevel += 1;
          if (variableOrRule in this.CONDITIONAL_GROUP_RULE) {
            enteringConditionalGroup = true;
          }
          // might be less variable
        } else if (!insideRule && parenLevel === 0 && variableOrRule.indexOf(':') !== -1) {
          insidePropertyValue = true;
          this.indent();
        }
      }
    } else if (this._ch === '#' && this._input.peek() === '{') {
      this.preserveSingleSpace(isAfterSpace);
      this.print_string(this._ch + this.eatString('}'));
    } else if (this._ch === '{') {
      if (insidePropertyValue) {
        insidePropertyValue = false;
        this.outdent();
      }

      // when entering conditional groups, only rulesets are allowed
      if (enteringConditionalGroup) {
        enteringConditionalGroup = false;
        insideRule = (this._indentLevel >= this._nestedLevel);
      } else {
        // otherwise, declarations are also allowed
        insideRule = (this._indentLevel >= this._nestedLevel - 1);
      }
      if (this._options.newline_between_rules && insideRule) {
        if (this._output.previous_line && this._output.previous_line.item(-1) !== '{') {
          this._output.ensure_empty_line_above('/', ',');
        }
      }

      this._output.space_before_token = true;

      // The difference in print_string and indent order is necessary to indent the '{' correctly
      if (this._options.brace_style === 'expand') {
        this._output.add_new_line();
        this.print_string(this._ch);
        this.indent();
        this._output.set_indent(this._indentLevel);
      } else {
        this.indent();
        this.print_string(this._ch);
      }

      this.eatWhitespace(true);
      this._output.add_new_line();
    } else if (this._ch === '}') {
      this.outdent();
      this._output.add_new_line();
      if (previous_ch === '{') {
        this._output.trim(true);
      }
      insideAtImport = false;
      insideAtExtend = false;
      if (insidePropertyValue) {
        this.outdent();
        insidePropertyValue = false;
      }
      this.print_string(this._ch);
      insideRule = false;
      if (this._nestedLevel) {
        this._nestedLevel--;
      }

      this.eatWhitespace(true);
      this._output.add_new_line();

      if (this._options.newline_between_rules && !this._output.just_added_blankline()) {
        if (this._input.peek() !== '}') {
          this._output.add_new_line(true);
        }
      }
    } else if (this._ch === ":") {
      if ((insideRule || enteringConditionalGroup) && !(this._input.lookBack("&") || this.foundNestedPseudoClass()) && !this._input.lookBack("(") && !insideAtExtend && parenLevel === 0) {
        // 'property: value' delimiter
        // which could be in a conditional group query
        this.print_string(':');
        if (!insidePropertyValue) {
          insidePropertyValue = true;
          this._output.space_before_token = true;
          this.eatWhitespace(true);
          this.indent();
        }
      } else {
        // sass/less parent reference don't use a space
        // sass nested pseudo-class don't use a space

        // preserve space before pseudoclasses/pseudoelements, as it means "in any child"
        if (this._input.lookBack(" ")) {
          this._output.space_before_token = true;
        }
        if (this._input.peek() === ":") {
          // pseudo-element
          this._ch = this._input.next();
          this.print_string("::");
        } else {
          // pseudo-class
          this.print_string(':');
        }
      }
    } else if (this._ch === '"' || this._ch === '\'') {
      this.preserveSingleSpace(isAfterSpace);
      this.print_string(this._ch + this.eatString(this._ch));
      this.eatWhitespace(true);
    } else if (this._ch === ';') {
      if (parenLevel === 0) {
        if (insidePropertyValue) {
          this.outdent();
          insidePropertyValue = false;
        }
        insideAtExtend = false;
        insideAtImport = false;
        this.print_string(this._ch);
        this.eatWhitespace(true);

        // This maintains single line comments on the same
        // line. Block comments are also affected, but
        // a new line is always output before one inside
        // that section
        if (this._input.peek() !== '/') {
          this._output.add_new_line();
        }
      } else {
        this.print_string(this._ch);
        this.eatWhitespace(true);
        this._output.space_before_token = true;
      }
    } else if (this._ch === '(') { // may be a url
      if (this._input.lookBack("url")) {
        this.print_string(this._ch);
        this.eatWhitespace();
        parenLevel++;
        this.indent();
        this._ch = this._input.next();
        if (this._ch === ')' || this._ch === '"' || this._ch === '\'') {
          this._input.back();
        } else if (this._ch) {
          this.print_string(this._ch + this.eatString(')'));
          if (parenLevel) {
            parenLevel--;
            this.outdent();
          }
        }
      } else {
        this.preserveSingleSpace(isAfterSpace);
        this.print_string(this._ch);
        this.eatWhitespace();
        parenLevel++;
        this.indent();
      }
    } else if (this._ch === ')') {
      if (parenLevel) {
        parenLevel--;
        this.outdent();
      }
      this.print_string(this._ch);
    } else if (this._ch === ',') {
      this.print_string(this._ch);
      this.eatWhitespace(true);
      if (this._options.selector_separator_newline && !insidePropertyValue && parenLevel === 0 && !insideAtImport && !insideAtExtend) {
        this._output.add_new_line();
      } else {
        this._output.space_before_token = true;
      }
    } else if ((this._ch === '>' || this._ch === '+' || this._ch === '~') && !insidePropertyValue && parenLevel === 0) {
      //handle combinator spacing
      if (this._options.space_around_combinator) {
        this._output.space_before_token = true;
        this.print_string(this._ch);
        this._output.space_before_token = true;
      } else {
        this.print_string(this._ch);
        this.eatWhitespace();
        // squash extra whitespace
        if (this._ch && whitespaceChar.test(this._ch)) {
          this._ch = '';
        }
      }
    } else if (this._ch === ']') {
      this.print_string(this._ch);
    } else if (this._ch === '[') {
      this.preserveSingleSpace(isAfterSpace);
      this.print_string(this._ch);
    } else if (this._ch === '=') { // no whitespace before or after
      this.eatWhitespace();
      this.print_string('=');
      if (whitespaceChar.test(this._ch)) {
        this._ch = '';
      }
    } else if (this._ch === '!' && !this._input.lookBack("\\")) { // !important
      this.print_string(' ');
      this.print_string(this._ch);
    } else {
      this.preserveSingleSpace(isAfterSpace);
      this.print_string(this._ch);
    }
  }

  var sweetCode = this._output.get_code(eol);

  return sweetCode;
};

beautifier$1.Beautifier = Beautifier$3;

/*jshint node:true */

var Beautifier$2 = beautifier$1.Beautifier,
  Options$3 = options$1.Options;

function css_beautify$1(source_text, options) {
  var beautifier = new Beautifier$2(source_text, options);
  return beautifier.beautify();
}

css$1.exports = css_beautify$1;
css$1.exports.defaultOptions = function() {
  return new Options$3();
};

var html = {exports: {}};

var beautifier = {};

var options = {};

/*jshint node:true */

var BaseOptions = options$2.Options;

function Options$2(options) {
  BaseOptions.call(this, options, 'html');
  if (this.templating.length === 1 && this.templating[0] === 'auto') {
    this.templating = ['django', 'erb', 'handlebars', 'php'];
  }

  this.indent_inner_html = this._get_boolean('indent_inner_html');
  this.indent_body_inner_html = this._get_boolean('indent_body_inner_html', true);
  this.indent_head_inner_html = this._get_boolean('indent_head_inner_html', true);

  this.indent_handlebars = this._get_boolean('indent_handlebars', true);
  this.wrap_attributes = this._get_selection('wrap_attributes',
    ['auto', 'force', 'force-aligned', 'force-expand-multiline', 'aligned-multiple', 'preserve', 'preserve-aligned']);
  this.wrap_attributes_indent_size = this._get_number('wrap_attributes_indent_size', this.indent_size);
  this.extra_liners = this._get_array('extra_liners', ['head', 'body', '/html']);

  // Block vs inline elements
  // https://developer.mozilla.org/en-US/docs/Web/HTML/Block-level_elements
  // https://developer.mozilla.org/en-US/docs/Web/HTML/Inline_elements
  // https://www.w3.org/TR/html5/dom.html#phrasing-content
  this.inline = this._get_array('inline', [
    'a', 'abbr', 'area', 'audio', 'b', 'bdi', 'bdo', 'br', 'button', 'canvas', 'cite',
    'code', 'data', 'datalist', 'del', 'dfn', 'em', 'embed', 'i', 'iframe', 'img',
    'input', 'ins', 'kbd', 'keygen', 'label', 'map', 'mark', 'math', 'meter', 'noscript',
    'object', 'output', 'progress', 'q', 'ruby', 's', 'samp', /* 'script', */ 'select', 'small',
    'span', 'strong', 'sub', 'sup', 'svg', 'template', 'textarea', 'time', 'u', 'var',
    'video', 'wbr', 'text',
    // obsolete inline tags
    'acronym', 'big', 'strike', 'tt'
  ]);
  this.void_elements = this._get_array('void_elements', [
    // HTLM void elements - aka self-closing tags - aka singletons
    // https://www.w3.org/html/wg/drafts/html/master/syntax.html#void-elements
    'area', 'base', 'br', 'col', 'embed', 'hr', 'img', 'input', 'keygen',
    'link', 'menuitem', 'meta', 'param', 'source', 'track', 'wbr',
    // NOTE: Optional tags are too complex for a simple list
    // they are hard coded in _do_optional_end_element

    // Doctype and xml elements
    '!doctype', '?xml',

    // obsolete tags
    // basefont: https://www.computerhope.com/jargon/h/html-basefont-tag.htm
    // isndex: https://developer.mozilla.org/en-US/docs/Web/HTML/Element/isindex
    'basefont', 'isindex'
  ]);
  this.unformatted = this._get_array('unformatted', []);
  this.content_unformatted = this._get_array('content_unformatted', [
    'pre', 'textarea'
  ]);
  this.unformatted_content_delimiter = this._get_characters('unformatted_content_delimiter');
  this.indent_scripts = this._get_selection('indent_scripts', ['normal', 'keep', 'separate']);

}
Options$2.prototype = new BaseOptions();



options.Options = Options$2;

var tokenizer = {};

/*jshint node:true */

var BaseTokenizer = tokenizer$1.Tokenizer;
var BASETOKEN = tokenizer$1.TOKEN;
var Directives = directives.Directives;
var TemplatablePattern = templatablepattern.TemplatablePattern;
var Pattern = pattern.Pattern;

var TOKEN$1 = {
  TAG_OPEN: 'TK_TAG_OPEN',
  TAG_CLOSE: 'TK_TAG_CLOSE',
  ATTRIBUTE: 'TK_ATTRIBUTE',
  EQUALS: 'TK_EQUALS',
  VALUE: 'TK_VALUE',
  COMMENT: 'TK_COMMENT',
  TEXT: 'TK_TEXT',
  UNKNOWN: 'TK_UNKNOWN',
  START: BASETOKEN.START,
  RAW: BASETOKEN.RAW,
  EOF: BASETOKEN.EOF
};

var directives_core = new Directives(/<\!--/, /-->/);

var Tokenizer$1 = function(input_string, options) {
  BaseTokenizer.call(this, input_string, options);
  this._current_tag_name = '';

  // Words end at whitespace or when a tag starts
  // if we are indenting handlebars, they are considered tags
  var templatable_reader = new TemplatablePattern(this._input).read_options(this._options);
  var pattern_reader = new Pattern(this._input);

  this.__patterns = {
    word: templatable_reader.until(/[\n\r\t <]/),
    single_quote: templatable_reader.until_after(/'/),
    double_quote: templatable_reader.until_after(/"/),
    attribute: templatable_reader.until(/[\n\r\t =>]|\/>/),
    element_name: templatable_reader.until(/[\n\r\t >\/]/),

    handlebars_comment: pattern_reader.starting_with(/{{!--/).until_after(/--}}/),
    handlebars: pattern_reader.starting_with(/{{/).until_after(/}}/),
    handlebars_open: pattern_reader.until(/[\n\r\t }]/),
    handlebars_raw_close: pattern_reader.until(/}}/),
    comment: pattern_reader.starting_with(/<!--/).until_after(/-->/),
    cdata: pattern_reader.starting_with(/<!\[CDATA\[/).until_after(/]]>/),
    // https://en.wikipedia.org/wiki/Conditional_comment
    conditional_comment: pattern_reader.starting_with(/<!\[/).until_after(/]>/),
    processing: pattern_reader.starting_with(/<\?/).until_after(/\?>/)
  };

  if (this._options.indent_handlebars) {
    this.__patterns.word = this.__patterns.word.exclude('handlebars');
  }

  this._unformatted_content_delimiter = null;

  if (this._options.unformatted_content_delimiter) {
    var literal_regexp = this._input.get_literal_regexp(this._options.unformatted_content_delimiter);
    this.__patterns.unformatted_content_delimiter =
      pattern_reader.matching(literal_regexp)
      .until_after(literal_regexp);
  }
};
Tokenizer$1.prototype = new BaseTokenizer();

Tokenizer$1.prototype._is_comment = function(current_token) { // jshint unused:false
  return false; //current_token.type === TOKEN.COMMENT || current_token.type === TOKEN.UNKNOWN;
};

Tokenizer$1.prototype._is_opening = function(current_token) {
  return current_token.type === TOKEN$1.TAG_OPEN;
};

Tokenizer$1.prototype._is_closing = function(current_token, open_token) {
  return current_token.type === TOKEN$1.TAG_CLOSE &&
    (open_token && (
      ((current_token.text === '>' || current_token.text === '/>') && open_token.text[0] === '<') ||
      (current_token.text === '}}' && open_token.text[0] === '{' && open_token.text[1] === '{')));
};

Tokenizer$1.prototype._reset = function() {
  this._current_tag_name = '';
};

Tokenizer$1.prototype._get_next_token = function(previous_token, open_token) { // jshint unused:false
  var token = null;
  this._readWhitespace();
  var c = this._input.peek();

  if (c === null) {
    return this._create_token(TOKEN$1.EOF, '');
  }

  token = token || this._read_open_handlebars(c, open_token);
  token = token || this._read_attribute(c, previous_token, open_token);
  token = token || this._read_close(c, open_token);
  token = token || this._read_raw_content(c, previous_token, open_token);
  token = token || this._read_content_word(c);
  token = token || this._read_comment_or_cdata(c);
  token = token || this._read_processing(c);
  token = token || this._read_open(c, open_token);
  token = token || this._create_token(TOKEN$1.UNKNOWN, this._input.next());

  return token;
};

Tokenizer$1.prototype._read_comment_or_cdata = function(c) { // jshint unused:false
  var token = null;
  var resulting_string = null;
  var directives = null;

  if (c === '<') {
    var peek1 = this._input.peek(1);
    // We treat all comments as literals, even more than preformatted tags
    // we only look for the appropriate closing marker
    if (peek1 === '!') {
      resulting_string = this.__patterns.comment.read();

      // only process directive on html comments
      if (resulting_string) {
        directives = directives_core.get_directives(resulting_string);
        if (directives && directives.ignore === 'start') {
          resulting_string += directives_core.readIgnored(this._input);
        }
      } else {
        resulting_string = this.__patterns.cdata.read();
      }
    }

    if (resulting_string) {
      token = this._create_token(TOKEN$1.COMMENT, resulting_string);
      token.directives = directives;
    }
  }

  return token;
};

Tokenizer$1.prototype._read_processing = function(c) { // jshint unused:false
  var token = null;
  var resulting_string = null;
  var directives = null;

  if (c === '<') {
    var peek1 = this._input.peek(1);
    if (peek1 === '!' || peek1 === '?') {
      resulting_string = this.__patterns.conditional_comment.read();
      resulting_string = resulting_string || this.__patterns.processing.read();
    }

    if (resulting_string) {
      token = this._create_token(TOKEN$1.COMMENT, resulting_string);
      token.directives = directives;
    }
  }

  return token;
};

Tokenizer$1.prototype._read_open = function(c, open_token) {
  var resulting_string = null;
  var token = null;
  if (!open_token) {
    if (c === '<') {

      resulting_string = this._input.next();
      if (this._input.peek() === '/') {
        resulting_string += this._input.next();
      }
      resulting_string += this.__patterns.element_name.read();
      token = this._create_token(TOKEN$1.TAG_OPEN, resulting_string);
    }
  }
  return token;
};

Tokenizer$1.prototype._read_open_handlebars = function(c, open_token) {
  var resulting_string = null;
  var token = null;
  if (!open_token) {
    if (this._options.indent_handlebars && c === '{' && this._input.peek(1) === '{') {
      if (this._input.peek(2) === '!') {
        resulting_string = this.__patterns.handlebars_comment.read();
        resulting_string = resulting_string || this.__patterns.handlebars.read();
        token = this._create_token(TOKEN$1.COMMENT, resulting_string);
      } else {
        resulting_string = this.__patterns.handlebars_open.read();
        token = this._create_token(TOKEN$1.TAG_OPEN, resulting_string);
      }
    }
  }
  return token;
};


Tokenizer$1.prototype._read_close = function(c, open_token) {
  var resulting_string = null;
  var token = null;
  if (open_token) {
    if (open_token.text[0] === '<' && (c === '>' || (c === '/' && this._input.peek(1) === '>'))) {
      resulting_string = this._input.next();
      if (c === '/') { //  for close tag "/>"
        resulting_string += this._input.next();
      }
      token = this._create_token(TOKEN$1.TAG_CLOSE, resulting_string);
    } else if (open_token.text[0] === '{' && c === '}' && this._input.peek(1) === '}') {
      this._input.next();
      this._input.next();
      token = this._create_token(TOKEN$1.TAG_CLOSE, '}}');
    }
  }

  return token;
};

Tokenizer$1.prototype._read_attribute = function(c, previous_token, open_token) {
  var token = null;
  var resulting_string = '';
  if (open_token && open_token.text[0] === '<') {

    if (c === '=') {
      token = this._create_token(TOKEN$1.EQUALS, this._input.next());
    } else if (c === '"' || c === "'") {
      var content = this._input.next();
      if (c === '"') {
        content += this.__patterns.double_quote.read();
      } else {
        content += this.__patterns.single_quote.read();
      }
      token = this._create_token(TOKEN$1.VALUE, content);
    } else {
      resulting_string = this.__patterns.attribute.read();

      if (resulting_string) {
        if (previous_token.type === TOKEN$1.EQUALS) {
          token = this._create_token(TOKEN$1.VALUE, resulting_string);
        } else {
          token = this._create_token(TOKEN$1.ATTRIBUTE, resulting_string);
        }
      }
    }
  }
  return token;
};

Tokenizer$1.prototype._is_content_unformatted = function(tag_name) {
  // void_elements have no content and so cannot have unformatted content
  // script and style tags should always be read as unformatted content
  // finally content_unformatted and unformatted element contents are unformatted
  return this._options.void_elements.indexOf(tag_name) === -1 &&
    (this._options.content_unformatted.indexOf(tag_name) !== -1 ||
      this._options.unformatted.indexOf(tag_name) !== -1);
};


Tokenizer$1.prototype._read_raw_content = function(c, previous_token, open_token) { // jshint unused:false
  var resulting_string = '';
  if (open_token && open_token.text[0] === '{') {
    resulting_string = this.__patterns.handlebars_raw_close.read();
  } else if (previous_token.type === TOKEN$1.TAG_CLOSE &&
    previous_token.opened.text[0] === '<' && previous_token.text[0] !== '/') {
    // ^^ empty tag has no content 
    var tag_name = previous_token.opened.text.substr(1).toLowerCase();
    if (tag_name === 'script' || tag_name === 'style') {
      // Script and style tags are allowed to have comments wrapping their content
      // or just have regular content.
      var token = this._read_comment_or_cdata(c);
      if (token) {
        token.type = TOKEN$1.TEXT;
        return token;
      }
      resulting_string = this._input.readUntil(new RegExp('</' + tag_name + '[\\n\\r\\t ]*?>', 'ig'));
    } else if (this._is_content_unformatted(tag_name)) {

      resulting_string = this._input.readUntil(new RegExp('</' + tag_name + '[\\n\\r\\t ]*?>', 'ig'));
    }
  }

  if (resulting_string) {
    return this._create_token(TOKEN$1.TEXT, resulting_string);
  }

  return null;
};

Tokenizer$1.prototype._read_content_word = function(c) {
  var resulting_string = '';
  if (this._options.unformatted_content_delimiter) {
    if (c === this._options.unformatted_content_delimiter[0]) {
      resulting_string = this.__patterns.unformatted_content_delimiter.read();
    }
  }

  if (!resulting_string) {
    resulting_string = this.__patterns.word.read();
  }
  if (resulting_string) {
    return this._create_token(TOKEN$1.TEXT, resulting_string);
  }
};

tokenizer.Tokenizer = Tokenizer$1;
tokenizer.TOKEN = TOKEN$1;

/*jshint node:true */

var Options$1 = options.Options;
var Output = output.Output;
var Tokenizer = tokenizer.Tokenizer;
var TOKEN = tokenizer.TOKEN;

var lineBreak = /\r\n|[\r\n]/;
var allLineBreaks = /\r\n|[\r\n]/g;

var Printer = function(options, base_indent_string) { //handles input/output and some other printing functions

  this.indent_level = 0;
  this.alignment_size = 0;
  this.max_preserve_newlines = options.max_preserve_newlines;
  this.preserve_newlines = options.preserve_newlines;

  this._output = new Output(options, base_indent_string);

};

Printer.prototype.current_line_has_match = function(pattern) {
  return this._output.current_line.has_match(pattern);
};

Printer.prototype.set_space_before_token = function(value, non_breaking) {
  this._output.space_before_token = value;
  this._output.non_breaking_space = non_breaking;
};

Printer.prototype.set_wrap_point = function() {
  this._output.set_indent(this.indent_level, this.alignment_size);
  this._output.set_wrap_point();
};


Printer.prototype.add_raw_token = function(token) {
  this._output.add_raw_token(token);
};

Printer.prototype.print_preserved_newlines = function(raw_token) {
  var newlines = 0;
  if (raw_token.type !== TOKEN.TEXT && raw_token.previous.type !== TOKEN.TEXT) {
    newlines = raw_token.newlines ? 1 : 0;
  }

  if (this.preserve_newlines) {
    newlines = raw_token.newlines < this.max_preserve_newlines + 1 ? raw_token.newlines : this.max_preserve_newlines + 1;
  }
  for (var n = 0; n < newlines; n++) {
    this.print_newline(n > 0);
  }

  return newlines !== 0;
};

Printer.prototype.traverse_whitespace = function(raw_token) {
  if (raw_token.whitespace_before || raw_token.newlines) {
    if (!this.print_preserved_newlines(raw_token)) {
      this._output.space_before_token = true;
    }
    return true;
  }
  return false;
};

Printer.prototype.previous_token_wrapped = function() {
  return this._output.previous_token_wrapped;
};

Printer.prototype.print_newline = function(force) {
  this._output.add_new_line(force);
};

Printer.prototype.print_token = function(token) {
  if (token.text) {
    this._output.set_indent(this.indent_level, this.alignment_size);
    this._output.add_token(token.text);
  }
};

Printer.prototype.indent = function() {
  this.indent_level++;
};

Printer.prototype.get_full_indent = function(level) {
  level = this.indent_level + (level || 0);
  if (level < 1) {
    return '';
  }

  return this._output.get_indent_string(level);
};

var get_type_attribute = function(start_token) {
  var result = null;
  var raw_token = start_token.next;

  // Search attributes for a type attribute
  while (raw_token.type !== TOKEN.EOF && start_token.closed !== raw_token) {
    if (raw_token.type === TOKEN.ATTRIBUTE && raw_token.text === 'type') {
      if (raw_token.next && raw_token.next.type === TOKEN.EQUALS &&
        raw_token.next.next && raw_token.next.next.type === TOKEN.VALUE) {
        result = raw_token.next.next.text;
      }
      break;
    }
    raw_token = raw_token.next;
  }

  return result;
};

var get_custom_beautifier_name = function(tag_check, raw_token) {
  var typeAttribute = null;
  var result = null;

  if (!raw_token.closed) {
    return null;
  }

  if (tag_check === 'script') {
    typeAttribute = 'text/javascript';
  } else if (tag_check === 'style') {
    typeAttribute = 'text/css';
  }

  typeAttribute = get_type_attribute(raw_token) || typeAttribute;

  // For script and style tags that have a type attribute, only enable custom beautifiers for matching values
  // For those without a type attribute use default;
  if (typeAttribute.search('text/css') > -1) {
    result = 'css';
  } else if (typeAttribute.search(/module|((text|application|dojo)\/(x-)?(javascript|ecmascript|jscript|livescript|(ld\+)?json|method|aspect))/) > -1) {
    result = 'javascript';
  } else if (typeAttribute.search(/(text|application|dojo)\/(x-)?(html)/) > -1) {
    result = 'html';
  } else if (typeAttribute.search(/test\/null/) > -1) {
    // Test only mime-type for testing the beautifier when null is passed as beautifing function
    result = 'null';
  }

  return result;
};

function in_array(what, arr) {
  return arr.indexOf(what) !== -1;
}

function TagFrame(parent, parser_token, indent_level) {
  this.parent = parent || null;
  this.tag = parser_token ? parser_token.tag_name : '';
  this.indent_level = indent_level || 0;
  this.parser_token = parser_token || null;
}

function TagStack(printer) {
  this._printer = printer;
  this._current_frame = null;
}

TagStack.prototype.get_parser_token = function() {
  return this._current_frame ? this._current_frame.parser_token : null;
};

TagStack.prototype.record_tag = function(parser_token) { //function to record a tag and its parent in this.tags Object
  var new_frame = new TagFrame(this._current_frame, parser_token, this._printer.indent_level);
  this._current_frame = new_frame;
};

TagStack.prototype._try_pop_frame = function(frame) { //function to retrieve the opening tag to the corresponding closer
  var parser_token = null;

  if (frame) {
    parser_token = frame.parser_token;
    this._printer.indent_level = frame.indent_level;
    this._current_frame = frame.parent;
  }

  return parser_token;
};

TagStack.prototype._get_frame = function(tag_list, stop_list) { //function to retrieve the opening tag to the corresponding closer
  var frame = this._current_frame;

  while (frame) { //till we reach '' (the initial value);
    if (tag_list.indexOf(frame.tag) !== -1) { //if this is it use it
      break;
    } else if (stop_list && stop_list.indexOf(frame.tag) !== -1) {
      frame = null;
      break;
    }
    frame = frame.parent;
  }

  return frame;
};

TagStack.prototype.try_pop = function(tag, stop_list) { //function to retrieve the opening tag to the corresponding closer
  var frame = this._get_frame([tag], stop_list);
  return this._try_pop_frame(frame);
};

TagStack.prototype.indent_to_tag = function(tag_list) {
  var frame = this._get_frame(tag_list);
  if (frame) {
    this._printer.indent_level = frame.indent_level;
  }
};

function Beautifier$1(source_text, options, js_beautify, css_beautify) {
  //Wrapper function to invoke all the necessary constructors and deal with the output.
  this._source_text = source_text || '';
  options = options || {};
  this._js_beautify = js_beautify;
  this._css_beautify = css_beautify;
  this._tag_stack = null;

  // Allow the setting of language/file-type specific options
  // with inheritance of overall settings
  var optionHtml = new Options$1(options, 'html');

  this._options = optionHtml;

  this._is_wrap_attributes_force = this._options.wrap_attributes.substr(0, 'force'.length) === 'force';
  this._is_wrap_attributes_force_expand_multiline = (this._options.wrap_attributes === 'force-expand-multiline');
  this._is_wrap_attributes_force_aligned = (this._options.wrap_attributes === 'force-aligned');
  this._is_wrap_attributes_aligned_multiple = (this._options.wrap_attributes === 'aligned-multiple');
  this._is_wrap_attributes_preserve = this._options.wrap_attributes.substr(0, 'preserve'.length) === 'preserve';
  this._is_wrap_attributes_preserve_aligned = (this._options.wrap_attributes === 'preserve-aligned');
}

Beautifier$1.prototype.beautify = function() {

  // if disabled, return the input unchanged.
  if (this._options.disabled) {
    return this._source_text;
  }

  var source_text = this._source_text;
  var eol = this._options.eol;
  if (this._options.eol === 'auto') {
    eol = '\n';
    if (source_text && lineBreak.test(source_text)) {
      eol = source_text.match(lineBreak)[0];
    }
  }

  // HACK: newline parsing inconsistent. This brute force normalizes the input.
  source_text = source_text.replace(allLineBreaks, '\n');

  var baseIndentString = source_text.match(/^[\t ]*/)[0];

  var last_token = {
    text: '',
    type: ''
  };

  var last_tag_token = new TagOpenParserToken();

  var printer = new Printer(this._options, baseIndentString);
  var tokens = new Tokenizer(source_text, this._options).tokenize();

  this._tag_stack = new TagStack(printer);

  var parser_token = null;
  var raw_token = tokens.next();
  while (raw_token.type !== TOKEN.EOF) {

    if (raw_token.type === TOKEN.TAG_OPEN || raw_token.type === TOKEN.COMMENT) {
      parser_token = this._handle_tag_open(printer, raw_token, last_tag_token, last_token);
      last_tag_token = parser_token;
    } else if ((raw_token.type === TOKEN.ATTRIBUTE || raw_token.type === TOKEN.EQUALS || raw_token.type === TOKEN.VALUE) ||
      (raw_token.type === TOKEN.TEXT && !last_tag_token.tag_complete)) {
      parser_token = this._handle_inside_tag(printer, raw_token, last_tag_token, tokens);
    } else if (raw_token.type === TOKEN.TAG_CLOSE) {
      parser_token = this._handle_tag_close(printer, raw_token, last_tag_token);
    } else if (raw_token.type === TOKEN.TEXT) {
      parser_token = this._handle_text(printer, raw_token, last_tag_token);
    } else {
      // This should never happen, but if it does. Print the raw token
      printer.add_raw_token(raw_token);
    }

    last_token = parser_token;

    raw_token = tokens.next();
  }
  var sweet_code = printer._output.get_code(eol);

  return sweet_code;
};

Beautifier$1.prototype._handle_tag_close = function(printer, raw_token, last_tag_token) {
  var parser_token = {
    text: raw_token.text,
    type: raw_token.type
  };
  printer.alignment_size = 0;
  last_tag_token.tag_complete = true;

  printer.set_space_before_token(raw_token.newlines || raw_token.whitespace_before !== '', true);
  if (last_tag_token.is_unformatted) {
    printer.add_raw_token(raw_token);
  } else {
    if (last_tag_token.tag_start_char === '<') {
      printer.set_space_before_token(raw_token.text[0] === '/', true); // space before />, no space before >
      if (this._is_wrap_attributes_force_expand_multiline && last_tag_token.has_wrapped_attrs) {
        printer.print_newline(false);
      }
    }
    printer.print_token(raw_token);

  }

  if (last_tag_token.indent_content &&
    !(last_tag_token.is_unformatted || last_tag_token.is_content_unformatted)) {
    printer.indent();

    // only indent once per opened tag
    last_tag_token.indent_content = false;
  }

  if (!last_tag_token.is_inline_element &&
    !(last_tag_token.is_unformatted || last_tag_token.is_content_unformatted)) {
    printer.set_wrap_point();
  }

  return parser_token;
};

Beautifier$1.prototype._handle_inside_tag = function(printer, raw_token, last_tag_token, tokens) {
  var wrapped = last_tag_token.has_wrapped_attrs;
  var parser_token = {
    text: raw_token.text,
    type: raw_token.type
  };

  printer.set_space_before_token(raw_token.newlines || raw_token.whitespace_before !== '', true);
  if (last_tag_token.is_unformatted) {
    printer.add_raw_token(raw_token);
  } else if (last_tag_token.tag_start_char === '{' && raw_token.type === TOKEN.TEXT) {
    // For the insides of handlebars allow newlines or a single space between open and contents
    if (printer.print_preserved_newlines(raw_token)) {
      raw_token.newlines = 0;
      printer.add_raw_token(raw_token);
    } else {
      printer.print_token(raw_token);
    }
  } else {
    if (raw_token.type === TOKEN.ATTRIBUTE) {
      printer.set_space_before_token(true);
      last_tag_token.attr_count += 1;
    } else if (raw_token.type === TOKEN.EQUALS) { //no space before =
      printer.set_space_before_token(false);
    } else if (raw_token.type === TOKEN.VALUE && raw_token.previous.type === TOKEN.EQUALS) { //no space before value
      printer.set_space_before_token(false);
    }

    if (raw_token.type === TOKEN.ATTRIBUTE && last_tag_token.tag_start_char === '<') {
      if (this._is_wrap_attributes_preserve || this._is_wrap_attributes_preserve_aligned) {
        printer.traverse_whitespace(raw_token);
        wrapped = wrapped || raw_token.newlines !== 0;
      }


      if (this._is_wrap_attributes_force) {
        var force_attr_wrap = last_tag_token.attr_count > 1;
        if (this._is_wrap_attributes_force_expand_multiline && last_tag_token.attr_count === 1) {
          var is_only_attribute = true;
          var peek_index = 0;
          var peek_token;
          do {
            peek_token = tokens.peek(peek_index);
            if (peek_token.type === TOKEN.ATTRIBUTE) {
              is_only_attribute = false;
              break;
            }
            peek_index += 1;
          } while (peek_index < 4 && peek_token.type !== TOKEN.EOF && peek_token.type !== TOKEN.TAG_CLOSE);

          force_attr_wrap = !is_only_attribute;
        }

        if (force_attr_wrap) {
          printer.print_newline(false);
          wrapped = true;
        }
      }
    }
    printer.print_token(raw_token);
    wrapped = wrapped || printer.previous_token_wrapped();
    last_tag_token.has_wrapped_attrs = wrapped;
  }
  return parser_token;
};

Beautifier$1.prototype._handle_text = function(printer, raw_token, last_tag_token) {
  var parser_token = {
    text: raw_token.text,
    type: 'TK_CONTENT'
  };
  if (last_tag_token.custom_beautifier_name) { //check if we need to format javascript
    this._print_custom_beatifier_text(printer, raw_token, last_tag_token);
  } else if (last_tag_token.is_unformatted || last_tag_token.is_content_unformatted) {
    printer.add_raw_token(raw_token);
  } else {
    printer.traverse_whitespace(raw_token);
    printer.print_token(raw_token);
  }
  return parser_token;
};

Beautifier$1.prototype._print_custom_beatifier_text = function(printer, raw_token, last_tag_token) {
  var local = this;
  if (raw_token.text !== '') {

    var text = raw_token.text,
      _beautifier,
      script_indent_level = 1,
      pre = '',
      post = '';
    if (last_tag_token.custom_beautifier_name === 'javascript' && typeof this._js_beautify === 'function') {
      _beautifier = this._js_beautify;
    } else if (last_tag_token.custom_beautifier_name === 'css' && typeof this._css_beautify === 'function') {
      _beautifier = this._css_beautify;
    } else if (last_tag_token.custom_beautifier_name === 'html') {
      _beautifier = function(html_source, options) {
        var beautifier = new Beautifier$1(html_source, options, local._js_beautify, local._css_beautify);
        return beautifier.beautify();
      };
    }

    if (this._options.indent_scripts === "keep") {
      script_indent_level = 0;
    } else if (this._options.indent_scripts === "separate") {
      script_indent_level = -printer.indent_level;
    }

    var indentation = printer.get_full_indent(script_indent_level);

    // if there is at least one empty line at the end of this text, strip it
    // we'll be adding one back after the text but before the containing tag.
    text = text.replace(/\n[ \t]*$/, '');

    // Handle the case where content is wrapped in a comment or cdata.
    if (last_tag_token.custom_beautifier_name !== 'html' &&
      text[0] === '<' && text.match(/^(<!--|<!\[CDATA\[)/)) {
      var matched = /^(<!--[^\n]*|<!\[CDATA\[)(\n?)([ \t\n]*)([\s\S]*)(-->|]]>)$/.exec(text);

      // if we start to wrap but don't finish, print raw
      if (!matched) {
        printer.add_raw_token(raw_token);
        return;
      }

      pre = indentation + matched[1] + '\n';
      text = matched[4];
      if (matched[5]) {
        post = indentation + matched[5];
      }

      // if there is at least one empty line at the end of this text, strip it
      // we'll be adding one back after the text but before the containing tag.
      text = text.replace(/\n[ \t]*$/, '');

      if (matched[2] || matched[3].indexOf('\n') !== -1) {
        // if the first line of the non-comment text has spaces
        // use that as the basis for indenting in null case.
        matched = matched[3].match(/[ \t]+$/);
        if (matched) {
          raw_token.whitespace_before = matched[0];
        }
      }
    }

    if (text) {
      if (_beautifier) {

        // call the Beautifier if avaliable
        var Child_options = function() {
          this.eol = '\n';
        };
        Child_options.prototype = this._options.raw_options;
        var child_options = new Child_options();
        text = _beautifier(indentation + text, child_options);
      } else {
        // simply indent the string otherwise
        var white = raw_token.whitespace_before;
        if (white) {
          text = text.replace(new RegExp('\n(' + white + ')?', 'g'), '\n');
        }

        text = indentation + text.replace(/\n/g, '\n' + indentation);
      }
    }

    if (pre) {
      if (!text) {
        text = pre + post;
      } else {
        text = pre + text + '\n' + post;
      }
    }

    printer.print_newline(false);
    if (text) {
      raw_token.text = text;
      raw_token.whitespace_before = '';
      raw_token.newlines = 0;
      printer.add_raw_token(raw_token);
      printer.print_newline(true);
    }
  }
};

Beautifier$1.prototype._handle_tag_open = function(printer, raw_token, last_tag_token, last_token) {
  var parser_token = this._get_tag_open_token(raw_token);

  if ((last_tag_token.is_unformatted || last_tag_token.is_content_unformatted) &&
    !last_tag_token.is_empty_element &&
    raw_token.type === TOKEN.TAG_OPEN && raw_token.text.indexOf('</') === 0) {
    // End element tags for unformatted or content_unformatted elements
    // are printed raw to keep any newlines inside them exactly the same.
    printer.add_raw_token(raw_token);
    parser_token.start_tag_token = this._tag_stack.try_pop(parser_token.tag_name);
  } else {
    printer.traverse_whitespace(raw_token);
    this._set_tag_position(printer, raw_token, parser_token, last_tag_token, last_token);
    if (!parser_token.is_inline_element) {
      printer.set_wrap_point();
    }
    printer.print_token(raw_token);
  }

  //indent attributes an auto, forced, aligned or forced-align line-wrap
  if (this._is_wrap_attributes_force_aligned || this._is_wrap_attributes_aligned_multiple || this._is_wrap_attributes_preserve_aligned) {
    parser_token.alignment_size = raw_token.text.length + 1;
  }

  if (!parser_token.tag_complete && !parser_token.is_unformatted) {
    printer.alignment_size = parser_token.alignment_size;
  }

  return parser_token;
};

var TagOpenParserToken = function(parent, raw_token) {
  this.parent = parent || null;
  this.text = '';
  this.type = 'TK_TAG_OPEN';
  this.tag_name = '';
  this.is_inline_element = false;
  this.is_unformatted = false;
  this.is_content_unformatted = false;
  this.is_empty_element = false;
  this.is_start_tag = false;
  this.is_end_tag = false;
  this.indent_content = false;
  this.multiline_content = false;
  this.custom_beautifier_name = null;
  this.start_tag_token = null;
  this.attr_count = 0;
  this.has_wrapped_attrs = false;
  this.alignment_size = 0;
  this.tag_complete = false;
  this.tag_start_char = '';
  this.tag_check = '';

  if (!raw_token) {
    this.tag_complete = true;
  } else {
    var tag_check_match;

    this.tag_start_char = raw_token.text[0];
    this.text = raw_token.text;

    if (this.tag_start_char === '<') {
      tag_check_match = raw_token.text.match(/^<([^\s>]*)/);
      this.tag_check = tag_check_match ? tag_check_match[1] : '';
    } else {
      tag_check_match = raw_token.text.match(/^{{(?:[\^]|#\*?)?([^\s}]+)/);
      this.tag_check = tag_check_match ? tag_check_match[1] : '';

      // handle "{{#> myPartial}}
      if (raw_token.text === '{{#>' && this.tag_check === '>' && raw_token.next !== null) {
        this.tag_check = raw_token.next.text;
      }
    }
    this.tag_check = this.tag_check.toLowerCase();

    if (raw_token.type === TOKEN.COMMENT) {
      this.tag_complete = true;
    }

    this.is_start_tag = this.tag_check.charAt(0) !== '/';
    this.tag_name = !this.is_start_tag ? this.tag_check.substr(1) : this.tag_check;
    this.is_end_tag = !this.is_start_tag ||
      (raw_token.closed && raw_token.closed.text === '/>');

    // handlebars tags that don't start with # or ^ are single_tags, and so also start and end.
    this.is_end_tag = this.is_end_tag ||
      (this.tag_start_char === '{' && (this.text.length < 3 || (/[^#\^]/.test(this.text.charAt(2)))));
  }
};

Beautifier$1.prototype._get_tag_open_token = function(raw_token) { //function to get a full tag and parse its type
  var parser_token = new TagOpenParserToken(this._tag_stack.get_parser_token(), raw_token);

  parser_token.alignment_size = this._options.wrap_attributes_indent_size;

  parser_token.is_end_tag = parser_token.is_end_tag ||
    in_array(parser_token.tag_check, this._options.void_elements);

  parser_token.is_empty_element = parser_token.tag_complete ||
    (parser_token.is_start_tag && parser_token.is_end_tag);

  parser_token.is_unformatted = !parser_token.tag_complete && in_array(parser_token.tag_check, this._options.unformatted);
  parser_token.is_content_unformatted = !parser_token.is_empty_element && in_array(parser_token.tag_check, this._options.content_unformatted);
  parser_token.is_inline_element = in_array(parser_token.tag_name, this._options.inline) || parser_token.tag_start_char === '{';

  return parser_token;
};

Beautifier$1.prototype._set_tag_position = function(printer, raw_token, parser_token, last_tag_token, last_token) {

  if (!parser_token.is_empty_element) {
    if (parser_token.is_end_tag) { //this tag is a double tag so check for tag-ending
      parser_token.start_tag_token = this._tag_stack.try_pop(parser_token.tag_name); //remove it and all ancestors
    } else { // it's a start-tag
      // check if this tag is starting an element that has optional end element
      // and do an ending needed
      if (this._do_optional_end_element(parser_token)) {
        if (!parser_token.is_inline_element) {
          printer.print_newline(false);
        }
      }

      this._tag_stack.record_tag(parser_token); //push it on the tag stack

      if ((parser_token.tag_name === 'script' || parser_token.tag_name === 'style') &&
        !(parser_token.is_unformatted || parser_token.is_content_unformatted)) {
        parser_token.custom_beautifier_name = get_custom_beautifier_name(parser_token.tag_check, raw_token);
      }
    }
  }

  if (in_array(parser_token.tag_check, this._options.extra_liners)) { //check if this double needs an extra line
    printer.print_newline(false);
    if (!printer._output.just_added_blankline()) {
      printer.print_newline(true);
    }
  }

  if (parser_token.is_empty_element) { //if this tag name is a single tag type (either in the list or has a closing /)

    // if you hit an else case, reset the indent level if you are inside an:
    // 'if', 'unless', or 'each' block.
    if (parser_token.tag_start_char === '{' && parser_token.tag_check === 'else') {
      this._tag_stack.indent_to_tag(['if', 'unless', 'each']);
      parser_token.indent_content = true;
      // Don't add a newline if opening {{#if}} tag is on the current line
      var foundIfOnCurrentLine = printer.current_line_has_match(/{{#if/);
      if (!foundIfOnCurrentLine) {
        printer.print_newline(false);
      }
    }

    // Don't add a newline before elements that should remain where they are.
    if (parser_token.tag_name === '!--' && last_token.type === TOKEN.TAG_CLOSE &&
      last_tag_token.is_end_tag && parser_token.text.indexOf('\n') === -1) ; else {
      if (!(parser_token.is_inline_element || parser_token.is_unformatted)) {
        printer.print_newline(false);
      }
      this._calcluate_parent_multiline(printer, parser_token);
    }
  } else if (parser_token.is_end_tag) { //this tag is a double tag so check for tag-ending
    var do_end_expand = false;

    // deciding whether a block is multiline should not be this hard
    do_end_expand = parser_token.start_tag_token && parser_token.start_tag_token.multiline_content;
    do_end_expand = do_end_expand || (!parser_token.is_inline_element &&
      !(last_tag_token.is_inline_element || last_tag_token.is_unformatted) &&
      !(last_token.type === TOKEN.TAG_CLOSE && parser_token.start_tag_token === last_tag_token) &&
      last_token.type !== 'TK_CONTENT'
    );

    if (parser_token.is_content_unformatted || parser_token.is_unformatted) {
      do_end_expand = false;
    }

    if (do_end_expand) {
      printer.print_newline(false);
    }
  } else { // it's a start-tag
    parser_token.indent_content = !parser_token.custom_beautifier_name;

    if (parser_token.tag_start_char === '<') {
      if (parser_token.tag_name === 'html') {
        parser_token.indent_content = this._options.indent_inner_html;
      } else if (parser_token.tag_name === 'head') {
        parser_token.indent_content = this._options.indent_head_inner_html;
      } else if (parser_token.tag_name === 'body') {
        parser_token.indent_content = this._options.indent_body_inner_html;
      }
    }

    if (!(parser_token.is_inline_element || parser_token.is_unformatted) &&
      (last_token.type !== 'TK_CONTENT' || parser_token.is_content_unformatted)) {
      printer.print_newline(false);
    }

    this._calcluate_parent_multiline(printer, parser_token);
  }
};

Beautifier$1.prototype._calcluate_parent_multiline = function(printer, parser_token) {
  if (parser_token.parent && printer._output.just_added_newline() &&
    !((parser_token.is_inline_element || parser_token.is_unformatted) && parser_token.parent.is_inline_element)) {
    parser_token.parent.multiline_content = true;
  }
};

//To be used for <p> tag special case:
var p_closers = ['address', 'article', 'aside', 'blockquote', 'details', 'div', 'dl', 'fieldset', 'figcaption', 'figure', 'footer', 'form', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'header', 'hr', 'main', 'nav', 'ol', 'p', 'pre', 'section', 'table', 'ul'];
var p_parent_excludes = ['a', 'audio', 'del', 'ins', 'map', 'noscript', 'video'];

Beautifier$1.prototype._do_optional_end_element = function(parser_token) {
  var result = null;
  // NOTE: cases of "if there is no more content in the parent element"
  // are handled automatically by the beautifier.
  // It assumes parent or ancestor close tag closes all children.
  // https://www.w3.org/TR/html5/syntax.html#optional-tags
  if (parser_token.is_empty_element || !parser_token.is_start_tag || !parser_token.parent) {
    return;

  }

  if (parser_token.tag_name === 'body') {
    // A head element’s end tag may be omitted if the head element is not immediately followed by a space character or a comment.
    result = result || this._tag_stack.try_pop('head');

    //} else if (parser_token.tag_name === 'body') {
    // DONE: A body element’s end tag may be omitted if the body element is not immediately followed by a comment.

  } else if (parser_token.tag_name === 'li') {
    // An li element’s end tag may be omitted if the li element is immediately followed by another li element or if there is no more content in the parent element.
    result = result || this._tag_stack.try_pop('li', ['ol', 'ul']);

  } else if (parser_token.tag_name === 'dd' || parser_token.tag_name === 'dt') {
    // A dd element’s end tag may be omitted if the dd element is immediately followed by another dd element or a dt element, or if there is no more content in the parent element.
    // A dt element’s end tag may be omitted if the dt element is immediately followed by another dt element or a dd element.
    result = result || this._tag_stack.try_pop('dt', ['dl']);
    result = result || this._tag_stack.try_pop('dd', ['dl']);


  } else if (parser_token.parent.tag_name === 'p' && p_closers.indexOf(parser_token.tag_name) !== -1) {
    // IMPORTANT: this else-if works because p_closers has no overlap with any other element we look for in this method
    // check for the parent element is an HTML element that is not an <a>, <audio>, <del>, <ins>, <map>, <noscript>, or <video> element,  or an autonomous custom element.
    // To do this right, this needs to be coded as an inclusion of the inverse of the exclusion above.
    // But to start with (if we ignore "autonomous custom elements") the exclusion would be fine.
    var p_parent = parser_token.parent.parent;
    if (!p_parent || p_parent_excludes.indexOf(p_parent.tag_name) === -1) {
      result = result || this._tag_stack.try_pop('p');
    }
  } else if (parser_token.tag_name === 'rp' || parser_token.tag_name === 'rt') {
    // An rt element’s end tag may be omitted if the rt element is immediately followed by an rt or rp element, or if there is no more content in the parent element.
    // An rp element’s end tag may be omitted if the rp element is immediately followed by an rt or rp element, or if there is no more content in the parent element.
    result = result || this._tag_stack.try_pop('rt', ['ruby', 'rtc']);
    result = result || this._tag_stack.try_pop('rp', ['ruby', 'rtc']);

  } else if (parser_token.tag_name === 'optgroup') {
    // An optgroup element’s end tag may be omitted if the optgroup element is immediately followed by another optgroup element, or if there is no more content in the parent element.
    // An option element’s end tag may be omitted if the option element is immediately followed by another option element, or if it is immediately followed by an optgroup element, or if there is no more content in the parent element.
    result = result || this._tag_stack.try_pop('optgroup', ['select']);
    //result = result || this._tag_stack.try_pop('option', ['select']);

  } else if (parser_token.tag_name === 'option') {
    // An option element’s end tag may be omitted if the option element is immediately followed by another option element, or if it is immediately followed by an optgroup element, or if there is no more content in the parent element.
    result = result || this._tag_stack.try_pop('option', ['select', 'datalist', 'optgroup']);

  } else if (parser_token.tag_name === 'colgroup') {
    // DONE: A colgroup element’s end tag may be omitted if the colgroup element is not immediately followed by a space character or a comment.
    // A caption element's end tag may be ommitted if a colgroup, thead, tfoot, tbody, or tr element is started.
    result = result || this._tag_stack.try_pop('caption', ['table']);

  } else if (parser_token.tag_name === 'thead') {
    // A colgroup element's end tag may be ommitted if a thead, tfoot, tbody, or tr element is started.
    // A caption element's end tag may be ommitted if a colgroup, thead, tfoot, tbody, or tr element is started.
    result = result || this._tag_stack.try_pop('caption', ['table']);
    result = result || this._tag_stack.try_pop('colgroup', ['table']);

    //} else if (parser_token.tag_name === 'caption') {
    // DONE: A caption element’s end tag may be omitted if the caption element is not immediately followed by a space character or a comment.

  } else if (parser_token.tag_name === 'tbody' || parser_token.tag_name === 'tfoot') {
    // A thead element’s end tag may be omitted if the thead element is immediately followed by a tbody or tfoot element.
    // A tbody element’s end tag may be omitted if the tbody element is immediately followed by a tbody or tfoot element, or if there is no more content in the parent element.
    // A colgroup element's end tag may be ommitted if a thead, tfoot, tbody, or tr element is started.
    // A caption element's end tag may be ommitted if a colgroup, thead, tfoot, tbody, or tr element is started.
    result = result || this._tag_stack.try_pop('caption', ['table']);
    result = result || this._tag_stack.try_pop('colgroup', ['table']);
    result = result || this._tag_stack.try_pop('thead', ['table']);
    result = result || this._tag_stack.try_pop('tbody', ['table']);

    //} else if (parser_token.tag_name === 'tfoot') {
    // DONE: A tfoot element’s end tag may be omitted if there is no more content in the parent element.

  } else if (parser_token.tag_name === 'tr') {
    // A tr element’s end tag may be omitted if the tr element is immediately followed by another tr element, or if there is no more content in the parent element.
    // A colgroup element's end tag may be ommitted if a thead, tfoot, tbody, or tr element is started.
    // A caption element's end tag may be ommitted if a colgroup, thead, tfoot, tbody, or tr element is started.
    result = result || this._tag_stack.try_pop('caption', ['table']);
    result = result || this._tag_stack.try_pop('colgroup', ['table']);
    result = result || this._tag_stack.try_pop('tr', ['table', 'thead', 'tbody', 'tfoot']);

  } else if (parser_token.tag_name === 'th' || parser_token.tag_name === 'td') {
    // A td element’s end tag may be omitted if the td element is immediately followed by a td or th element, or if there is no more content in the parent element.
    // A th element’s end tag may be omitted if the th element is immediately followed by a td or th element, or if there is no more content in the parent element.
    result = result || this._tag_stack.try_pop('td', ['table', 'thead', 'tbody', 'tfoot', 'tr']);
    result = result || this._tag_stack.try_pop('th', ['table', 'thead', 'tbody', 'tfoot', 'tr']);
  }

  // Start element omission not handled currently
  // A head element’s start tag may be omitted if the element is empty, or if the first thing inside the head element is an element.
  // A tbody element’s start tag may be omitted if the first thing inside the tbody element is a tr element, and if the element is not immediately preceded by a tbody, thead, or tfoot element whose end tag has been omitted. (It can’t be omitted if the element is empty.)
  // A colgroup element’s start tag may be omitted if the first thing inside the colgroup element is a col element, and if the element is not immediately preceded by another colgroup element whose end tag has been omitted. (It can’t be omitted if the element is empty.)

  // Fix up the parent of the parser token
  parser_token.parent = this._tag_stack.get_parser_token();

  return result;
};

beautifier.Beautifier = Beautifier$1;

/*jshint node:true */

var Beautifier = beautifier.Beautifier,
  Options = options.Options;

function style_html$1(html_source, options, js_beautify, css_beautify) {
  var beautifier = new Beautifier(html_source, options, js_beautify, css_beautify);
  return beautifier.beautify();
}

html.exports = style_html$1;
html.exports.defaultOptions = function() {
  return new Options();
};

/*jshint node:true */

var js_beautify = javascript.exports;
var css_beautify = css$1.exports;
var html_beautify = html.exports;

function style_html(html_source, options, js, css) {
  js = js || js_beautify;
  css = css || css_beautify;
  return html_beautify(html_source, options, js, css);
}
style_html.defaultOptions = html_beautify.defaultOptions;

src.js = js_beautify;
src.css = css_beautify;
src.html = style_html;

/*jshint node:true */

(function (module) {

/**
The following batches are equivalent:

var beautify_js = require('js-beautify');
var beautify_js = require('js-beautify').js;
var beautify_js = require('js-beautify').js_beautify;

var beautify_css = require('js-beautify').css;
var beautify_css = require('js-beautify').css_beautify;

var beautify_html = require('js-beautify').html;
var beautify_html = require('js-beautify').html_beautify;

All methods returned accept two arguments, the source string and an options object.
**/

function get_beautify(js_beautify, css_beautify, html_beautify) {
  // the default is js
  var beautify = function(src, config) {
    return js_beautify.js_beautify(src, config);
  };

  // short aliases
  beautify.js = js_beautify.js_beautify;
  beautify.css = css_beautify.css_beautify;
  beautify.html = html_beautify.html_beautify;

  // legacy aliases
  beautify.js_beautify = js_beautify.js_beautify;
  beautify.css_beautify = css_beautify.css_beautify;
  beautify.html_beautify = html_beautify.html_beautify;

  return beautify;
}

{
  (function(mod) {
    var beautifier = src;
    beautifier.js_beautify = beautifier.js;
    beautifier.css_beautify = beautifier.css;
    beautifier.html_beautify = beautifier.html;

    mod.exports = get_beautify(beautifier, beautifier, beautifier);

  })(module);
}
}(js$1));

var JsBeautify = js$1.exports;

function _defineProperty$3(obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }

  return obj;
}

function ownKeys$3(object, enumerableOnly) {
  var keys = Object.keys(object);

  if (Object.getOwnPropertySymbols) {
    var symbols = Object.getOwnPropertySymbols(object);
    if (enumerableOnly) symbols = symbols.filter(function (sym) {
      return Object.getOwnPropertyDescriptor(object, sym).enumerable;
    });
    keys.push.apply(keys, symbols);
  }

  return keys;
}

function _objectSpread2$3(target) {
  for (var i = 1; i < arguments.length; i++) {
    var source = arguments[i] != null ? arguments[i] : {};

    if (i % 2) {
      ownKeys$3(Object(source), true).forEach(function (key) {
        _defineProperty$3(target, key, source[key]);
      });
    } else if (Object.getOwnPropertyDescriptors) {
      Object.defineProperties(target, Object.getOwnPropertyDescriptors(source));
    } else {
      ownKeys$3(Object(source)).forEach(function (key) {
        Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key));
      });
    }
  }

  return target;
}

function _objectWithoutPropertiesLoose$1(source, excluded) {
  if (source == null) return {};
  var target = {};
  var sourceKeys = Object.keys(source);
  var key, i;

  for (i = 0; i < sourceKeys.length; i++) {
    key = sourceKeys[i];
    if (excluded.indexOf(key) >= 0) continue;
    target[key] = source[key];
  }

  return target;
}

function _objectWithoutProperties$1(source, excluded) {
  if (source == null) return {};

  var target = _objectWithoutPropertiesLoose$1(source, excluded);

  var key, i;

  if (Object.getOwnPropertySymbols) {
    var sourceSymbolKeys = Object.getOwnPropertySymbols(source);

    for (i = 0; i < sourceSymbolKeys.length; i++) {
      key = sourceSymbolKeys[i];
      if (excluded.indexOf(key) >= 0) continue;
      if (!Object.prototype.propertyIsEnumerable.call(source, key)) continue;
      target[key] = source[key];
    }
  }

  return target;
}

function _slicedToArray(arr, i) {
  return _arrayWithHoles(arr) || _iterableToArrayLimit(arr, i) || _unsupportedIterableToArray$1(arr, i) || _nonIterableRest();
}

function _arrayWithHoles(arr) {
  if (Array.isArray(arr)) return arr;
}

function _iterableToArrayLimit(arr, i) {
  if (typeof Symbol === "undefined" || !(Symbol.iterator in Object(arr))) return;
  var _arr = [];
  var _n = true;
  var _d = false;
  var _e = undefined;

  try {
    for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) {
      _arr.push(_s.value);

      if (i && _arr.length === i) break;
    }
  } catch (err) {
    _d = true;
    _e = err;
  } finally {
    try {
      if (!_n && _i["return"] != null) _i["return"]();
    } finally {
      if (_d) throw _e;
    }
  }

  return _arr;
}

function _unsupportedIterableToArray$1(o, minLen) {
  if (!o) return;
  if (typeof o === "string") return _arrayLikeToArray$1(o, minLen);
  var n = Object.prototype.toString.call(o).slice(8, -1);
  if (n === "Object" && o.constructor) n = o.constructor.name;
  if (n === "Map" || n === "Set") return Array.from(o);
  if (n === "Arguments" || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(n)) return _arrayLikeToArray$1(o, minLen);
}

function _arrayLikeToArray$1(arr, len) {
  if (len == null || len > arr.length) len = arr.length;

  for (var i = 0, arr2 = new Array(len); i < len; i++) arr2[i] = arr[i];

  return arr2;
}

function _nonIterableRest() {
  throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.");
}

function _defineProperty$2(obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }

  return obj;
}

function ownKeys$2(object, enumerableOnly) {
  var keys = Object.keys(object);

  if (Object.getOwnPropertySymbols) {
    var symbols = Object.getOwnPropertySymbols(object);
    if (enumerableOnly) symbols = symbols.filter(function (sym) {
      return Object.getOwnPropertyDescriptor(object, sym).enumerable;
    });
    keys.push.apply(keys, symbols);
  }

  return keys;
}

function _objectSpread2$2(target) {
  for (var i = 1; i < arguments.length; i++) {
    var source = arguments[i] != null ? arguments[i] : {};

    if (i % 2) {
      ownKeys$2(Object(source), true).forEach(function (key) {
        _defineProperty$2(target, key, source[key]);
      });
    } else if (Object.getOwnPropertyDescriptors) {
      Object.defineProperties(target, Object.getOwnPropertyDescriptors(source));
    } else {
      ownKeys$2(Object(source)).forEach(function (key) {
        Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key));
      });
    }
  }

  return target;
}

function compose$1() {
  for (var _len = arguments.length, fns = new Array(_len), _key = 0; _key < _len; _key++) {
    fns[_key] = arguments[_key];
  }

  return function (x) {
    return fns.reduceRight(function (y, f) {
      return f(y);
    }, x);
  };
}

function curry$1(fn) {
  return function curried() {
    var _this = this;

    for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
      args[_key2] = arguments[_key2];
    }

    return args.length >= fn.length ? fn.apply(this, args) : function () {
      for (var _len3 = arguments.length, nextArgs = new Array(_len3), _key3 = 0; _key3 < _len3; _key3++) {
        nextArgs[_key3] = arguments[_key3];
      }

      return curried.apply(_this, [].concat(args, nextArgs));
    };
  };
}

function isObject$3(value) {
  return {}.toString.call(value).includes('Object');
}

function isEmpty(obj) {
  return !Object.keys(obj).length;
}

function isFunction$2(value) {
  return typeof value === 'function';
}

function hasOwnProperty$1(object, property) {
  return Object.prototype.hasOwnProperty.call(object, property);
}

function validateChanges(initial, changes) {
  if (!isObject$3(changes)) errorHandler$1('changeType');
  if (Object.keys(changes).some(function (field) {
    return !hasOwnProperty$1(initial, field);
  })) errorHandler$1('changeField');
  return changes;
}

function validateSelector(selector) {
  if (!isFunction$2(selector)) errorHandler$1('selectorType');
}

function validateHandler(handler) {
  if (!(isFunction$2(handler) || isObject$3(handler))) errorHandler$1('handlerType');
  if (isObject$3(handler) && Object.values(handler).some(function (_handler) {
    return !isFunction$2(_handler);
  })) errorHandler$1('handlersType');
}

function validateInitial(initial) {
  if (!initial) errorHandler$1('initialIsRequired');
  if (!isObject$3(initial)) errorHandler$1('initialType');
  if (isEmpty(initial)) errorHandler$1('initialContent');
}

function throwError$1(errorMessages, type) {
  throw new Error(errorMessages[type] || errorMessages["default"]);
}

var errorMessages$1 = {
  initialIsRequired: 'initial state is required',
  initialType: 'initial state should be an object',
  initialContent: 'initial state shouldn\'t be an empty object',
  handlerType: 'handler should be an object or a function',
  handlersType: 'all handlers should be a functions',
  selectorType: 'selector should be a function',
  changeType: 'provided value of changes should be an object',
  changeField: 'it seams you want to change a field in the state which is not specified in the "initial" state',
  "default": 'an unknown error accured in `state-local` package'
};
var errorHandler$1 = curry$1(throwError$1)(errorMessages$1);
var validators$1 = {
  changes: validateChanges,
  selector: validateSelector,
  handler: validateHandler,
  initial: validateInitial
};

function create(initial) {
  var handler = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  validators$1.initial(initial);
  validators$1.handler(handler);
  var state = {
    current: initial
  };
  var didUpdate = curry$1(didStateUpdate)(state, handler);
  var update = curry$1(updateState)(state);
  var validate = curry$1(validators$1.changes)(initial);
  var getChanges = curry$1(extractChanges)(state);

  function getState() {
    var selector = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : function (state) {
      return state;
    };
    validators$1.selector(selector);
    return selector(state.current);
  }

  function setState(causedChanges) {
    compose$1(didUpdate, update, validate, getChanges)(causedChanges);
  }

  return [getState, setState];
}

function extractChanges(state, causedChanges) {
  return isFunction$2(causedChanges) ? causedChanges(state.current) : causedChanges;
}

function updateState(state, changes) {
  state.current = _objectSpread2$2(_objectSpread2$2({}, state.current), changes);
  return changes;
}

function didStateUpdate(state, handler, changes) {
  isFunction$2(handler) ? handler(state.current) : Object.keys(changes).forEach(function (field) {
    var _handler$field;

    return (_handler$field = handler[field]) === null || _handler$field === void 0 ? void 0 : _handler$field.call(handler, state.current[field]);
  });
  return changes;
}

var index$1 = {
  create: create
};

var config$1 = {
  paths: {
    vs: 'https://cdn.jsdelivr.net/npm/monaco-editor@0.33.0/min/vs'
  }
};

function curry(fn) {
  return function curried() {
    var _this = this;

    for (var _len = arguments.length, args = new Array(_len), _key = 0; _key < _len; _key++) {
      args[_key] = arguments[_key];
    }

    return args.length >= fn.length ? fn.apply(this, args) : function () {
      for (var _len2 = arguments.length, nextArgs = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
        nextArgs[_key2] = arguments[_key2];
      }

      return curried.apply(_this, [].concat(args, nextArgs));
    };
  };
}

function isObject$2(value) {
  return {}.toString.call(value).includes('Object');
}

/**
 * validates the configuration object and informs about deprecation
 * @param {Object} config - the configuration object 
 * @return {Object} config - the validated configuration object
 */

function validateConfig(config) {
  if (!config) errorHandler('configIsRequired');
  if (!isObject$2(config)) errorHandler('configType');

  if (config.urls) {
    informAboutDeprecation();
    return {
      paths: {
        vs: config.urls.monacoBase
      }
    };
  }

  return config;
}
/**
 * logs deprecation message
 */


function informAboutDeprecation() {
  console.warn(errorMessages.deprecation);
}

function throwError(errorMessages, type) {
  throw new Error(errorMessages[type] || errorMessages["default"]);
}

var errorMessages = {
  configIsRequired: 'the configuration object is required',
  configType: 'the configuration object should be an object',
  "default": 'an unknown error accured in `@monaco-editor/loader` package',
  deprecation: "Deprecation warning!\n    You are using deprecated way of configuration.\n\n    Instead of using\n      monaco.config({ urls: { monacoBase: '...' } })\n    use\n      monaco.config({ paths: { vs: '...' } })\n\n    For more please check the link https://github.com/suren-atoyan/monaco-loader#config\n  "
};
var errorHandler = curry(throwError)(errorMessages);
var validators = {
  config: validateConfig
};

var compose = function compose() {
  for (var _len = arguments.length, fns = new Array(_len), _key = 0; _key < _len; _key++) {
    fns[_key] = arguments[_key];
  }

  return function (x) {
    return fns.reduceRight(function (y, f) {
      return f(y);
    }, x);
  };
};

function merge(target, source) {
  Object.keys(source).forEach(function (key) {
    if (source[key] instanceof Object) {
      if (target[key]) {
        Object.assign(source[key], merge(target[key], source[key]));
      }
    }
  });
  return _objectSpread2$3(_objectSpread2$3({}, target), source);
}

// The source (has been changed) is https://github.com/facebook/react/issues/5465#issuecomment-157888325
var CANCELATION_MESSAGE = {
  type: 'cancelation',
  msg: 'operation is manually canceled'
};

function makeCancelable(promise) {
  var hasCanceled_ = false;
  var wrappedPromise = new Promise(function (resolve, reject) {
    promise.then(function (val) {
      return hasCanceled_ ? reject(CANCELATION_MESSAGE) : resolve(val);
    });
    promise["catch"](reject);
  });
  return wrappedPromise.cancel = function () {
    return hasCanceled_ = true;
  }, wrappedPromise;
}

/** the local state of the module */

var _state$create = index$1.create({
  config: config$1,
  isInitialized: false,
  resolve: null,
  reject: null,
  monaco: null
}),
    _state$create2 = _slicedToArray(_state$create, 2),
    getState = _state$create2[0],
    setState = _state$create2[1];
/**
 * set the loader configuration
 * @param {Object} config - the configuration object
 */


function config(globalConfig) {
  var _validators$config = validators.config(globalConfig),
      monaco = _validators$config.monaco,
      config = _objectWithoutProperties$1(_validators$config, ["monaco"]);

  setState(function (state) {
    return {
      config: merge(state.config, config),
      monaco: monaco
    };
  });
}
/**
 * handles the initialization of the monaco-editor
 * @return {Promise} - returns an instance of monaco (with a cancelable promise)
 */


function init() {
  var state = getState(function (_ref) {
    var monaco = _ref.monaco,
        isInitialized = _ref.isInitialized,
        resolve = _ref.resolve;
    return {
      monaco: monaco,
      isInitialized: isInitialized,
      resolve: resolve
    };
  });

  if (!state.isInitialized) {
    setState({
      isInitialized: true
    });

    if (state.monaco) {
      state.resolve(state.monaco);
      return makeCancelable(wrapperPromise);
    }

    if (window.monaco && window.monaco.editor) {
      storeMonacoInstance(window.monaco);
      state.resolve(window.monaco);
      return makeCancelable(wrapperPromise);
    }

    compose(injectScripts, getMonacoLoaderScript)(configureLoader);
  }

  return makeCancelable(wrapperPromise);
}
/**
 * injects provided scripts into the document.body
 * @param {Object} script - an HTML script element
 * @return {Object} - the injected HTML script element
 */


function injectScripts(script) {
  return document.body.appendChild(script);
}
/**
 * creates an HTML script element with/without provided src
 * @param {string} [src] - the source path of the script
 * @return {Object} - the created HTML script element
 */


function createScript(src) {
  var script = document.createElement('script');
  return src && (script.src = src), script;
}
/**
 * creates an HTML script element with the monaco loader src
 * @return {Object} - the created HTML script element
 */


function getMonacoLoaderScript(configureLoader) {
  var state = getState(function (_ref2) {
    var config = _ref2.config,
        reject = _ref2.reject;
    return {
      config: config,
      reject: reject
    };
  });
  var loaderScript = createScript("".concat(state.config.paths.vs, "/loader.js"));

  loaderScript.onload = function () {
    return configureLoader();
  };

  loaderScript.onerror = state.reject;
  return loaderScript;
}
/**
 * configures the monaco loader
 */


function configureLoader() {
  var state = getState(function (_ref3) {
    var config = _ref3.config,
        resolve = _ref3.resolve,
        reject = _ref3.reject;
    return {
      config: config,
      resolve: resolve,
      reject: reject
    };
  });
  var require = window.require;

  require.config(state.config);

  require(['vs/editor/editor.main'], function (monaco) {
    storeMonacoInstance(monaco);
    state.resolve(monaco);
  }, function (error) {
    state.reject(error);
  });
}
/**
 * store monaco instance in local state
 */


function storeMonacoInstance(monaco) {
  if (!getState().monaco) {
    setState({
      monaco: monaco
    });
  }
}
/**
 * internal helper function
 * extracts stored monaco instance
 * @return {Object|null} - the monaco instance
 */


function __getMonacoInstance() {
  return getState(function (_ref4) {
    var monaco = _ref4.monaco;
    return monaco;
  });
}

var wrapperPromise = new Promise(function (resolve, reject) {
  return setState({
    resolve: resolve,
    reject: reject
  });
});
var loader = {
  config: config,
  init: init,
  __getMonacoInstance: __getMonacoInstance
};

const dom$j = new Dom();
let hash$1 = {};
class HtmlUtil {
  constructor(builder) {
    this.builder = builder;
    const util = this.builder.util;
    const builderStuff = this.builder.builderStuff;
    let viewhtml = builderStuff.querySelector('.viewhtml');

    if (!viewhtml) {
      /*
      Note:
      - viewhtml => non syntax-highlighted. 
      - viewhtmlmonaco => syntax-highlighted.
       - viewhtmlexternal => external use.
      */
      const html = `<div class="is-modal viewhtml" tabindex="-1" role="dialog" aria-modal="true" aria-hidden="true">
                <div class="is-modal-content">
                    <textarea class="tabSupport" style="width:100%;height:calc(100% - 50px);border:none;margin:0;box-sizing:border-box;"></textarea>
                    <div class="is-modal-footer" style="border-top:1px solid ${this.builder.styleSeparatorColor};width:100%;height:50px;position:absolute;left:0;bottom:0;overflow:hidden;text-align:right">
                        <button title="${util.out('Cancel')}" class="input-cancel classic-secondary">${util.out('Cancel')}</button>
                        <button title="${util.out('Ok')}" class="input-ok classic-primary">${util.out('Ok')}</button>
                    </div>
                </div>
            </div>

            <div class="is-modal viewhtmlmonaco" tabindex="-1" role="dialog" aria-modal="true" aria-hidden="true">
                <div class="is-modal-content" style="max-width: 55vw;padding:0">
                    <textarea style="position:absolute;display:none;"></textarea>
                    <div class="input-code-editor" style="width:100%;height:65vh;"></div>
                    <div class="is-modal-footer" style="display:flex;justify-content:flex-end;border-top:1px solid ${this.builder.styleSeparatorColor};">
                        <button title="${util.out('Cancel')}" class="input-cancel classic-secondary">${util.out('Cancel')}</button>
                        <button title="${util.out('Ok')}" class="input-ok classic-primary">${util.out('Ok')}</button>
                    </div>
                </div>
            </div>

            <div class="is-modal viewhtmlexternal" tabindex="-1" role="dialog" aria-modal="true" aria-hidden="true">
                <div class="is-modal-content" style="max-width: 55vw;padding:0">
                    <div class="input-code-editor" style="width:100%;height:65vh;"></div>
                    <div class="is-modal-footer" style="display:flex;justify-content:flex-end;border-top:1px solid ${this.builder.styleSeparatorColor};">
                        <button title="${util.out('Cancel')}" class="input-cancel classic-secondary">${util.out('Cancel')}</button>
                        <button title="${util.out('Ok')}" class="input-ok classic-primary">${util.out('Ok')}</button>
                    </div>
                </div>
            </div>
            
            `;
      dom$j.appendHtml(builderStuff, html);
      viewhtml = builderStuff.querySelector('.viewhtml');
      let viewhtmlmonaco = builderStuff.querySelector('.viewhtmlmonaco');
      let viewhtmlexternal = builderStuff.querySelector('.viewhtmlexternal');
      let elm = viewhtml.querySelector('.input-ok');
      dom$j.addEventListener(elm, 'click', () => {
        this.applyHtml(viewhtml);
      });
      elm = viewhtml.querySelector('.input-cancel');
      dom$j.addEventListener(elm, 'click', () => {
        this.builder.hideModal(viewhtml);
      });
      elm = viewhtmlmonaco.querySelector('.input-ok');
      dom$j.addEventListener(elm, 'click', () => {
        this.applyHtml(viewhtmlmonaco);
      });
      elm = viewhtmlmonaco.querySelector('.input-cancel');
      dom$j.addEventListener(elm, 'click', () => {
        this.builder.hideModal(viewhtmlmonaco);
      });
      elm = viewhtmlexternal.querySelector('.input-ok');
      dom$j.addEventListener(elm, 'click', () => {
        const source = document.querySelector('textarea[data-source-active]');
        const selectorOk = source.getAttribute('data-source-ok');
        source.removeAttribute('data-source-active');
        source.removeAttribute('data-source-ok');
        source.removeAttribute('data-source-cancel');
        document.querySelector(selectorOk).click();
        this.builder.hideModal(viewhtmlexternal);
      });
      elm = viewhtmlexternal.querySelector('.input-cancel');
      dom$j.addEventListener(elm, 'click', () => {
        const source = document.querySelector('textarea[data-source-active]');
        const selectorCancel = source.getAttribute('data-source-cancel');
        source.removeAttribute('data-source-active');
        source.removeAttribute('data-source-ok');
        source.removeAttribute('data-source-cancel');
        document.querySelector(selectorCancel).click();
        this.builder.hideModal(viewhtmlexternal);
      });
    }
  }

  applyHtml(modal) {
    const util = this.builder.util;
    let mode = this.builder.codeEditorMode;
    let area = this.builder.codeEditorArea;
    this.builder.uo.saveForUndo();
    let textarea = modal.querySelector('textarea');
    let html = textarea.value;

    if (mode === 'code') {
      let codeblock = this.builder.activeCodeBlock; //this.builder.renderCustomCodeBlock($block, html);

      codeblock.setAttribute('data-html', encodeURIComponent(html)); // => important

      html = html.replace(/{id}/g, util.makeId());

      for (let i = 1; i <= 20; i++) {
        html = html.replace('[%HTML' + i + '%]', codeblock.getAttribute('data-html-' + i) === undefined ? '' : decodeURIComponent(codeblock.getAttribute('data-html-' + i))); //render editable area
      } //codeblock.innerHTML = html;


      codeblock.innerHTML = ''; // Use createContextualFragment() to make embedded script executable
      // https://ghinda.net/article/script-tags/

      let range = document.createRange();
      range.setStart(codeblock, 0);
      codeblock.appendChild(range.createContextualFragment(html));
      let builderActive = this.builder.doc.querySelector('.builder-active');
      if (builderActive) this.builder.applyBehaviorOn(builderActive); //Trigger Change event

      this.builder.opts.onChange(); //Trigger Render event

      this.builder.opts.onRender();
      this.builder.hideModal(modal);
      util.hideControls();
      return;
    }

    html = this.fromViewToActual(html);

    if (mode === 'cell') {
      const cell = util.cellSelected();
      cell.innerHTML = html;
      let builderActive = this.builder.doc.querySelector('.builder-active');
      if (builderActive) this.builder.applyBehaviorOn(builderActive); //Trigger Change event

      this.builder.opts.onChange(); //Trigger Render event

      this.builder.opts.onRender();
    }

    if (mode === 'row') {
      let row;
      let cell = util.cellSelected();

      if (cell) {
        row = cell.parentNode;
      } else {
        row = util.rowSelected();
      }

      if (!row) return;
      row.innerHTML = html;
      let builderActive = this.builder.doc.querySelector('.builder-active');
      if (builderActive) this.builder.applyBehaviorOn(builderActive); //Trigger Change event

      this.builder.opts.onChange(); //Trigger Render event

      this.builder.opts.onRender();
    }

    if (mode === 'full') {
      // area.innerHTML = html;
      area.innerHTML = ''; // Use createContextualFragment() to make embedded script executable
      // https://ghinda.net/article/script-tags/

      let range = document.createRange();
      range.setStart(area, 0);
      area.appendChild(range.createContextualFragment(html)); // let builderActive = this.builder.doc.querySelector('.builder-active');
      // if(builderActive) this.builder.applyBehaviorOn(builderActive);

      this.builder.applyBehaviorOn(area);
      /*else {
           const builders = this.builder.doc.querySelectorAll(this.builder.opts.container);
          if(builders.length > 1) {
              const cell = util.cellSelected();
              if(!cell) {
                  // Return first instance
                  area = builders[0];
              } else {
                  // Return active instance
                  area = cell.parentNode.parentNode;
              }
          } else {
              // Single instance
              area = builders[0];
          }
          this.builder.applyBehaviorOn(area);
       }*/
      //Trigger Change event

      this.builder.opts.onChange(); //Trigger Render event

      this.builder.opts.onRender();
      util.clearActiveCell();
    }

    util.clearControls(); // NEW

    this.builder.hideModal(modal);
  }

  viewHtmlExternal() {
    const util = this.builder.util;
    const builderStuff = this.builder.builderStuff;
    const viewhtmlexternal = builderStuff.querySelector('.viewhtmlexternal');
    if (document.activeElement) this.elmFocus = document.activeElement;
    util.showModal(viewhtmlexternal, true, () => {
      if (this.elmFocus) this.elmFocus.focus();
    }, false);
    let codeEditor = viewhtmlexternal.querySelector('.input-code-editor');
    codeEditor.style.opacity = 0.01;

    this._viewHtmlExternal();
  }

  _viewHtmlExternal() {
    const source = document.querySelector('textarea[data-source-active]');
    const builderStuff = this.builder.builderStuff;
    const viewhtmlexternal = builderStuff.querySelector('.viewhtmlexternal');
    let codeEditor = viewhtmlexternal.querySelector('.input-code-editor');

    if (!codeEditor.querySelector('.monaco-editor')) {
      // First time code editor load
      loader.init().then(monaco => {
        this.builder.monaco = monaco;
        if (!codeEditor.getAttribute('data-mode-id')) this.builder.codeEditorExternal = monaco.editor.create(codeEditor, {
          value: '',
          minimap: {
            enabled: false
          },
          automaticLayout: true,
          contextmenu: false,
          fontSize: 13,
          wordWrap: true,
          scrollbar: {
            useShadows: false,
            vertical: 'visible',
            horizontal: 'visible',
            horizontalScrollbarSize: 12,
            verticalScrollbarSize: 12
          },
          language: 'html',
          theme: this.builder.styleDark || this.builder.styleColoredDark ? 'vs-dark' : ''
        });
        this.builder.codeEditorExternal.onKeyUp(() => {
          const source = document.querySelector('textarea[data-source-active]');
          source.value = this.builder.codeEditorExternal.getModel().getValue();
        });
      });
      setTimeout(() => {
        this._viewHtmlExternal();
      }, 100);
      return;
    } // code editor


    const model = this.builder.monaco.editor.createModel(source.value, 'html');
    this.builder.codeEditorExternal.setModel(model);
    codeEditor.style.opacity = '';
  }

  view(mode, area, btn) {
    const util = this.builder.util;
    const builderStuff = this.builder.builderStuff;
    let viewhtml;

    if (this.builder.opts.htmlSyntaxHighlighting) {
      viewhtml = builderStuff.querySelector('.viewhtmlmonaco');
    } else {
      viewhtml = builderStuff.querySelector('.viewhtml');
    }

    util.showModal(viewhtml, true, () => {
      if (btn) {
        btn.removeAttribute('data-focus');
        btn.focus();
      }
    });
    if (btn) btn.setAttribute('data-focus', true);
    let codeEditor = viewhtml.querySelector('.input-code-editor');
    codeEditor.style.opacity = 0.01;

    this._view(mode, area);
  }

  _view(mode, area) {
    const util = this.builder.util;
    const builderStuff = this.builder.builderStuff;
    let viewhtml;

    if (this.builder.opts.htmlSyntaxHighlighting) {
      viewhtml = builderStuff.querySelector('.viewhtmlmonaco');
      let codeEditor = viewhtml.querySelector('.input-code-editor');

      if (!codeEditor.querySelector('.monaco-editor')) {
        // First time code editor load
        loader.init().then(monaco => {
          this.builder.monaco = monaco;
          if (!codeEditor.getAttribute('data-mode-id')) this.builder.codeEditor = monaco.editor.create(codeEditor, {
            value: '',
            minimap: {
              enabled: false
            },
            automaticLayout: true,
            contextmenu: false,
            fontSize: 13,
            wordWrap: true,
            scrollbar: {
              useShadows: false,
              vertical: 'visible',
              horizontal: 'visible',
              horizontalScrollbarSize: 12,
              verticalScrollbarSize: 12
            },
            language: 'html',
            theme: this.builder.styleDark || this.builder.styleColoredDark ? 'vs-dark' : ''
          });
          let textarea = viewhtml.querySelector('textarea');
          this.builder.codeEditor.onKeyUp(() => {
            textarea.value = this.builder.codeEditor.getModel().getValue();
          });
        });
        setTimeout(() => {
          this._view(mode, area);
        }, 100);
        return;
      }
    } else {
      viewhtml = builderStuff.querySelector('.viewhtml');
    }

    let textarea = viewhtml.querySelector('textarea'); // util.showModal(viewhtml, true);

    if (mode === 'cell') {
      // old: 3550
      const cell = util.cellSelected();
      if (!cell) return;
      let textarea = viewhtml.querySelector('textarea');
      this.builder.cleanHtmlFormatting = true;
      textarea.value = this.readHtml(cell, true); // for view=true

      this.builder.cleanHtmlFormatting = false;
    }

    if (mode === 'row') {
      // old: 15534
      // const cell = util.cellSelected();
      // if(!cell) return;
      // const row = cell.parentNode;
      let row;
      let cell = util.cellSelected();

      if (cell) {
        row = cell.parentNode;
      } else {
        row = util.rowSelected();
      }

      if (!row) return;
      let textarea = viewhtml.querySelector('textarea');
      this.builder.cleanHtmlFormatting = true;
      textarea.value = this.readHtml(row, true); // for view=true

      this.builder.cleanHtmlFormatting = false; //Change to row selection

      dom$j.removeClass(row, 'row-outline');
    }

    if (mode === 'full') {
      // 
      if (area) ; else {
        const builders = this.builder.doc.querySelectorAll(this.builder.opts.container);

        if (builders.length > 1) {
          const cell = util.cellSelected();

          if (!cell) {
            // Return first instance
            area = builders[0];
          } else {
            // Return active instance
            area = cell.parentNode.parentNode;
          }
        } else {
          // Single instance
          area = builders[0];
        }

        if (this.builder.opts.page !== '') {
          const wrapper = this.builder.doc.querySelector(this.builder.opts.page);

          if (wrapper) {
            //return wrapper
            area = wrapper;
          }
        }
      }

      let textarea = viewhtml.querySelector('textarea');
      this.builder.cleanHtmlFormatting = true; // textarea.value = this.readHtml(area, true); // for view=true

      if (this.builder.shortenHTML) {
        textarea.value = this.readHtml(area, true); // for view=true
      } else {
        textarea.value = this.readHtml(area, false); // actual
      }

      this.builder.cleanHtmlFormatting = false;
    }

    if (this.builder.opts.htmlSyntaxHighlighting) {
      // code editor
      const model = this.builder.monaco.editor.createModel(textarea.value, 'html');
      this.builder.codeEditor.setModel(model);
    } //Hide popup


    let columnMore = builderStuff.querySelector('.is-pop.columnmore');
    if (columnMore) columnMore.style.display = '';
    let rowMore = builderStuff.querySelector('.is-pop.rowmore');
    if (rowMore) rowMore.style.display = '';
    this.builder.codeEditorMode = mode;
    this.builder.codeEditorArea = area;
    let codeEditor = viewhtml.querySelector('.input-code-editor');
    codeEditor.style.opacity = '';
  }

  fromViewToActual(html) {
    for (var key in hash$1) {
      html = html.replace(key, hash$1[key]);
    }

    return html;
  }

  readHtml(content, view, multiple) {
    //view = true (hide internal attributes). view = false (actual html)
    //Make absolute
    if (this.builder.opts.absolutePath) {
      let links = content.querySelectorAll('a');
      Array.prototype.forEach.call(links, link => {
        let href = link.href;
        link.setAttribute('href', href);
      });
      let imgs = content.querySelectorAll('img');
      Array.prototype.forEach.call(imgs, img => {
        let src = img.src;
        img.setAttribute('src', src);
      });
    }

    if (this.builder.cleanHtmlFormatting) ;

    const util = this.builder.util; // const builderStuff = this.builder.builderStuff;
    // //Prepare temporary helpers: #tmp_content & #tmp_buildercontent
    // let elm = builderStuff.querySelector('#tmp_content');
    // if(elm) builderStuff.removeChild(elm);
    // elm = builderStuff.querySelector('#tmp_buildercontent');
    // if(elm) builderStuff.removeChild(elm);
    // let html = `<div id="tmp_content" style="position:absolute;top:0;left:0;width:1px;height:1px;overflow:hidden;visibility:hidden;"></div>
    //     <div id="tmp_buildercontent" style="position:absolute;top:0;left:0;width:1px;height:1px;overflow:hidden;visibility:hidden;"></div>`;
    // dom.appendHtml(builderStuff, html);

    let html; // let tmp = builderStuff.querySelector('#tmp_content');

    let tmp = document.createElement('div');
    tmp.innerHTML = content.innerHTML; //Find subblocks (previously is-builder) in custom code blocks and save them to data-html-1, data-html-2, and so on.

    let blocks = tmp.querySelectorAll('[data-html]');
    Array.prototype.forEach.call(blocks, block => {
      // NEW
      let index = 1;
      let subblocks = block.querySelectorAll('[data-subblock]');
      Array.prototype.forEach.call(subblocks, subblock => {
        if (subblock.closest('.glide__slide--clone')) return;
        let html = subblock.innerHTML;
        block.setAttribute('data-html-' + index, encodeURIComponent(html));
        index++;
      });
    }); //Render custom code blocks (including any editable areas within)

    blocks = tmp.querySelectorAll('[data-html]');
    Array.prototype.forEach.call(blocks, block => {
      let blockhtml = decodeURIComponent(block.getAttribute('data-html'));
      blockhtml = blockhtml.replace(/{id}/g, util.makeId());
      /* OLD
      for(var i=1;i<=20;i++){ 
          blockhtml = blockhtml.replace('[%HTML'+i+'%]', (block.getAttribute('data-html-'+i) === undefined ? '' : decodeURIComponent(block.getAttribute('data-html-'+i))));//render editable area
      }
      block.innerHTML = blockhtml; //embedded script is not (and should not) executed here
      */
      // NEW

      block.innerHTML = blockhtml; //embedded script is not (and should not) executed here  

      let subblocks = block.querySelectorAll('[data-subblock]');
      var i = 1;
      Array.prototype.forEach.call(subblocks, subblock => {
        subblock.innerHTML = block.getAttribute('data-html-' + i) === undefined ? '' : decodeURIComponent(block.getAttribute('data-html-' + i));
        i++;
      });
    }); //For Viewing, hide data-html, data-html-1.., data-settings

    if (view) {
      hash$1 = {};
      blocks = tmp.querySelectorAll('[data-html]');
      Array.prototype.forEach.call(blocks, block => {
        let uniqueID = util.makeId();
        hash$1[uniqueID] = block.getAttribute('data-html');
        block.setAttribute('data-html', uniqueID);

        for (let i = 1; i <= 20; i++) {
          if (block.getAttribute('data-html-' + i) !== undefined) {
            uniqueID = util.makeId();
            hash$1[uniqueID] = block.getAttribute('data-html-' + i);
            block.getAttribute('data-html-' + i, uniqueID);
          }
        }

        if (block.getAttribute('data-settings') !== undefined) {
          uniqueID = util.makeId();
          hash$1[uniqueID] = block.getAttribute('data-settings');
          block.getAttribute('data-settings', uniqueID);
        }
      });
    } //Make absolute (for Export/Download)


    if (this.builder.makeAbsolute) {
      const convertMediaUrl = imgUrl => {
        // make absolute
        let img = document.createElement('img');
        img.src = imgUrl;
        img.setAttribute('data-absoluteurl', img.src);
        imgUrl = img.getAttribute('data-absoluteurl');
        return imgUrl;
      };

      const convertBgUrl = imgUrl => {
        // make absolute
        let img = document.createElement('img');
        img.src = imgUrl;
        img.setAttribute('data-absoluteurl', img.src);
        imgUrl = img.getAttribute('data-absoluteurl'); // Do not use imgUrl directly, change base to [%PATH%] instead 
        // (to prevent auto converting back to relative path)

        let urlBase = location.href.substring(0, location.href.lastIndexOf('/'));
        return imgUrl.replace(urlBase, '[%PATH%]'); // Example:
        // url = http://localhost:8080/uploads/image.jpg 
        // urlBase = http://localhost:8080
        // become: [%PATH%]/uploads/image.jpg 
      };

      const convertMedia = (elm, attrName) => {
        if (elm.hasAttribute(attrName)) {
          if (!(elm.parentNode.tagName.toLowerCase() === 'video' || elm.parentNode.tagName.toLowerCase() === 'audio')) return;
          let imgUrl = elm.getAttribute(attrName);
          imgUrl = convertMediaUrl(imgUrl);
          elm.setAttribute(attrName, `${imgUrl}`);
        }
      };

      let imgs = tmp.querySelectorAll('img');
      Array.prototype.forEach.call(imgs, img => {
        let src = img.src;
        img.setAttribute('src', src);
      });
      let elms = tmp.querySelectorAll('*');
      elms.forEach(elm => {
        if (elm.style.backgroundImage) {
          let s = elm.style.backgroundImage;

          if (s.indexOf('url(') !== -1) {
            let imgUrl = s.slice(4, -1).replace(/["']/g, '');
            imgUrl = convertBgUrl(imgUrl);
            elm.style.backgroundImage = `url(${imgUrl})`;
          }
        }

        convertMedia(elm, 'src');
        convertMedia(elm, 'data-default');
        convertMedia(elm, 'data-240');
        convertMedia(elm, 'data-360');
        convertMedia(elm, 'data-480');
        convertMedia(elm, 'data-540');
        convertMedia(elm, 'data-720');
        convertMedia(elm, 'data-1080');
        convertMedia(elm, 'data-1440');
        convertMedia(elm, 'data-2160');

        if (elm.hasAttribute('href')) {
          if (elm.tagName.toLowerCase() === 'link') {
            let imgUrl = elm.getAttribute('href');
            imgUrl = convertMediaUrl(imgUrl);
            elm.setAttribute('href', `${imgUrl}`);
          }
        }
      });
    } //Cleaning


    let _builders = tmp.querySelectorAll('.is-builder');

    Array.prototype.forEach.call(_builders, _builder => {
      _builder.style.transform = '';
      _builder.style.WebkitTransform = '';
      _builder.style.MozTransform = '';

      _builder.removeAttribute('data-sort');

      dom$j.removeClass(_builder, 'builder-active');
      const rows = dom$j.elementChildren(_builder);
      rows.forEach(row => {
        const cols = dom$j.elementChildren(row);
        cols.forEach(col => {
          col.style.cursor = '';
        });
      });
    }); // cleanup cursor from drag to resize columns

    const rows = dom$j.elementChildren(tmp);
    rows.forEach(row => {
      row.style.cursor = ''; //if tmp is row html, then this is a column

      const cols = dom$j.elementChildren(row);
      cols.forEach(col => {
        col.style.cursor = '';
      });
    });
    let elms = tmp.querySelectorAll('.sortable-chosen');
    dom$j.removeClasses(elms, 'sortable-chosen');
    elms = tmp.querySelectorAll('.sortable-ghost');
    dom$j.removeClasses(elms, 'sortable-ghost');
    elms = tmp.querySelectorAll('.elm-active');
    dom$j.removeClasses(elms, 'elm-active');
    elms = tmp.querySelectorAll('.elm-inspected');
    dom$j.removeClasses(elms, 'elm-inspected');
    elms = tmp.querySelectorAll('.cell-active');
    dom$j.removeClasses(elms, 'cell-active');
    elms = tmp.querySelectorAll('.row-active');
    dom$j.removeClasses(elms, 'row-active');
    elms = tmp.querySelectorAll('.row-outline');
    dom$j.removeClasses(elms, 'row-outline');
    elms = tmp.querySelectorAll('.is-builder');
    dom$j.removeClasses(elms, 'is-builder');
    elms = tmp.querySelectorAll('.row-outline');
    dom$j.removeClasses(elms, 'row-outline');
    elms = tmp.querySelectorAll('[data-click]');
    dom$j.removeAttributes(elms, 'data-click');
    elms = tmp.querySelectorAll('[contenteditable]');
    dom$j.removeAttributes(elms, 'contenteditable');
    elms = tmp.querySelectorAll('[draggridoutline]');
    dom$j.removeAttributes(elms, 'draggridoutline');
    elms = tmp.querySelectorAll('[between-blocks-left]');
    dom$j.removeAttributes(elms, 'between-blocks-left');
    elms = tmp.querySelectorAll('[between-blocks-center]');
    dom$j.removeAttributes(elms, 'between-blocks-center');
    elms = tmp.querySelectorAll('[hideelementhighlight]');
    dom$j.removeAttributes(elms, 'hideelementhighlight');
    elms = tmp.querySelectorAll('[data-module-active]');
    dom$j.removeAttributes(elms, 'data-module-active');
    elms = tmp.querySelectorAll('[draggable]');
    dom$j.removeAttributes(elms, 'draggable');
    elms = tmp.querySelectorAll('[data-animated]');
    dom$j.removeAttributes(elms, 'data-animated');
    elms = tmp.querySelectorAll('[data-saveforundo]');
    dom$j.removeAttributes(elms, 'data-saveforundo');
    elms = tmp.querySelectorAll('[hidesnippetaddtool]');
    dom$j.removeAttributes(elms, 'hidesnippetaddtool');
    elms = tmp.querySelectorAll('[gray]');
    dom$j.removeAttributes(elms, 'gray');
    elms = tmp.querySelectorAll('[rowoutline]');
    dom$j.removeAttributes(elms, 'rowoutline');
    elms = tmp.querySelectorAll('[hidecolumntool]');
    dom$j.removeAttributes(elms, 'hidecolumntool');
    elms = tmp.querySelectorAll('[grayoutline]');
    dom$j.removeAttributes(elms, 'grayoutline');
    elms = tmp.querySelectorAll('[hideoutline]');
    dom$j.removeAttributes(elms, 'hideoutline');
    elms = tmp.querySelectorAll('[leftrowtool]');
    dom$j.removeAttributes(elms, 'leftrowtool');
    elms = tmp.querySelectorAll('[minimal]');
    dom$j.removeAttributes(elms, 'minimal');
    elms = tmp.querySelectorAll('[clean]');
    dom$j.removeAttributes(elms, 'clean');
    elms = tmp.querySelectorAll('[grideditor]');
    dom$j.removeAttributes(elms, 'grideditor');
    elms = tmp.querySelectorAll('[gridoutline]');
    dom$j.removeAttributes(elms, 'gridoutline');
    dom$j.removeElements(tmp.querySelectorAll('.is-row-tool'));
    dom$j.removeElements(tmp.querySelectorAll('.is-col-tool'));
    dom$j.removeElements(tmp.querySelectorAll('.is-rowadd-tool'));
    dom$j.removeElements(tmp.querySelectorAll('.ovl'));
    dom$j.removeElements(tmp.querySelectorAll('.row-add-initial')); //Extra cleaning

    elms = tmp.querySelectorAll('.aos-init');
    dom$j.removeClasses(elms, 'aos-init');
    elms = tmp.querySelectorAll('.aos-animate');
    dom$j.removeClasses(elms, 'aos-animate');
    elms = tmp.querySelectorAll('.skrollable');
    dom$j.removeClasses(elms, 'skrollable');
    elms = tmp.querySelectorAll('.skrollable-after');
    dom$j.removeClasses(elms, 'skrollable-after');
    elms = tmp.querySelectorAll('.skrollable-before');
    dom$j.removeClasses(elms, 'skrollable-before');
    elms = tmp.querySelectorAll('.skrollable-between');
    dom$j.removeClasses(elms, 'skrollable-between');
    elms = tmp.querySelectorAll('.is-inview');
    dom$j.removeClasses(elms, 'is-inview');
    let emptyclasses = tmp.querySelectorAll('[class=""]');
    Array.prototype.forEach.call(emptyclasses, emptyclass => {
      emptyclass.removeAttribute('class');
    });
    let unusedOverlays = tmp.querySelectorAll('.is-row-overlay');
    Array.prototype.forEach.call(unusedOverlays, unusedOverlay => {
      if (!unusedOverlay.hasAttribute('style')) unusedOverlay.parentNode.removeChild(unusedOverlay);
    });
    dom$j.removeEmptyStyle(tmp);
    elms = tmp.querySelectorAll('[data-keep]');
    dom$j.removeAttributes(elms, 'data-keep'); //Backward compatible: cleanup button <span contenteditable="false"><a contenteditable="true">button</a></span>

    let links = tmp.querySelectorAll('a');
    Array.prototype.forEach.call(links, link => {
      if (link.style.display === 'inline-block') {
        if (link.parentNode.childElementCount === 1 && link.parentNode.tagName.toLowerCase() === 'span') {
          link.parentNode.outerHTML = link.parentNode.innerHTML;
        }
      }
    });
    html = '';

    if (multiple) {
      //ContentBox
      // Remove dummy DIV after last section
      let elms = tmp.querySelectorAll('.is-dummy');
      dom$j.removeElements(elms);
      elms = tmp.querySelectorAll('[data-scroll]');
      dom$j.removeAttributes(elms, 'data-scroll');
      elms = tmp.querySelectorAll('[data-scroll-once]');
      dom$j.removeAttributes(elms, 'data-scroll-once');

      if (this.builder.shortenOutput) {
        elms = tmp.querySelectorAll('[data-noedit]');
        dom$j.removeAttributes(elms, 'data-noedit');
        elms = tmp.querySelectorAll('[data-module]');
        dom$j.removeAttributes(elms, 'data-module');
        elms = tmp.querySelectorAll('[data-module-desc]');
        dom$j.removeAttributes(elms, 'data-module-desc');
        elms = tmp.querySelectorAll('[data-dialog-width]');
        dom$j.removeAttributes(elms, 'data-dialog-width');
        elms = tmp.querySelectorAll('[data-dialog-height]');
        dom$j.removeAttributes(elms, 'data-dialog-height');
        elms = tmp.querySelectorAll('[data-html]');
        dom$j.removeAttributes(elms, 'data-html');
        elms = tmp.querySelectorAll('[data-settings]');
        dom$j.removeAttributes(elms, 'data-settings');
        elms = tmp.querySelectorAll('[data-html-1]');
        dom$j.removeAttributes(elms, 'data-html-1');
        elms = tmp.querySelectorAll('[data-html-2]');
        dom$j.removeAttributes(elms, 'data-html-2');
        elms = tmp.querySelectorAll('[data-html-3]');
        dom$j.removeAttributes(elms, 'data-html-3');
        elms = tmp.querySelectorAll('[data-html-4]');
        dom$j.removeAttributes(elms, 'data-html-4');
        elms = tmp.querySelectorAll('[data-html-5]');
        dom$j.removeAttributes(elms, 'data-html-5');
        elms = tmp.querySelectorAll('[data-html-6]');
        dom$j.removeAttributes(elms, 'data-html-6');
        elms = tmp.querySelectorAll('[data-html-7]');
        dom$j.removeAttributes(elms, 'data-html-7');
        elms = tmp.querySelectorAll('[data-html-8]');
        dom$j.removeAttributes(elms, 'data-html-8');
        elms = tmp.querySelectorAll('[data-html-9]');
        dom$j.removeAttributes(elms, 'data-html-9');
        elms = tmp.querySelectorAll('[data-html-10]');
        dom$j.removeAttributes(elms, 'data-html-10');
        elms = tmp.querySelectorAll('[data-html-12]');
        dom$j.removeAttributes(elms, 'data-html-12');
        elms = tmp.querySelectorAll('[data-html-13]');
        dom$j.removeAttributes(elms, 'data-html-13');
        elms = tmp.querySelectorAll('[data-html-14]');
        dom$j.removeAttributes(elms, 'data-html-14');
        elms = tmp.querySelectorAll('[data-html-15]');
        dom$j.removeAttributes(elms, 'data-html-15');
        elms = tmp.querySelectorAll('[data-html-16]');
        dom$j.removeAttributes(elms, 'data-html-16');
        elms = tmp.querySelectorAll('[data-html-17]');
        dom$j.removeAttributes(elms, 'data-html-17');
        elms = tmp.querySelectorAll('[data-html-18]');
        dom$j.removeAttributes(elms, 'data-html-18');
        elms = tmp.querySelectorAll('[data-html-19]');
        dom$j.removeAttributes(elms, 'data-html-19');
        elms = tmp.querySelectorAll('[data-html-20]');
        dom$j.removeAttributes(elms, 'data-html-20');
        elms = tmp.querySelectorAll('[data-html-21]');
        dom$j.removeAttributes(elms, 'data-html-21');
        elms = tmp.querySelectorAll('[data-html-21]');
        dom$j.removeAttributes(elms, 'data-html-21');
        elms = tmp.querySelectorAll('[data-html-22]');
        dom$j.removeAttributes(elms, 'data-html-22');
        elms = tmp.querySelectorAll('[data-html-23]');
        dom$j.removeAttributes(elms, 'data-html-23');
        elms = tmp.querySelectorAll('[data-html-24]');
        dom$j.removeAttributes(elms, 'data-html-24');
        elms = tmp.querySelectorAll('[data-html-25]');
        dom$j.removeAttributes(elms, 'data-html-25');
      } // cleaning


      elms = tmp.querySelectorAll('[data-bottom-top],[data-center],[data-center-bottom],[data-100-top],[data-50-top],[data-top],[data-top-bottom],[data-center-top],[data--300-bottom],[data--150-bottom],[data--50-bottom],[data-bottom],[data-100-bottom],[data-150-bottom],[data-400-bottom],' + '[data--400-bottom],[data--200-bottom],[data--100-bottom],[data-50-bottom],[data-200-bottom],[data-300-bottom],' + '[data-in],[data-in-150],[data-in-300],' + '[data-cen--150],[data-cen],[data-cen-150],[data-out--300],[data-out--150],[data-out]' + '[data-t],[data-t-100],[data-t-200],[data-t-300],[data-t-400],' + '[data-t-500],[data-t-600],[data-t-700],[data-t-800],[data-t-900],[data-t-1000],' + '[data-t-1100],[data-t-1200],[data-t-1300],[data-t-1400],[data-t-1500],[data-t-1600],' + '[data-t-1700],[data-t-1800],[data-t-1900],[data-t-2000],[data-t-2100],[data-t-2200],' + '[data-t-2300],[data-t-2400],[data-t-2500],[data-t-2600],[data-t-2700],[data-t-2800]'); // elms = tmp.querySelectorAll('[data-bottom-top],[data-center],[data-center-bottom],[data-100-top],[data-50-top],[data-top],[data-top-bottom],[data-center-top],[data--300-bottom],[data--150-bottom],[data--50-bottom],[data-bottom],[data-100-bottom],[data-150-bottom],[data-400-bottom],' +
      //     '[data--400-bottom],[data--200-bottom],[data--100-bottom],[data-50-bottom],[data-200-bottom],[data-300-bottom],' +
      //     '[data-in],[data-in-150],[data-in-300],[data-cen--150],[data-cen],[data-cen-150],[data-out--300],[data-out--150],[data-out]');

      Array.prototype.forEach.call(elms, elm => {
        elm.style.transition = '';
        elm.style.transform = '';
        elm.style.opacity = '';
      });
      elms = tmp.querySelectorAll('.is-animated');
      Array.prototype.forEach.call(elms, elm => {
        dom$j.removeClass(elm, 'animated');
        dom$j.removeClass(elm, 'pulse');
        dom$j.removeClass(elm, 'bounceIn');
        dom$j.removeClass(elm, 'fadeIn');
        dom$j.removeClass(elm, 'fadeOut'); //new

        dom$j.removeClass(elm, 'fadeInDown');
        dom$j.removeClass(elm, 'fadeInLeft');
        dom$j.removeClass(elm, 'fadeInRight');
        dom$j.removeClass(elm, 'fadeInUp');
        dom$j.removeClass(elm, 'flipInX');
        dom$j.removeClass(elm, 'flipInY');
        dom$j.removeClass(elm, 'slideInUp');
        dom$j.removeClass(elm, 'slideInDown');
        dom$j.removeClass(elm, 'slideInLeft');
        dom$j.removeClass(elm, 'slideInRight');
        dom$j.removeClass(elm, 'zoomIn');
        elm.style.animationDelay = '';
        elm.style.transitionDelay = '';
        elm.style.transitionDuration = '';
        elm.style.transitionTimingFunction = '';
        elm.style.transitionProperty = '';
      });
      let emptystyles = tmp.querySelectorAll('[style=""]');
      Array.prototype.forEach.call(emptystyles, emptystyle => {
        emptystyle.removeAttribute('style');
      }); //Cleanup utils

      elms = tmp.querySelectorAll('.is-appeared');
      Array.prototype.forEach.call(elms, elm => {
        dom$j.removeClass(elm, 'is-appeared');
      });
      elms = tmp.querySelectorAll('.box-active');
      Array.prototype.forEach.call(elms, elm => {
        dom$j.removeClass(elm, 'box-active');
      });
      elms = tmp.querySelectorAll('.section-active');
      Array.prototype.forEach.call(elms, elm => {
        dom$j.removeClass(elm, 'section-active');
      });
      elms = tmp.querySelectorAll('.section-select');
      Array.prototype.forEach.call(elms, elm => {
        dom$j.removeClass(elm, 'section-select');
      });
      elms = tmp.querySelectorAll('.box-select');
      Array.prototype.forEach.call(elms, elm => {
        dom$j.removeClass(elm, 'box-select');
      });
      elms = tmp.querySelectorAll('.is-section-tool');
      dom$j.removeElements(elms);
      elms = tmp.querySelectorAll('.is-box-tool');
      dom$j.removeElements(elms);
      elms = tmp.querySelectorAll('.is-section-info');
      dom$j.removeElements(elms);
      var html_content = '';
      var html_footer = '';
      var html_others = ''; // Apply behavior on each row

      const sections = dom$j.elementChildren(tmp);
      sections.forEach(section => {
        let currentSection = section;

        if (dom$j.hasClass(currentSection, 'is-section')) {
          var secclass = ''; // var secstyle = '';

          if (currentSection.getAttribute('class')) secclass = ' class="' + currentSection.getAttribute('class') + '"'; // if (currentSection.getAttribute('style')) secstyle = ' style="' + currentSection.getAttribute('style') + '"';

          var copySection = currentSection.cloneNode(true);
          var htmlSection = copySection.outerHTML;
          html += htmlSection; //content & footer

          if (secclass.indexOf('is-static') === -1) {
            html_content += htmlSection + '\n\n';
          } else {
            html_footer += htmlSection + '\n\n';
          }
        } else {
          copySection = currentSection.cloneNode(true);
          htmlSection = copySection.outerHTML;
          html += htmlSection; //others

          html_others += htmlSection;
        }
      });
      if (html_footer !== '') html_footer = '<!---FOOTER--->\n' + html_footer;
      if (html_others !== '') html_others = '<!---OTHERS--->\n' + html_others;
      let contentbox = document.querySelector('[data-contentbox]');
      let disableStaticSection = false;

      if (contentbox) {
        disableStaticSection = contentbox.settings.disableStaticSection;
      }

      if (!disableStaticSection) {
        html = html_content + html_footer + html_others;
      }
    } else {
      let emptystyles = tmp.querySelectorAll('[style=""]');
      Array.prototype.forEach.call(emptystyles, emptystyle => {
        emptystyle.removeAttribute('style');
      });

      if (this.builder.shortenOutput) {
        elms = tmp.querySelectorAll('[data-noedit]');
        dom$j.removeAttributes(elms, 'data-noedit');
        elms = tmp.querySelectorAll('[data-module]');
        dom$j.removeAttributes(elms, 'data-module');
        elms = tmp.querySelectorAll('[data-module-desc]');
        dom$j.removeAttributes(elms, 'data-module-desc');
        elms = tmp.querySelectorAll('[data-dialog-width]');
        dom$j.removeAttributes(elms, 'data-dialog-width');
        elms = tmp.querySelectorAll('[data-dialog-height]');
        dom$j.removeAttributes(elms, 'data-dialog-height');
        elms = tmp.querySelectorAll('[data-html]');
        dom$j.removeAttributes(elms, 'data-html');
        elms = tmp.querySelectorAll('[data-settings]');
        dom$j.removeAttributes(elms, 'data-settings');
        elms = tmp.querySelectorAll('[data-html-1]');
        dom$j.removeAttributes(elms, 'data-html-1');
        elms = tmp.querySelectorAll('[data-html-2]');
        dom$j.removeAttributes(elms, 'data-html-2');
        elms = tmp.querySelectorAll('[data-html-3]');
        dom$j.removeAttributes(elms, 'data-html-3');
        elms = tmp.querySelectorAll('[data-html-4]');
        dom$j.removeAttributes(elms, 'data-html-4');
        elms = tmp.querySelectorAll('[data-html-5]');
        dom$j.removeAttributes(elms, 'data-html-5');
        elms = tmp.querySelectorAll('[data-html-6]');
        dom$j.removeAttributes(elms, 'data-html-6');
        elms = tmp.querySelectorAll('[data-html-7]');
        dom$j.removeAttributes(elms, 'data-html-7');
        elms = tmp.querySelectorAll('[data-html-8]');
        dom$j.removeAttributes(elms, 'data-html-8');
        elms = tmp.querySelectorAll('[data-html-9]');
        dom$j.removeAttributes(elms, 'data-html-9');
        elms = tmp.querySelectorAll('[data-html-10]');
        dom$j.removeAttributes(elms, 'data-html-10');
        elms = tmp.querySelectorAll('[data-html-12]');
        dom$j.removeAttributes(elms, 'data-html-12');
        elms = tmp.querySelectorAll('[data-html-13]');
        dom$j.removeAttributes(elms, 'data-html-13');
        elms = tmp.querySelectorAll('[data-html-14]');
        dom$j.removeAttributes(elms, 'data-html-14');
        elms = tmp.querySelectorAll('[data-html-15]');
        dom$j.removeAttributes(elms, 'data-html-15');
        elms = tmp.querySelectorAll('[data-html-16]');
        dom$j.removeAttributes(elms, 'data-html-16');
        elms = tmp.querySelectorAll('[data-html-17]');
        dom$j.removeAttributes(elms, 'data-html-17');
        elms = tmp.querySelectorAll('[data-html-18]');
        dom$j.removeAttributes(elms, 'data-html-18');
        elms = tmp.querySelectorAll('[data-html-19]');
        dom$j.removeAttributes(elms, 'data-html-19');
        elms = tmp.querySelectorAll('[data-html-20]');
        dom$j.removeAttributes(elms, 'data-html-20');
        elms = tmp.querySelectorAll('[data-html-21]');
        dom$j.removeAttributes(elms, 'data-html-21');
        elms = tmp.querySelectorAll('[data-html-21]');
        dom$j.removeAttributes(elms, 'data-html-21');
        elms = tmp.querySelectorAll('[data-html-22]');
        dom$j.removeAttributes(elms, 'data-html-22');
        elms = tmp.querySelectorAll('[data-html-23]');
        dom$j.removeAttributes(elms, 'data-html-23');
        elms = tmp.querySelectorAll('[data-html-24]');
        dom$j.removeAttributes(elms, 'data-html-24');
        elms = tmp.querySelectorAll('[data-html-25]');
        dom$j.removeAttributes(elms, 'data-html-25');
      }

      html = tmp.innerHTML.trim();
      html = html.replace(/<font/g, '<span').replace(/<\/font/g, '</span');
    } // elm = builderStuff.querySelector('#tmp_content');
    // if(elm) builderStuff.removeChild(elm);
    // elm = builderStuff.querySelector('#tmp_buildercontent');
    // if(elm) builderStuff.removeChild(elm);
    // return html;


    let beautify = JsBeautify.html;
    html = beautify(html); // https://stackoverflow.com/questions/22962220/remove-multiple-line-breaks-n-in-javascript
    // html = html.replace(/(\r\n|\r|\n){2}/g, '$1').replace(/(\r\n|\r|\n){3,}/g, '$1\n')

    html = html.replace(/(\r\n|\r|\n){3,}/g, '$1\n');
    return html;
  }

  beautify(html) {
    let beautify = JsBeautify.html;
    html = beautify(html);
    html = html.replace(/(\r\n|\r|\n){3,}/g, '$1\n');
    return html;
  }

}

class UndoRedo {
  constructor(builder) {
    this.builder = builder;
    const dom = this.builder.dom;
    this.dom = dom;
    this.undoList = [];
  }

  readStyles() {
    if (this.builder.opts.undoContainerOnly) return '';
    let css = '';
    let i, src;
    let links = this.builder.doc.getElementsByTagName('link');

    for (i = 0; i < links.length; i++) {
      src = links[i].href.toLowerCase();

      if (src.indexOf('basetype-') !== -1) {
        css += links[i].outerHTML;
      }
    }

    links = this.builder.doc.getElementsByTagName('link');

    for (i = 0; i < links.length; i++) {
      src = links[i].href.toLowerCase();

      if (src.indexOf('basetype-') !== -1) ; else if (src.indexOf('type-') !== -1) {
        css += links[i].outerHTML;
      }
    }

    return css;
  }

  writeStyles(styles) {
    const dom = this.dom;
    if (this.builder.opts.undoContainerOnly) return;
    let i, src;
    let links = this.builder.doc.getElementsByTagName('link');

    for (i = 0; i < links.length; i++) {
      src = links[i].href.toLowerCase();

      if (src.indexOf('basetype-') !== -1) {
        links[i].parentNode.removeChild(links[i]);
      } else if (src.indexOf('type-') !== -1) {
        links[i].parentNode.removeChild(links[i]);
      }
    }

    let head = this.builder.doc.getElementsByTagName('head')[0];
    dom.appendHtml(head, styles);
  }

  readHtml() {
    // const htmlutil = new HtmlUtil(this.builder); 
    if (this.builder.opts.page !== '' && !this.builder.opts.undoContainerOnly) {
      let wrapper = this.builder.doc.querySelector(this.builder.opts.page); // return htmlutil.readHtml(wrapper, false);

      return wrapper.innerHTML;
    } else {
      const builders = this.builder.doc.querySelectorAll(this.builder.opts.container);
      let html = '';
      Array.prototype.forEach.call(builders, builder => {
        // if(html==='') {
        //     html = htmlutil.readHtml(builder, false);
        // } else {
        //     html+= '####|####' + htmlutil.readHtml(builder, false);
        // }
        // html += htmlutil.readHtml(builder, false) + '####|####'; //new
        html += builder.innerHTML + '####|####'; //new
      });
      html = html.substr(0, html.length - 9); //new

      return html;
    }
  }

  cleaning(tmp) {
    // See html.js readHtml's cleaning
    const dom = this.dom;
    let builders = tmp.querySelectorAll('.is-builder');
    Array.prototype.forEach.call(builders, builder => {
      builder.removeAttribute('data-sort');
    });
    let elms = tmp.querySelectorAll('[data-click]');
    dom.removeAttributes(elms, 'data-click');
    dom.removeElements(tmp.querySelectorAll('.is-row-tool'));
    dom.removeElements(tmp.querySelectorAll('.is-rowadd-tool'));
    dom.removeElements(tmp.querySelectorAll('.is-col-tool'));
    dom.removeElements(tmp.querySelectorAll('.ovl'));
    dom.removeElements(tmp.querySelectorAll('.row-add-initial')); //ContentBox

    elms = tmp.querySelectorAll('.is-section-tool');
    dom.removeElements(elms);
  }

  writeHtml(html) {
    if (this.builder.opts.page !== '' && !this.builder.opts.undoContainerOnly) {
      let wrapper = this.builder.doc.querySelector(this.builder.opts.page); // wrapper.innerHTML = html;
      // Use createContextualFragment() to make embedded script executable
      // https://ghinda.net/article/script-tags/

      var range = document.createRange();
      wrapper.innerHTML = '';
      wrapper.appendChild(range.createContextualFragment(html)); // applyBehavior (in ContentBox, unUndo will call pageSetup/applyBehavior)

      this.cleaning(wrapper);
    } else {
      const builders = this.builder.doc.querySelectorAll(this.builder.opts.container);
      let n = 0;
      Array.prototype.forEach.call(builders, builder => {
        // builder.innerHTML = html.split('####|####')[n];
        // Use createContextualFragment() to make embedded script executable
        // https://ghinda.net/article/script-tags/
        var range = document.createRange();
        builder.innerHTML = '';
        builder.appendChild(range.createContextualFragment(html.split('####|####')[n])); // applyBehavior

        this.cleaning(builder);
        this.builder.applyBehaviorOn(builder);
        n++;
      });
    }

    this.builder.opts.onRender();
  }

  saveForUndo(checkLater) {
    if (this.builder.undoRedoStyles) {
      if (this.undoList[120]) {
        let saves = this.undoList[120][0];
        let html = saves.split('###$###')[1]; // Styles doesn't need to be checked. saveForUndoCheck is true only after colorpicker or gradient picker opened
        // (because after opened, user may not make changes, so checking for content change is needed).
        // This relates to html content, not styles. So we always check html content, not styles. 

        if (this.builder.saveForUndoCheck) {
          if (html === this.readHtml()) {
            // no change
            // console.log('no change');
            if (checkLater === true) this.builder.saveForUndoCheck = true;else this.builder.saveForUndoCheck = false;
            return;
          }
        }
      }
    } else {
      if (this.undoList[120]) {
        let html = this.undoList[120][0];

        if (this.builder.saveForUndoCheck) {
          if (html === this.readHtml()) {
            // no change
            // console.log('no change');
            if (checkLater === true) this.builder.saveForUndoCheck = true;else this.builder.saveForUndoCheck = false;
            return;
          }
        }
      }
    }

    if (checkLater === true) this.builder.saveForUndoCheck = true;else this.builder.saveForUndoCheck = false; // console.log('save');

    this.undoList[140] = this.undoList[139]; // addition

    this.undoList[139] = this.undoList[138];
    this.undoList[138] = this.undoList[137];
    this.undoList[137] = this.undoList[136];
    this.undoList[136] = this.undoList[135];
    this.undoList[135] = this.undoList[134];
    this.undoList[134] = this.undoList[133];
    this.undoList[133] = this.undoList[132];
    this.undoList[132] = this.undoList[131];
    this.undoList[131] = this.undoList[130];
    this.undoList[130] = this.undoList[129];
    this.undoList[129] = this.undoList[128];
    this.undoList[128] = this.undoList[127];
    this.undoList[127] = this.undoList[126];
    this.undoList[126] = this.undoList[125];
    this.undoList[125] = this.undoList[124];
    this.undoList[124] = this.undoList[123];
    this.undoList[123] = this.undoList[122];
    this.undoList[122] = this.undoList[121];
    this.undoList[121] = this.undoList[120];

    if (this.builder.undoRedoStyles) {
      let styles = this.readStyles();
      let html = this.readHtml();
      let saves = ' ' + styles + '###$###' + html;
      this.undoList[120] = [saves, null];
    } else {
      this.undoList[120] = [this.readHtml(), null];
    }
  }

  doUndo() {
    const dom = this.dom;
    if (!this.undoList[120]) return;
    if (this.builder.undoRedoInProcess === true) return; // do not precess if previous operation is still running
    // console.log('undo');

    this.builder.undoRedoInProcess = true;
    this.undoList[99] = this.undoList[100];
    this.undoList[100] = this.undoList[101];
    this.undoList[101] = this.undoList[102];
    this.undoList[102] = this.undoList[103];
    this.undoList[103] = this.undoList[104];
    this.undoList[104] = this.undoList[105];
    this.undoList[105] = this.undoList[106];
    this.undoList[106] = this.undoList[107];
    this.undoList[107] = this.undoList[108];
    this.undoList[108] = this.undoList[109];
    this.undoList[109] = this.undoList[110];
    this.undoList[110] = this.undoList[111];
    this.undoList[111] = this.undoList[112];
    this.undoList[112] = this.undoList[113];
    this.undoList[113] = this.undoList[114];
    this.undoList[114] = this.undoList[115];
    this.undoList[115] = this.undoList[116];
    this.undoList[116] = this.undoList[117];
    this.undoList[117] = this.undoList[118];
    this.undoList[118] = this.undoList[119];

    if (this.builder.undoRedoStyles) {
      let styles = this.readStyles();
      let html = this.readHtml();
      let saves = ' ' + styles + '###$###' + html;
      this.undoList[119] = [saves, null]; // -

      saves = this.undoList[120][0];
      styles = saves.split('###$###')[0].trim();
      html = saves.split('###$###')[1];
      this.writeStyles(styles);
      this.writeHtml(html);
    } else {
      this.undoList[119] = [this.readHtml(), null];
      let html = this.undoList[120][0];
      this.writeHtml(html);
    }

    this.builder.applyBehavior();
    this.builder.opts.onChange();
    this.undoList[120] = this.undoList[121];
    this.undoList[121] = this.undoList[122];
    this.undoList[122] = this.undoList[123];
    this.undoList[123] = this.undoList[124];
    this.undoList[124] = this.undoList[125];
    this.undoList[125] = this.undoList[126];
    this.undoList[126] = this.undoList[127];
    this.undoList[127] = this.undoList[128];
    this.undoList[128] = this.undoList[129];
    this.undoList[129] = this.undoList[130];
    this.undoList[130] = this.undoList[131];
    this.undoList[131] = this.undoList[132];
    this.undoList[132] = this.undoList[133];
    this.undoList[133] = this.undoList[134];
    this.undoList[134] = this.undoList[135];
    this.undoList[135] = this.undoList[136];
    this.undoList[136] = this.undoList[137];
    this.undoList[137] = this.undoList[138];
    this.undoList[138] = this.undoList[139];
    this.undoList[139] = this.undoList[140];
    this.undoList[140] = this.undoList[141];
    const util = new Util(this.builder);
    util.clearActiveCell();
    util.clearAfterUndoRedo();
    let elm = this.builder.doc.querySelector('[data-saveforundo]');

    if (elm) {
      let panel = document.querySelector('.is-side.elementstyles');
      dom.addClass(panel, 'active');
      setTimeout(() => {
        elm.click();
      }, 700);
    } else {
      let panel = document.querySelector('.is-side.elementstyles');
      dom.removeClass(panel, 'active');
    }

    if (this.builder.opts.onUndo) {
      this.builder.opts.onUndo();
    }

    this.builder.undoRedoInProcess = false;
  }

  doRedo() {
    const dom = this.dom;
    if (!this.undoList[119]) return;
    if (this.builder.undoRedoInProcess === true) return; // do not precess if previous operation is still running
    // console.log('redo');

    this.builder.undoRedoInProcess = true;
    this.undoList[141] = this.undoList[140];
    this.undoList[140] = this.undoList[139];
    this.undoList[139] = this.undoList[138];
    this.undoList[138] = this.undoList[137];
    this.undoList[137] = this.undoList[136];
    this.undoList[136] = this.undoList[135];
    this.undoList[135] = this.undoList[134];
    this.undoList[134] = this.undoList[133];
    this.undoList[133] = this.undoList[132];
    this.undoList[132] = this.undoList[131];
    this.undoList[131] = this.undoList[130];
    this.undoList[130] = this.undoList[129];
    this.undoList[129] = this.undoList[128];
    this.undoList[128] = this.undoList[127];
    this.undoList[127] = this.undoList[126];
    this.undoList[126] = this.undoList[125];
    this.undoList[125] = this.undoList[124];
    this.undoList[124] = this.undoList[123];
    this.undoList[123] = this.undoList[122];
    this.undoList[122] = this.undoList[121];
    this.undoList[121] = this.undoList[120];

    if (this.builder.undoRedoStyles) {
      let styles = this.readStyles();
      let html = this.readHtml();
      let saves = ' ' + styles + '###$###' + html;
      this.undoList[120] = [saves, null]; // -

      saves = this.undoList[119][0];
      styles = saves.split('###$###')[0].trim();
      html = saves.split('###$###')[1];
      this.writeStyles(styles);
      this.writeHtml(html);
    } else {
      this.undoList[120] = [this.readHtml(), null];
      let html = this.undoList[119][0];
      this.writeHtml(html);
    }

    this.builder.applyBehavior();
    this.builder.opts.onChange();
    this.undoList[119] = this.undoList[118];
    this.undoList[118] = this.undoList[117];
    this.undoList[117] = this.undoList[116];
    this.undoList[116] = this.undoList[115];
    this.undoList[115] = this.undoList[114];
    this.undoList[114] = this.undoList[113];
    this.undoList[113] = this.undoList[112];
    this.undoList[112] = this.undoList[111];
    this.undoList[111] = this.undoList[110];
    this.undoList[110] = this.undoList[109];
    this.undoList[109] = this.undoList[108];
    this.undoList[108] = this.undoList[107];
    this.undoList[107] = this.undoList[106];
    this.undoList[106] = this.undoList[105];
    this.undoList[105] = this.undoList[104];
    this.undoList[104] = this.undoList[103];
    this.undoList[103] = this.undoList[102];
    this.undoList[102] = this.undoList[101];
    this.undoList[101] = this.undoList[100];
    this.undoList[100] = this.undoList[99];
    this.undoList[99] = null;
    const util = new Util(this.builder);
    util.clearActiveCell();
    util.clearAfterUndoRedo();
    let elm = this.builder.doc.querySelector('[data-saveforundo]');

    if (elm) {
      let panel = document.querySelector('.is-side.elementstyles');
      dom.addClass(panel, 'active');
      setTimeout(() => {
        elm.click();
      }, 700);
    } else {
      let panel = document.querySelector('.is-side.elementstyles');
      dom.removeClass(panel, 'active');
    }

    if (this.builder.opts.onRedo) {
      this.builder.opts.onRedo();
    }

    this.builder.undoRedoInProcess = false;
  }

}

/*!
Ionicons
License: MIT
Copyright (c) 2015-present Ionic (http://ionic.io/)
https://ionic.io/ionicons/v1
*/

/*!
Tabler Icons
License: MIT
Copyright (c) 2020 Paweł Kuna
https://github.com/tabler/tabler-icons
*/
const prepareSvgIcons = builder => {
  const html = `<svg width="0" height="0" style="position:absolute;display:none;">
        <defs>
            <symbol viewBox="0 0 512 512" id="ion-ios-arrow-left"><path d="M352 115.4L331.3 96 160 256l171.3 160 20.7-19.3L201.5 256z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-arrow-right"><path d="M160 115.4L180.7 96 352 256 180.7 416 160 396.7 310.5 256z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-plus-outline"><path d="M256 48C141.1 48 48 141.1 48 256s93.1 208 208 208 208-93.1 208-208S370.9 48 256 48zm0 398.7c-105.1 0-190.7-85.5-190.7-190.7S150.9 65.3 256 65.3 446.7 150.9 446.7 256 361.1 446.7 256 446.7z"></path><path d="M264.1 128h-16.8v119.9H128v16.8h119.3V384h16.8V264.7H384v-16.8H264.1z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-image"><path d="M368 224c26.5 0 48-21.5 48-48s-21.5-48-48-48-48 21.5-48 48 21.5 48 48 48z"></path><path d="M452 64H60c-15.6 0-28 12.7-28 28.3v327.4c0 15.6 12.4 28.3 28 28.3h392c15.6 0 28-12.7 28-28.3V92.3c0-15.6-12.4-28.3-28-28.3zM348.9 261.7c-3-3.5-7.6-6.2-12.8-6.2-5.1 0-8.7 2.4-12.8 5.7L304.6 277c-3.9 2.8-7 4.7-11.5 4.7-4.3 0-8.2-1.6-11-4.1-1-.9-2.8-2.6-4.3-4.1L224 215.3c-4-4.6-10-7.5-16.7-7.5-6.7 0-12.9 3.3-16.8 7.8L64 368.2V107.7c1-6.8 6.3-11.7 13.1-11.7h357.7c6.9 0 12.5 5.1 12.9 12l.3 260.4-99.1-106.7z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-photos-outline"><path d="M96 128v320h384V128H96zm368 304H112V144h352v288z"></path><path d="M32 64v320h48v-16H48V80h352v32h16V64z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-settings-strong"><path d="M32 376h283.35c6.186-14.112 20.281-24 36.65-24s30.465 9.888 36.65 24H480v32h-91.35c-6.186 14.112-20.281 24-36.65 24s-30.465-9.888-36.65-24H32M32 240h91.35c6.186-14.112 20.281-24 36.65-24s30.465 9.888 36.65 24H480v32H196.65c-6.186 14.112-20.281 24-36.65 24s-30.465-9.888-36.65-24H32M32 104h283.35c6.186-14.112 20.281-24 36.65-24s30.465 9.888 36.65 24H480v32h-91.35c-6.186 14.112-20.281 24-36.65 24s-30.465-9.888-36.65-24H32"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-settings"><path d="M352 104c8.837 0 16 7.163 16 16s-7.163 16-16 16-16-7.163-16-16 7.163-16 16-16m0-16c-17.645 0-32 14.355-32 32s14.355 32 32 32 32-14.355 32-32-14.355-32-32-32zM352 376c8.837 0 16 7.163 16 16s-7.163 16-16 16-16-7.163-16-16 7.163-16 16-16m0-16c-17.645 0-32 14.355-32 32s14.355 32 32 32 32-14.355 32-32-14.355-32-32-32zM160 240c8.837 0 16 7.163 16 16s-7.163 16-16 16-16-7.163-16-16 7.163-16 16-16m0-16c-17.645 0-32 14.355-32 32s14.355 32 32 32 32-14.355 32-32-14.355-32-32-32zM207.32 248H480v16H207.32c.439-2.604.68-5.273.68-8s-.24-5.396-.68-8zM112 256c0 2.727.24 5.396.68 8H32v-16h80.68a47.955 47.955 0 0 0-.68 8zM399.32 384H480v16h-80.68c.439-2.604.68-5.273.68-8s-.24-5.396-.68-8zM304 392c0 2.727.24 5.396.68 8H32v-16h272.68a47.955 47.955 0 0 0-.68 8zM399.32 112H480v16h-80.68c.439-2.604.68-5.273.68-8s-.24-5.396-.68-8zM304.68 112c-.439 2.604-.68 5.273-.68 8s.24 5.396.68 8H32v-16h272.68z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-android-options"><path d="M32 384h272v32H32zM400 384h80v32h-80zM384 447.5c0 17.949-14.327 32.5-32 32.5-17.673 0-32-14.551-32-32.5v-95c0-17.949 14.327-32.5 32-32.5 17.673 0 32 14.551 32 32.5v95z"></path><g><path d="M32 240h80v32H32zM208 240h272v32H208zM192 303.5c0 17.949-14.327 32.5-32 32.5-17.673 0-32-14.551-32-32.5v-95c0-17.949 14.327-32.5 32-32.5 17.673 0 32 14.551 32 32.5v95z"></path></g><g><path d="M32 96h272v32H32zM400 96h80v32h-80zM384 159.5c0 17.949-14.327 32.5-32 32.5-17.673 0-32-14.551-32-32.5v-95c0-17.949 14.327-32.5 32-32.5 17.673 0 32 14.551 32 32.5v95z"></path></g></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-list-number"><rect x="0" y="0" width="2048.00" height="2048.00" fill="none" /><g transform="matrix(1,0,0,1,1024.0,1024.0)"><path d="M-1043.45,1024 C-1039.25,946.283 -1023.18,878.648 -995.249,821.096 C-967.313,763.544 -912.806,711.242 -831.728,664.192 L-710.742,594.247 C-656.55,562.74 -618.532,535.854 -596.687,513.589 C-562.24,478.722 -545.016,438.813 -545.016,393.863 C-545.016,341.352 -560.769,299.658 -592.276,268.781 C-623.783,237.904 -665.792,222.466 -718.304,222.466 C-796.02,222.466 -849.792,251.873 -879.619,310.685 C-895.582,342.192 -904.404,385.882 -906.084,441.754 L-1021.4,441.754 C-1020.14,363.197 -1005.65,299.133 -977.92,249.562 C-928.769,162.183 -842.02,118.494 -717.673,118.494 C-614.331,118.494 -538.82,146.43 -491.139,202.302 C-443.459,258.174 -419.619,320.347 -419.619,388.822 C-419.619,461.078 -445.034,522.831 -495.865,574.082 C-525.272,603.909 -577.993,640.037 -654.03,682.466 L-740.358,730.356 C-781.527,753.041 -813.874,774.676 -837.399,795.26 C-879.408,831.808 -905.874,872.347 -916.797,916.877 L-424.03,916.877 L-424.03,1024 L-1043.45,1024 Z "  /><path d="M-922.391,-764.384 L-922.391,-851.343 C-840.474,-859.324 -783.341,-872.662 -750.994,-891.356 C-718.647,-910.05 -694.492,-954.265 -678.529,-1024 L-589.049,-1024 L-589.049,-125.425 L-710.035,-125.425 L-710.035,-764.384 L-922.391,-764.384 Z "  /><path d="M-198.618,-510.942 L-198.618,-667.156 L1004.57,-667.156 L1004.57,-510.942 L-198.618,-510.942 Z "  /><path d="M-198.618,78.1071 L-198.618,-78.1071 L1004.57,-78.1071 L1004.57,78.1071 L-198.618,78.1071 Z "  /><path d="M-179.185,649.354 L-179.185,493.14 L1024,493.14 L1024,649.354 L-179.185,649.354 Z "  /></g></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-list-bullet"><rect x="0" y="0" width="2048.00" height="2048.00" fill="none" /><g transform="matrix(1,0,0,1,1024.0,1024.0)"><path d="M-379.801,-514.33 L-379.801,-670.545 L914.662,-670.545 L914.662,-514.33 L-379.801,-514.33 Z "  /><path d="M-379.801,78.1071 L-379.801,-78.1071 L914.662,-78.1071 L914.662,78.1071 L-379.801,78.1071 Z "  /><path d="M-379.801,670.545 L-379.801,514.33 L914.662,514.33 L914.662,670.545 L-379.801,670.545 Z "  /><path d="M-929.642,-469.441 L-929.642,-715.434 L-669.092,-715.434 L-669.092,-469.441 L-929.642,-469.441 Z "  /><path d="M-929.642,127.109 L-929.642,-118.885 L-669.092,-118.885 L-669.092,127.109 L-929.642,127.109 Z "  /><path d="M-929.642,715.434 L-929.642,469.441 L-669.092,469.441 L-669.092,715.434 L-929.642,715.434 Z "  /></g></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-clean">
                <g transform="matrix(1,0,0,1,1024.0,1024.0)">
                    <path d="M75.0013,893.849 L-1030.73,900.993 L-32.1518,-880.838 L1009.54,-880.838 L75.0013,893.849 Z " fill="currentColor" />
                    <path d="M-30.8571,780.685 L-845.2,787.828 L-508.893,193.749 L305.26,194.963 L-30.8571,780.685 Z " fill="#ffffff" fill-opacity="1.00" />
                </g>
            </symbol>
            <symbol viewBox="0 0 512 512" id="ion-location"><path d="M256 64c-65.9 0-119.3 53.7-119.3 120 0 114.6 119.3 264 119.3 264s119.3-149.4 119.3-264c0-66.3-53.4-120-119.3-120zm0 178.2c-31.2 0-56.4-25.4-56.4-56.7 0-31.3 25.3-56.8 56.4-56.8 31.2 0 56.4 25.4 56.4 56.8 0 31.3-25.2 56.7-56.4 56.7z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-film-outline"><path d="M0 360h400v-336h-400v336zM72 40v48h-56v-48h56zM72 104v48h-56v-48h56zM72 168v48h-56v-48h56zM72 232v48h-56v-48h56zM72 296v48h-56v-48h56zM312 40v144h-224v-144h224zM312 200v144h-224v-144h224zM384 40v48h-56v-48h56zM384 104v48h-56v-48h56zM384 168v48h-56v-48
                h56zM384 232v48h-56v-48h56zM384 296v48h-56v-48h56z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-social-youtube-outline"><path d="M265 96c65.3 0 118.7 1.1 168.1 3.3h1.4c23.1 0 42 22 42 49.1v1.1l.1 1.1c2.3 34 3.4 69.3 3.4 104.9.1 35.6-1.1 70.9-3.4 104.9l-.1 1.1v1.1c0 13.8-4.7 26.6-13.4 36.1-7.8 8.6-18 13.4-28.6 13.4h-1.6c-52.9 2.5-108.8 3.8-166.4 3.8h-10.6.1-10.9c-57.8 0-113.7-1.3-166.2-3.7h-1.6c-10.6 0-20.7-4.8-28.5-13.4-8.6-9.5-13.4-22.3-13.4-36.1v-1.1l-.1-1.1c-2.4-34.1-3.5-69.4-3.3-104.7v-.2c-.1-35.3 1-70.5 3.3-104.6l.1-1.1v-1.1c0-27.2 18.8-49.3 41.9-49.3h1.4c49.5-2.3 102.9-3.3 168.2-3.3H265m0-32.2h-18c-57.6 0-114.2.8-169.6 3.3-40.8 0-73.9 36.3-73.9 81.3C1 184.4-.1 220 0 255.7c-.1 35.7.9 71.3 3.4 107 0 45 33.1 81.6 73.9 81.6 54.8 2.6 110.7 3.8 167.8 3.8h21.6c57.1 0 113-1.2 167.9-3.8 40.9 0 74-36.6 74-81.6 2.4-35.7 3.5-71.4 3.4-107.1.1-35.7-1-71.3-3.4-107.1 0-45-33.1-81.1-74-81.1C379.2 64.8 322.7 64 265 64z"></path><path d="M207 353.8V157.4l145 98.2-145 98.2z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-play"><path d="M128 96v320l256-160L128 96z"></path></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-align-full"><rect x="0" y="0" width="2048.00" height="2048.00" fill="none" /><g transform="matrix(1,0,0,1,1024.0,1024.0)"><path d="M-770.727,738.299 L-770.727,582.085 L769.712,582.085 L769.712,738.299 L-770.727,738.299 Z " /><path d="M-770.727,-534.628 L-770.727,-690.842 L769.712,-690.842 L769.712,-534.628 L-770.727,-534.628 Z " /><path d="M-770.219,-115.563 L-770.219,-271.777 L770.219,-271.777 L770.219,-115.563 L-770.219,-115.563 Z " /><path d="M-770.219,303.503 L-770.219,147.288 L770.219,147.288 L770.219,303.503 L-770.219,303.503 Z " /></g></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-align-center"><rect x="0" y="0" width="2048.00" height="2048.00" fill="none" /><g transform="matrix(1,0,0,1,1024.0,1024.0)"><path d="M-770.727,738.299 L-770.727,582.085 L769.712,582.085 L769.712,738.299 L-770.727,738.299 Z " /><path d="M-552.286,-107.697 L-552.286,-263.911 L552.286,-263.911 L552.286,-107.697 L-552.286,-107.697 Z " /><path d="M-467.355,319.234 L-467.355,163.02 L466.34,163.02 L466.34,319.234 L-467.355,319.234 Z " /><path d="M-770.727,-534.628 L-770.727,-690.842 L769.712,-690.842 L769.712,-534.628 L-770.727,-534.628 Z " /></g></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-align-left"><rect x="0" y="0" width="2048.00" height="2048.00" fill="none" /><g transform="matrix(1,0,0,1,1024.0,1024.0)"><path d="M-770.727,738.299 L-770.727,582.085 L769.712,582.085 L769.712,738.299 L-770.727,738.299 Z " /><path d="M-770.727,-534.628 L-770.727,-690.842 L769.712,-690.842 L769.712,-534.628 L-770.727,-534.628 Z " /><path d="M-770.219,-115.563 L-770.219,-271.777 L482.839,-271.777 L482.839,-115.563 L-770.219,-115.563 Z " /><path d="M-770.219,303.503 L-770.219,147.288 L122.787,147.288 L122.787,303.503 L-770.219,303.503 Z " /></g></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-align-right"><rect x="0" y="0" width="2048.00" height="2048.00" fill="none" /><g transform="matrix(1,0,0,1,1024.0,1024.0)"><path d="M-770.727,738.299 L-770.727,582.085 L769.712,582.085 L769.712,738.299 L-770.727,738.299 Z " /><path d="M-770.727,-534.628 L-770.727,-690.842 L769.712,-690.842 L769.712,-534.628 L-770.727,-534.628 Z " /><path d="M-483.346,-118.081 L-483.346,-274.295 L769.712,-274.295 L769.712,-118.081 L-483.346,-118.081 Z " /><path d="M-123.871,303.503 L-123.871,147.288 L769.136,147.288 L769.136,303.503 L-123.871,303.503 Z " /></g></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-indent"><rect x="0" y="0" width="2048.00" height="2048.00" fill="none" /><g transform="matrix(1,0,0,1,1024.0,1024.0)"><path d="M-829.04,-514.33 L-829.04,-670.545 L808.959,-670.545 L808.959,-514.33 L-829.04,-514.33 Z " /><path d="M-829.04,670.545 L-829.04,514.33 L808.959,514.33 L808.959,670.545 L-829.04,670.545 Z " /><path d="M-254.279,-110.244 L-254.279,-266.458 L808.959,-266.458 L808.959,-110.244 L-254.279,-110.244 Z " /><path d="M-254.279,266.458 L-254.279,110.244 L808.959,110.244 L808.959,266.458 L-254.279,266.458 Z " /><path d="M-829.04,-195.117 L-490.958,-1.03508e-14 L-829.04,195.117 L-829.04,-195.117 Z " /></g></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-outdent"><rect x="0" y="0" width="2048.00" height="2048.00" fill="none" /><g transform="matrix(1,0,0,1,1024.0,1024.0)"><path d="M-829.04,-514.33 L-829.04,-670.545 L808.959,-670.545 L808.959,-514.33 L-829.04,-514.33 Z " /><path d="M-829.04,670.545 L-829.04,514.33 L808.959,514.33 L808.959,670.545 L-829.04,670.545 Z " /><path d="M-829.04,-110.244 L-829.04,-266.458 L234.198,-266.458 L234.198,-110.244 L-829.04,-110.244 Z " /><path d="M-829.04,266.458 L-829.04,110.244 L234.198,110.244 L234.198,266.458 L-829.04,266.458 Z " /><path d="M808.959,-195.117 L470.877,-1.03508e-14 L808.959,195.117 L808.959,-195.117 Z " /></g></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-table"><rect x="0" y="0" width="2048.00" height="2048.00" fill="none" /><g transform="matrix(1,0,0,1,1024.0,1024.0)"><path d="M-660.783,660.783 L660.783,660.783 L660.783,-660.783 L-660.783,-660.783 L-660.783,660.783 Z " fill="none" stroke-width="75.82" stroke="currentColor" stroke-linecap="square" stroke-linejoin="miter" /><path d="M-37.9645,698.933 L37.9645,698.933 L37.9645,-698.569 L-37.9645,-698.569 L-37.9645,698.933 Z " fill="currentColor" fill-opacity="1.00" stroke-width="0.25" stroke="currentColor" stroke-linecap="square" stroke-linejoin="miter" /><path d="M-698.933,-37.7825 L-698.933,38.1465 L698.569,38.1465 L698.569,-37.7825 L-698.933,-37.7825 Z " fill="currentColor" fill-opacity="1.00" stroke-width="0.25" stroke="currentColor" stroke-linecap="square" stroke-linejoin="miter" /></g></symbol>
            <symbol viewBox="0 0 512 512" id="ion-android-happy"><path d="M256 48C140.563 48 48 141.6 48 256s92.563 208 208 208 208-93.6 208-208S370.401 48 256 48zm0 374.4c-91.518 0-166.404-74.883-166.404-166.4 0-91.518 74.887-166.4 166.404-166.4S422.404 164.482 422.404 256 347.518 422.4 256 422.4zm72.8-187.2c17.683 0 31.201-13.518 31.201-31.2s-13.519-31.2-31.201-31.2c-17.682 0-31.2 13.518-31.2 31.2s13.518 31.2 31.2 31.2zm-145.6 0c17.682 0 31.2-13.518 31.2-31.2s-13.519-31.2-31.2-31.2c-17.683 0-31.201 13.518-31.201 31.2s13.519 31.2 31.201 31.2zM256 370.4c48.883 0 89.436-30.164 106.081-72.801H149.919C166.564 340.236 207.117 370.4 256 370.4z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-android-create"><path d="M64 368v80h80l235.727-235.729-79.999-79.998L64 368zm377.602-217.602c8.531-8.531 8.531-21.334 0-29.865l-50.135-50.135c-8.531-8.531-21.334-8.531-29.865 0l-39.468 39.469 79.999 79.998 39.469-39.467z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-minus-empty"><path d="M384 265H128v-17h256v17z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-eye"><path d="M256 128c-81.9 0-145.7 48.8-224 128 67.4 67.7 124 128 224 128 99.9 0 173.4-76.4 224-126.6C428.2 198.6 354.8 128 256 128zm0 219.3c-49.4 0-89.6-41-89.6-91.3 0-50.4 40.2-91.3 89.6-91.3s89.6 41 89.6 91.3c0 50.4-40.2 91.3-89.6 91.3z"></path><path d="M256 224c0-7.9 2.9-15.1 7.6-20.7-2.5-.4-5-.6-7.6-.6-28.8 0-52.3 23.9-52.3 53.3s23.5 53.3 52.3 53.3 52.3-23.9 52.3-53.3c0-2.3-.2-4.6-.4-6.9-5.5 4.3-12.3 6.9-19.8 6.9-17.8 0-32.1-14.3-32.1-32z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-reply"><path d="M448 400s-36.8-208-224-208v-80L64 256l160 134.4v-92.3c101.6 0 171 8.9 224 101.9z"></path></symbol>
            
            <symbol viewBox="0 0 512 512" id="ion-wrench"><path d="M461.9 114.9l-56.5 56.7-55.1-10-9.9-55.1 56.5-56.7c-12.7-12.7-30.8-18.5-44.2-17.8-13.5.7-42.3 8.3-64.6 32-21.6 22.8-44.3 65.3-24.2 112.5 2.4 5.7 5.1 13.2-2.9 21.2-8.1 8-215 202.8-215 202.8-19.4 16.7-18 47.6-.1 65.6 18.2 17.9 48.9 19 65.6-.3 0 0 193.2-205.8 202.7-215.1 8.5-8.3 16.1-5.5 21.2-2.9 35.6 18.4 86.3 2.4 112.6-23.9 26.3-26.3 31.1-51.7 31.9-64.7.8-12.9-3.7-30-18-44.3zM91.3 443.2c-6.3 6.2-16.5 6.2-22.7 0-6.2-6.3-6.2-16.5 0-22.7 6.3-6.2 16.5-6.2 22.7 0 6.2 6.3 6.2 16.5 0 22.7z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-more"><path d="M113.7 304C86.2 304 64 282.6 64 256c0-26.5 22.2-48 49.7-48 27.6 0 49.8 21.5 49.8 48 0 26.6-22.2 48-49.8 48zM256 304c-27.5 0-49.8-21.4-49.8-48 0-26.5 22.3-48 49.8-48 27.5 0 49.7 21.5 49.7 48 0 26.6-22.2 48-49.7 48zM398.2 304c-27.5 0-49.8-21.4-49.8-48 0-26.5 22.2-48 49.8-48 27.5 0 49.8 21.5 49.8 48 0 26.6-22.2 48-49.8 48z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-code-working"><circle cx="184.166" cy="256.166" r="24"></circle><circle cx="256.166" cy="256.166" r="24"></circle><circle cx="328.166" cy="256.166" r="24"></circle><g><path d="M168 392a23.929 23.929 0 0 1-16.971-7.029l-112-112c-9.373-9.373-9.373-24.569 0-33.941l112-112c9.373-9.372 24.568-9.372 33.941 0 9.371 9.372 9.371 24.568 0 33.941L89.941 256l95.029 95.029c9.371 9.372 9.371 24.568 0 33.941A23.925 23.925 0 0 1 168 392zM344 392a23.929 23.929 0 0 0 16.971-7.029l112-112c9.373-9.373 9.373-24.569 0-33.941l-112-112c-9.373-9.372-24.568-9.372-33.941 0-9.371 9.372-9.371 24.568 0 33.941L422.059 256l-95.029 95.029c-9.371 9.372-9.371 24.568 0 33.941A23.925 23.925 0 0 0 344 392z"></path></g></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-gear"><path d="M416.349 256.046c-.001-21.013 13.143-38.948 31.651-46.062a196.302 196.302 0 0 0-23.664-57.139 49.423 49.423 0 0 1-20.082 4.254c-12.621 0-25.238-4.811-34.871-14.442-14.863-14.863-18.248-36.846-10.18-54.97A196.274 196.274 0 0 0 302.074 64C294.971 82.529 277.027 95.69 256 95.69c-21.025 0-38.969-13.161-46.073-31.69a196.243 196.243 0 0 0-57.128 23.688c8.068 18.122 4.683 40.104-10.181 54.97-9.631 9.631-22.25 14.443-34.871 14.443a49.429 49.429 0 0 1-20.083-4.255A196.273 196.273 0 0 0 64 209.984c18.509 7.112 31.652 25.049 31.652 46.062 0 21.008-13.132 38.936-31.63 46.054a196.318 196.318 0 0 0 23.692 57.128 49.428 49.428 0 0 1 20.032-4.232c12.622 0 25.239 4.812 34.871 14.443 14.841 14.841 18.239 36.781 10.215 54.889a196.257 196.257 0 0 0 57.13 23.673c7.128-18.479 25.046-31.596 46.038-31.596 20.992 0 38.91 13.115 46.037 31.596a196.234 196.234 0 0 0 57.132-23.675c-8.023-18.106-4.626-40.046 10.216-54.887 9.629-9.632 22.248-14.444 34.868-14.444 6.836 0 13.67 1.411 20.033 4.233a196.318 196.318 0 0 0 23.692-57.128c-18.498-7.119-31.629-25.048-31.629-46.054zM256.9 335.9c-44.3 0-80-35.9-80-80 0-44.101 35.7-80 80-80 44.299 0 80 35.899 80 80 0 44.1-35.701 80-80 80z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-arrow-expand"><path d="M274 209.7l63.9-63.8L288 96h128v128l-49.9-49.9-63.8 63.9zM274 302.3l63.9 63.8L288 416h128V288l-49.9 49.9-63.8-63.9zM238 302.3l-63.9 63.8L224 416H96V288l49.9 49.9 63.8-63.9zM238 209.7l-63.9-63.8L224 96H96v128l49.9-49.9 63.8 63.9z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-android-expand"><path d="M396.795 396.8H320V448h128V320h-51.205zM396.8 115.205V192H448V64H320v51.205zM115.205 115.2H192V64H64v128h51.205zM115.2 396.795V320H64v128h128v-51.205z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-arrow-move"><path d="M480 256l-96-96v76H276V128h76l-96-96-96 96h76v108H128v-76l-96 96 96 96v-76h108v108h-76l96 96 96-96h-76.2l-.4-108.5 108.6.3V352z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-drag"><path d="M0 144h512v32H0zM0 240h512v32H0zM0 336h512v32H0z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-link"><path d="M74.6 256c0-38.3 31.1-69.4 69.4-69.4h88V144h-88c-61.8 0-112 50.2-112 112s50.2 112 112 112h88v-42.6h-88c-38.3 0-69.4-31.1-69.4-69.4zm85.4 22h192v-44H160v44zm208-134h-88v42.6h88c38.3 0 69.4 31.1 69.4 69.4s-31.1 69.4-69.4 69.4h-88V368h88c61.8 0 112-50.2 112-112s-50.2-112-112-112z"/></symbol>
            <symbol viewBox="0 0 512 512" id="ion-contrast"><path d="M256 32C132.3 32 32 132.3 32 256s100.3 224 224 224 224-100.3 224-224S379.7 32 256 32zm135.8 359.8C355.5 428 307 448 256 448V64c51 0 99.5 20 135.8 56.2C428 156.5 448 204.7 448 256c0 51.3-20 99.5-56.2 135.8z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-plus-empty"><path d="M384 265H264v119h-17V265H128v-17h119V128h17v120h120v17z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-arrow-thin-up"><path d="M349.7 189.8c-3.1 3.1-8 3-11.3 0L264 123.4V408c0 4.4-3.6 8-8 8s-8-3.6-8-8V123.4l-74.4 66.3c-3.4 2.9-8.1 3.2-11.2.1-3.1-3.1-3.3-8.5-.1-11.4 0 0 87-79.2 88-80s2.8-2.4 5.7-2.4 4.9 1.6 5.7 2.4 88 80 88 80c1.5 1.5 2.3 3.6 2.3 5.7s-.8 4.1-2.3 5.7z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-arrow-thin-down"><path d="M349.7 322.2c-3.1-3.1-8-3-11.3 0L264 388.6V104c0-4.4-3.6-8-8-8s-8 3.6-8 8v284.6l-74.4-66.3c-3.4-2.9-8.1-3.2-11.2-.1-3.1 3.1-3.3 8.5-.1 11.4 0 0 87 79.2 88 80s2.8 2.4 5.7 2.4 4.9-1.6 5.7-2.4 88-80 88-80c1.5-1.5 2.3-3.6 2.3-5.7s-.8-4.1-2.3-5.7z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-arrow-thin-left"><path d="M189.8 349.7c3.1-3.1 3-8 0-11.3L123.4 264H408c4.4 0 8-3.6 8-8s-3.6-8-8-8H123.4l66.3-74.4c2.9-3.4 3.2-8.1.1-11.2-3.1-3.1-8.5-3.3-11.4-.1 0 0-79.2 87-80 88S96 253.1 96 256s1.6 4.9 2.4 5.7 80 88 80 88c1.5 1.5 3.6 2.3 5.7 2.3s4.1-.8 5.7-2.3z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-arrow-thin-right"><path d="M322.2 349.7c-3.1-3.1-3-8 0-11.3l66.4-74.4H104c-4.4 0-8-3.6-8-8s3.6-8 8-8h284.6l-66.3-74.4c-2.9-3.4-3.2-8.1-.1-11.2 3.1-3.1 8.5-3.3 11.4-.1 0 0 79.2 87 80 88s2.4 2.8 2.4 5.7-1.6 4.9-2.4 5.7-80 88-80 88c-1.5 1.5-3.6 2.3-5.7 2.3s-4.1-.8-5.7-2.3z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-close-empty"><path d="M340.2 160l-84.4 84.3-84-83.9-11.8 11.8 84 83.8-84 83.9 11.8 11.7 84-83.8 84.4 84.2 11.8-11.7-84.4-84.3 84.4-84.2z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-android-more-vertical"><path d="M296 136c0-22.002-17.998-40-40-40s-40 17.998-40 40 17.998 40 40 40 40-17.998 40-40zm0 240c0-22.002-17.998-40-40-40s-40 17.998-40 40 17.998 40 40 40 40-17.998 40-40zm0-120c0-22.002-17.998-40-40-40s-40 17.998-40 40 17.998 40 40 40 40-17.998 40-40z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-quote"><path d="M192 64c-40.646 0-72.483 11.229-94.627 33.373C75.229 119.517 64 151.354 64 192v256h160V192h-96c0-23.056 4.922-39.666 14.627-49.373C152.334 132.922 168.944 128 192 128M416 64c-40.646 0-72.483 11.229-94.627 33.373C299.229 119.517 288 151.354 288 192v256h160V192h-96c0-23.056 4.922-39.666 14.627-49.373C376.334 132.922 392.944 128 416 128"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-code"><path d="M168 392a23.929 23.929 0 0 1-16.971-7.029l-112-112c-9.373-9.373-9.373-24.569 0-33.941l112-112c9.373-9.372 24.568-9.372 33.941 0 9.371 9.372 9.371 24.568 0 33.941L89.941 256l95.029 95.029c9.371 9.373 9.371 24.568 0 33.941A23.925 23.925 0 0 1 168 392zM344 392a23.929 23.929 0 0 0 16.971-7.029l112-112c9.373-9.373 9.373-24.569 0-33.941l-112-112c-9.373-9.372-24.568-9.372-33.941 0-9.371 9.372-9.371 24.568 0 33.941L422.059 256l-95.029 95.029c-9.371 9.373-9.371 24.568 0 33.941A23.925 23.925 0 0 0 344 392z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-camera"><path d="M430.4 147h-67.5l-40.4-40.8s-.2-.2-.3-.2l-.2-.2c-6-6-14.1-9.8-23.3-9.8h-84c-9.8 0-18.5 4.2-24.6 10.9v.1l-39.5 40h-69C63 147 48 161.6 48 180.2v202.1c0 18.6 15 33.7 33.6 33.7h348.8c18.5 0 33.6-15.1 33.6-33.7V180.2c0-18.6-15.1-33.2-33.6-33.2zM256 365.5c-50.9 0-92.4-41.6-92.4-92.6 0-51.1 41.5-92.6 92.4-92.6 51 0 92.4 41.5 92.4 92.6 0 51-41.4 92.6-92.4 92.6zm168.1-165c-7.7 0-14-6.3-14-14.1s6.3-14.1 14-14.1 14 6.3 14 14.1-6.3 14.1-14 14.1z"></path><path d="M256 202.9c-38.6 0-69.8 31.3-69.8 70 0 38.6 31.2 70 69.8 70 38.5 0 69.8-31.3 69.8-70s-31.3-70-69.8-70z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-move"><path d="M475.9 246.2l-79.4-79.4c-5.4-5.4-14.2-5.4-19.6 0l-.2.2c-5.4 5.4-5.4 14.2 0 19.6l54.9 54.9-161.8.5.5-161.8 54.9 54.9c5.4 5.4 14.2 5.4 19.6 0l.2-.2c5.4-5.4 5.4-14.2 0-19.6l-79.4-79.4c-5.4-5.4-14.2-5.4-19.6 0l-79.4 79.4c-5.4 5.4-5.4 14.2 0 19.6l.2.2c5.4 5.4 14.2 5.4 19.6 0l54.9-54.9.5 161.8-161.8-.5 54.9-54.9c5.4-5.4 5.4-14.2 0-19.6l-.2-.2c-5.4-5.4-14.2-5.4-19.6 0l-79.4 79.4c-5.4 5.4-5.4 14.2 0 19.6l79.4 79.4c5.4 5.4 14.2 5.4 19.6 0l.2-.2c5.4-5.4 5.4-14.2 0-19.6L80 270.5l161.8-.5-.5 161.8-54.9-54.9c-5.4-5.4-14.2-5.4-19.6 0l-.2.2c-5.4 5.4-5.4 14.2 0 19.6l79.4 79.4c5.4 5.4 14.2 5.4 19.6 0l79.4-79.4c5.4-5.4 5.4-14.2 0-19.6l-.2-.2c-5.4-5.4-14.2-5.4-19.6 0l-54.9 54.9-.5-161.8 161.8.5-54.9 54.9c-5.4 5.4-5.4 14.2 0 19.6l.2.2c5.4 5.4 14.2 5.4 19.6 0l79.4-79.4c5.5-5.4 5.5-14.2 0-19.6z"/></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-ok">
                <rect x="0" y="0" width="2048.00" height="2048.00" fill="none" />
                <g transform="matrix(1,0,0,1,1024.0,1024.0)">
                <path d="M330.323,493.628 L330.323,398.406 L-330.323,398.406 L-330.323,493.628 L330.323,493.628 Z " fill="currentColor" fill-opacity="1.00" />
                <path d="M230.718,468.568 L328.59,468.568 L328.59,-599.718 L230.718,-599.718 L230.718,468.568 Z " fill="currentColor" fill-opacity="1.00" />
                <path d="M-300.714,376.053 L-373.748,449.088 L-68.5805,754.255 L4.45387,681.221 L-300.714,376.053 Z " fill="currentColor" fill-opacity="1.00" />
                <path d="M-9.9476e-14,216.241 L-73.0344,143.207 L-378.202,448.375 L-305.168,521.409 L-9.9476e-14,216.241 Z " fill="currentColor" fill-opacity="1.00" />
                </g>
            </symbol>
            <symbol viewBox="0 0 512 512" id="ion-grid"><path d="M160 153.3c0 3.7-3 6.7-6.7 6.7h-50.5c-3.7 0-6.7-3-6.7-6.7v-50.5c0-3.7 3-6.7 6.7-6.7h50.5c3.7 0 6.7 3 6.7 6.7v50.5zM288 153.3c0 3.7-3 6.7-6.7 6.7h-50.5c-3.7 0-6.7-3-6.7-6.7v-50.5c0-3.7 3-6.7 6.7-6.7h50.5c3.7 0 6.7 3 6.7 6.7v50.5zM416 153.3c0 3.7-3 6.7-6.7 6.7h-50.5c-3.7 0-6.7-3-6.7-6.7v-50.5c0-3.7 3-6.7 6.7-6.7h50.5c3.7 0 6.7 3 6.7 6.7v50.5z"></path><g><path d="M160 281.3c0 3.7-3 6.7-6.7 6.7h-50.5c-3.7 0-6.7-3-6.7-6.7v-50.5c0-3.7 3-6.7 6.7-6.7h50.5c3.7 0 6.7 3 6.7 6.7v50.5zM288 281.3c0 3.7-3 6.7-6.7 6.7h-50.5c-3.7 0-6.7-3-6.7-6.7v-50.5c0-3.7 3-6.7 6.7-6.7h50.5c3.7 0 6.7 3 6.7 6.7v50.5zM416 281.3c0 3.7-3 6.7-6.7 6.7h-50.5c-3.7 0-6.7-3-6.7-6.7v-50.5c0-3.7 3-6.7 6.7-6.7h50.5c3.7 0 6.7 3 6.7 6.7v50.5z"></path></g><g><path d="M160 409.3c0 3.7-3 6.7-6.7 6.7h-50.5c-3.7 0-6.7-3-6.7-6.7v-50.5c0-3.7 3-6.7 6.7-6.7h50.5c3.7 0 6.7 3 6.7 6.7v50.5zM288 409.3c0 3.7-3 6.7-6.7 6.7h-50.5c-3.7 0-6.7-3-6.7-6.7v-50.5c0-3.7 3-6.7 6.7-6.7h50.5c3.7 0 6.7 3 6.7 6.7v50.5zM416 409.3c0 3.7-3 6.7-6.7 6.7h-50.5c-3.7 0-6.7-3-6.7-6.7v-50.5c0-3.7 3-6.7 6.7-6.7h50.5c3.7 0 6.7 3 6.7 6.7v50.5z"></path></g></symbol>
            <symbol viewBox="0 0 512 512" id="ion-gear-b"><path d="M448 294.4v-76.8h-42.8c-3.4-14.4-8.9-28-16.1-40.5l29.8-29.7-54.3-54.3-29.1 29.1c-12.6-7.7-26.4-13.5-41.1-17.3V64h-76.8v40.9c-14.7 3.8-28.5 9.7-41.1 17.3l-29.1-29.1-54.3 54.3 29.8 29.7c-7.2 12.5-12.6 26.1-16.1 40.5H64v76.8h44.1c3.8 13.7 9.5 26.6 16.7 38.6l-31.7 31.7 54.3 54.3 32.3-32.3c11.7 6.8 24.5 11.9 37.9 15.4v46h76.8v-46c13.5-3.5 26.2-8.6 37.9-15.4l32.3 32.3 54.3-54.3-31.6-31.7c7.2-11.9 12.9-24.8 16.7-38.6h44zm-192 15.4c-29.7 0-53.7-24.1-53.7-53.8s24-53.8 53.7-53.8 53.8 24.1 53.8 53.8-24.1 53.8-53.8 53.8z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-grid-view-outline"><path d="M448 192v-16H336V64h-16v112H192V64h-16v112H64v16h112v128H64v16h112v112h16V336h128v112h16V336h112v-16H336V192h112zM320 320H192V192h128v128z"></path></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-increase"><g transform="matrix(1,0,0,1,1024.0,1024.0)"><path d="M852.574,595.004 L852.574,507.837 L-852.574,507.837 L-852.574,595.004 L852.574,595.004 Z " /><path d="M852.574,224.232 L852.574,137.066 L-852.574,137.066 L-852.574,224.232 L852.574,224.232 Z " /><path d="M852.574,-134.971 L852.574,-222.138 L-852.574,-222.138 L-852.574,-134.971 L852.574,-134.971 Z " /><path d="M852.574,-505.743 L852.574,-592.909 L-852.574,-592.909 L-852.574,-505.743 L852.574,-505.743 Z " /></g></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-decrease"><g transform="matrix(1,0,0,1,1024.0,1024.0)"><path d="M509.832,595.004 L509.832,507.837 L-509.832,507.837 L-509.832,595.004 L509.832,595.004 Z " /><path d="M509.832,224.232 L509.832,137.066 L-509.832,137.066 L-509.832,224.232 L509.832,224.232 Z " /><path d="M509.832,-136.947 L509.832,-224.113 L-509.832,-224.113 L-509.832,-136.947 L509.832,-136.947 Z " /><path d="M509.832,-505.743 L509.832,-592.909 L-509.832,-592.909 L-509.832,-505.743 L509.832,-505.743 Z " /></g></symbol>
            <symbol viewBox="0 0 2048.0 2048.0" id="icon-strike">
                <g transform="matrix(1,0,0,1,1024.0,1024.0)">
                <path d="M298.298,-653.766 C292.151,-624.873 284.005,-605.663 273.862,-596.135 C263.719,-586.607 250.656,-581.842 234.673,-581.842 C220.535,-581.842 196.253,-589.526 161.828,-604.895 C87.4454,-637.475 17.0588,-653.766 -49.3321,-653.766 C-155.68,-653.766 -243.28,-621.339 -312.129,-556.485 C-380.979,-491.631 -415.404,-414.328 -415.404,-324.578 C-415.404,-272.94 -403.724,-225.606 -380.364,-182.575 C-357.005,-139.544 -322.733,-100.201 -277.551,-64.5467 C-232.368,-28.8923 -156.295,18.903 -49.3321,78.8392 C57.631,138.775 123.1,177.964 147.074,196.406 C182.729,223.455 209.008,252.654 225.913,284.005 C242.819,315.357 251.271,346.401 251.271,377.137 C251.271,432.463 228.987,480.412 184.419,520.984 C139.851,561.556 79.1465,581.842 2.30524,581.842 C-64.0856,581.842 -125.098,567.089 -180.731,537.582 C-236.364,508.075 -277.704,471.037 -304.753,426.469 C-331.801,381.901 -353.316,314.742 -369.299,224.991 L-403.417,224.991 L-403.417,653.766 L-369.299,653.766 C-364.996,624.873 -358.388,605.817 -349.474,596.596 C-340.561,587.375 -328.42,582.764 -313.051,582.764 C-297.068,582.764 -259.109,592.446 -199.173,611.81 C-139.236,631.174 -99.74,642.393 -80.6834,645.467 C-48.7174,651 -14.5998,653.766 21.6692,653.766 C137.239,653.766 231.753,619.495 305.214,550.952 C378.674,482.41 415.404,400.804 415.404,306.136 C415.404,256.343 403.878,208.701 380.826,163.211 C357.773,117.721 324.885,78.2244 282.161,44.7216 C239.438,11.2188 159.676,-36.8838 42.8774,-99.5863 C-100.355,-176.428 -191.027,-237.901 -229.141,-284.005 C-255.574,-315.357 -268.791,-350.089 -268.791,-388.202 C-268.791,-437.995 -247.89,-482.41 -206.088,-521.445 C-164.287,-560.48 -111.42,-579.998 -47.4879,-579.998 C9.06727,-579.998 63.7783,-565.552 116.645,-536.66 C169.512,-507.767 210.238,-468.732 238.823,-419.554 C267.408,-370.375 287.233,-304.292 298.298,-221.303 L332.415,-221.303 L332.415,-653.766 L298.298,-653.766 Z " fill="currentColor" fill-opacity="1.00" /><path d="M-530.954,-41.4477 L-530.954,41.4477 L530.954,41.4477 L530.954,-41.4477 L-530.954,-41.4477 Z " fill="currentColor" fill-opacity="1.00" /></g></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-undo"><path d="M447.9 368.2c0-16.8 3.6-83.1-48.7-135.7-35.2-35.4-80.3-53.4-143.3-56.2V96L64 224l192 128v-79.8c40 1.1 62.4 9.1 86.7 20 30.9 13.8 55.3 44 75.8 76.6l19.2 31.2H448c0-10.1-.1-22.9-.1-31.8z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-redo"><path d="M64 400h10.3l19.2-31.2c20.5-32.7 44.9-62.8 75.8-76.6 24.4-10.9 46.7-18.9 86.7-20V352l192-128L256 96v80.3c-63 2.8-108.1 20.7-143.3 56.2C60.4 285.2 64 351.5 64 368.2c.1 8.9 0 21.7 0 31.8z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-android-arrow-dropdown"><path d="M128 192l128 128 128-128z"></path></symbol>
            <symbol viewBox="0 0 512 512" id="ion-social-twitter"><path d="M492 109.5c-17.4 7.7-36 12.9-55.6 15.3 20-12 35.4-31 42.6-53.6-18.7 11.1-39.4 19.2-61.5 23.5C399.8 75.8 374.6 64 346.8 64c-53.5 0-96.8 43.4-96.8 96.9 0 7.6.8 15 2.5 22.1-80.5-4-151.9-42.6-199.6-101.3-8.3 14.3-13.1 31-13.1 48.7 0 33.6 17.2 63.3 43.2 80.7-16-.4-31-4.8-44-12.1v1.2c0 47 33.4 86.1 77.7 95-8.1 2.2-16.7 3.4-25.5 3.4-6.2 0-12.3-.6-18.2-1.8 12.3 38.5 48.1 66.5 90.5 67.3-33.1 26-74.9 41.5-120.3 41.5-7.8 0-15.5-.5-23.1-1.4C62.8 432 113.7 448 168.3 448 346.6 448 444 300.3 444 172.2c0-4.2-.1-8.4-.3-12.5C462.6 146 479 129 492 109.5z"></path></symbol>
            <symbol viewBox="0 0 24 24" id="icon-zoom-in" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                <circle cx="10" cy="10" r="7"></circle>
                <line x1="7" y1="10" x2="13" y2="10"></line>
                <line x1="10" y1="7" x2="10" y2="13"></line>
                <line x1="21" y1="21" x2="15" y2="15"></line>
            </symbol>
            <symbol viewBox="0 0 512 512" id="ion-android-contract"><path d="M64 371.2h76.795V448H192V320H64v51.2zm76.795-230.4H64V192h128V64h-51.205v76.8zM320 448h51.2v-76.8H448V320H320v128zm51.2-307.2V64H320v128h128v-51.2h-76.8z"></path></symbol>
            
            <symbol viewBox="0 0 24 24" id="icon-lock-off" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"/>
                <line x1="3" y1="3" x2="21" y2="21" />
                <path d="M19 19a2 2 0 0 1 -2 2h-10a2 2 0 0 1 -2 -2v-6a2 2 0 0 1 2 -2h4m4 0h2a2 2 0 0 1 2 2v2" />
                <circle cx="12" cy="16" r="1" />
                <path d="M8 11v-3m.712 -3.278a4 4 0 0 1 7.288 2.278v4" />
            </symbol>
            <symbol viewBox="0 0 24 24" id="icon-lock" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"/>
                <rect x="5" y="11" width="14" height="10" rx="2" />
                <circle cx="12" cy="16" r="1" />
                <path d="M8 11v-4a4 4 0 0 1 8 0v4" />
            </symbol>
            <symbol viewBox="0 0 24 24" id="icon-code" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z"/>
                <polyline points="7 8 3 12 7 16" />
                <polyline points="17 8 21 12 17 16" />
                <line x1="14" y1="4" x2="10" y2="20" />
            </symbol>
            <symbol viewBox="0 0 512 512" id="ion-ios-cloud-upload-outline">
                <path d="M193.3 260.4l-11.6-11.6 74.5-74.3 74.5 74.3-11.7 11.6-54.6-54.6v241.8h-16.5V205.8z"/>
                <path d="M399.3 183.6c0-1.2.2-2.4.2-3.6 0-64.3-52.8-116.4-116.8-116.4-46.1 0-85.8 27.1-104.4 66.3-8.1-4.1-17.1-6.4-26.8-6.4-29.6 0-54.1 23.6-58.9 52C57.4 187.6 32 222.2 32 261.8c0 49.7 40.1 90.2 89.6 90.2H213v-16h-90.6c-40.9 0-74.2-33.5-74.2-74.6 0-31.8 20.2-61.2 50.2-71.6l8.4-2.9 1.5-8.8c3.6-21.6 22.1-39.3 43.9-39.3 6.9 0 13.7 1.6 19.9 4.8l13.5 6.8 6.5-13.7c16.6-34.9 52.1-57.4 90.4-57.4 55.3 0 100.9 43.3 100.9 98.9 0 13.3-.2 20.3-.2 20.3l15.2.1c36.6.5 65.6 33.4 65.6 70.3 0 36.8-29.8 66.9-66.5 67.1H297v16h101c45 0 82-37.3 82-82.8s-35.5-85.5-80.7-85.6z"/>
            </symbol>
            <symbol viewBox="0 0 512 512" id="ion-volume-medium">
                <path d="M270 407.7V104.4L175.3 192H71v128h104.3zm56.3-52.1c20.5-27.8 32.8-62.3 32.8-99.6 0-37.4-12.3-71.8-32.8-99.6l-20.4 15.3c17.4 23.6 27.8 52.7 27.8 84.3 0 31.6-10.4 60.7-27.8 84.3l20.4 15.3zm66.5 46c30-40.7 48-91 48-145.6s-18-104.9-48-145.6l-20.4 15.3c26.9 36.4 43 81.4 43 130.3 0 48.9-16.1 93.8-43 130.3l20.4 15.3z"/>
            </symbol>



            <symbol id="icon-reload" viewBox="0 0 24 24" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                <path d="M19.933 13.041a8 8 0 1 1 -9.925 -8.788c3.899 -1.002 7.935 1.007 9.425 4.747"></path>
                <path d="M20 4v5h-5"></path>
            </symbol>
            <symbol id="icon-devices" viewBox="0 0 24 24" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                <rect x="13" y="8" width="8" height="12" rx="1"></rect>
                <path d="M18 8v-3a1 1 0 0 0 -1 -1h-13a1 1 0 0 0 -1 1v12a1 1 0 0 0 1 1h9"></path>
                <line x1="16" y1="9" x2="18" y2="9"></line>
            </symbol>  
            <symbol id="icon-device-desktop" viewBox="0 0 24 24" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                <rect x="3" y="4" width="18" height="12" rx="1"></rect>
                <line x1="7" y1="20" x2="17" y2="20"></line>
                <line x1="9" y1="16" x2="9" y2="20"></line>
                <line x1="15" y1="16" x2="15" y2="20"></line>
            </symbol>
            <symbol id="icon-device-mobile" viewBox="0 0 24 24" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                <rect x="6" y="3" width="12" height="18" rx="2"></rect>
                <line x1="11" y1="4" x2="13" y2="4"></line>
                <line x1="12" y1="17" x2="12" y2="17.01"></line>
            </symbol>
            <symbol id="icon-device-laptop" viewBox="0 0 24 24" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                <line x1="3" y1="19" x2="21" y2="19"></line>
                <rect x="5" y="6" width="14" height="10" rx="1"></rect>
            </symbol>
            <symbol id="icon-device-tablet" viewBox="0 0 24 24" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                <rect x="5" y="3" width="14" height="18" rx="1"></rect>
                <circle cx="12" cy="17" r="1"></circle>
            </symbol>

            <symbol id="icon-eye" viewBox="0 0 24 24" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                <path d="M12 12m-3 0a3 3 0 1 0 6 0a3 3 0 1 0 -6 0m13 0c-2.667 4.667 -6 7 -10 7s-7.333 -2.333 -10 -7c2.667 -4.667 6 -7 10 -7s7.333 2.333 10 7"></path>
            </symbol>
            <symbol id="icon-eye-off" viewBox="0 0 24 24" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                <path d="M9.88 9.878a3 3 0 1 0 4.243 4.242m.581 -3.42a3.012 3.012 0 0 0 -1.45 -1.426m-3.877 -3.913a9.469 9.469 0 0 1 2.623 -.361c4 0 7.333 2.333 10 7c-.778 1.362 -1.613 2.524 -2.504 3.489m-2.138 1.859c-1.629 1.101 -3.415 1.652 -5.358 1.652c-4 0 -7.333 -2.333 -10 -7c1.374 -2.404 2.924 -4.189 4.652 -5.354m-3.652 -3.646l18 18"></path>
            </symbol>

            <symbol id="icon-download" viewBox="0 0 24 24" stroke-width="1" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round">
                <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                <path d="M4 17v2a2 2 0 0 0 2 2h12a2 2 0 0 0 2 -2v-2"></path>
                <path d="M7 11l5 5l5 -5"></path>
                <path d="M12 4l0 12"></path>
            </symbol>

            </defs>
        </svg>`;
  builder.dom.appendHtml(builder.builderStuff, html);
};

class Snippets {
  constructor(builder) {
    this.builder = builder;
    const util = this.builder.util;
    this.util = util;
  }

  getSnippetsHtml() {
    const util = this.util;
    const snippetUrl = this.builder.opts.snippetUrl;
    const snippetPath = this.builder.opts.snippetPath;
    let snippetCategoriesString = '[';

    for (let i = 0; i < this.builder.opts.snippetCategories.length; i++) {
      snippetCategoriesString += `[${this.builder.opts.snippetCategories[i][0]},'${util.out(this.builder.opts.snippetCategories[i][1])}'],`;
    }

    snippetCategoriesString += ']';
    const defaultSnippetCategory = this.builder.opts.defaultSnippetCategory;
    const html = `
        <!DOCTYPE HTML>
        <html>
        
        <head>
            <meta charset="utf-8">
            <title></title>
            <meta name="viewport" content="width=device-width, initial-scale=1">
            <meta name="description" content="">
            <script src="${snippetUrl}" type="text/javascript"></script>
            <style>
                body {
                    // background: #fff;
                    background: ${this.builder.styleSnippetBackground};
                    margin: 0;
                }

                .is-pop-close {
                    display: flex;
                    justify-content: center;
                    align-items: center;
                    border: none;
                    background: transparent;
                    z-index:10;width:30px;height:30px;position:absolute;
                    top:2px;right:2px;box-sizing:border-box;padding:0;line-height:40px;font-size: 12px;text-align:center;cursor:pointer;
                }
                .is-pop-close:focus-visible {
                    outline:  ${this.builder.styleOutlineColor} 2px solid;
                }

                /*
                body.dark {
                    background: #111;
                }
                .dark .is-categories {
                    background: #333;
                }
                .dark .is-category-list a {
                    background: #333;
                    color: #fff;
                }
                .dark .is-category-list a.active {
                    background: #525252;
                    color: #fff;
                }
                .dark .is-category-list a:hover {
                    color: #fff;
                }
                .dark .is-design-list>li img {
                    opacity: 0.93;
                }
                .dark .is-more-categories {
                    background: #333;
                }
                .dark .is-more-categories a {
                    background: #333;
                    color: #fff;
                }
                .dark .is-more-categories a:hover,
                .dark .is-more-categories a:focus {
                    background: #4c4c4c;
                }
                .dark .is-more-categories a.active {
                    background: #4c4c4c;
                }
                */

                
                .dark .is-design-list>li {
                    outline: transparent 1px solid;
                }

                
                        
                /* Scrollbar for modal */

                /* Darker color, because background for snippet thumbnails is always light. */
                .dark * {
                    scrollbar-width: thin;
                    scrollbar-color: rgb(78 78 78 / 62%) auto;
                }
                .dark *::-webkit-scrollbar {
                    width: 12px;
                }
                .dark *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .dark *::-webkit-scrollbar-thumb {
                    background-color:rgb(78 78 78 / 62%);
                } 

                .colored-dark * {
                    scrollbar-width: thin;
                    scrollbar-color: rgb(100, 100, 100) auto;
                }
                .colored-dark *::-webkit-scrollbar {
                    width: 12px;
                }
                .colored-dark *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .colored-dark *::-webkit-scrollbar-thumb {
                    background-color:rgb(100, 100, 100);
                } 

                .colored * {
                    scrollbar-width: thin;
                    scrollbar-color: rgba(0, 0, 0, 0.4) auto;
                }
                .colored *::-webkit-scrollbar {
                    width: 12px;
                }
                .colored *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .colored *::-webkit-scrollbar-thumb {
                    background-color: rgba(0, 0, 0, 0.4);
                } 

                .light * {
                    scrollbar-width: thin;
                    scrollbar-color: rgba(0, 0, 0, 0.4) auto;
                }
                .light *::-webkit-scrollbar {
                    width: 12px;
                }
                .light *::-webkit-scrollbar-track {
                    background: transparent;
                }
                .light *::-webkit-scrollbar-thumb {
                    background-color: rgba(0, 0, 0, 0.4);
                } 
        
                svg {
                    fill: ${this.builder.styleSnippetColor};
                }

                .is-design-list {
                    position: fixed;
                    top: 0px;
                    left: 0px;
                    border-top: transparent 68px solid;
                    width: 100%;
                    height: 100%;
                    overflow-y: auto;
                    padding: 0px 0px 30px 30px;
                    box-sizing: border-box;
                    overflow: auto;
                    list-style: none;
                    margin: 0;
                }
        
                .is-design-list>li {
                    width: 250px;
                    min-height:120px;
                    position:relative;
                    background: #fff;
                    // background: ${this.builder.styleToolBackground};
                    overflow: hidden;
                    margin: 32px 40px 0 0;
                    cursor: pointer;
                    display: inline-block;
                    outline: #ececec 1px solid;
                    box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
                    border-radius: 2px;
                }

                .is-design-list>li:focus-visible {
                    outline:  ${this.builder.styleOutlineColor} 2px solid;
                }

                .is-design-list>li img {
                    box-shadow: none;
                    opacity: 1;
                    display: block;
                    box-sizing: border-box;
                    transition: all 0.2s ease-in-out;
                    max-width: 400px;
                    width: 100%
                }
        
                // .is-design-list>li:hover img {
                //     opacity: 0.95;
                // }
                
                // .is-design-list>li:hover {
                //     background: #999;
                // }
                .is-overlay {
                    position:absolute;left:0px;top:0px;width:100%;height:100%;
                }
                .is-design-list>li .is-overlay:after {
                    background: rgba(0, 0, 0, 0.02);
                    position: absolute;
                    content: "";
                    display: block;
                    top: 0;
                    left: 0;
                    right: 0;
                    bottom: 0;
                    transition: all 0.3s ease-in-out;
                    opacity: 0;
                }
                .is-design-list>li:hover .is-overlay:after,
                .is-design-list>li:focus .is-overlay:after {
                    opacity: 0.9;
                }
                .dark .is-design-list>li .is-overlay:after {
                    background: rgb(78 78 78 / 13%);
                }
        
                .is-category-list {
                    position: relative;
                    top: 0px;
                    left: 0px;
                    width: 100%;
                    height: 80px;
                    box-sizing: border-box;
                    z-index: 1;
                }
        
                .is-category-list>div {
                    white-space: nowrap;
                    padding: 0 30px;
                    box-sizing: border-box;
                    font-family: sans-serif;
                    font-size: 10px;
                    text-transform: uppercase;
                    letter-spacing: 2px;
                    // background: #f5f5f5;
                    background: ${this.builder.styleSnippetTabsBackground};
                    // box-shadow: 0 5px 8px rgb(0 0 0 / 4%);
                }
        
                .is-category-list a {
                    display: inline-block;
                    padding: 10px 20px;
                    // background: #fefefe;
                    // color: #000;
                    background: ${this.builder.styleSnippetTabItemBackground};
                    color: ${this.builder.styleSnippetTabItemColor};
                    border-radius: 50px;
        
                    margin: 0 12px 0 0;
                    text-decoration: none;
                    box-shadow: 0 5px 15px rgba(0, 0, 0, 0.03);
                    transition: box-shadow ease 0.3s;
                }
        
                .is-category-list a:hover {
                    /*background: #fafafa;*/
                    background: ${this.builder.styleSnippetTabItemBackgroundHover};
                    box-shadow: 0 5px 15px rgba(0, 0, 0, 0.06);
                    // color: #000;
                    color: ${this.builder.styleSnippetTabItemColor};
                }
        
                .is-category-list a.active {
                    // background: #f5f5f5;
                    background: ${this.builder.styleSnippetTabItemBackgroundActive};
                    // color: #000;
                    color: ${this.builder.styleSnippetTabItemColor};
                    box-shadow: none;
                    cursor: default;
                }

                .is-category-list a:focus-visible {
                    outline:  ${this.builder.styleOutlineColor} 2px solid;
                }
        
                .is-more-categories {
                    display: none;
                    position: absolute;
                    width: 400px;
                    box-sizing: border-box;
                    padding: 0;
                    z-index: 1;
                    font-family: sans-serif;
                    font-size: 10px;
                    text-transform: uppercase;
                    letter-spacing: 2px;
                    box-shadow: 0 5px 15px rgba(0, 0, 0, 0.05);
                    // background: #fff;
                    background: ${this.builder.styleSnippetMoreItemBackground};
                }
        
                .is-more-categories a {
                    width: 200px;
                    float: left;
                    display: block;
                    box-sizing: border-box;
                    padding: 12px 20px;
                    // background: #fff;
                    background: ${this.builder.styleSnippetMoreItemBackground};
                    text-decoration: none;
                    // color: #000;
                    color: ${this.builder.styleSnippetMoreItemColor};
                    line-height: 1.6;
                    outline: none;
                }
        
                .is-more-categories a:hover,
                .is-more-categories a:focus {
                    // background: #eee;
                    background: ${this.builder.styleSnippetMoreItemBackgroundHover};
                }
        
                .is-more-categories a.active {
                    // background: #eee;
                    background: ${this.builder.styleSnippetMoreItemBackgroundActive};
                }
        
                .is-more-categories.active {
                    display: block;
                }
        
                /* First Loading */
                /* .is-category-list {
                    display: none;
                }
        
                .is-design-list {
                    display: none;
                }
        
                .pace {
                    -webkit-pointer-events: none;
                    pointer-events: none;
                    -webkit-user-select: none;
                    -moz-user-select: none;
                    user-select: none;
                }
        
                .pace-inactive {
                    display: none;
                }
        
                .pace .pace-progress {
                    background: #000000;
                    position: fixed;
                    z-index: 2000;
                    top: 0;
                    right: 100%;
                    width: 100%;
                    height: 2px;
                } */
        
                .is-more-categories>a:nth-child(0) {
                    display: none
                }
        
                .is-more-categories>a:nth-child(1) {
                    display: none
                }
        
                .is-more-categories>a:nth-child(2) {
                    display: none
                }
        
                .is-more-categories>a:nth-child(3) {
                    display: none
                }
        
                .is-more-categories>a:nth-child(4) {
                    display: none
                }
        
                .is-more-categories>a:nth-child(5) {
                    display: none
                }
        
                .is-more-categories>a:nth-child(6) {
                    display: none
                }
        
                .is-more-categories>a:nth-child(7) {
                    display: none
                }
        
                @media all and (max-width: 1212px) {
                    .is-categories>a:nth-child(7):not(.more-snippets) {
                        display: none
                    }
        
                    .is-more-categories>a:nth-child(7) {
                        display: block
                    }
                }
        
                @media all and (max-width: 1070px) {
                    .is-categories>a:nth-child(6):not(.more-snippets) {
                        display: none
                    }
        
                    .is-more-categories>a:nth-child(6) {
                        display: block
                    }
                }
        
                @media all and (max-width: 940px) {
                    .is-categories>a:nth-child(5):not(.more-snippets) {
                        display: none
                    }
        
                    .is-more-categories>a:nth-child(5) {
                        display: block
                    }
                }
        
                @media all and (max-width: 700px) {
                    .is-categories>a:nth-child(4):not(.more-snippets) {
                        display: none
                    }
        
                    .is-more-categories>a:nth-child(4) {
                        display: block
                    }
                }
        
                @media all and (max-width: 555px) {
                    .is-categories>a:nth-child(3):not(.more-snippets) {
                        display: none
                    }
        
                    .is-more-categories>a:nth-child(3) {
                        display: block
                    }
                }
        
                @media all and (max-width: 415px) {
                    .is-categories>a:nth-child(2):not(.more-snippets) {
                        display: none
                    }
        
                    .is-more-categories>a:nth-child(2) {
                        display: block
                    }
                }
        
                @media all and (max-width: 640px) {
                    .is-more-categories a {
                        width: 150px;
                        padding: 10px 5px 10px 15px;
                        font-size: 10px;
                    }
        
                    .is-more-categories {
                        left: 0 !important;
                        width: 100% !important;
                    }
                }
            </style>
        </head>
        
        <body${this.builder.styleDark ? ' class="dark"' : ''}${this.builder.styleColored ? ' class="colored"' : ''}${this.builder.styleColoredDark ? ' class="colored-dark"' : ''}${this.builder.styleLight ? ' class="light"' : ''}>

            <svg style="display:none">
                <defs>
                    <symbol viewBox="0 0 512 512" id="ion-ios-close-empty">
                        <path d="M340.2 160l-84.4 84.3-84-83.9-11.8 11.8 84 83.8-84 83.9 11.8 11.7 84-83.8 84.4 84.2 11.8-11.7-84.4-84.3 84.4-84.2z"></path>
                    </symbol>
                </defs>
            </svg>
        
            <button class="is-pop-close" tabindex="-1">
                <svg class="is-icon-flex" style="width:30px;height:30px;">
                    <use xlink:href="#ion-ios-close-empty"></use>
                </svg>
            </button>
        
            <div class="is-category-list">
                <div class="is-categories" style="position:fixed;top:0;left:0;right:0;height:68px;padding-top:17px;box-sizing:border-box;">
                </div>
            </div>

            <div class="is-more-categories" tabindex="-1" role="dialog" aria-modal="true" aria-hidden="true">
            </div>
        
            <ul class="is-design-list" tabindex="-1" role="dialog" aria-modal="true" aria-hidden="true">
            </ul>
        
            <script>
        
                var snippetPath = '${snippetPath}';
                var snippetCategories = ${snippetCategoriesString};
                var defaultSnippetCategory = ${defaultSnippetCategory};

                ${typeof this.builder.slider !== 'undefined' && this.builder.slider !== null ? `
                var slider='${this.builder.slider}';` : 'var slider=null;'};
        
                var numOfCat = snippetCategories.length;
                if (numOfCat <= 7) {
                    document.querySelector('.is-more-categories').style.width = '200px';
                }
        
                var categorytabs = document.querySelector('.is-categories');
                categorytabs.innerHTML = '';
                let html_catselect = '';
                for (var i = 0; i < numOfCat; i++) {
                    if (i < 7) {
                        html_catselect += '<a href="" data-cat="' + snippetCategories[i][0] + '">' + parent._cb.out(snippetCategories[i][1]) + '</a>';
                    }
                }
                html_catselect += '<a href="" class="more-snippets">' + parent._cb.out('More') + '</a>';
                categorytabs.innerHTML = html_catselect;
        
                var categorymore = document.querySelector('.is-more-categories');
                html_catselect = '';
                for (var i = 0; i < numOfCat; i++) {
                    html_catselect += '<a href="" data-cat="' + snippetCategories[i][0] + '">' + parent._cb.out(snippetCategories[i][1]) + '</a>';
                }
                categorymore.innerHTML = html_catselect;
        
                // Show/hide "More" button
                if (numOfCat <= 7) {
                    var bHasMore = false;
        
                    const childNodes = categorymore.childNodes;
                    let i = childNodes.length;
                    while (i--) {
                        if(childNodes[i].style.display === 'block') {
                            bHasMore = true;
                        }
                    }
                    var more = document.querySelector('.more-snippets');
                    if (!bHasMore) more.style.display = 'none';
                    else more.style.display = '';
                }
        
                var elms = categorytabs.querySelectorAll('a[data-cat="' + defaultSnippetCategory + '"]'); //.classList.add('active');
                Array.prototype.forEach.call(elms, function(elm){
                    elm.classList.add('active');
                });
                elms = categorymore.querySelectorAll('a[data-cat="' + defaultSnippetCategory + '"]'); //.classList.add('active');
                Array.prototype.forEach.call(elms, function(elm){
                    elm.classList.add('active');
                });


                document.addEventListener('keydown', function(e){
                    if(e.keyCode === 27) { // escape key
                        const moreCategories = document.querySelector('.is-more-categories');
                        if(moreCategories.classList.contains('active')) {
                            moreCategories.classList.remove('active');
                            const activeTab = categorylist.querySelector('.more-snippets');
                            activeTab.classList.remove('active');
                            activeTab.focus();
                        } else {
                            const modal = parent.document.querySelector('.is-modal.snippets');
                            removeClass(modal, 'active');
                        }
                    }
                });


                let categorylist = document.querySelector('.is-category-list');
                let designlist = document.querySelector('.is-design-list');

                const select = (elm) => {

                    if(elm.classList.contains('active')) {
                        return false;
                    }
    
                    var cat = elm.getAttribute('data-cat');
                    if (cat) if(designlist.querySelectorAll('[data-cat="' + cat + '"]').length === 0) {

                        for (let i = 0; i <snippets.length; i++) {
                    
                            var thumb = snippets[i].thumbnail;
                            
                            thumb = snippetPath + thumb;    
    
                            if (snippets[i].category === cat) {
                                designlist.insertAdjacentHTML('beforeend', '<li role="button" tabindex="0" data-id="' + snippets[i].id + '" data-cat="' + snippets[i].category + '"><img src="' + thumb + '"><span class="is-overlay"></span></li>');
                            
                                var newitem = designlist.querySelector('[data-id="' + snippets[i].id + '"]');
                                newitem.addEventListener('click', function(e){
                                    
                                    var snippetid = e.target.parentNode.getAttribute('data-id');
                                    addSnippet(snippetid);
    
                                });

                                newitem.addEventListener('keydown', (e)=>{
                                    if ((e.which === 9 && !e.shiftKey)) { // tab
                                        let last = false;
                                        if(e.target.nextElementSibling) {
                                            if(!e.target.nextElementSibling.classList.contains('active')) {
                                                last = true;
                                            }
                                        } else {
                                            last = true;
                                        }
                                        if(last) {
                                            e.preventDefault();
                                            let activeTab = categorylist.querySelector('.active');
                                            if(activeTab) activeTab.focus();
                                            else {
                                                activeTab = categorylist.querySelector('.more-snippets');
                                                activeTab.focus();
                                            }
                                        }
                                    }
                                    if ((e.which === 9 && e.shiftKey)) { // shift + tab
                                        let first = false;
                                        if(e.target.previousElementSibling) {
                                            if(!e.target.previousElementSibling.classList.contains('active')) {
                                                first = true;
                                            }
                                        } else {
                                            first = true;
                                        }
                                        if(first) {
                                            e.preventDefault();
                                            let activeTab = categorylist.querySelector('.active');
                                            if(activeTab) activeTab.focus();
                                            else {
                                                activeTab = categorylist.querySelector('.more-snippets');
                                                activeTab.focus();
                                            }
                                        }
                                    }
                                });

                            }
    
                        }    
                    }
    
                    if (cat) {
                        // Hide all, show items from selected category
                        var categorylist_items = categorylist.querySelectorAll('a');    
                        Array.prototype.forEach.call(categorylist_items, function(elm){
                            elm.className = elm.className.replace('active', '');
                        });
                        categorymore.className = categorymore.className.replace('active', ''); 
                        var categorymore_items = categorymore.querySelectorAll('a');
                        Array.prototype.forEach.call(categorymore_items, function(elm){
                            elm.className = elm.className.replace('active', '');
                        });
    
                        var items = designlist.querySelectorAll('li');
                        Array.prototype.forEach.call(items, function(elm){
                            elm.style.display = 'none';
                            elm.classList.remove('active');
                        });
                        Array.prototype.forEach.call(items, function(elm){
                            var catSplit = elm.getAttribute('data-cat').split(',');
                            for (var j = 0; j < catSplit.length; j++) {
                                if (catSplit[j] == cat) {
                                    elm.style.display = ''; // TODO: hide & show snippets => animated
                                    elm.classList.add('active');
                                }
                            }
                        });
                        
                    } else {
                        // show dropdown
                        var more = document.querySelector('.more-snippets');
                        var moreCategories = document.querySelector('.is-more-categories');
    
                        var _width = moreCategories.offsetWidth;
                        more.classList.add('active');
                        moreCategories.classList.add('active');
                        var top = more.getBoundingClientRect().top;
                        var left = more.getBoundingClientRect().left;
                        top = top + 50;
                        moreCategories.style.top = top + 'px';
                        moreCategories.style.left = left + 'px';
                    }
                    elm.classList.add('active');
    
                };

                let tabs = document.querySelectorAll('.is-categories a');
                Array.prototype.forEach.call(tabs, (tab) => {
        
                    tab.addEventListener('keydown', (e)=>{
                        e.preventDefault();
        
                        if ((e.which === 39 && e.target.nextElementSibling)) { // arrow right key pressed
                            e.target.nextElementSibling.focus();
                        } else if ((e.which === 37 && e.target.previousElementSibling)) { // arrow left key pressed
                            e.target.previousElementSibling.focus();
                        } else if(e.keyCode === 13 || e.keyCode === 32) { // enter or spacebar key
                            select(e.target);
                        } else if ((e.which === 9 && !e.shiftKey)) { // tab key pressed
        
                            let moreCategories = document.querySelector('.is-more-categories');
                            if(moreCategories.classList.contains('active')) {
                 
                                // Redirect to dropdown list
                                let firstItem;
                                let activeItem;
                                let moreCategories = document.querySelector('.is-more-categories');
                                let items = moreCategories.querySelectorAll('a');
                                items.forEach(item=>{
                                    let display = window.getComputedStyle(item).getPropertyValue('display');
                                    if(display==='block') {
                                        item.classList.add('show');
                                        if(item.classList.contains('active')) {
                                            activeItem=item;
                                        }
                                    } else {
                                        item.classList.remove('show');
                                    }
                                });
                                firstItem = moreCategories.querySelector('.show');
                                if(activeItem) {
                                    activeItem.focus();
                                } else {
                                    firstItem.focus();
                                }
                                return;

                            }

                            // Redirect to tab content
                            let inputs = [];
                            let controls = designlist.querySelectorAll('.active');
                            controls.forEach(control=>{
                                inputs.push(control);
                            });
        
                            if(inputs.length===0) return;
        
                            let firstInput = inputs[0];
        
                            firstInput.focus();
                        } else if(e.which === 40 && e.target.classList.contains('more-snippets')) { // down
                 
                            // Redirect to dropdown list
                            let firstItem;
                            let activeItem;
                            let moreCategories = document.querySelector('.is-more-categories');
                            let items = moreCategories.querySelectorAll('a');
                            items.forEach(item=>{
                                let display = window.getComputedStyle(item).getPropertyValue('display');
                                if(display==='block') {
                                    item.classList.add('show');
                                    if(item.classList.contains('active')) {
                                        activeItem=item;
                                    }
                                } else {
                                    item.classList.remove('show');
                                }
                            });
                            firstItem = moreCategories.querySelector('.show');
                            if(activeItem) {
                                activeItem.focus();
                            } else {
                                firstItem.focus();
                            }

                        } 

                        
                    });
        
                    tab.addEventListener('click', (e)=>{
                        e.preventDefault();

                        e.target.focus();
        
                        select(e.target);
                    });
        
                });

                let dropdownItems = document.querySelectorAll('.is-more-categories a');
                Array.prototype.forEach.call(dropdownItems, (item) => {

                    item.addEventListener('keydown', (e)=>{
                        e.preventDefault();
                        if(e.keyCode === 38 && e.target.previousElementSibling && window.getComputedStyle(e.target.previousElementSibling).getPropertyValue('display')==='block') { // up
                            e.target.previousElementSibling.focus();
                        } else if(e.keyCode === 40 && e.target.nextElementSibling && window.getComputedStyle(e.target.nextElementSibling).getPropertyValue('display')==='block') { // down
                            e.target.nextElementSibling.focus();
                        } else if(e.keyCode === 13 || e.keyCode === 32) { // enter or spacebar key
                            select(e.target);
                        } else if ((e.which === 9 && !e.shiftKey)) { // tab key pressed
        
                            // Redirect to tab content
                            let inputs = [];
                            let controls = designlist.querySelectorAll('.active');
                            controls.forEach(control=>{
                                inputs.push(control);
                            });
        
                            if(inputs.length===0) return;
        
                            let firstInput = inputs[0];
        
                            firstInput.focus();
                        } 
                        
                    });
        
                    item.addEventListener('click', (e)=>{
                        e.preventDefault();

                        e.target.focus();
        
                        select(e.target);
                    });
                });


        
                var snippets = data_basic.snippets; //DATA

                if (slider !== null){
                    if(slider==='slick') {
                        //remove glide
                        const predicate = (item) => (item.type!=='glide');
                        snippets = snippets.filter(predicate);
                    } else if(slider==='glide') {
                        //remove slick
                        const predicate = (item) => (item.type!=='slick');
                        snippets = snippets.filter(predicate);
                    } else if(slider==='all') {
                        // Do Nothing
                    } else {
                        // remove all slider (if incorrect settings)
                        const predicate = (item) => (item.type!=='glide' && item.type!=='slick');
                        snippets = snippets.filter(predicate);
                    }

                    // for (let i = 0; i < snippets.length; i++) {
                    //     console.log(snippets[i].type)
                    // }

                } else {
                    // Backward compatible OR if slider param not set

                    // Hide slider snippet if slick is not included
                    var bHideSliderSnippet = true;
                    if(parent._cb.win.jQuery) {
                        if(parent._cb.win.jQuery.fn.slick) {
                            bHideSliderSnippet = false;
                        }
                    }
                    if(bHideSliderSnippet){
                        const result = snippets.filter((item)=>{
                            return item.type !== 'slick';
                        });
                        snippets = [...result];
                    } 

                    if(!(parent._cb.win.Glide)){
                        const result = snippets.filter((item)=>{
                            return !(item.glide || item.type === 'glide');
                        });
                        snippets = [...result];
                    }

                    // for (let i = 0; i < snippets.length; i++) {
                    //     console.log(snippets[i].type)
                    // }

                }
                

                for (let i = 0; i <snippets.length; i++) {
                    
                    snippets[i].id = i+1;
                    var thumb = snippets[i].thumbnail;
        
                    thumb = snippetPath + thumb;
        
                    if (snippets[i].category === defaultSnippetCategory + '') {
                        designlist.insertAdjacentHTML('beforeend', '<li class="active" role="button" tabindex="0" data-id="' + snippets[i].id + '" data-cat="' + snippets[i].category + '"><img src="' + thumb + '"><span class="is-overlay"></span></li>');
                    
                        var newitem = designlist.querySelector('[data-id="' + snippets[i].id + '"]');
                        newitem.addEventListener('click', function(e){
        
                            var snippetid = e.target.parentNode.getAttribute('data-id');
                            addSnippet(snippetid);
        
                        });
        
                        newitem.addEventListener('keydown', (e)=>{

                            if ((e.which === 9 && !e.shiftKey)) { // tab
                                let last = false;
                                if(e.target.nextElementSibling) {
                                    if(!e.target.nextElementSibling.classList.contains('active')) {
                                        last = true;
                                    }
                                } else {
                                    last = true;
                                }
                                if(last) {
                                    e.preventDefault();
                                    let activeTab = categorylist.querySelector('.active');
                                    if(activeTab) activeTab.focus();
                                    else {
                                        activeTab = categorylist.querySelector('.more-snippets');
                                        activeTab.focus();
                                    }
                                }
                            }
                            if ((e.which === 9 && e.shiftKey)) { // shift + tab
                                let first = false;
                                if(e.target.previousElementSibling) {
                                    if(!e.target.previousElementSibling.classList.contains('active')) {
                                        first = true;
                                    }
                                } else {
                                    first = true;
                                }
                                if(first) {
                                    e.preventDefault();
                                    let activeTab = categorylist.querySelector('.active');
                                    if(activeTab) activeTab.focus();
                                    else {
                                        activeTab = categorylist.querySelector('.more-snippets');
                                        activeTab.focus();
                                    }
                                }
                            }
                        });

                    }

                }
        



                /*
                elms = categorymore.querySelectorAll('a');
                Array.prototype.forEach.call(elms, function(elm){
        
                    elm.addEventListener('click', function(e){
                        
                        var cat = elm.getAttribute('data-cat');
                        if(designlist.querySelectorAll('[data-cat="' + cat + '"]').length === 0) {
        
                            for (let i = 0; i <snippets.length; i++) {
                        
                                var thumb = snippets[i].thumbnail;
                                
                                thumb = snippetPath + thumb;
        
                                if (snippets[i].category === cat) {
                      
                                    designlist.insertAdjacentHTML('beforeend', '<li role="button" tabindex="0" data-id="' + snippets[i].id + '" data-cat="' + snippets[i].category + '"><img src="' + thumb + '"><span class="is-overlay"></span></li>');
                                
                                    var newitem = designlist.querySelector('[data-id="' + snippets[i].id + '"]');
                                    newitem.addEventListener('click', function(e){
                                        
                                        var snippetid = e.target.parentNode.getAttribute('data-id');
                                        addSnippet(snippetid);
        
                                    });
                                }
        
                            }    
                        }
        
                        // Hide all, show items from selected category
                        Array.prototype.forEach.call(elms, function(elm){
                            elm.className = elm.className.replace('active', '');
                        });
                        categorymore.className = categorymore.className.replace('active', ''); // hide popup
                        //var categorymore_items = categorymore.querySelectorAll('a');
                        
                        var categorylist = document.querySelector('.is-category-list');
                        var categorylist_items = categorylist.querySelectorAll('a');                
                        Array.prototype.forEach.call(categorylist_items, function(elm){
                            elm.className = elm.className.replace('active', '');
                        });
                            
                        var more = document.querySelector('.more-snippets');
                        more.className = more.className.replace('active', '');
        
                        var items = designlist.querySelectorAll('div');
                        Array.prototype.forEach.call(items, function(elm){
                            elm.style.display = 'none';
                            elm.classList.remove('active');
                        });
                        Array.prototype.forEach.call(items, function(elm){
                            var catSplit = elm.getAttribute('data-cat').split(',');
                            for (var j = 0; j < catSplit.length; j++) {
                                if (catSplit[j] == cat) {
                                    elm.style.display = '';
                                    elm.classList.add('active');
                                }
                            }
                        });
        
                        elm.classList.add('active');
        
                        e.preventDefault();
                    });
        
                });
                */
        
                var close = document.querySelector('.is-pop-close');
                close.addEventListener('click', function(e){
                    var modal = parent.document.querySelector('.is-modal.snippets');
                    removeClass(modal, 'active');
                });
        
                // Add document Click event
                document.addEventListener('click', function(e){
                    e = e || window.event;
                    var target = e.target || e.srcElement;  
        
                    if(parentsHasClass(target, 'more-snippets')) return;
                    if(hasClass(target, 'more-snippets')) return;
                    
                    var more = document.querySelector('.more-snippets');
                    var moreCategories = document.querySelector('.is-more-categories');
                    
                    more.className = more.className.replace('active', '');
                    moreCategories.className = moreCategories.className.replace('active', '');
                });
        
                parent.document.addEventListener('click', function(e){
                    var more = document.querySelector('.more-snippets');
                    var moreCategories = document.querySelector('.is-more-categories');
                    
                    more.className = more.className.replace('active', '');
                    moreCategories.className = moreCategories.className.replace('active', '');
                });
        
                function addSnippet(snippetid) {
                    
                    // TODO: var framework = parent._cb.opts.framework;
                    var snippetPathReplace = parent._cb.opts.snippetPathReplace;
                    var emailMode = parent._cb.opts.emailMode;
                    
                    // 
                    for (let i = 0; i <snippets.length; i++) {
                        if(snippets[i].id + ''=== snippetid) {
                            
                            var html = snippets[i].html;
                            var noedit = snippets[i].noedit;
                            break;
                        }
                    }
        
                    var bSnippet;
                    if (html.indexOf('"row') === -1) {
                        bSnippet = true; // Just snippet (without row/column grid)
                    } else {
                        bSnippet = false; // Snippet is wrapped in row/colum
                    }
                    if (emailMode) bSnippet = false;

                    if(bSnippet) {
                        var quickadd = parent._cb.builderStuff.querySelector('.quickadd');
                        var mode = quickadd.getAttribute('data-mode');
                        if(!mode) {
                            // in case of using viewSnippets() to open the dialog (mode=null) => change to non snippet.
                            html = '<div class="row">' +
                                '<div class="column full">' +
                                html +
                                '</div>' +
                            '</div>';
                            bSnippet=false;
                        }
                    }
         
                    // Convert snippet into your defined 12 columns grid   
                    var rowClass = parent._cb.opts.row; //row
                    var colClass = parent._cb.opts.cols; //['col s1', 'col s2', 'col s3', 'col s4', 'col s5', 'col s6', 'col s7', 'col s8', 'col s9', 'col s10', 'col s11', 'col s12']
                    if(rowClass!=='' && colClass.length===12){

                        //html = html.replace(new RegExp('row clearfix', 'g'), rowClass);
                        html = html.replace(new RegExp('row clearfix', 'g'), 'row'); // backward
                        html = html.replace(new RegExp('"row', 'g'), '"' + rowClass);
                        
                        html = html.replace(new RegExp('column full', 'g'), colClass[11]);
                        html = html.replace(new RegExp('column half', 'g'), colClass[5]);
                        html = html.replace(new RegExp('column third', 'g'), colClass[3]);
                        html = html.replace(new RegExp('column fourth', 'g'), colClass[2]);
                        html = html.replace(new RegExp('column fifth', 'g'), colClass[1]);
                        html = html.replace(new RegExp('column sixth', 'g'), colClass[1]);
                        html = html.replace(new RegExp('column two-third', 'g'), colClass[7]);
                        html = html.replace(new RegExp('column two-fourth', 'g'), colClass[8]);
                        html = html.replace(new RegExp('column two-fifth', 'g'), colClass[9]);
                        html = html.replace(new RegExp('column two-sixth', 'g'), colClass[9]);
                    }
                    
                    html = html.replace(/{id}/g, makeid()); // Replace {id} with auto generated id (for custom code snippet)
                    
                    if(parent._cb.opts.onAdd){
                        html = parent._cb.opts.onAdd(html);
                    }

                    if(snippetPathReplace.length>0) {
                        if (snippetPathReplace[0] != '') {
                            var regex = new RegExp(snippetPathReplace[0], 'g');
                            html = html.replace(regex, snippetPathReplace[1]);

                            /* for encoded replace, change / to %2F  */
                            var slash = new RegExp('/', 'g');
                            var snippetPathReplace_0 = snippetPathReplace[0].replace(slash, '%2F');
                            var snippetPathReplace_1 = snippetPathReplace[1].replace(slash, '%2F');
                            regex = new RegExp(snippetPathReplace_0, 'g');
                            html = html.replace(regex, snippetPathReplace_1);
                        }
                    }
                    
                    parent._cb.addSnippet(html, bSnippet, noedit);
        
                    var modal = parent.document.querySelector('.is-modal.snippets');
                    removeClass(modal, 'active');
        
                }
        
                function hasClass(element, classname) {
                    if(!element) return false;
                    return element.classList ? element.classList.contains(classname) : new RegExp('\\b'+ classname+'\\b').test(element.className);
                }
        
                function removeClass(element, classname) {
                    if(!element) return;
                    if(element.classList.length>0) {
                        element.className = element.className.replace(classname, '');
                    }
                }
                
                function parentsHasClass(element, classname) {
                    while (element) {
                        if(element.tagName === 'BODY' || element.tagName === 'HTML') return false;
                        if(!element.classList) return false;
                        if (element.classList.contains(classname)) {
                            return true;
                        }
                        element = element.parentNode;
                    }
                }
        
                function makeid() {//http://stackoverflow.com/questions/1349404/generate-a-string-of-5-random-characters-in-javascript
                    var text = "";
                    var possible = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
                    for (var i = 0; i < 2; i++)
                        text += possible.charAt(Math.floor(Math.random() * possible.length));
        
                    var text2 = "";
                    var possible2 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
                    for (var i = 0; i < 5; i++)
                        text2 += possible2.charAt(Math.floor(Math.random() * possible2.length));
        
                    return text + text2;
                }
        
            </script>
        
        </textarea>
        </body>
        
        </html>
        `;
    return html;
  }

}

const renderQuickAdd = builder => {
  const util = builder.util;
  const builderStuff = builder.builderStuff;
  const dom = builder.dom;
  let quickadd = builderStuff.querySelector('.quickadd');

  if (!quickadd) {
    const html = `<div class="is-pop quickadd arrow-right" style="z-index:10003;" tabindex="-1" role="dialog" aria-modal="true" aria-hidden="true">
        <div class="is-pop-close" style="display:none;z-index:1;width:40px;height:40px;position:absolute;top:0px;right:0px;box-sizing:border-box;padding:0;line-height:40px;font-size: 12px;color:#777;text-align:center;cursor:pointer;"><svg class="is-icon-flex" style="width:40px;height:40px;"><use xlink:href="#ion-ios-close-empty"></use></svg></div>
        <div class="is-pop-tabs">
            <div class="is-pop-tab-item" data-value="left">${util.out('Add to Left')}</div>
            <div class="is-pop-tab-item active" data-value="right">${util.out('Add to Right')}</div>
        </div>
        <div style="padding:12px;display:flex;flex-direction:row;flex-flow: wrap;justify-content: center;align-items: center;">
            <button title="${util.out('Paragraph')}" class="add-paragraph"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:12px;height:12px;"><use xlink:href="#icon-align-full"></use></svg></span>${util.out('Paragraph')}</button>
            <button title="${util.out('Headline')}" class="add-headline"><span style="font-family:serif;display:block;margin:0 0 8px;font-size:11px;">H</span>${util.out('Headline')}</button>
            <button title="${util.out('Image')}" class="add-image"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex ion-image" style="width:14px;height:14px;"><use xlink:href="#ion-image"></use></svg></span>${util.out('Image')}</button>
            <button title="${util.out('List')}" class="add-list"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#icon-list-bullet"></use></svg></span>${util.out('List')}</button>
            <button title="${util.out('Heading 1')}" class="add-heading1"><span style="font-family:serif;display:block;margin:0 0 8px;">H1</span>${util.out('Heading 1')}</button>
            <button title="${util.out('Heading 2')}" class="add-heading2"><span style="font-family:serif;display:block;margin:0 0 8px;">H2</span>${util.out('Heading 2')}</button>
            <button title="${util.out('Heading 3')}" class="add-heading3"><span style="font-family:serif;display:block;margin:0 0 8px;">H3</span>${util.out('Heading 3')}</button>
            <button title="${util.out('Heading 4')}" class="add-heading4"><span style="font-family:serif;display:block;margin:0 0 8px;">H4</span>${util.out('Heading 4')}</button>
            <button title="${util.out('Quote')}" class="add-quote"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#ion-quote"></use></svg></span>${util.out('Quote')}</button>
            <button title="${util.out('Preformatted')}" class="add-preformatted"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#ion-code"></use></svg></span>${util.out('Preformatted')}</button>
            ${builder.opts.emailMode ? '' : `<button title="${util.out('Button')}" class="add-button"><span style="display:block;margin:0 0 8px;"><span style="display:inline-block;border:#a1a1a1 1px solid;background:#f3f3f3;width:10px;height:4px;"></span></span>${util.out('Button')}</button>`}
            ${builder.opts.emailMode ? '' : `<button title="${util.out('Two Button')}" class="add-twobutton"><span style="display:block;margin:0 0 8px;"><span style="display:inline-block;border:#a1a1a1 1px solid;background:#f3f3f3;width:10px;height:4px;"></span></span>${util.out('Two Button')}</button>`}
            <button title="${util.out('Youtube')}" class="add-youtube"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#ion-social-youtube-outline"></use></svg></use></svg></svg></span>${util.out('Youtube')}</button>
            <button title="${util.out('Video')}" class="add-video"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#ion-ios-play"></use></svg></use></svg></svg></span>${util.out('Video')}</button>
            <button style="display:none" title="${util.out('Audio')}" class="add-audio"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#ion-volume-medium"></use></svg></use></svg></svg></span>${util.out('Audio')}</button>
            <button title="${util.out('Map')}" class="add-map"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#ion-location"></use></svg></use></svg></svg></span>${util.out('Map')}</button>
            <button title="${util.out('Table')}" class="add-table"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:15px;height:15px;"><use xlink:href="#icon-table"></use></svg></use></svg></svg></span>${util.out('Table')}</button>
            <button title="${util.out('Icon')}" class="add-icon"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#ion-android-happy"></use></svg></use></svg></svg></span>${util.out('Icon')}</button>
            <button title="${util.out('Social Links')}" class="add-social"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#ion-social-twitter"></use></svg></use></svg></svg></span>${util.out('Social Links')}</button>
            <button title="${util.out('HTML/JS')}" class="add-code"><span style="display:block;margin:0 0 8px;"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#icon-code"></use></svg></use></svg></svg></span>${util.out('HTML/JS')}</button>
            <button title="${util.out('Spacer')}" class="add-spacer"><span style="display:block;margin:0 0 8px;"><span style="display:inline-block;background:#eee;width:30px;height:5px;"></span></span>${util.out('Spacer')}</button>
            <button style="display:none" title="${util.out('Line')}" class="add-line"><span style="display:block;margin:0 0 8px;"><span style="display:inline-block;background:#ddd;width:30px;height:2px;"></span></span>${util.out('Line')}</button>
            <div class="pop-separator"></div>
            <button title="${util.out('More...')}" class="add-more" style="flex-direction:initial;">${util.out('More...')}</button>
        </div>
        </div>
        
        <div class="is-modal snippets" tabindex="-1" role="dialog" aria-modal="true" aria-hidden="true">
            <div class="is-modal-content" style="max-width:1250px;height:90%;padding:0;">
                <iframe tabindex="0" style="width:100%;height:100%;border: none;display: block;" src="about:blank"></iframe>
            </div>
        </div>
        `;
    dom.appendHtml(builderStuff, html);
    quickadd = builderStuff.querySelector('.quickadd');
    document.addEventListener('click', e => {
      e = e || window.event;
      var target = e.target || e.srcElement;

      if (quickadd.style.display === 'flex') {
        let a = dom.parentsHasClass(target, 'quickadd');
        let b = dom.parentsHasClass(target, 'row-add');
        let c = dom.parentsHasClass(target, 'is-rowadd-tool');
        let d = dom.parentsHasClass(target, 'cell-add');
        let f = dom.parentsHasClass(target, 'elm-add');
        let g = dom.parentsHasClass(target, 'row-add-initial');

        if (a || b || c || d || f || g) {
          return;
        } else {
          // quickadd.style.display = '';
          util.hidePop(quickadd);
        }
      }
    });
    let tabs = quickadd.querySelectorAll('.is-pop-tab-item');
    Array.prototype.forEach.call(tabs, tab => {
      dom.addEventListener(tab, 'click', e => {
        let elms = quickadd.querySelectorAll('.is-pop-tab-item');
        Array.prototype.forEach.call(elms, elm => {
          dom.removeClass(elm, 'active');
        });
        dom.addClass(e.target, 'active');
        let val = quickadd.querySelector('.active').getAttribute('data-value');

        if (val === 'left') {
          quickadd.setAttribute('data-mode', 'cell-left');
        } else {
          quickadd.setAttribute('data-mode', 'cell-right');
        }
      });
    });
    let elm = quickadd.querySelector('.add-paragraph');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = `<p>Lorem Ipsum is simply dummy text of the printing and typesetting industry. 
            Lorem Ipsum has been the industry's standard dummy text ever since the 1500s, 
            when an unknown printer took a galley of type and scrambled it to make a type specimen book.</p>`;
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-headline');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = `<div class="display">
                <h1>Headline Goes Here</h1>
                <p>Lorem Ipsum is simply dummy text</p>
            </div>`;
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-image');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      let html = `<img src="${builder.opts.snippetPath}example.png" alt="" />`;

      if (builder.opts.snippetSampleImage) {
        html = `<img src="${builder.opts.snippetSampleImage}" alt="" />`;
      }

      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-heading1');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = '<h1>Heading 1 here</h1>';
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-heading2');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = '<h2>Heading 2 here</h2>';
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-heading3');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = '<h3>Heading 3 here</h3>';
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-heading4');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = '<h4>Heading 4 here</h4>';
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-preformatted');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = `<pre>Lorem Ipsum is simply dummy text of the printing and typesetting industry. 
            Lorem Ipsum has been the industry's standard dummy text ever since the 1500s, 
            when an unknown printer took a galley of type and scrambled it to make a type specimen book.</pre>`;
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-list');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = `<ul style="list-style: initial;padding-left: 20px;">
                <li>Lorem Ipsum is simply dummy text</li>
                <li>Lorem Ipsum is simply dummy text</li>
            </ul>`;
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-quote');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = '<blockquote>Lorem Ipsum is simply dummy text</blockquote>';
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-button');
    if (elm) dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      let html = `<div>
                <a href="#" role="button" class="transition-all inline-block cursor-pointer no-underline border-2 border-solid border-transparent ml-1 mr-1 mt-2 mb-2 hover:border-transparent rounded bg-gray-200 hover:bg-gray-300 tracking-50 uppercase py-2 size-14 px-6 font-semibold text-gray-600 leading-relaxed">Read More</a>
            </div>`;

      if (builder.opts.emailMode) {
        html = '<div><a href="#" role="button" style="margin-top: ;margin-right: ;margin-bottom: ;margin-left: ;display: inline-block; text-decoration: none; transition: all 0.16s ease 0s; border-style: solid; cursor: pointer; background-color: rgb(220, 220, 220); color: rgb(0, 0, 0); border-color: rgb(220, 220, 220); border-width: 2px; border-radius: 0px; padding: 13px 28px; line-height: 21px; text-transform: uppercase; font-weight: 400; font-size: 14px; letter-spacing: 3px;">Read More</a></div>';
      }

      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-twobutton');
    if (elm) dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      let html = `<div>
                <a href="#" role="button" class="transition-all inline-block cursor-pointer no-underline border-2 border-solid border-transparent ml-1 mr-1 mt-2 mb-2 hover:border-transparent rounded bg-gray-200 hover:bg-gray-300 tracking-50 uppercase py-2 size-14 px-6 font-semibold text-gray-600 leading-relaxed">Read More</a>
                <a href="#" role="button" class="transition-all inline-block cursor-pointer no-underline border-2 border-solid ml-1 mr-1 mt-2 mb-2 rounded tracking-50 uppercase py-2 size-14 px-6 border-current hover:border-transparent hover:text-white font-semibold text-indigo-500 hover:bg-indigo-500 leading-relaxed">Get Started</a>
            </div>`;

      if (builder.opts.emailMode) {
        html = `<div>
                    <a href="#" role="button" style="margin-top: ;margin-right: ;margin-bottom: ;margin-left: ;display: inline-block; text-decoration: none; transition: all 0.16s ease 0s; border-style: solid; cursor: pointer; background-color: rgb(220, 220, 220); color: rgb(0, 0, 0); border-color: rgb(220, 220, 220); border-width: 2px; border-radius: 0px; padding: 13px 28px; line-height: 21px; text-transform: uppercase; font-weight: 400; font-size: 14px; letter-spacing: 3px;">Read More</a> &nbsp;
                    <a href="#" role="button" style="display: inline-block; text-decoration: none; transition: all 0.16s ease 0s; border-style: solid; cursor: pointer; background-color: rgba(0, 0, 0, 0); border-color: rgb(53, 53, 53); border-width: 2px; border-radius: 0px; padding: 13px 28px; line-height: 21px; text-transform: uppercase; font-weight: 600; font-size: 14px; letter-spacing: 3px; color: rgb(53, 53, 53);">Get Started</a>
                </div>`;
      }

      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-spacer');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = '<div class="spacer height-80"></div>'; // util.addContent(html, mode, 'data-noedit');

      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-line');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = '<hr>'; // util.addContent(html, mode, 'data-noedit');

      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-code');
    dom.addEventListener(elm, 'click', () => {
      // const mode = quickadd.getAttribute('data-mode');
      let html = '<div class="row">' + '<div class="column full" data-noedit data-html="' + encodeURIComponent('<h1 id="{id}">Lorem ipsum</h1>' + '<p>This is a code block. You can edit this block using the source dialog.</p>' + '<scr' + 'ipt>' + 'var docReady = function (fn) {' + 'var stateCheck = setInterval(function () {' + 'if (document.readyState !== "complete") return;' + 'clearInterval(stateCheck);' + 'try{fn()}catch(e){}' + '}, 1);' + '};' + 'docReady(function() {' + 'document.querySelector(\'#{id}\').innerHTML =\'<b>Hello World..!</b>\';' + '});' + '</scr' + 'ipt>') + '">' + '</div>' + '</div>';
      let noedit = true;
      let bSnippet = false;
      let snippetPathReplace = builder.opts.snippetPathReplace; // Convert snippet into your defined 12 columns grid   

      let rowClass = builder.opts.row; //row

      let colClass = builder.opts.cols; //['col s1', 'col s2', 'col s3', 'col s4', 'col s5', 'col s6', 'col s7', 'col s8', 'col s9', 'col s10', 'col s11', 'col s12']

      if (rowClass !== '' && colClass.length === 12) {
        // html = html.replace(new RegExp('row clearfix', 'g'), rowClass);
        html = html.replace(new RegExp('row clearfix', 'g'), 'row'); // backward

        html = html.replace(new RegExp('"row', 'g'), '"' + rowClass);
        html = html.replace(new RegExp('column full', 'g'), colClass[11]);
        html = html.replace(new RegExp('column half', 'g'), colClass[5]);
        html = html.replace(new RegExp('column third', 'g'), colClass[3]);
        html = html.replace(new RegExp('column fourth', 'g'), colClass[2]);
        html = html.replace(new RegExp('column fifth', 'g'), colClass[1]);
        html = html.replace(new RegExp('column sixth', 'g'), colClass[1]);
        html = html.replace(new RegExp('column two-third', 'g'), colClass[7]);
        html = html.replace(new RegExp('column two-fourth', 'g'), colClass[8]);
        html = html.replace(new RegExp('column two-fifth', 'g'), colClass[9]);
        html = html.replace(new RegExp('column two-sixth', 'g'), colClass[9]);
      }

      html = html.replace(/{id}/g, util.makeId()); // Replace {id} with auto generated id (for custom code snippet)

      if (builder.opts.onAdd) {
        html = builder.opts.onAdd(html);
      }

      if (snippetPathReplace.length > 0) {
        if (snippetPathReplace[0] !== '') {
          let regex = new RegExp(snippetPathReplace[0], 'g');
          html = html.replace(regex, snippetPathReplace[1]);
        }
      }

      builder.addSnippet(html, bSnippet, noedit);
    });
    elm = quickadd.querySelector('.add-table');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = `<table class="default" style="border-collapse:collapse;width:100%;">
                <thead>
                <tr>
                    <td style="vertical-align:top;"><br></td>
                    <td style="vertical-align:top;"><br></td>
                </tr>
                </thead>
                <tr>
                    <td style="vertical-align:top;"><br></td>
                    <td style="vertical-align:top;"><br></td>
                </tr>
            </table>`;
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-icon');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = `<div style="text-align: center;">
            <i class="icon ion-android-alarm-clock size-80"></i>
            </div>`;
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-social');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = `<div class="is-social" style="text-align: center;">
                <a href="https://twitter.com/"><i class="icon ion-social-twitter" style="margin-right: 1em"></i></a>
                <a href="https://www.facebook.com/"><i class="icon ion-social-facebook" style="margin-right: 1em"></i></a>
                <a href="mailto:you@example.com"><i class="icon ion-android-drafts"></i></a>
            </div>`;
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-map');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = `<div class="embed-responsive embed-responsive-16by9">
            <iframe width="100%" height="400" frameborder="0" scrolling="no" marginheight="0" marginwidth="0" class="mg1" src="https://maps.google.com/maps?q=Melbourne,+Victoria,+Australia&amp;hl=en&amp;sll=-7.981898,112.626504&amp;sspn=0.009084,0.016512&amp;oq=melbourne&amp;hnear=Melbourne+Victoria,+Australia&amp;t=m&amp;z=10&amp;output=embed"></iframe>
            </div>`;
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-youtube');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      const html = `<div class="embed-responsive embed-responsive-16by9">
                <iframe width="560" height="315" src="https://www.youtube.com/embed/P5yHEKqx86U?rel=0" frameborder="0" allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture" allowfullscreen=""></iframe>
            </div>`;
      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-video');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      let html = `<video style="width: 100%;" loop="" autoplay="">
                <source src="${builder.opts.snippetPath}example.mp4" type="video/mp4"></video>`;

      if (builder.opts.snippetSampleVideo) {
        html = `<video style="width: 100%;" loop="" autoplay="">
                    <source src="${builder.opts.snippetSampleVideo}" type="video/mp4"></video>`;
      }

      util.addContent(html, mode);
    });
    elm = quickadd.querySelector('.add-audio');
    dom.addEventListener(elm, 'click', () => {
      const mode = quickadd.getAttribute('data-mode');
      let html = `<div style="display:flex;width:100%;position:relative;margin:15px 0;background:transparent;">
                        <audio controls="" style="width:100%">
                            <source src="${builder.opts.snippetPath}example.mp3" type="audio/mpeg">
                            Your browser does not support the audio element.
                        </audio>
                    </div>`;

      if (builder.opts.snippetSampleAudio) {
        html = `<div style="display:flex;width:100%;position:relative;margin:15px 0;background:transparent;">
                            <audio controls="" style="width:100%">
                                <source src="${builder.opts.snippetSampleAudio}" type="audio/mpeg">
                                Your browser does not support the audio element.
                            </audio>
                    </div>`;
      }

      util.addContent(html, mode);
    });
    const snippets = new Snippets(builder);
    elm = quickadd.querySelector('.add-more');
    dom.addEventListener(elm, 'click', () => {
      let modal = builderStuff.querySelector('.snippets');
      util.showModal(modal, false, null, false); // let iframe = modal.querySelector('iframe');
      // if(iframe.src==='about:blank') {
      //     iframe.src = builder.opts.snippetData;
      // }

      const ifr = modal.querySelector('iframe');
      var doc = ifr.contentWindow.document;

      if (doc.body.innerHTML === '') {
        doc.open();
        doc.write(snippets.getSnippetsHtml());
        doc.close();
      } // quickadd.style.display = '';


      util.hidePop(quickadd);
    });
  }

  return quickadd;
};

class Grid {
  constructor(builder) {
    this.builder = builder;
    const util = this.builder.util;
    const builderStuff = this.builder.builderStuff;
    this.util = util;
    this.builderStuff = builderStuff;
    const dom = this.builder.dom;
    this.dom = dom;
    this.rowTool = new RowTool$1(builder);
  }

  moveColumnPrevious() {
    let util = this.util;
    const cell = util.cellSelected();
    if (!cell) return;

    if (cell.previousElementSibling) {
      this.builder.uo.saveForUndo();
      cell.parentElement.insertBefore(cell, cell.previousElementSibling);
      this.builder.opts.onChange();
    }
  }

  moveColumnNext() {
    let util = this.util;
    const cell = util.cellSelected();
    if (!cell) return;
    const cellnext = util.cellNext(cell);

    if (cellnext) {
      this.builder.uo.saveForUndo();
      cell.parentElement.insertBefore(cellnext, cell);
      this.builder.opts.onChange();
    }
  }

  moveColumnUp() {
    let builder = this.builder;
    let util = this.util;
    const cell = util.cellSelected();
    if (!cell) return;
    let row = cell.parentNode;
    let num = 3; //is-row-tool, is-col-tool & is-rowadd-tool

    if (row.querySelector('.is-row-overlay')) {
      num = 4; //is-row-tool, is-col-tool, is-rowadd-tool & is-row-overlay
    }

    if (row.childElementCount - num === 1) {
      //-num => minus is-row-tool, is-col-tool, is-rowadd-tool & is-row-overlay
      if (row.previousElementSibling) {
        let maxCols = 4;

        if (this.builder.maxColumns) {
          maxCols = this.builder.maxColumns;
        }

        if (row.previousElementSibling.childElementCount >= maxCols + num || row.previousElementSibling.hasAttribute('data-protected')) {
          //+2 => includes is-row-tool & is-rowaddtool
          this.builder.uo.saveForUndo(); //Move row up

          row.parentNode.insertBefore(row, row.previousElementSibling);
          this.builder.opts.onChange();
          return;
        } else {
          this.builder.uo.saveForUndo(); //Add inside prev row

          let tool = row.previousElementSibling.querySelector('.is-row-tool');
          row.previousElementSibling.removeChild(tool); //remove next row tool

          tool = row.previousElementSibling.querySelector('.is-rowadd-tool');
          row.previousElementSibling.removeChild(tool); //remove

          tool = row.previousElementSibling.querySelector('.is-col-tool');
          row.previousElementSibling.removeChild(tool); //remove

          row.previousElementSibling.appendChild(cell); //add

          row.parentNode.removeChild(row); //remove current (empty) row (including its overlay)
          //re-add tool

          let builderActive = this.builder.doc.querySelector('.builder-active');
          builder.applyBehaviorOn(builderActive);
          setTimeout(() => {
            cell.click(); //refresh active cell/row
          }, 30);
          this.builder.opts.onChange();
        }
      } else {
        //move outside container (move to previous container)
        return;
      }
    } else {
      this.builder.uo.saveForUndo();
      var rowElement = row.cloneNode(true);
      rowElement.innerHTML = '';
      rowElement.appendChild(cell);
      row.parentNode.insertBefore(rowElement, row); //re-add tool

      let builderActive = this.builder.doc.querySelector('.builder-active');
      builder.applyBehaviorOn(builderActive);
      setTimeout(() => {
        cell.click(); //refresh active cell/row
      }, 30);
      this.builder.opts.onChange();
    } //fix layout


    row = cell.parentNode; //update active row

    util.fixLayout(row);
    if (row.nextElementSibling) util.fixLayout(row.nextElementSibling);
  }

  moveColumnDown() {
    let builder = this.builder;
    let util = this.util;
    let dom = this.dom;
    const cell = util.cellSelected();
    if (!cell) return;
    let row = cell.parentNode;
    let num = 3; //is-row-tool, is-col-tool & is-rowadd-tool

    if (row.querySelector('.is-row-overlay')) {
      num = 4; //is-row-tool, is-col-tool, is-rowadd-tool & is-row-overlay
    }

    if (row.childElementCount - num === 1) {
      //-2 => minus is-row-tool & is-rowadd-tool
      if (row.nextElementSibling) {
        let maxCols = 4;

        if (this.builder.maxColumns) {
          maxCols = this.builder.maxColumns;
        }

        if (row.nextElementSibling.childElementCount >= maxCols + num || row.nextElementSibling.hasAttribute('data-protected')) {
          //+2 => includes is-row-tool & is-rowadd-tool
          this.builder.uo.saveForUndo(); //Move row down

          row.parentNode.insertBefore(row.nextElementSibling, row);
          this.builder.opts.onChange();
          return;
        } else {
          this.builder.uo.saveForUndo(); //Add inside next row

          let tool = row.nextElementSibling.querySelector('.is-row-tool');
          row.nextElementSibling.removeChild(tool); //remove next row tool

          tool = row.nextElementSibling.querySelector('.is-rowadd-tool');
          row.nextElementSibling.removeChild(tool); //remove

          tool = row.nextElementSibling.querySelector('.is-col-tool');
          row.nextElementSibling.removeChild(tool); //remove

          row.nextElementSibling.appendChild(cell); //add

          row.parentNode.removeChild(row); //remove current (empty) row (including its overlay)
          //re-add tool

          let builderActive = this.builder.doc.querySelector('.builder-active');
          builder.applyBehaviorOn(builderActive);
          setTimeout(() => {
            cell.click(); //refresh active cell/row
          }, 30);
          this.builder.opts.onChange();
        }
      } else {
        //move outside container (move to next container)
        return;
      }
    } else {
      this.builder.uo.saveForUndo();
      var rowElement = row.cloneNode(true);
      rowElement.innerHTML = '';
      rowElement.appendChild(cell);
      dom.moveAfter(rowElement, row); //re-add tool

      let builderActive = this.builder.doc.querySelector('.builder-active');
      builder.applyBehaviorOn(builderActive);
      setTimeout(() => {
        cell.click(); //refresh active cell/row
      }, 30);
      this.builder.opts.onChange();
    } //fix layout


    row = cell.parentNode; //update active row

    util.fixLayout(row);
    if (row.previousElementSibling) util.fixLayout(row.previousElementSibling);
  }

  duplicateColumn() {
    let builder = this.builder;
    let util = this.util;
    let dom = this.dom;
    const cell = util.cellSelected();
    if (!cell) return;
    this.builder.uo.saveForUndo();
    var cellElement = cell.cloneNode(true);
    dom.removeClass(cellElement, 'cell-active');
    cellElement.removeAttribute('data-click');

    if (this.builder.opts.enableDragResize) {
      cell.style.width = '';
      cell.style.flex = '';
    }

    let row = cell.parentNode;
    let num = 3;

    if (row.querySelector('.is-row-overlay')) {
      num = 4;
    }

    let maxCols = 4;

    if (this.builder.maxColumns) {
      maxCols = this.builder.maxColumns;
    }

    if (row.childElementCount >= maxCols + num) {
      alert(util.out('You have reached the maximum number of columns'));
      return false;
    }

    row.insertBefore(cellElement, cell);
    util.fixLayout(row); //re-add tool

    let builderActive = this.builder.doc.querySelector('.builder-active');
    builder.applyBehaviorOn(builderActive);
    setTimeout(() => {
      cell.click(); //refresh active cell/row
    }, 30);
    this.builder.opts.onChange();
  }

  removeColumn() {
    let util = this.util;
    const cell = util.cellSelected();
    if (!cell) return;
    let row = cell.parentNode;
    let num = 3;

    if (row.querySelector('.is-row-overlay')) {
      num = 4;
    }

    util.confirm(util.out('Are you sure you want to delete this block?'), ok => {
      if (ok) {
        this.builder.uo.saveForUndo();

        if (row.childElementCount - num === 1) {
          row.parentNode.removeChild(row);
          util.checkEmpty();
        } else {
          row.removeChild(cell);
          util.fixLayout(row);
        }

        util.clearActiveCell();
        this.builder.opts.onChange();
      }
    });
  }

  increaseColumn() {
    let builder = this.builder;
    let util = this.util;
    let dom = this.dom;
    const cell = util.cellSelected();
    if (!cell) return;
    let cellnext = util.cellNext(cell);
    let cellnext2;

    if (!cellnext) {
      cellnext = cell.previousElementSibling;
      if (!cellnext) return;
      cellnext2 = cellnext.previousElementSibling;
    } else {
      cellnext2 = util.cellNext(cellnext);
      if (!cellnext2) cellnext2 = cell.previousElementSibling;
    }

    const rowClass = builder.opts.row;
    let colClass = builder.opts.cols;
    const colSizes = builder.opts.colsizes;

    if (rowClass !== '' && colClass.length > 0 && colSizes.length > 0) {
      if (cellnext2) {
        for (let i = 0; i < colSizes.length; i++) {
          let group = colSizes[i];

          for (let j = 0; j < group.length; j++) {
            if (group[j].length === 3) {
              if (dom.hasClass(cell, group[j][0]) && dom.hasClass(cellnext, group[j][1]) && dom.hasClass(cellnext2, group[j][2])) {
                if (j + 1 === group.length) ; else {
                  this.builder.uo.saveForUndo();
                  dom.removeClass(cell, group[j][0]);
                  dom.removeClass(cellnext, group[j][1]);
                  dom.removeClass(cellnext2, group[j][2]);
                  dom.addClass(cell, group[j + 1][0]);
                  dom.addClass(cellnext, group[j + 1][1]);
                  dom.addClass(cellnext2, group[j + 1][2]);
                  this.builder.opts.onChange();
                  return;
                }
              }
            }
          }
        }
      }

      for (let i = 0; i < colSizes.length; i++) {
        let group = colSizes[i];

        for (let j = 0; j < group.length; j++) {
          if (group[j].length === 2) {
            if (dom.hasClass(cell, group[j][0]) && dom.hasClass(cellnext, group[j][1])) {
              if (j + 1 === group.length) ; else {
                this.builder.uo.saveForUndo();
                dom.removeClass(cell, group[j][0]);
                dom.removeClass(cellnext, group[j][1]);
                dom.addClass(cell, group[j + 1][0]);
                dom.addClass(cellnext, group[j + 1][1]);
                this.builder.opts.onChange();
                return;
              }
            }
          }
        }
      }

      return;
    } //others (12 columns grid)       


    if (rowClass !== '' && colClass.length > 0) {
      // Bootstrap stuff
      if (cell.className.indexOf('col-md-') !== -1) ; else if (cell.className.indexOf('col-sm-') !== -1) {
        colClass = ['col-sm-1', 'col-sm-2', 'col-sm-3', 'col-sm-4', 'col-sm-5', 'col-sm-6', 'col-sm-7', 'col-sm-8', 'col-sm-9', 'col-sm-10', 'col-sm-11', 'col-sm-12'];
      } else if (cell.className.indexOf('col-xs-') !== -1) {
        colClass = ['col-xs-1', 'col-xs-2', 'col-xs-3', 'col-xs-4', 'col-xs-5', 'col-xs-6', 'col-xs-7', 'col-xs-8', 'col-xs-9', 'col-xs-10', 'col-xs-11', 'col-xs-12'];
      } else if (cell.className.indexOf('col-lg-') !== -1) {
        colClass = ['col-lg-1', 'col-lg-2', 'col-lg-3', 'col-lg-4', 'col-lg-5', 'col-lg-6', 'col-lg-7', 'col-lg-8', 'col-lg-9', 'col-lg-10', 'col-lg-11', 'col-lg-12'];
      } else if (cell.className.indexOf('col-xl-') !== -1) {
        colClass = ['col-xl-1', 'col-xl-2', 'col-xl-3', 'col-xl-4', 'col-xl-5', 'col-xl-6', 'col-xl-7', 'col-xl-8', 'col-xl-9', 'col-xl-10', 'col-xl-11', 'col-xl-12'];
      } else if (cell.className.indexOf('col-xxl-') !== -1) {
        colClass = ['col-xxl-1', 'col-xxl-2', 'col-xxl-3', 'col-xxl-4', 'col-xxl-5', 'col-xxl-6', 'col-xxl-7', 'col-xxl-8', 'col-xxl-9', 'col-xxl-10', 'col-xxl-11', 'col-xxl-12'];
      }

      if (!dom.hasClass(cell, colClass[11])) {
        //if not column full
        if (dom.hasClass(cell, colClass[11])) {
          return;
        }

        if (dom.hasClass(cellnext, colClass[0])) {
          return;
        }

        this.builder.uo.saveForUndo();

        if (dom.hasClass(cell, colClass[10])) {
          dom.removeClass(cell, colClass[10]);
          dom.addClass(cell, colClass[11]);
        } else if (dom.hasClass(cell, colClass[9])) {
          dom.removeClass(cell, colClass[9]);
          dom.addClass(cell, colClass[10]);
        } else if (dom.hasClass(cell, colClass[8])) {
          dom.removeClass(cell, colClass[8]);
          dom.addClass(cell, colClass[9]);
        } else if (dom.hasClass(cell, colClass[7])) {
          dom.removeClass(cell, colClass[7]);
          dom.addClass(cell, colClass[8]);
        } else if (dom.hasClass(cell, colClass[6])) {
          dom.removeClass(cell, colClass[6]);
          dom.addClass(cell, colClass[7]);
        } else if (dom.hasClass(cell, colClass[5])) {
          dom.removeClass(cell, colClass[5]);
          dom.addClass(cell, colClass[6]);
        } else if (dom.hasClass(cell, colClass[4])) {
          dom.removeClass(cell, colClass[4]);
          dom.addClass(cell, colClass[5]);
        } else if (dom.hasClass(cell, colClass[3])) {
          dom.removeClass(cell, colClass[3]);
          dom.addClass(cell, colClass[4]);
        } else if (dom.hasClass(cell, colClass[2])) {
          dom.removeClass(cell, colClass[2]);
          dom.addClass(cell, colClass[3]);
        } else if (dom.hasClass(cell, colClass[1])) {
          dom.removeClass(cell, colClass[1]);
          dom.addClass(cell, colClass[2]);
        } else if (dom.hasClass(cell, colClass[0])) {
          dom.removeClass(cell, colClass[0]);
          dom.addClass(cell, colClass[1]);
        }

        if (dom.hasClass(cellnext, colClass[1])) {
          dom.removeClass(cellnext, colClass[1]);
          dom.addClass(cellnext, colClass[0]);
        } else if (dom.hasClass(cellnext, colClass[2])) {
          dom.removeClass(cellnext, colClass[2]);
          dom.addClass(cellnext, colClass[1]);
        } else if (dom.hasClass(cellnext, colClass[3])) {
          dom.removeClass(cellnext, colClass[3]);
          dom.addClass(cellnext, colClass[2]);
        } else if (dom.hasClass(cellnext, colClass[4])) {
          dom.removeClass(cellnext, colClass[4]);
          dom.addClass(cellnext, colClass[3]);
        } else if (dom.hasClass(cellnext, colClass[5])) {
          dom.removeClass(cellnext, colClass[5]);
          dom.addClass(cellnext, colClass[4]);
        } else if (dom.hasClass(cellnext, colClass[6])) {
          dom.removeClass(cellnext, colClass[6]);
          dom.addClass(cellnext, colClass[5]);
        } else if (dom.hasClass(cellnext, colClass[7])) {
          dom.removeClass(cellnext, colClass[7]);
          dom.addClass(cellnext, colClass[6]);
        } else if (dom.hasClass(cellnext, colClass[8])) {
          dom.removeClass(cellnext, colClass[8]);
          dom.addClass(cellnext, colClass[7]);
        } else if (dom.hasClass(cellnext, colClass[9])) {
          dom.removeClass(cellnext, colClass[9]);
          dom.addClass(cellnext, colClass[8]);
        } else if (dom.hasClass(cellnext, colClass[10])) {
          dom.removeClass(cellnext, colClass[10]);
          dom.addClass(cellnext, colClass[9]);
        } else if (dom.hasClass(cellnext, colClass[11])) {
          dom.removeClass(cellnext, colClass[11]);
          dom.addClass(cellnext, colClass[10]);
        }

        this.builder.opts.onChange();
      }
    }
  }

  decreaseColumn() {
    let builder = this.builder;
    let util = this.util;
    let dom = this.dom;
    const cell = util.cellSelected();
    if (!cell) return;
    let cellnext = util.cellNext(cell);
    let cellnext2;

    if (!cellnext) {
      cellnext = cell.previousElementSibling;
      if (!cellnext) return;
      cellnext2 = cellnext.previousElementSibling;
    } else {
      cellnext2 = util.cellNext(cellnext);
      if (!cellnext2) cellnext2 = cell.previousElementSibling;
    }

    const rowClass = builder.opts.row;
    let colClass = builder.opts.cols;
    const colSizes = builder.opts.colsizes;

    if (rowClass !== '' && colClass.length > 0 && colSizes.length > 0) {
      if (cellnext2) {
        for (let i = 0; i < colSizes.length; i++) {
          let group = colSizes[i];

          for (let j = 0; j < group.length; j++) {
            if (group[j].length === 3) {
              if (dom.hasClass(cell, group[j][0]) && dom.hasClass(cellnext, group[j][1]) && dom.hasClass(cellnext2, group[j][2])) {
                if (j === 0) ; else {
                  this.builder.uo.saveForUndo();
                  dom.removeClass(cell, group[j][0]);
                  dom.removeClass(cellnext, group[j][1]);
                  dom.removeClass(cellnext2, group[j][2]);
                  dom.addClass(cell, group[j - 1][0]);
                  dom.addClass(cellnext, group[j - 1][1]);
                  dom.addClass(cellnext2, group[j - 1][2]);
                  this.builder.opts.onChange();
                  return;
                }
              }
            }
          }
        }
      }

      for (let i = 0; i < colSizes.length; i++) {
        let group = colSizes[i];

        for (let j = 0; j < group.length; j++) {
          if (group[j].length === 2) {
            if (dom.hasClass(cell, group[j][0]) && dom.hasClass(cellnext, group[j][1])) {
              if (j === 0) ; else {
                this.builder.uo.saveForUndo();
                dom.removeClass(cell, group[j][0]);
                dom.removeClass(cellnext, group[j][1]);
                dom.addClass(cell, group[j - 1][0]);
                dom.addClass(cellnext, group[j - 1][1]);
                this.builder.opts.onChange();
                return;
              }
            }
          }
        }
      }

      return;
    } //others (12 columns grid)       
    // const rowClass = builder.opts.row; //row
    // const colClass = builder.opts.cols; //['col s1', 'col s2', 'col s3', 'col s4', 'col s5', 'col s6', 'col s7', 'col s8', 'col s9', 'col s10', 'col s11', 'col s12']


    if (rowClass !== '' && colClass.length > 0) {
      // Bootstrap stuff
      if (cell.className.indexOf('col-md-') !== -1) ; else if (cell.className.indexOf('col-sm-') !== -1) {
        colClass = ['col-sm-1', 'col-sm-2', 'col-sm-3', 'col-sm-4', 'col-sm-5', 'col-sm-6', 'col-sm-7', 'col-sm-8', 'col-sm-9', 'col-sm-10', 'col-sm-11', 'col-sm-12'];
      } else if (cell.className.indexOf('col-xs-') !== -1) {
        colClass = ['col-xs-1', 'col-xs-2', 'col-xs-3', 'col-xs-4', 'col-xs-5', 'col-xs-6', 'col-xs-7', 'col-xs-8', 'col-xs-9', 'col-xs-10', 'col-xs-11', 'col-xs-12'];
      } else if (cell.className.indexOf('col-lg-') !== -1) {
        colClass = ['col-lg-1', 'col-lg-2', 'col-lg-3', 'col-lg-4', 'col-lg-5', 'col-lg-6', 'col-lg-7', 'col-lg-8', 'col-lg-9', 'col-lg-10', 'col-lg-11', 'col-lg-12'];
      } else if (cell.className.indexOf('col-xl-') !== -1) {
        colClass = ['col-xl-1', 'col-xl-2', 'col-xl-3', 'col-xl-4', 'col-xl-5', 'col-xl-6', 'col-xl-7', 'col-xl-8', 'col-xl-9', 'col-xl-10', 'col-xl-11', 'col-xl-12'];
      } else if (cell.className.indexOf('col-xxl-') !== -1) {
        colClass = ['col-xxl-1', 'col-xxl-2', 'col-xxl-3', 'col-xxl-4', 'col-xxl-5', 'col-xxl-6', 'col-xxl-7', 'col-xxl-8', 'col-xxl-9', 'col-xxl-10', 'col-xxl-11', 'col-xxl-12'];
      }

      if (!dom.hasClass(cell, colClass[11])) {
        //if not column full
        if (dom.hasClass(cell, colClass[0])) {
          return;
        }

        if (dom.hasClass(cellnext, colClass[11])) {
          return;
        }

        this.builder.uo.saveForUndo();

        if (dom.hasClass(cell, colClass[11])) {
          dom.removeClass(cell, colClass[11]);
          dom.addClass(cell, colClass[10]);
        } else if (dom.hasClass(cell, colClass[10])) {
          dom.removeClass(cell, colClass[10]);
          dom.addClass(cell, colClass[9]);
        } else if (dom.hasClass(cell, colClass[9])) {
          dom.removeClass(cell, colClass[9]);
          dom.addClass(cell, colClass[8]);
        } else if (dom.hasClass(cell, colClass[8])) {
          dom.removeClass(cell, colClass[8]);
          dom.addClass(cell, colClass[7]);
        } else if (dom.hasClass(cell, colClass[7])) {
          dom.removeClass(cell, colClass[7]);
          dom.addClass(cell, colClass[6]);
        } else if (dom.hasClass(cell, colClass[6])) {
          dom.removeClass(cell, colClass[6]);
          dom.addClass(cell, colClass[5]);
        } else if (dom.hasClass(cell, colClass[5])) {
          dom.removeClass(cell, colClass[5]);
          dom.addClass(cell, colClass[4]);
        } else if (dom.hasClass(cell, colClass[4])) {
          dom.removeClass(cell, colClass[4]);
          dom.addClass(cell, colClass[3]);
        } else if (dom.hasClass(cell, colClass[3])) {
          dom.removeClass(cell, colClass[3]);
          dom.addClass(cell, colClass[2]);
        } else if (dom.hasClass(cell, colClass[2])) {
          dom.removeClass(cell, colClass[2]);
          dom.addClass(cell, colClass[1]);
        } else if (dom.hasClass(cell, colClass[1])) {
          dom.removeClass(cell, colClass[1]);
          dom.addClass(cell, colClass[0]);
        }

        if (dom.hasClass(cellnext, colClass[0])) {
          dom.removeClass(cellnext, colClass[0]);
          dom.addClass(cellnext, colClass[1]);
        } else if (dom.hasClass(cellnext, colClass[1])) {
          dom.removeClass(cellnext, colClass[1]);
          dom.addClass(cellnext, colClass[2]);
        } else if (dom.hasClass(cellnext, colClass[2])) {
          dom.removeClass(cellnext, colClass[2]);
          dom.addClass(cellnext, colClass[3]);
        } else if (dom.hasClass(cellnext, colClass[3])) {
          dom.removeClass(cellnext, colClass[3]);
          dom.addClass(cellnext, colClass[4]);
        } else if (dom.hasClass(cellnext, colClass[4])) {
          dom.removeClass(cellnext, colClass[4]);
          dom.addClass(cellnext, colClass[5]);
        } else if (dom.hasClass(cellnext, colClass[5])) {
          dom.removeClass(cellnext, colClass[5]);
          dom.addClass(cellnext, colClass[6]);
        } else if (dom.hasClass(cellnext, colClass[6])) {
          dom.removeClass(cellnext, colClass[6]);
          dom.addClass(cellnext, colClass[7]);
        } else if (dom.hasClass(cellnext, colClass[7])) {
          dom.removeClass(cellnext, colClass[7]);
          dom.addClass(cellnext, colClass[8]);
        } else if (dom.hasClass(cellnext, colClass[8])) {
          dom.removeClass(cellnext, colClass[8]);
          dom.addClass(cellnext, colClass[9]);
        } else if (dom.hasClass(cellnext, colClass[9])) {
          dom.removeClass(cellnext, colClass[9]);
          dom.addClass(cellnext, colClass[10]);
        } else if (dom.hasClass(cellnext, colClass[10])) {
          dom.removeClass(cellnext, colClass[10]);
          dom.addClass(cellnext, colClass[11]);
        }

        this.builder.opts.onChange();
      }
    }
  } // ROW


  removeRow() {
    // let builder = this.builder;
    let util = this.util;
    let dom = this.dom; // const cell = util.cellSelected();
    // if(!cell) return;
    // const row = cell.parentNode;

    let row;
    let cell = util.cellSelected();

    if (cell) {
      row = cell.parentNode;
    } else {
      row = util.rowSelected();
    }

    if (!row) return; //Change to row selection

    dom.removeClass(row, 'row-outline');
    util.confirm(util.out('Are you sure you want to delete this block?'), ok => {
      // const cell = util.cellSelected();
      // if(!cell) return;
      // const row = cell.parentNode; 
      let row;
      let cell = util.cellSelected();

      if (cell) {
        row = cell.parentNode;
      } else {
        row = util.rowSelected();
      }

      if (!row) return;

      if (ok) {
        this.builder.uo.saveForUndo();
        row.parentNode.removeChild(row);
        util.checkEmpty();
        this.builder.opts.onChange();
      }
    });
  }

  moveRowUp() {
    // let builder = this.builder;
    let util = this.util;
    let dom = this.dom; // const cell = util.cellSelected();
    // if(!cell) return;
    // let row = cell.parentNode;

    let row;
    let cell = util.cellSelected();

    if (cell) {
      row = cell.parentNode;
    } else {
      row = util.rowSelected();
    }

    if (!row) return; //Change to row selection

    dom.removeClass(row, 'row-outline');

    if (row.previousElementSibling) {
      this.builder.uo.saveForUndo();
      row.parentNode.insertBefore(row, row.previousElementSibling);
      this.rowTool.position(row);
      this.builder.opts.onChange();
    } else {
      // Move to previous container
      let currContainer = row.parentNode;
      let prev = null;
      const builders = document.querySelectorAll(this.builder.opts.container);
      Array.prototype.forEach.call(builders, builder => {
        if (builder.innerHTML === currContainer.innerHTML) {
          if (prev) {
            dom.moveAfter(row, prev.lastChild);
            this.rowTool.position(row);
            util.checkEmpty();
            this.builder.opts.onChange();
            return false;
          }
        }

        prev = builder;
      });
    }
  }

  moveRowDown() {
    // let builder = this.builder;
    let util = this.util;
    let dom = this.dom; // const cell = util.cellSelected();
    // if(!cell) return;
    // let row = cell.parentNode;

    let row;
    let cell = util.cellSelected();

    if (cell) {
      row = cell.parentNode;
    } else {
      row = util.rowSelected();
    }

    if (!row) return; //Change to row selection

    dom.removeClass(row, 'row-outline');

    if (row.nextElementSibling) {
      this.builder.uo.saveForUndo();
      row.parentNode.insertBefore(row.nextElementSibling, row);
      this.rowTool.position(row);
      this.builder.opts.onChange();
    } else {
      // Move to next container
      let currContainer = row.parentNode;
      let flag = false;
      const builders = document.querySelectorAll(this.builder.opts.container);
      Array.prototype.forEach.call(builders, builder => {
        if (flag) {
          builder.insertBefore(row, builder.firstChild);
          this.rowTool.position(row);
          util.checkEmpty();
          this.builder.opts.onChange();
          return false;
        }

        if (builder.innerHTML === currContainer.innerHTML) {
          flag = true;
        }
      });
    }
  }

  duplicateRow() {
    let builder = this.builder;
    let util = this.util;
    let dom = this.dom; // const cell = util.cellSelected();
    // if(!cell) return;
    // let row = cell.parentNode;

    let row;
    let cell = util.cellSelected();

    if (cell) {
      row = cell.parentNode;
    } else {
      row = util.rowSelected();
    }

    if (!row) return; //Change to row selection

    dom.removeClass(row, 'row-outline');
    this.builder.uo.saveForUndo(); //Clone row & cleanup attached tool & event

    const rowElement = row.cloneNode(true);
    rowElement.removeChild(rowElement.querySelector('.is-row-tool'));
    rowElement.removeChild(rowElement.querySelector('.is-col-tool'));
    let cols = dom.elementChildren(rowElement);
    cols.forEach(col => {
      col.removeAttribute('data-click');

      if (col.classList.contains('cell-active')) {
        builder.activeCol = col;
      }
    });
    dom.moveAfter(rowElement, row); //Unselect current row

    dom.removeClass(row, 'row-active');
    dom.removeClass(row, 'row-outline');
    cols = dom.elementChildren(row);
    cols.forEach(col => {
      dom.removeClass(col, 'cell-active');
    }); //re-add tool

    let builderActive = this.builder.doc.querySelector('.builder-active');
    builder.applyBehaviorOn(builderActive);
    this.rowTool.position(rowElement);
    this.builder.opts.onChange();
  }

}

class RowTool$1 {
  constructor(builder) {
    this.builder = builder;
    const dom = this.builder.dom;
    this.dom = dom;
  }

  position(row) {
    let dom = this.dom;
    const builderStuff = this.builder.builderStuff;
    let rowTool = row.querySelector('.is-row-tool');
    let rowMore = builderStuff.querySelector('.rowmore');
    dom.addClass(rowMore, 'transition1');
    const elm = rowTool.querySelector('.row-more');
    let top, left;

    if (!this.builder.iframe) {
      top = elm.getBoundingClientRect().top + window.pageYOffset;
      left = elm.getBoundingClientRect().left + window.pageXOffset;
    } else {
      let adjY = this.builder.iframe.getBoundingClientRect().top + window.pageYOffset;
      let adjX = this.builder.iframe.getBoundingClientRect().left + window.pageXOffset;
      top = elm.getBoundingClientRect().top;
      left = elm.getBoundingClientRect().left;
      top = top + adjY;
      left = left + adjX;
    } // const w = rowMore.offsetWidth; 


    rowMore.style.top = top - 8 + 'px';
    dom.removeClass(rowMore, 'arrow-bottom');
    dom.removeClass(rowMore, 'arrow-left');
    dom.removeClass(rowMore, 'arrow-right');
    dom.removeClass(rowMore, 'center');
    dom.removeClass(rowMore, 'right');
    dom.removeClass(rowMore, 'left');

    if (this.builder.opts.rowTool === 'right') {
      rowMore.style.left = left - rowMore.offsetWidth - 10 + 'px';
      dom.addClass(rowMore, 'arrow-right');
      dom.addClass(rowMore, 'left');
    } else {
      rowMore.style.left = left + 35 + 'px';
      dom.addClass(rowMore, 'arrow-left');
      dom.addClass(rowMore, 'left');
    }

    setTimeout(() => {
      dom.removeClass(rowMore, 'transition1');
    }, 300);
  }

}

/*
Draggable
Insipred by: https://www.kirupa.com/html5/drag.htm
*/
let initialX, initialY, currentX, currentY, xOffset, yOffset, dragActive, activeDraggableBox;
class Draggable$1 {
  constructor(opts = {}) {
    this.opts = opts;
    const elms = document.querySelectorAll(this.opts.selector);
    Array.prototype.forEach.call(elms, elm => {
      elm.setAttribute('draggable', '');
      elm.addEventListener('touchstart', this.dragStart, false);
      elm.addEventListener('touchend', this.dragEnd, false);
      elm.addEventListener('mousedown', this.dragStart, false);
      elm.addEventListener('mouseup', this.dragEnd, false);
    });
    document.addEventListener('touchmove', this.drag, false);
    document.addEventListener('mousemove', this.drag, false); // if(this.isTouchSupport) {
    //     window.addEventListener('touchmove', (e)=>{
    //         e.returnValue = true;
    //         if (dragActive) {
    //             e.preventDefault();
    //         }
    //     },
    //         {
    //             passive: dragActive
    //         }
    //     ); 
    // }
  }

  dragStart(e) {
    if (!e.target.hasAttribute('draggable')) return; //any child element (ex. close button) should not be draggable. LATER: revew.

    e.target.parentNode.style.transition = 'none';
    dragActive = true;
    activeDraggableBox = e.target.parentElement;
    var xOffset;
    var yOffset;

    if (!activeDraggableBox.getAttribute('data-xOffset')) {
      activeDraggableBox.setAttribute('data-xOffset', 0);
      xOffset = 0;
    } else {
      xOffset = activeDraggableBox.getAttribute('data-xOffset');
    }

    if (!activeDraggableBox.getAttribute('data-yOffset')) {
      activeDraggableBox.setAttribute('data-yOffset', 0);
      yOffset = 0;
    } else {
      yOffset = activeDraggableBox.getAttribute('data-yOffset');
    }

    if (e.type === 'touchstart') {
      initialX = e.touches[0].clientX - xOffset;
      initialY = e.touches[0].clientY - yOffset;
    } else {
      initialX = e.clientX - xOffset;
      initialY = e.clientY - yOffset;
    }

    activeDraggableBox.setAttribute('data-initialX', initialX);
    activeDraggableBox.setAttribute('data-initialY', initialY);
  }

  dragEnd(e) {
    if (!e.target.hasAttribute('draggable')) return; //any child element (ex. close button) should not be draggable. LATER: revew.

    e.target.parentNode.style.transition = ''; //Update

    currentX = activeDraggableBox.getAttribute('data-currentX');
    currentY = activeDraggableBox.getAttribute('data-currentY');
    initialX = currentX;
    initialY = currentY;
    activeDraggableBox.setAttribute('data-initialX', initialX);
    activeDraggableBox.setAttribute('data-initialY', initialY);
    dragActive = false;
  }

  drag(e) {
    if (dragActive) {
      e.preventDefault();
      var initialX = activeDraggableBox.getAttribute('data-initialX');
      var initialY = activeDraggableBox.getAttribute('data-initialY');

      if (e.type === 'touchmove') {
        currentX = e.touches[0].clientX - initialX;
        currentY = e.touches[0].clientY - initialY;
      } else {
        currentX = e.clientX - initialX;
        currentY = e.clientY - initialY;
      }

      activeDraggableBox.style.transform = 'translate3d(' + currentX + 'px, ' + currentY + 'px, 0)'; //Save

      activeDraggableBox.setAttribute('data-currentX', currentX);
      activeDraggableBox.setAttribute('data-currentY', currentY);
      xOffset = currentX;
      yOffset = currentY;
      activeDraggableBox.setAttribute('data-xOffset', xOffset);
      activeDraggableBox.setAttribute('data-yOffset', yOffset);
    }
  }

  isTouchSupport() {
    if ('ontouchstart' in window || navigator.MaxTouchPoints > 0 || navigator.msMaxTouchPoints > 0) {
      return true;
    } else {
      return false;
    }
  }

}

const renderGridEditor = builder => {
  const util = builder.util;
  const builderStuff = builder.builderStuff;
  const grid = new Grid(builder);
  const htmlutil = new HtmlUtil(builder);
  const dom = builder.dom;
  let rowhtmlbutton = '';
  if (builder.opts.rowHtmlEditor) rowhtmlbutton = `<button tabindex="-1" title="${util.out('HTML')}" class="row-html">
            <svg class="is-icon-flex" style="margin-right:-3px;width:12px;height:12px;"><use xlink:href="#ion-ios-arrow-left"></use></svg><svg class="is-icon-flex" style="margin-left:-2px;width:12px;height:12px;"><use xlink:href="#ion-ios-arrow-right"></use></svg>
        </button>`;
  let colhtmlbutton = '';
  if (builder.opts.columnHtmlEditor) colhtmlbutton = `<button tabindex="-1" title="${util.out('HTML')}" class="cell-html">
            <svg class="is-icon-flex" style="margin-right:-3px;width:12px;height:12px;"><use xlink:href="#ion-ios-arrow-left"></use></svg><svg class="is-icon-flex" style="margin-left:-2px;width:12px;height:12px;"><use xlink:href="#ion-ios-arrow-right"></use></svg>
        </button>`;
  const html = `<div class="is-modal is-modal-content grideditor" tabindex="-1" role="dialog" aria-modal="true" aria-hidden="true">
        <div class="is-modal-bar is-draggable">
            <button class="is-modal-close" tabindex="-1" title="${util.out('Close')}">&#10005;</button>
        </div>
        <div style="padding:13px 0 5px 18px;font-size:10px;text-transform:uppercase;letter-spacing:1px;">${util.out('Row')}</div>
        <div style="display:flex;flex-flow:wrap;">
            <button tabindex="-1" title="${util.out('Add')}" class="row-add"><svg class="is-icon-flex" style="width:19px;height:19px;"><use xlink:href="#ion-ios-plus-empty"></use></svg></button>
            <button tabindex="-1" title="${util.out('Duplicate')}" class="row-duplicate" style="display: block;"><svg class="is-icon-flex" style="width:12px;height:12px;"><use xlink:href="#ion-ios-photos-outline"></use></svg></button>
            <button tabindex="-1" title="${util.out('Move Up')}" class="row-up" style="display: block;"><svg class="is-icon-flex" style="width:15px;height:15px;"><use xlink:href="#ion-ios-arrow-thin-up"></use></svg></button>
            <button tabindex="-1" title="${util.out('Move Down')}" class="row-down" style="display: block;"><svg class="is-icon-flex" style="width:15px;height:15px;"><use xlink:href="#ion-ios-arrow-thin-down"></use></svg></button>
            ${rowhtmlbutton}
            <button tabindex="-1" title="${util.out('Delete')}" class="row-remove"><svg class="is-icon-flex" style="width:20px;height:20px;"><use xlink:href="#ion-ios-close-empty"></use></svg></button>
        </div>
        <div style="padding:8px 0 5px 18px;font-size:11px;text-transform:uppercase;letter-spacing:1px;">${util.out('Column')}</div>
        <div style="display:flex;flex-flow:wrap;">
            <button tabindex="-1" title="${util.out('Add')}" class="cell-add"><svg class="is-icon-flex" style="width:19px;height:19px;"><use xlink:href="#ion-ios-plus-empty"></use></svg></button>
            <button tabindex="-1" title="${util.out('Duplicate')}" class="cell-duplicate"><svg class="is-icon-flex" style="width:12px;height:12px;"><use xlink:href="#ion-ios-photos-outline"></use></svg></button>
            <button tabindex="-1" title="${util.out('Move Up')}" class="cell-up"><svg class="is-icon-flex" style="width:15px;height:15px;"><use xlink:href="#ion-ios-arrow-thin-up"></use></svg></button>
            <button tabindex="-1" title="${util.out('Move Down')}" class="cell-down"><svg class="is-icon-flex" style="width:15px;height:15px;"><use xlink:href="#ion-ios-arrow-thin-down"></use></svg></button>
            <button tabindex="-1" title="${util.out('Move Left')}" class="cell-prev"><svg class="is-icon-flex" style="width:15px;height:15px;"><use xlink:href="#ion-ios-arrow-thin-left"></use></svg></button>
            <button tabindex="-1" title="${util.out('Move Right')}" class="cell-next"><svg class="is-icon-flex" style="width:15px;height:15px;"><use xlink:href="#ion-ios-arrow-thin-right"></use></svg></button>
            <button tabindex="-1" title="${util.out('Increase')}" class="cell-increase"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#icon-increase"></use></svg></button>
            <button tabindex="-1" title="${util.out('Decrease')}" class="cell-decrease"><svg class="is-icon-flex" style="width:13px;height:13px;"><use xlink:href="#icon-decrease"></use></svg></button>
            ${colhtmlbutton}
            <button tabindex="-1" title="${util.out('Delete')}" class="cell-remove"><svg class="is-icon-flex" style="width:20px;height:20px;"><use xlink:href="#ion-ios-close-empty"></use></svg></button>
            <button tabindex="-1" title="${util.out('Lock')}" class="cell-locking"><svg class="is-icon-flex" style="width:12px;height:12px;"><use xlink:href="#icon-lock"></use></svg></button>
            <button tabindex="-1" title="${util.out('Column Settings')}" class="cell-settings"><svg class="is-icon-flex" style="width:12px;height:12px;"><use xlink:href="#ion-ios-gear"></use></svg></button>
               
            <div class="is-separator">
                <button tabindex="-1" title="${util.out('Outline')}" class="grid-outline"><svg class="is-icon-flex" style="width:12px;height:12px;"><use xlink:href="#ion-ios-grid-view-outline"></use></svg></button>
                <!--<button tabindex="-1" title="${util.out('Element Tool')}" class="cell-elmtool"><svg class="is-icon-flex" style="width:12px;height:12px;"><use xlink:href="#ion-ios-gear"></use></svg></button>-->
            </div>
        </div>
    </div>`; //LATER:
  //<button title="${util.out('Element Tool')}" class="cell-elmtool"><svg class="is-icon-flex" style="width:12px;height:12px;"><use xlink:href="#ion-ios-gear"></use></svg></button>

  dom.appendHtml(builderStuff, html);
  new Draggable$1({
    selector: '.is-draggable'
  });
  const grideditor = document.querySelector('.grideditor');
  document.addEventListener('click', e => {
    e = e || window.event;
    var target = e.target || e.srcElement;

    if (dom.hasClass(grideditor, 'active')) {
      let a = dom.parentsHasClass(target, 'is-builder');
      let b = dom.parentsHasClass(target, 'grideditor');
      let c = dom.parentsHasClass(target, 'is-modal');
      let d = dom.parentsHasClass(target, 'is-pop');
      let f = dom.parentsHasClass(target, 'rte-grideditor') || dom.hasClass(target, 'rte-grideditor');

      if (a || b || c || d || f) {
        grideditor.style.display = '';
        return;
      } else {
        grideditor.style.display = 'none';
      }
    }
  }, false);
  let elm = grideditor.querySelector('.is-modal-close');
  dom.addEventListener(elm, 'click', () => {
    dom.removeClass(grideditor, 'active');
    const builders = builder.doc.querySelectorAll(builder.opts.container);
    Array.prototype.forEach.call(builders, builder => {
      builder.removeAttribute('grideditor');
    });
  });
  const btnGridOutline = grideditor.querySelector('.grid-outline');
  dom.addEventListener(btnGridOutline, 'click', () => {
    const builders = builder.doc.querySelectorAll(builder.opts.container);
    Array.prototype.forEach.call(builders, builder => {
      if (builder.hasAttribute('gridoutline')) {
        builder.removeAttribute('gridoutline');
        dom.removeClass(btnGridOutline, 'on');
      } else {
        builder.setAttribute('gridoutline', '');
        dom.addClass(btnGridOutline, 'on');
      }
    });
  });
  const quickadd = renderQuickAdd(builder); // CELL

  const btnAdd = grideditor.querySelector('.cell-add');
  dom.addEventListener(btnAdd, 'click', () => {
    let tabs = quickadd.querySelector('.is-pop-tabs');
    tabs.style.display = 'flex';
    const top = btnAdd.getBoundingClientRect().top + window.pageYOffset;
    const left = btnAdd.getBoundingClientRect().left + window.pageXOffset; // quickadd.style.display = 'flex';

    util.showPop(quickadd, false, btnAdd);
    const w = quickadd.offsetWidth; //to get value, element must not hidden (display:none). So set display:flex before this.
    //const h = quickadd.offsetHeight;

    quickadd.style.top = top + 'px';
    quickadd.style.left = left - w - 8 + 'px';
    dom.removeClass(quickadd, 'arrow-bottom');
    dom.removeClass(quickadd, 'arrow-left');
    dom.removeClass(quickadd, 'arrow-top');
    dom.removeClass(quickadd, 'center');
    dom.removeClass(quickadd, 'left');
    dom.addClass(quickadd, 'arrow-right');
    dom.addClass(quickadd, 'right');
    let val = quickadd.querySelector('.active').getAttribute('data-value');

    if (val === 'left') {
      quickadd.setAttribute('data-mode', 'cell-left');
    } else {
      quickadd.setAttribute('data-mode', 'cell-right');
    }
  });
  elm = grideditor.querySelector('.cell-prev');
  dom.addEventListener(elm, 'click', () => {
    grid.moveColumnPrevious();
    util.clearControls();
  });
  elm = grideditor.querySelector('.cell-next');
  dom.addEventListener(elm, 'click', () => {
    grid.moveColumnNext();
    util.clearControls();
  });
  elm = grideditor.querySelector('.cell-increase');
  dom.addEventListener(elm, 'click', () => {
    grid.increaseColumn();
    util.clearControls();
  });
  elm = grideditor.querySelector('.cell-decrease');
  dom.addEventListener(elm, 'click', () => {
    grid.decreaseColumn();
    util.clearControls();
  });
  elm = grideditor.querySelector('.cell-up');
  dom.addEventListener(elm, 'click', () => {
    grid.moveColumnUp();
    util.clearControls();
  });
  elm = grideditor.querySelector('.cell-down');
  dom.addEventListener(elm, 'click', () => {
    grid.moveColumnDown();
    util.clearControls();
  });
  elm = grideditor.querySelector('.cell-duplicate');
  dom.addEventListener(elm, 'click', () => {
    grid.duplicateColumn();
    util.clearControls();
  });
  elm = grideditor.querySelector('.cell-remove');
  dom.addEventListener(elm, 'click', () => {
    grid.removeColumn();
    util.clearControls();
  });
  const btnCellHtml = grideditor.querySelector('.cell-html');
  if (btnCellHtml) dom.addEventListener(btnCellHtml, 'click', () => {
    const cell = util.cellSelected();
    if (!cell) return;
    htmlutil.view('cell', false, btnCellHtml);
  });
  const btnCellSettings = grideditor.querySelector('.cell-settings');
  if (btnCellSettings) dom.addEventListener(btnCellSettings, 'click', () => {
    const cell = util.cellSelected();
    if (!cell) return;
    builder.colTool.readCellStyles(cell); //save selection

    util.saveSelection();
    const cellSettings = document.querySelector('.is-modal.columnsettings');
    util.showModal(cellSettings, false, () => {
      let rtetool = builder.builderStuff.querySelector('.is-rte-tool');

      if (rtetool.style.display === 'flex') {
        // means text selection
        util.restoreSelection(); // if(this.builder.activeElement) {
        //     dom.addClass(this.builder.activeElement, 'elm-active');
        //     this.builder.elmTool.repositionElementTool();
        // }

        btnCellSettings.removeAttribute('data-focus');
        btnCellSettings.focus();
      }
    });
    btnCellSettings.setAttribute('data-focus', true);
  }); // Lock/Unlock Column

  const btnCellLocking = grideditor.querySelector('.cell-locking');
  if (btnCellLocking) dom.addEventListener(btnCellLocking, 'click', e => {
    let cell = util.cellSelected();
    if (!cell) return;

    if (!cell.hasAttribute('data-noedit')) {
      cell.setAttribute('data-noedit', '');
      cell.contentEditable = false;
      dom.addClass(btnCellLocking, 'on');
      util.clearActiveElement(true);
    } else {
      cell.removeAttribute('data-noedit');
      cell.contentEditable = true;
      dom.removeClass(btnCellLocking, 'on');
    }

    builder.colTool.showHideLockIndicator(cell);
    builder.element.applyBehavior(cell);
    e.preventDefault();
  }); // ROW

  const btnRowAdd = grideditor.querySelector('.row-add');
  dom.addEventListener(btnRowAdd, 'click', () => {
    let tabs = quickadd.querySelector('.is-pop-tabs');
    tabs.style.display = 'none';
    const top = btnRowAdd.getBoundingClientRect().top + window.pageYOffset;
    const left = btnRowAdd.getBoundingClientRect().left + window.pageXOffset; // quickadd.style.display = 'flex';

    util.showPop(quickadd, false, btnRowAdd);
    const w = quickadd.offsetWidth; //to get value, element must not hidden (display:none). So set display:flex before this.
    //const h = quickadd.offsetHeight;

    quickadd.style.top = top + 'px';
    quickadd.style.left = left - w - 8 + 'px';
    dom.removeClass(quickadd, 'arrow-bottom');
    dom.removeClass(quickadd, 'arrow-left');
    dom.removeClass(quickadd, 'arrow-top');
    dom.removeClass(quickadd, 'center');
    dom.removeClass(quickadd, 'left');
    dom.addClass(quickadd, 'arrow-right');
    dom.addClass(quickadd, 'right');
    quickadd.setAttribute('data-mode', 'row');
  });
  elm = grideditor.querySelector('.row-up');
  dom.addEventListener(elm, 'click', () => {
    grid.moveRowUp();
    util.clearControls();
  });
  elm = grideditor.querySelector('.row-down');
  dom.addEventListener(elm, 'click', () => {
    grid.moveRowDown();
    util.clearControls();
  });
  elm = grideditor.querySelector('.row-duplicate');
  dom.addEventListener(elm, 'click', () => {
    grid.duplicateRow();
    util.clearControls();
  });
  elm = grideditor.querySelector('.row-remove');
  dom.addEventListener(elm, 'click', () => {
    grid.removeRow();
    util.clearControls();
  });
  const btnRowHtml = grideditor.querySelector('.row-html');
  if (btnRowHtml) dom.addEventListener(btnRowHtml, 'click', () => {
    const cell = util.cellSelected();
    if (!cell) return;
    htmlutil.view('row', false, btnRowHtml);
  });
};

/**!
 * Sortable 1.14.0
 * @author	RubaXa   <trash@rubaxa.org>
 * @author	owenm    <owen23355@gmail.com>
 * @license MIT
 */
function ownKeys$1(object, enumerableOnly) {
  var keys = Object.keys(object);

  if (Object.getOwnPropertySymbols) {
    var symbols = Object.getOwnPropertySymbols(object);

    if (enumerableOnly) {
      symbols = symbols.filter(function (sym) {
        return Object.getOwnPropertyDescriptor(object, sym).enumerable;
      });
    }

    keys.push.apply(keys, symbols);
  }

  return keys;
}

function _objectSpread2$1(target) {
  for (var i = 1; i < arguments.length; i++) {
    var source = arguments[i] != null ? arguments[i] : {};

    if (i % 2) {
      ownKeys$1(Object(source), true).forEach(function (key) {
        _defineProperty$1(target, key, source[key]);
      });
    } else if (Object.getOwnPropertyDescriptors) {
      Object.defineProperties(target, Object.getOwnPropertyDescriptors(source));
    } else {
      ownKeys$1(Object(source)).forEach(function (key) {
        Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key));
      });
    }
  }

  return target;
}

function _typeof$1(obj) {
  "@babel/helpers - typeof";

  if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
    _typeof$1 = function (obj) {
      return typeof obj;
    };
  } else {
    _typeof$1 = function (obj) {
      return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
    };
  }

  return _typeof$1(obj);
}

function _defineProperty$1(obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }

  return obj;
}

function _extends() {
  _extends = Object.assign || function (target) {
    for (var i = 1; i < arguments.length; i++) {
      var source = arguments[i];

      for (var key in source) {
        if (Object.prototype.hasOwnProperty.call(source, key)) {
          target[key] = source[key];
        }
      }
    }

    return target;
  };

  return _extends.apply(this, arguments);
}

function _objectWithoutPropertiesLoose(source, excluded) {
  if (source == null) return {};
  var target = {};
  var sourceKeys = Object.keys(source);
  var key, i;

  for (i = 0; i < sourceKeys.length; i++) {
    key = sourceKeys[i];
    if (excluded.indexOf(key) >= 0) continue;
    target[key] = source[key];
  }

  return target;
}

function _objectWithoutProperties(source, excluded) {
  if (source == null) return {};

  var target = _objectWithoutPropertiesLoose(source, excluded);

  var key, i;

  if (Object.getOwnPropertySymbols) {
    var sourceSymbolKeys = Object.getOwnPropertySymbols(source);

    for (i = 0; i < sourceSymbolKeys.length; i++) {
      key = sourceSymbolKeys[i];
      if (excluded.indexOf(key) >= 0) continue;
      if (!Object.prototype.propertyIsEnumerable.call(source, key)) continue;
      target[key] = source[key];
    }
  }

  return target;
}

var version$1 = "1.14.0";

function userAgent(pattern) {
  if (typeof window !== 'undefined' && window.navigator) {
    return !! /*@__PURE__*/navigator.userAgent.match(pattern);
  }
}

var IE11OrLess = userAgent(/(?:Trident.*rv[ :]?11\.|msie|iemobile|Windows Phone)/i);
var Edge = userAgent(/Edge/i);
var FireFox = userAgent(/firefox/i);
var Safari = userAgent(/safari/i) && !userAgent(/chrome/i) && !userAgent(/android/i);
var IOS = userAgent(/iP(ad|od|hone)/i);
var ChromeForAndroid = userAgent(/chrome/i) && userAgent(/android/i);

var captureMode = {
  capture: false,
  passive: false
};

function on(el, event, fn) {
  el.addEventListener(event, fn, !IE11OrLess && captureMode);
}

function off(el, event, fn) {
  el.removeEventListener(event, fn, !IE11OrLess && captureMode);
}

function matches(
/**HTMLElement*/
el,
/**String*/
selector) {
  if (!selector) return;
  selector[0] === '>' && (selector = selector.substring(1));

  if (el) {
    try {
      if (el.matches) {
        return el.matches(selector);
      } else if (el.msMatchesSelector) {
        return el.msMatchesSelector(selector);
      } else if (el.webkitMatchesSelector) {
        return el.webkitMatchesSelector(selector);
      }
    } catch (_) {
      return false;
    }
  }

  return false;
}

function getParentOrHost(el) {
  return el.host && el !== document && el.host.nodeType ? el.host : el.parentNode;
}

function closest(
/**HTMLElement*/
el,
/**String*/
selector,
/**HTMLElement*/
ctx, includeCTX) {
  if (el) {
    ctx = ctx || document;

    do {
      if (selector != null && (selector[0] === '>' ? el.parentNode === ctx && matches(el, selector) : matches(el, selector)) || includeCTX && el === ctx) {
        return el;
      }

      if (el === ctx) break;
      /* jshint boss:true */
    } while (el = getParentOrHost(el));
  }

  return null;
}

var R_SPACE = /\s+/g;

function toggleClass$1(el, name, state) {
  if (el && name) {
    if (el.classList) {
      el.classList[state ? 'add' : 'remove'](name);
    } else {
      var className = (' ' + el.className + ' ').replace(R_SPACE, ' ').replace(' ' + name + ' ', ' ');
      el.className = (className + (state ? ' ' + name : '')).replace(R_SPACE, ' ');
    }
  }
}

function css(el, prop, val) {
  var style = el && el.style;

  if (style) {
    if (val === void 0) {
      if (document.defaultView && document.defaultView.getComputedStyle) {
        val = document.defaultView.getComputedStyle(el, '');
      } else if (el.currentStyle) {
        val = el.currentStyle;
      }

      return prop === void 0 ? val : val[prop];
    } else {
      if (!(prop in style) && prop.indexOf('webkit') === -1) {
        prop = '-webkit-' + prop;
      }

      style[prop] = val + (typeof val === 'string' ? '' : 'px');
    }
  }
}

function matrix(el, selfOnly) {
  var appliedTransforms = '';

  if (typeof el === 'string') {
    appliedTransforms = el;
  } else {
    do {
      var transform = css(el, 'transform');

      if (transform && transform !== 'none') {
        appliedTransforms = transform + ' ' + appliedTransforms;
      }
      /* jshint boss:true */

    } while (!selfOnly && (el = el.parentNode));
  }

  var matrixFn = window.DOMMatrix || window.WebKitCSSMatrix || window.CSSMatrix || window.MSCSSMatrix;
  /*jshint -W056 */

  return matrixFn && new matrixFn(appliedTransforms);
}

function find$2(ctx, tagName, iterator) {
  if (ctx) {
    var list = ctx.getElementsByTagName(tagName),
        i = 0,
        n = list.length;

    if (iterator) {
      for (; i < n; i++) {
        iterator(list[i], i);
      }
    }

    return list;
  }

  return [];
}

function getWindowScrollingElement() {
  var scrollingElement = document.scrollingElement;

  if (scrollingElement) {
    return scrollingElement;
  } else {
    return document.documentElement;
  }
}
/**
 * Returns the "bounding client rect" of given element
 * @param  {HTMLElement} el                       The element whose boundingClientRect is wanted
 * @param  {[Boolean]} relativeToContainingBlock  Whether the rect should be relative to the containing block of (including) the container
 * @param  {[Boolean]} relativeToNonStaticParent  Whether the rect should be relative to the relative parent of (including) the contaienr
 * @param  {[Boolean]} undoScale                  Whether the container's scale() should be undone
 * @param  {[HTMLElement]} container              The parent the element will be placed in
 * @return {Object}                               The boundingClientRect of el, with specified adjustments
 */


function getRect$1(el, relativeToContainingBlock, relativeToNonStaticParent, undoScale, container) {
  if (!el.getBoundingClientRect && el !== window) return;
  var elRect, top, left, bottom, right, height, width;

  if (el !== window && el.parentNode && el !== getWindowScrollingElement()) {
    elRect = el.getBoundingClientRect();
    top = elRect.top;
    left = elRect.left;
    bottom = elRect.bottom;
    right = elRect.right;
    height = elRect.height;
    width = elRect.width;
  } else {
    top = 0;
    left = 0;
    bottom = window.innerHeight;
    right = window.innerWidth;
    height = window.innerHeight;
    width = window.innerWidth;
  }

  if ((relativeToContainingBlock || relativeToNonStaticParent) && el !== window) {
    // Adjust for translate()
    container = container || el.parentNode; // solves #1123 (see: https://stackoverflow.com/a/37953806/6088312)
    // Not needed on <= IE11

    if (!IE11OrLess) {
      do {
        if (container && container.getBoundingClientRect && (css(container, 'transform') !== 'none' || relativeToNonStaticParent && css(container, 'position') !== 'static')) {
          var containerRect = container.getBoundingClientRect(); // Set relative to edges of padding box of container

          top -= containerRect.top + parseInt(css(container, 'border-top-width'));
          left -= containerRect.left + parseInt(css(container, 'border-left-width'));
          bottom = top + elRect.height;
          right = left + elRect.width;
          break;
        }
        /* jshint boss:true */

      } while (container = container.parentNode);
    }
  }

  if (undoScale && el !== window) {
    // Adjust for scale()
    var elMatrix = matrix(container || el),
        scaleX = elMatrix && elMatrix.a,
        scaleY = elMatrix && elMatrix.d;

    if (elMatrix) {
      top /= scaleY;
      left /= scaleX;
      width /= scaleX;
      height /= scaleY;
      bottom = top + height;
      right = left + width;
    }
  }

  return {
    top: top,
    left: left,
    bottom: bottom,
    right: right,
    width: width,
    height: height
  };
}
/**
 * Checks if a side of an element is scrolled past a side of its parents
 * @param  {HTMLElement}  el           The element who's side being scrolled out of view is in question
 * @param  {String}       elSide       Side of the element in question ('top', 'left', 'right', 'bottom')
 * @param  {String}       parentSide   Side of the parent in question ('top', 'left', 'right', 'bottom')
 * @return {HTMLElement}               The parent scroll element that the el's side is scrolled past, or null if there is no such element
 */


function isScrolledPast(el, elSide, parentSide) {
  var parent = getParentAutoScrollElement(el, true),
      elSideVal = getRect$1(el)[elSide];
  /* jshint boss:true */

  while (parent) {
    var parentSideVal = getRect$1(parent)[parentSide],
        visible = void 0;

    if (parentSide === 'top' || parentSide === 'left') {
      visible = elSideVal >= parentSideVal;
    } else {
      visible = elSideVal <= parentSideVal;
    }

    if (!visible) return parent;
    if (parent === getWindowScrollingElement()) break;
    parent = getParentAutoScrollElement(parent, false);
  }

  return false;
}
/**
 * Gets nth child of el, ignoring hidden children, sortable's elements (does not ignore clone if it's visible)
 * and non-draggable elements
 * @param  {HTMLElement} el       The parent element
 * @param  {Number} childNum      The index of the child
 * @param  {Object} options       Parent Sortable's options
 * @return {HTMLElement}          The child at index childNum, or null if not found
 */


function getChild(el, childNum, options, includeDragEl) {
  var currentChild = 0,
      i = 0,
      children = el.children;

  while (i < children.length) {
    if (children[i].style.display !== 'none' && children[i] !== Sortable.ghost && (includeDragEl || children[i] !== Sortable.dragged) && closest(children[i], options.draggable, el, false)) {
      if (currentChild === childNum) {
        return children[i];
      }

      currentChild++;
    }

    i++;
  }

  return null;
}
/**
 * Gets the last child in the el, ignoring ghostEl or invisible elements (clones)
 * @param  {HTMLElement} el       Parent element
 * @param  {selector} selector    Any other elements that should be ignored
 * @return {HTMLElement}          The last child, ignoring ghostEl
 */


function lastChild(el, selector) {
  var last = el.lastElementChild;

  while (last && (last === Sortable.ghost || css(last, 'display') === 'none' || selector && !matches(last, selector))) {
    last = last.previousElementSibling;
  }

  return last || null;
}
/**
 * Returns the index of an element within its parent for a selected set of
 * elements
 * @param  {HTMLElement} el
 * @param  {selector} selector
 * @return {number}
 */


function index(el, selector) {
  var index = 0;

  if (!el || !el.parentNode) {
    return -1;
  }
  /* jshint boss:true */


  while (el = el.previousElementSibling) {
    if (el.nodeName.toUpperCase() !== 'TEMPLATE' && el !== Sortable.clone && (!selector || matches(el, selector))) {
      index++;
    }
  }

  return index;
}
/**
 * Returns the scroll offset of the given element, added with all the scroll offsets of parent elements.
 * The value is returned in real pixels.
 * @param  {HTMLElement} el
 * @return {Array}             Offsets in the format of [left, top]
 */


function getRelativeScrollOffset(el) {
  var offsetLeft = 0,
      offsetTop = 0,
      winScroller = getWindowScrollingElement();

  if (el) {
    do {
      var elMatrix = matrix(el),
          scaleX = elMatrix.a,
          scaleY = elMatrix.d;
      offsetLeft += el.scrollLeft * scaleX;
      offsetTop += el.scrollTop * scaleY;
    } while (el !== winScroller && (el = el.parentNode));
  }

  return [offsetLeft, offsetTop];
}
/**
 * Returns the index of the object within the given array
 * @param  {Array} arr   Array that may or may not hold the object
 * @param  {Object} obj  An object that has a key-value pair unique to and identical to a key-value pair in the object you want to find
 * @return {Number}      The index of the object in the array, or -1
 */


function indexOfObject(arr, obj) {
  for (var i in arr) {
    if (!arr.hasOwnProperty(i)) continue;

    for (var key in obj) {
      if (obj.hasOwnProperty(key) && obj[key] === arr[i][key]) return Number(i);
    }
  }

  return -1;
}

function getParentAutoScrollElement(el, includeSelf) {
  // skip to window
  if (!el || !el.getBoundingClientRect) return getWindowScrollingElement();
  var elem = el;
  var gotSelf = false;

  do {
    // we don't need to get elem css if it isn't even overflowing in the first place (performance)
    if (elem.clientWidth < elem.scrollWidth || elem.clientHeight < elem.scrollHeight) {
      var elemCSS = css(elem);

      if (elem.clientWidth < elem.scrollWidth && (elemCSS.overflowX == 'auto' || elemCSS.overflowX == 'scroll') || elem.clientHeight < elem.scrollHeight && (elemCSS.overflowY == 'auto' || elemCSS.overflowY == 'scroll')) {
        if (!elem.getBoundingClientRect || elem === document.body) return getWindowScrollingElement();
        if (gotSelf || includeSelf) return elem;
        gotSelf = true;
      }
    }
    /* jshint boss:true */

  } while (elem = elem.parentNode);

  return getWindowScrollingElement();
}

function extend(dst, src) {
  if (dst && src) {
    for (var key in src) {
      if (src.hasOwnProperty(key)) {
        dst[key] = src[key];
      }
    }
  }

  return dst;
}

function isRectEqual(rect1, rect2) {
  return Math.round(rect1.top) === Math.round(rect2.top) && Math.round(rect1.left) === Math.round(rect2.left) && Math.round(rect1.height) === Math.round(rect2.height) && Math.round(rect1.width) === Math.round(rect2.width);
}

var _throttleTimeout;

function throttle$1(callback, ms) {
  return function () {
    if (!_throttleTimeout) {
      var args = arguments,
          _this = this;

      if (args.length === 1) {
        callback.call(_this, args[0]);
      } else {
        callback.apply(_this, args);
      }

      _throttleTimeout = setTimeout(function () {
        _throttleTimeout = void 0;
      }, ms);
    }
  };
}

function cancelThrottle() {
  clearTimeout(_throttleTimeout);
  _throttleTimeout = void 0;
}

function scrollBy(el, x, y) {
  el.scrollLeft += x;
  el.scrollTop += y;
}

function clone(el) {
  var Polymer = window.Polymer;
  var $ = window.jQuery || window.Zepto;

  if (Polymer && Polymer.dom) {
    return Polymer.dom(el).cloneNode(true);
  } else if ($) {
    return $(el).clone(true)[0];
  } else {
    return el.cloneNode(true);
  }
}

var expando = 'Sortable' + new Date().getTime();

function AnimationStateManager() {
  var animationStates = [],
      animationCallbackId;
  return {
    captureAnimationState: function captureAnimationState() {
      animationStates = [];
      if (!this.options.animation) return;
      var children = [].slice.call(this.el.children);
      children.forEach(function (child) {
        if (css(child, 'display') === 'none' || child === Sortable.ghost) return;
        animationStates.push({
          target: child,
          rect: getRect$1(child)
        });

        var fromRect = _objectSpread2$1({}, animationStates[animationStates.length - 1].rect); // If animating: compensate for current animation


        if (child.thisAnimationDuration) {
          var childMatrix = matrix(child, true);

          if (childMatrix) {
            fromRect.top -= childMatrix.f;
            fromRect.left -= childMatrix.e;
          }
        }

        child.fromRect = fromRect;
      });
    },
    addAnimationState: function addAnimationState(state) {
      animationStates.push(state);
    },
    removeAnimationState: function removeAnimationState(target) {
      animationStates.splice(indexOfObject(animationStates, {
        target: target
      }), 1);
    },
    animateAll: function animateAll(callback) {
      var _this = this;

      if (!this.options.animation) {
        clearTimeout(animationCallbackId);
        if (typeof callback === 'function') callback();
        return;
      }

      var animating = false,
          animationTime = 0;
      animationStates.forEach(function (state) {
        var time = 0,
            target = state.target,
            fromRect = target.fromRect,
            toRect = getRect$1(target),
            prevFromRect = target.prevFromRect,
            prevToRect = target.prevToRect,
            animatingRect = state.rect,
            targetMatrix = matrix(target, true);

        if (targetMatrix) {
          // Compensate for current animation
          toRect.top -= targetMatrix.f;
          toRect.left -= targetMatrix.e;
        }

        target.toRect = toRect;

        if (target.thisAnimationDuration) {
          // Could also check if animatingRect is between fromRect and toRect
          if (isRectEqual(prevFromRect, toRect) && !isRectEqual(fromRect, toRect) && // Make sure animatingRect is on line between toRect & fromRect
          (animatingRect.top - toRect.top) / (animatingRect.left - toRect.left) === (fromRect.top - toRect.top) / (fromRect.left - toRect.left)) {
            // If returning to same place as started from animation and on same axis
            time = calculateRealTime(animatingRect, prevFromRect, prevToRect, _this.options);
          }
        } // if fromRect != toRect: animate


        if (!isRectEqual(toRect, fromRect)) {
          target.prevFromRect = fromRect;
          target.prevToRect = toRect;

          if (!time) {
            time = _this.options.animation;
          }

          _this.animate(target, animatingRect, toRect, time);
        }

        if (time) {
          animating = true;
          animationTime = Math.max(animationTime, time);
          clearTimeout(target.animationResetTimer);
          target.animationResetTimer = setTimeout(function () {
            target.animationTime = 0;
            target.prevFromRect = null;
            target.fromRect = null;
            target.prevToRect = null;
            target.thisAnimationDuration = null;
          }, time);
          target.thisAnimationDuration = time;
        }
      });
      clearTimeout(animationCallbackId);

      if (!animating) {
        if (typeof callback === 'function') callback();
      } else {
        animationCallbackId = setTimeout(function () {
          if (typeof callback === 'function') callback();
        }, animationTime);
      }

      animationStates = [];
    },
    animate: function animate(target, currentRect, toRect, duration) {
      if (duration) {
        css(target, 'transition', '');
        css(target, 'transform', '');
        var elMatrix = matrix(this.el),
            scaleX = elMatrix && elMatrix.a,
            scaleY = elMatrix && elMatrix.d,
            translateX = (currentRect.left - toRect.left) / (scaleX || 1),
            translateY = (currentRect.top - toRect.top) / (scaleY || 1);
        target.animatingX = !!translateX;
        target.animatingY = !!translateY;
        css(target, 'transform', 'translate3d(' + translateX + 'px,' + translateY + 'px,0)');
        this.forRepaintDummy = repaint(target); // repaint

        css(target, 'transition', 'transform ' + duration + 'ms' + (this.options.easing ? ' ' + this.options.easing : ''));
        css(target, 'transform', 'translate3d(0,0,0)');
        typeof target.animated === 'number' && clearTimeout(target.animated);
        target.animated = setTimeout(function () {
          css(target, 'transition', '');
          css(target, 'transform', '');
          target.animated = false;
          target.animatingX = false;
          target.animatingY = false;
        }, duration);
      }
    }
  };
}

function repaint(target) {
  return target.offsetWidth;
}

function calculateRealTime(animatingRect, fromRect, toRect, options) {
  return Math.sqrt(Math.pow(fromRect.top - animatingRect.top, 2) + Math.pow(fromRect.left - animatingRect.left, 2)) / Math.sqrt(Math.pow(fromRect.top - toRect.top, 2) + Math.pow(fromRect.left - toRect.left, 2)) * options.animation;
}

var plugins = [];
var defaults = {
  initializeByDefault: true
};
var PluginManager = {
  mount: function mount(plugin) {
    // Set default static properties
    for (var option in defaults) {
      if (defaults.hasOwnProperty(option) && !(option in plugin)) {
        plugin[option] = defaults[option];
      }
    }

    plugins.forEach(function (p) {
      if (p.pluginName === plugin.pluginName) {
        throw "Sortable: Cannot mount plugin ".concat(plugin.pluginName, " more than once");
      }
    });
    plugins.push(plugin);
  },
  pluginEvent: function pluginEvent(eventName, sortable, evt) {
    var _this = this;

    this.eventCanceled = false;

    evt.cancel = function () {
      _this.eventCanceled = true;
    };

    var eventNameGlobal = eventName + 'Global';
    plugins.forEach(function (plugin) {
      if (!sortable[plugin.pluginName]) return; // Fire global events if it exists in this sortable

      if (sortable[plugin.pluginName][eventNameGlobal]) {
        sortable[plugin.pluginName][eventNameGlobal](_objectSpread2$1({
          sortable: sortable
        }, evt));
      } // Only fire plugin event if plugin is enabled in this sortable,
      // and plugin has event defined


      if (sortable.options[plugin.pluginName] && sortable[plugin.pluginName][eventName]) {
        sortable[plugin.pluginName][eventName](_objectSpread2$1({
          sortable: sortable
        }, evt));
      }
    });
  },
  initializePlugins: function initializePlugins(sortable, el, defaults, options) {
    plugins.forEach(function (plugin) {
      var pluginName = plugin.pluginName;
      if (!sortable.options[pluginName] && !plugin.initializeByDefault) return;
      var initialized = new plugin(sortable, el, sortable.options);
      initialized.sortable = sortable;
      initialized.options = sortable.options;
      sortable[pluginName] = initialized; // Add default options from plugin

      _extends(defaults, initialized.defaults);
    });

    for (var option in sortable.options) {
      if (!sortable.options.hasOwnProperty(option)) continue;
      var modified = this.modifyOption(sortable, option, sortable.options[option]);

      if (typeof modified !== 'undefined') {
        sortable.options[option] = modified;
      }
    }
  },
  getEventProperties: function getEventProperties(name, sortable) {
    var eventProperties = {};
    plugins.forEach(function (plugin) {
      if (typeof plugin.eventProperties !== 'function') return;

      _extends(eventProperties, plugin.eventProperties.call(sortable[plugin.pluginName], name));
    });
    return eventProperties;
  },
  modifyOption: function modifyOption(sortable, name, value) {
    var modifiedValue;
    plugins.forEach(function (plugin) {
      // Plugin must exist on the Sortable
      if (!sortable[plugin.pluginName]) return; // If static option listener exists for this option, call in the context of the Sortable's instance of this plugin

      if (plugin.optionListeners && typeof plugin.optionListeners[name] === 'function') {
        modifiedValue = plugin.optionListeners[name].call(sortable[plugin.pluginName], value);
      }
    });
    return modifiedValue;
  }
};

function dispatchEvent$1(_ref) {
  var sortable = _ref.sortable,
      rootEl = _ref.rootEl,
      name = _ref.name,
      targetEl = _ref.targetEl,
      cloneEl = _ref.cloneEl,
      toEl = _ref.toEl,
      fromEl = _ref.fromEl,
      oldIndex = _ref.oldIndex,
      newIndex = _ref.newIndex,
      oldDraggableIndex = _ref.oldDraggableIndex,
      newDraggableIndex = _ref.newDraggableIndex,
      originalEvent = _ref.originalEvent,
      putSortable = _ref.putSortable,
      extraEventProperties = _ref.extraEventProperties;
  sortable = sortable || rootEl && rootEl[expando];
  if (!sortable) return;
  var evt,
      options = sortable.options,
      onName = 'on' + name.charAt(0).toUpperCase() + name.substr(1); // Support for new CustomEvent feature

  if (window.CustomEvent && !IE11OrLess && !Edge) {
    evt = new CustomEvent(name, {
      bubbles: true,
      cancelable: true
    });
  } else {
    evt = document.createEvent('Event');
    evt.initEvent(name, true, true);
  }

  evt.to = toEl || rootEl;
  evt.from = fromEl || rootEl;
  evt.item = targetEl || rootEl;
  evt.clone = cloneEl;
  evt.oldIndex = oldIndex;
  evt.newIndex = newIndex;
  evt.oldDraggableIndex = oldDraggableIndex;
  evt.newDraggableIndex = newDraggableIndex;
  evt.originalEvent = originalEvent;
  evt.pullMode = putSortable ? putSortable.lastPutMode : undefined;

  var allEventProperties = _objectSpread2$1(_objectSpread2$1({}, extraEventProperties), PluginManager.getEventProperties(name, sortable));

  for (var option in allEventProperties) {
    evt[option] = allEventProperties[option];
  }

  if (rootEl) {
    rootEl.dispatchEvent(evt);
  }

  if (options[onName]) {
    options[onName].call(sortable, evt);
  }
}

var _excluded = ["evt"];

var pluginEvent = function pluginEvent(eventName, sortable) {
  var _ref = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {},
      originalEvent = _ref.evt,
      data = _objectWithoutProperties(_ref, _excluded);

  PluginManager.pluginEvent.bind(Sortable)(eventName, sortable, _objectSpread2$1({
    dragEl: dragEl,
    parentEl: parentEl,
    ghostEl: ghostEl,
    rootEl: rootEl,
    nextEl: nextEl,
    lastDownEl: lastDownEl,
    cloneEl: cloneEl,
    cloneHidden: cloneHidden,
    dragStarted: moved,
    putSortable: putSortable,
    activeSortable: Sortable.active,
    originalEvent: originalEvent,
    oldIndex: oldIndex,
    oldDraggableIndex: oldDraggableIndex,
    newIndex: newIndex,
    newDraggableIndex: newDraggableIndex,
    hideGhostForTarget: _hideGhostForTarget,
    unhideGhostForTarget: _unhideGhostForTarget,
    cloneNowHidden: function cloneNowHidden() {
      cloneHidden = true;
    },
    cloneNowShown: function cloneNowShown() {
      cloneHidden = false;
    },
    dispatchSortableEvent: function dispatchSortableEvent(name) {
      _dispatchEvent({
        sortable: sortable,
        name: name,
        originalEvent: originalEvent
      });
    }
  }, data));
};

function _dispatchEvent(info) {
  dispatchEvent$1(_objectSpread2$1({
    putSortable: putSortable,
    cloneEl: cloneEl,
    targetEl: dragEl,
    rootEl: rootEl,
    oldIndex: oldIndex,
    oldDraggableIndex: oldDraggableIndex,
    newIndex: newIndex,
    newDraggableIndex: newDraggableIndex
  }, info));
}

var dragEl,
    parentEl,
    ghostEl,
    rootEl,
    nextEl,
    lastDownEl,
    cloneEl,
    cloneHidden,
    oldIndex,
    newIndex,
    oldDraggableIndex,
    newDraggableIndex,
    activeGroup,
    putSortable,
    awaitingDragStarted = false,
    ignoreNextClick = false,
    sortables = [],
    tapEvt,
    touchEvt,
    lastDx,
    lastDy,
    tapDistanceLeft,
    tapDistanceTop,
    moved,
    lastTarget,
    lastDirection,
    pastFirstInvertThresh = false,
    isCircumstantialInvert = false,
    targetMoveDistance,
    // For positioning ghost absolutely
ghostRelativeParent,
    ghostRelativeParentInitialScroll = [],
    // (left, top)
_silent = false,
    savedInputChecked = [];
/** @const */

var documentExists = typeof document !== 'undefined',
    PositionGhostAbsolutely = IOS,
    CSSFloatProperty = Edge || IE11OrLess ? 'cssFloat' : 'float',
    // This will not pass for IE9, because IE9 DnD only works on anchors
supportDraggable = documentExists && !ChromeForAndroid && !IOS && 'draggable' in document.createElement('div'),
    supportCssPointerEvents = function () {
  if (!documentExists) return; // false when <= IE11

  if (IE11OrLess) {
    return false;
  }

  var el = document.createElement('x');
  el.style.cssText = 'pointer-events:auto';
  return el.style.pointerEvents === 'auto';
}(),
    _detectDirection = function _detectDirection(el, options) {
  var elCSS = css(el),
      elWidth = parseInt(elCSS.width) - parseInt(elCSS.paddingLeft) - parseInt(elCSS.paddingRight) - parseInt(elCSS.borderLeftWidth) - parseInt(elCSS.borderRightWidth),
      child1 = getChild(el, 0, options),
      child2 = getChild(el, 1, options),
      firstChildCSS = child1 && css(child1),
      secondChildCSS = child2 && css(child2),
      firstChildWidth = firstChildCSS && parseInt(firstChildCSS.marginLeft) + parseInt(firstChildCSS.marginRight) + getRect$1(child1).width,
      secondChildWidth = secondChildCSS && parseInt(secondChildCSS.marginLeft) + parseInt(secondChildCSS.marginRight) + getRect$1(child2).width;

  if (elCSS.display === 'flex') {
    return elCSS.flexDirection === 'column' || elCSS.flexDirection === 'column-reverse' ? 'vertical' : 'horizontal';
  }

  if (elCSS.display === 'grid') {
    return elCSS.gridTemplateColumns.split(' ').length <= 1 ? 'vertical' : 'horizontal';
  }

  if (child1 && firstChildCSS["float"] && firstChildCSS["float"] !== 'none') {
    var touchingSideChild2 = firstChildCSS["float"] === 'left' ? 'left' : 'right';
    return child2 && (secondChildCSS.clear === 'both' || secondChildCSS.clear === touchingSideChild2) ? 'vertical' : 'horizontal';
  }

  return child1 && (firstChildCSS.display === 'block' || firstChildCSS.display === 'flex' || firstChildCSS.display === 'table' || firstChildCSS.display === 'grid' || firstChildWidth >= elWidth && elCSS[CSSFloatProperty] === 'none' || child2 && elCSS[CSSFloatProperty] === 'none' && firstChildWidth + secondChildWidth > elWidth) ? 'vertical' : 'horizontal';
},
    _dragElInRowColumn = function _dragElInRowColumn(dragRect, targetRect, vertical) {
  var dragElS1Opp = vertical ? dragRect.left : dragRect.top,
      dragElS2Opp = vertical ? dragRect.right : dragRect.bottom,
      dragElOppLength = vertical ? dragRect.width : dragRect.height,
      targetS1Opp = vertical ? targetRect.left : targetRect.top,
      targetS2Opp = vertical ? targetRect.right : targetRect.bottom,
      targetOppLength = vertical ? targetRect.width : targetRect.height;
  return dragElS1Opp === targetS1Opp || dragElS2Opp === targetS2Opp || dragElS1Opp + dragElOppLength / 2 === targetS1Opp + targetOppLength / 2;
},

/**
 * Detects first nearest empty sortable to X and Y position using emptyInsertThreshold.
 * @param  {Number} x      X position
 * @param  {Number} y      Y position
 * @return {HTMLElement}   Element of the first found nearest Sortable
 */
_detectNearestEmptySortable = function _detectNearestEmptySortable(x, y) {
  var ret;
  sortables.some(function (sortable) {
    var threshold = sortable[expando].options.emptyInsertThreshold;
    if (!threshold || lastChild(sortable)) return;
    var rect = getRect$1(sortable),
        insideHorizontally = x >= rect.left - threshold && x <= rect.right + threshold,
        insideVertically = y >= rect.top - threshold && y <= rect.bottom + threshold;

    if (insideHorizontally && insideVertically) {
      return ret = sortable;
    }
  });
  return ret;
},
    _prepareGroup = function _prepareGroup(options) {
  function toFn(value, pull) {
    return function (to, from, dragEl, evt) {
      var sameGroup = to.options.group.name && from.options.group.name && to.options.group.name === from.options.group.name;

      if (value == null && (pull || sameGroup)) {
        // Default pull value
        // Default pull and put value if same group
        return true;
      } else if (value == null || value === false) {
        return false;
      } else if (pull && value === 'clone') {
        return value;
      } else if (typeof value === 'function') {
        return toFn(value(to, from, dragEl, evt), pull)(to, from, dragEl, evt);
      } else {
        var otherGroup = (pull ? to : from).options.group.name;
        return value === true || typeof value === 'string' && value === otherGroup || value.join && value.indexOf(otherGroup) > -1;
      }
    };
  }

  var group = {};
  var originalGroup = options.group;

  if (!originalGroup || _typeof$1(originalGroup) != 'object') {
    originalGroup = {
      name: originalGroup
    };
  }

  group.name = originalGroup.name;
  group.checkPull = toFn(originalGroup.pull, true);
  group.checkPut = toFn(originalGroup.put);
  group.revertClone = originalGroup.revertClone;
  options.group = group;
},
    _hideGhostForTarget = function _hideGhostForTarget() {
  if (!supportCssPointerEvents && ghostEl) {
    css(ghostEl, 'display', 'none');
  }
},
    _unhideGhostForTarget = function _unhideGhostForTarget() {
  if (!supportCssPointerEvents && ghostEl) {
    css(ghostEl, 'display', '');
  }
}; // #1184 fix - Prevent click event on fallback if dragged but item not changed position


if (documentExists) {
  document.addEventListener('click', function (evt) {
    if (ignoreNextClick) {
      evt.preventDefault();
      evt.stopPropagation && evt.stopPropagation();
      evt.stopImmediatePropagation && evt.stopImmediatePropagation();
      ignoreNextClick = false;
      return false;
    }
  }, true);
}

var nearestEmptyInsertDetectEvent = function nearestEmptyInsertDetectEvent(evt) {
  if (dragEl) {
    evt = evt.touches ? evt.touches[0] : evt;

    var nearest = _detectNearestEmptySortable(evt.clientX, evt.clientY);

    if (nearest) {
      // Create imitation event
      var event = {};

      for (var i in evt) {
        if (evt.hasOwnProperty(i)) {
          event[i] = evt[i];
        }
      }

      event.target = event.rootEl = nearest;
      event.preventDefault = void 0;
      event.stopPropagation = void 0;

      nearest[expando]._onDragOver(event);
    }
  }
};

var _checkOutsideTargetEl = function _checkOutsideTargetEl(evt) {
  if (dragEl) {
    dragEl.parentNode[expando]._isOutsideThisEl(evt.target);
  }
};
/**
 * @class  Sortable
 * @param  {HTMLElement}  el
 * @param  {Object}       [options]
 */


function Sortable(el, options) {
  if (!(el && el.nodeType && el.nodeType === 1)) {
    throw "Sortable: `el` must be an HTMLElement, not ".concat({}.toString.call(el));
  }

  this.el = el; // root element

  this.options = options = _extends({}, options); // Export instance

  el[expando] = this;
  var defaults = {
    group: null,
    sort: true,
    disabled: false,
    store: null,
    handle: null,
    draggable: /^[uo]l$/i.test(el.nodeName) ? '>li' : '>*',
    swapThreshold: 1,
    // percentage; 0 <= x <= 1
    invertSwap: false,
    // invert always
    invertedSwapThreshold: null,
    // will be set to same as swapThreshold if default
    removeCloneOnHide: true,
    direction: function direction() {
      return _detectDirection(el, this.options);
    },
    ghostClass: 'sortable-ghost',
    chosenClass: 'sortable-chosen',
    dragClass: 'sortable-drag',
    ignore: 'a, img',
    filter: null,
    preventOnFilter: true,
    animation: 0,
    easing: null,
    setData: function setData(dataTransfer, dragEl) {
      dataTransfer.setData('Text', dragEl.textContent);
    },
    dropBubble: false,
    dragoverBubble: false,
    dataIdAttr: 'data-id',
    delay: 0,
    delayOnTouchOnly: false,
    touchStartThreshold: (Number.parseInt ? Number : window).parseInt(window.devicePixelRatio, 10) || 1,
    forceFallback: false,
    fallbackClass: 'sortable-fallback',
    fallbackOnBody: false,
    fallbackTolerance: 0,
    fallbackOffset: {
      x: 0,
      y: 0
    },
    supportPointer: Sortable.supportPointer !== false && 'PointerEvent' in window && !Safari,
    emptyInsertThreshold: 5
  };
  PluginManager.initializePlugins(this, el, defaults); // Set default options

  for (var name in defaults) {
    !(name in options) && (options[name] = defaults[name]);
  }

  _prepareGroup(options); // Bind all private methods


  for (var fn in this) {
    if (fn.charAt(0) === '_' && typeof this[fn] === 'function') {
      this[fn] = this[fn].bind(this);
    }
  } // Setup drag mode


  this.nativeDraggable = options.forceFallback ? false : supportDraggable;

  if (this.nativeDraggable) {
    // Touch start threshold cannot be greater than the native dragstart threshold
    this.options.touchStartThreshold = 1;
  } // Bind events


  if (options.supportPointer) {
    on(el, 'pointerdown', this._onTapStart);
  } else {
    on(el, 'mousedown', this._onTapStart);
    on(el, 'touchstart', this._onTapStart);
  }

  if (this.nativeDraggable) {
    on(el, 'dragover', this);
    on(el, 'dragenter', this);
  }

  sortables.push(this.el); // Restore sorting

  options.store && options.store.get && this.sort(options.store.get(this) || []); // Add animation state manager

  _extends(this, AnimationStateManager());
}

Sortable.prototype =
/** @lends Sortable.prototype */
{
  constructor: Sortable,
  _isOutsideThisEl: function _isOutsideThisEl(target) {
    if (!this.el.contains(target) && target !== this.el) {
      lastTarget = null;
    }
  },
  _getDirection: function _getDirection(evt, target) {
    return typeof this.options.direction === 'function' ? this.options.direction.call(this, evt, target, dragEl) : this.options.direction;
  },
  _onTapStart: function _onTapStart(
  /** Event|TouchEvent */
  evt) {
    if (!evt.cancelable) return;

    var _this = this,
        el = this.el,
        options = this.options,
        preventOnFilter = options.preventOnFilter,
        type = evt.type,
        touch = evt.touches && evt.touches[0] || evt.pointerType && evt.pointerType === 'touch' && evt,
        target = (touch || evt).target,
        originalTarget = evt.target.shadowRoot && (evt.path && evt.path[0] || evt.composedPath && evt.composedPath()[0]) || target,
        filter = options.filter;

    _saveInputCheckedState(el); // Don't trigger start event when an element is been dragged, otherwise the evt.oldindex always wrong when set option.group.


    if (dragEl) {
      return;
    }

    if (/mousedown|pointerdown/.test(type) && evt.button !== 0 || options.disabled) {
      return; // only left button and enabled
    } // cancel dnd if original target is content editable


    if (originalTarget.isContentEditable) {
      return;
    } // Safari ignores further event handling after mousedown


    if (!this.nativeDraggable && Safari && target && target.tagName.toUpperCase() === 'SELECT') {
      return;
    }

    target = closest(target, options.draggable, el, false);

    if (target && target.animated) {
      return;
    }

    if (lastDownEl === target) {
      // Ignoring duplicate `down`
      return;
    } // Get the index of the dragged element within its parent


    oldIndex = index(target);
    oldDraggableIndex = index(target, options.draggable); // Check filter

    if (typeof filter === 'function') {
      if (filter.call(this, evt, target, this)) {
        _dispatchEvent({
          sortable: _this,
          rootEl: originalTarget,
          name: 'filter',
          targetEl: target,
          toEl: el,
          fromEl: el
        });

        pluginEvent('filter', _this, {
          evt: evt
        });
        preventOnFilter && evt.cancelable && evt.preventDefault();
        return; // cancel dnd
      }
    } else if (filter) {
      filter = filter.split(',').some(function (criteria) {
        criteria = closest(originalTarget, criteria.trim(), el, false);

        if (criteria) {
          _dispatchEvent({
            sortable: _this,
            rootEl: criteria,
            name: 'filter',
            targetEl: target,
            fromEl: el,
            toEl: el
          });

          pluginEvent('filter', _this, {
            evt: evt
          });
          return true;
        }
      });

      if (filter) {
        preventOnFilter && evt.cancelable && evt.preventDefault();
        return; // cancel dnd
      }
    }

    if (options.handle && !closest(originalTarget, options.handle, el, false)) {
      return;
    } // Prepare `dragstart`


    this._prepareDragStart(evt, touch, target);
  },
  _prepareDragStart: function _prepareDragStart(
  /** Event */
  evt,
  /** Touch */
  touch,
  /** HTMLElement */
  target) {
    var _this = this,
        el = _this.el,
        options = _this.options,
        ownerDocument = el.ownerDocument,
        dragStartFn;

    if (target && !dragEl && target.parentNode === el) {
      var dragRect = getRect$1(target);
      rootEl = el;
      dragEl = target;
      parentEl = dragEl.parentNode;
      nextEl = dragEl.nextSibling;
      lastDownEl = target;
      activeGroup = options.group;
      Sortable.dragged = dragEl;
      tapEvt = {
        target: dragEl,
        clientX: (touch || evt).clientX,
        clientY: (touch || evt).clientY
      };
      tapDistanceLeft = tapEvt.clientX - dragRect.left;
      tapDistanceTop = tapEvt.clientY - dragRect.top;
      this._lastX = (touch || evt).clientX;
      this._lastY = (touch || evt).clientY;
      dragEl.style['will-change'] = 'all';

      dragStartFn = function dragStartFn() {
        pluginEvent('delayEnded', _this, {
          evt: evt
        });

        if (Sortable.eventCanceled) {
          _this._onDrop();

          return;
        } // Delayed drag has been triggered
        // we can re-enable the events: touchmove/mousemove


        _this._disableDelayedDragEvents();

        if (!FireFox && _this.nativeDraggable) {
          dragEl.draggable = true;
        } // Bind the events: dragstart/dragend


        _this._triggerDragStart(evt, touch); // Drag start event


        _dispatchEvent({
          sortable: _this,
          name: 'choose',
          originalEvent: evt
        }); // Chosen item


        toggleClass$1(dragEl, options.chosenClass, true);
      }; // Disable "draggable"


      options.ignore.split(',').forEach(function (criteria) {
        find$2(dragEl, criteria.trim(), _disableDraggable);
      });
      on(ownerDocument, 'dragover', nearestEmptyInsertDetectEvent);
      on(ownerDocument, 'mousemove', nearestEmptyInsertDetectEvent);
      on(ownerDocument, 'touchmove', nearestEmptyInsertDetectEvent);
      on(ownerDocument, 'mouseup', _this._onDrop);
      on(ownerDocument, 'touchend', _this._onDrop);
      on(ownerDocument, 'touchcancel', _this._onDrop); // Make dragEl draggable (must be before delay for FireFox)

      if (FireFox && this.nativeDraggable) {
        this.options.touchStartThreshold = 4;
        dragEl.draggable = true;
      }

      pluginEvent('delayStart', this, {
        evt: evt
      }); // Delay is impossible for native DnD in Edge or IE

      if (options.delay && (!options.delayOnTouchOnly || touch) && (!this.nativeDraggable || !(Edge || IE11OrLess))) {
        if (Sortable.eventCanceled) {
          this._onDrop();

          return;
        } // If the user moves the pointer or let go the click or touch
        // before the delay has been reached:
        // disable the delayed drag


        on(ownerDocument, 'mouseup', _this._disableDelayedDrag);
        on(ownerDocument, 'touchend', _this._disableDelayedDrag);
        on(ownerDocument, 'touchcancel', _this._disableDelayedDrag);
        on(ownerDocument, 'mousemove', _this._delayedDragTouchMoveHandler);
        on(ownerDocument, 'touchmove', _this._delayedDragTouchMoveHandler);
        options.supportPointer && on(ownerDocument, 'pointermove', _this._delayedDragTouchMoveHandler);
        _this._dragStartTimer = setTimeout(dragStartFn, options.delay);
      } else {
        dragStartFn();
      }
    }
  },
  _delayedDragTouchMoveHandler: function _delayedDragTouchMoveHandler(
  /** TouchEvent|PointerEvent **/
  e) {
    var touch = e.touches ? e.touches[0] : e;

    if (Math.max(Math.abs(touch.clientX - this._lastX), Math.abs(touch.clientY - this._lastY)) >= Math.floor(this.options.touchStartThreshold / (this.nativeDraggable && window.devicePixelRatio || 1))) {
      this._disableDelayedDrag();
    }
  },
  _disableDelayedDrag: function _disableDelayedDrag() {
    dragEl && _disableDraggable(dragEl);
    clearTimeout(this._dragStartTimer);

    this._disableDelayedDragEvents();
  },
  _disableDelayedDragEvents: function _disableDelayedDragEvents() {
    var ownerDocument = this.el.ownerDocument;
    off(ownerDocument, 'mouseup', this._disableDelayedDrag);
    off(ownerDocument, 'touchend', this._disableDelayedDrag);
    off(ownerDocument, 'touchcancel', this._disableDelayedDrag);
    off(ownerDocument, 'mousemove', this._delayedDragTouchMoveHandler);
    off(ownerDocument, 'touchmove', this._delayedDragTouchMoveHandler);
    off(ownerDocument, 'pointermove', this._delayedDragTouchMoveHandler);
  },
  _triggerDragStart: function _triggerDragStart(
  /** Event */
  evt,
  /** Touch */
  touch) {
    touch = touch || evt.pointerType == 'touch' && evt;

    if (!this.nativeDraggable || touch) {
      if (this.options.supportPointer) {
        on(document, 'pointermove', this._onTouchMove);
      } else if (touch) {
        on(document, 'touchmove', this._onTouchMove);
      } else {
        on(document, 'mousemove', this._onTouchMove);
      }
    } else {
      on(dragEl, 'dragend', this);
      on(rootEl, 'dragstart', this._onDragStart);
    }

    try {
      if (document.selection) {
        // Timeout neccessary for IE9
        _nextTick(function () {
          document.selection.empty();
        });
      } else {
        window.getSelection().removeAllRanges();
      }
    } catch (err) {}
  },
  _dragStarted: function _dragStarted(fallback, evt) {

    awaitingDragStarted = false;

    if (rootEl && dragEl) {
      pluginEvent('dragStarted', this, {
        evt: evt
      });

      if (this.nativeDraggable) {
        on(document, 'dragover', _checkOutsideTargetEl);
      }

      var options = this.options; // Apply effect

      !fallback && toggleClass$1(dragEl, options.dragClass, false);
      toggleClass$1(dragEl, options.ghostClass, true);
      Sortable.active = this;
      fallback && this._appendGhost(); // Drag start event

      _dispatchEvent({
        sortable: this,
        name: 'start',
        originalEvent: evt
      });
    } else {
      this._nulling();
    }
  },
  _emulateDragOver: function _emulateDragOver() {
    if (touchEvt) {
      this._lastX = touchEvt.clientX;
      this._lastY = touchEvt.clientY;

      _hideGhostForTarget();

      var target = document.elementFromPoint(touchEvt.clientX, touchEvt.clientY);
      var parent = target;

      while (target && target.shadowRoot) {
        target = target.shadowRoot.elementFromPoint(touchEvt.clientX, touchEvt.clientY);
        if (target === parent) break;
        parent = target;
      }

      dragEl.parentNode[expando]._isOutsideThisEl(target);

      if (parent) {
        do {
          if (parent[expando]) {
            var inserted = void 0;
            inserted = parent[expando]._onDragOver({
              clientX: touchEvt.clientX,
              clientY: touchEvt.clientY,
              target: target,
              rootEl: parent
            });

            if (inserted && !this.options.dragoverBubble) {
              break;
            }
          }

          target = parent; // store last element
        }
        /* jshint boss:true */
        while (parent = parent.parentNode);
      }

      _unhideGhostForTarget();
    }
  },
  _onTouchMove: function _onTouchMove(
  /**TouchEvent*/
  evt) {
    if (tapEvt) {
      var options = this.options,
          fallbackTolerance = options.fallbackTolerance,
          fallbackOffset = options.fallbackOffset,
          touch = evt.touches ? evt.touches[0] : evt,
          ghostMatrix = ghostEl && matrix(ghostEl, true),
          scaleX = ghostEl && ghostMatrix && ghostMatrix.a,
          scaleY = ghostEl && ghostMatrix && ghostMatrix.d,
          relativeScrollOffset = PositionGhostAbsolutely && ghostRelativeParent && getRelativeScrollOffset(ghostRelativeParent),
          dx = (touch.clientX - tapEvt.clientX + fallbackOffset.x) / (scaleX || 1) + (relativeScrollOffset ? relativeScrollOffset[0] - ghostRelativeParentInitialScroll[0] : 0) / (scaleX || 1),
          dy = (touch.clientY - tapEvt.clientY + fallbackOffset.y) / (scaleY || 1) + (relativeScrollOffset ? relativeScrollOffset[1] - ghostRelativeParentInitialScroll[1] : 0) / (scaleY || 1); // only set the status to dragging, when we are actually dragging

      if (!Sortable.active && !awaitingDragStarted) {
        if (fallbackTolerance && Math.max(Math.abs(touch.clientX - this._lastX), Math.abs(touch.clientY - this._lastY)) < fallbackTolerance) {
          return;
        }

        this._onDragStart(evt, true);
      }

      if (ghostEl) {
        if (ghostMatrix) {
          ghostMatrix.e += dx - (lastDx || 0);
          ghostMatrix.f += dy - (lastDy || 0);
        } else {
          ghostMatrix = {
            a: 1,
            b: 0,
            c: 0,
            d: 1,
            e: dx,
            f: dy
          };
        }

        var cssMatrix = "matrix(".concat(ghostMatrix.a, ",").concat(ghostMatrix.b, ",").concat(ghostMatrix.c, ",").concat(ghostMatrix.d, ",").concat(ghostMatrix.e, ",").concat(ghostMatrix.f, ")");
        css(ghostEl, 'webkitTransform', cssMatrix);
        css(ghostEl, 'mozTransform', cssMatrix);
        css(ghostEl, 'msTransform', cssMatrix);
        css(ghostEl, 'transform', cssMatrix);
        lastDx = dx;
        lastDy = dy;
        touchEvt = touch;
      }

      evt.cancelable && evt.preventDefault();
    }
  },
  _appendGhost: function _appendGhost() {
    // Bug if using scale(): https://stackoverflow.com/questions/2637058
    // Not being adjusted for
    if (!ghostEl) {
      var container = this.options.fallbackOnBody ? document.body : rootEl,
          rect = getRect$1(dragEl, true, PositionGhostAbsolutely, true, container),
          options = this.options; // Position absolutely

      if (PositionGhostAbsolutely) {
        // Get relatively positioned parent
        ghostRelativeParent = container;

        while (css(ghostRelativeParent, 'position') === 'static' && css(ghostRelativeParent, 'transform') === 'none' && ghostRelativeParent !== document) {
          ghostRelativeParent = ghostRelativeParent.parentNode;
        }

        if (ghostRelativeParent !== document.body && ghostRelativeParent !== document.documentElement) {
          if (ghostRelativeParent === document) ghostRelativeParent = getWindowScrollingElement();
          rect.top += ghostRelativeParent.scrollTop;
          rect.left += ghostRelativeParent.scrollLeft;
        } else {
          ghostRelativeParent = getWindowScrollingElement();
        }

        ghostRelativeParentInitialScroll = getRelativeScrollOffset(ghostRelativeParent);
      }

      ghostEl = dragEl.cloneNode(true);
      toggleClass$1(ghostEl, options.ghostClass, false);
      toggleClass$1(ghostEl, options.fallbackClass, true);
      toggleClass$1(ghostEl, options.dragClass, true);
      css(ghostEl, 'transition', '');
      css(ghostEl, 'transform', '');
      css(ghostEl, 'box-sizing', 'border-box');
      css(ghostEl, 'margin', 0);
      css(ghostEl, 'top', rect.top);
      css(ghostEl, 'left', rect.left);
      css(ghostEl, 'width', rect.width);
      css(ghostEl, 'height', rect.height);
      css(ghostEl, 'opacity', '0.8');
      css(ghostEl, 'position', PositionGhostAbsolutely ? 'absolute' : 'fixed');
      css(ghostEl, 'zIndex', '100000');
      css(ghostEl, 'pointerEvents', 'none');
      Sortable.ghost = ghostEl;
      container.appendChild(ghostEl); // Set transform-origin

      css(ghostEl, 'transform-origin', tapDistanceLeft / parseInt(ghostEl.style.width) * 100 + '% ' + tapDistanceTop / parseInt(ghostEl.style.height) * 100 + '%');
    }
  },
  _onDragStart: function _onDragStart(
  /**Event*/
  evt,
  /**boolean*/
  fallback) {
    var _this = this;

    var dataTransfer = evt.dataTransfer;
    var options = _this.options;
    pluginEvent('dragStart', this, {
      evt: evt
    });

    if (Sortable.eventCanceled) {
      this._onDrop();

      return;
    }

    pluginEvent('setupClone', this);

    if (!Sortable.eventCanceled) {
      cloneEl = clone(dragEl);
      cloneEl.draggable = false;
      cloneEl.style['will-change'] = '';

      this._hideClone();

      toggleClass$1(cloneEl, this.options.chosenClass, false);
      Sortable.clone = cloneEl;
    } // #1143: IFrame support workaround


    _this.cloneId = _nextTick(function () {
      pluginEvent('clone', _this);
      if (Sortable.eventCanceled) return;

      if (!_this.options.removeCloneOnHide) {
        rootEl.insertBefore(cloneEl, dragEl);
      }

      _this._hideClone();

      _dispatchEvent({
        sortable: _this,
        name: 'clone'
      });
    });
    !fallback && toggleClass$1(dragEl, options.dragClass, true); // Set proper drop events

    if (fallback) {
      ignoreNextClick = true;
      _this._loopId = setInterval(_this._emulateDragOver, 50);
    } else {
      // Undo what was set in _prepareDragStart before drag started
      off(document, 'mouseup', _this._onDrop);
      off(document, 'touchend', _this._onDrop);
      off(document, 'touchcancel', _this._onDrop);

      if (dataTransfer) {
        dataTransfer.effectAllowed = 'move';
        options.setData && options.setData.call(_this, dataTransfer, dragEl);
      }

      on(document, 'drop', _this); // #1276 fix:

      css(dragEl, 'transform', 'translateZ(0)');
    }

    awaitingDragStarted = true;
    _this._dragStartId = _nextTick(_this._dragStarted.bind(_this, fallback, evt));
    on(document, 'selectstart', _this);
    moved = true;

    if (Safari) {
      css(document.body, 'user-select', 'none');
    }
  },
  // Returns true - if no further action is needed (either inserted or another condition)
  _onDragOver: function _onDragOver(
  /**Event*/
  evt) {
    var el = this.el,
        target = evt.target,
        dragRect,
        targetRect,
        revert,
        options = this.options,
        group = options.group,
        activeSortable = Sortable.active,
        isOwner = activeGroup === group,
        canSort = options.sort,
        fromSortable = putSortable || activeSortable,
        vertical,
        _this = this,
        completedFired = false;

    if (_silent) return;

    function dragOverEvent(name, extra) {
      pluginEvent(name, _this, _objectSpread2$1({
        evt: evt,
        isOwner: isOwner,
        axis: vertical ? 'vertical' : 'horizontal',
        revert: revert,
        dragRect: dragRect,
        targetRect: targetRect,
        canSort: canSort,
        fromSortable: fromSortable,
        target: target,
        completed: completed,
        onMove: function onMove(target, after) {
          return _onMove(rootEl, el, dragEl, dragRect, target, getRect$1(target), evt, after);
        },
        changed: changed
      }, extra));
    } // Capture animation state


    function capture() {
      dragOverEvent('dragOverAnimationCapture');

      _this.captureAnimationState();

      if (_this !== fromSortable) {
        fromSortable.captureAnimationState();
      }
    } // Return invocation when dragEl is inserted (or completed)


    function completed(insertion) {
      dragOverEvent('dragOverCompleted', {
        insertion: insertion
      });

      if (insertion) {
        // Clones must be hidden before folding animation to capture dragRectAbsolute properly
        if (isOwner) {
          activeSortable._hideClone();
        } else {
          activeSortable._showClone(_this);
        }

        if (_this !== fromSortable) {
          // Set ghost class to new sortable's ghost class
          toggleClass$1(dragEl, putSortable ? putSortable.options.ghostClass : activeSortable.options.ghostClass, false);
          toggleClass$1(dragEl, options.ghostClass, true);
        }

        if (putSortable !== _this && _this !== Sortable.active) {
          putSortable = _this;
        } else if (_this === Sortable.active && putSortable) {
          putSortable = null;
        } // Animation


        if (fromSortable === _this) {
          _this._ignoreWhileAnimating = target;
        }

        _this.animateAll(function () {
          dragOverEvent('dragOverAnimationComplete');
          _this._ignoreWhileAnimating = null;
        });

        if (_this !== fromSortable) {
          fromSortable.animateAll();
          fromSortable._ignoreWhileAnimating = null;
        }
      } // Null lastTarget if it is not inside a previously swapped element


      if (target === dragEl && !dragEl.animated || target === el && !target.animated) {
        lastTarget = null;
      } // no bubbling and not fallback


      if (!options.dragoverBubble && !evt.rootEl && target !== document) {
        dragEl.parentNode[expando]._isOutsideThisEl(evt.target); // Do not detect for empty insert if already inserted


        !insertion && nearestEmptyInsertDetectEvent(evt);
      }

      !options.dragoverBubble && evt.stopPropagation && evt.stopPropagation();
      return completedFired = true;
    } // Call when dragEl has been inserted


    function changed() {
      newIndex = index(dragEl);
      newDraggableIndex = index(dragEl, options.draggable);

      _dispatchEvent({
        sortable: _this,
        name: 'change',
        toEl: el,
        newIndex: newIndex,
        newDraggableIndex: newDraggableIndex,
        originalEvent: evt
      });
    }

    if (evt.preventDefault !== void 0) {
      evt.cancelable && evt.preventDefault();
    }

    target = closest(target, options.draggable, el, true);
    dragOverEvent('dragOver');
    if (Sortable.eventCanceled) return completedFired;

    if (dragEl.contains(evt.target) || target.animated && target.animatingX && target.animatingY || _this._ignoreWhileAnimating === target) {
      return completed(false);
    }

    ignoreNextClick = false;

    if (activeSortable && !options.disabled && (isOwner ? canSort || (revert = parentEl !== rootEl) // Reverting item into the original list
    : putSortable === this || (this.lastPutMode = activeGroup.checkPull(this, activeSortable, dragEl, evt)) && group.checkPut(this, activeSortable, dragEl, evt))) {
      vertical = this._getDirection(evt, target) === 'vertical';
      dragRect = getRect$1(dragEl);
      dragOverEvent('dragOverValid');
      if (Sortable.eventCanceled) return completedFired;

      if (revert) {
        parentEl = rootEl; // actualization

        capture();

        this._hideClone();

        dragOverEvent('revert');

        if (!Sortable.eventCanceled) {
          if (nextEl) {
            rootEl.insertBefore(dragEl, nextEl);
          } else {
            rootEl.appendChild(dragEl);
          }
        }

        return completed(true);
      }

      var elLastChild = lastChild(el, options.draggable);

      if (!elLastChild || _ghostIsLast(evt, vertical, this) && !elLastChild.animated) {
        // Insert to end of list
        // If already at end of list: Do not insert
        if (elLastChild === dragEl) {
          return completed(false);
        } // if there is a last element, it is the target


        if (elLastChild && el === evt.target) {
          target = elLastChild;
        }

        if (target) {
          targetRect = getRect$1(target);
        }

        if (_onMove(rootEl, el, dragEl, dragRect, target, targetRect, evt, !!target) !== false) {
          capture();
          el.appendChild(dragEl);
          parentEl = el; // actualization

          changed();
          return completed(true);
        }
      } else if (elLastChild && _ghostIsFirst(evt, vertical, this)) {
        // Insert to start of list
        var firstChild = getChild(el, 0, options, true);

        if (firstChild === dragEl) {
          return completed(false);
        }

        target = firstChild;
        targetRect = getRect$1(target);

        if (_onMove(rootEl, el, dragEl, dragRect, target, targetRect, evt, false) !== false) {
          capture();
          el.insertBefore(dragEl, firstChild);
          parentEl = el; // actualization

          changed();
          return completed(true);
        }
      } else if (target.parentNode === el) {
        targetRect = getRect$1(target);
        var direction = 0,
            targetBeforeFirstSwap,
            differentLevel = dragEl.parentNode !== el,
            differentRowCol = !_dragElInRowColumn(dragEl.animated && dragEl.toRect || dragRect, target.animated && target.toRect || targetRect, vertical),
            side1 = vertical ? 'top' : 'left',
            scrolledPastTop = isScrolledPast(target, 'top', 'top') || isScrolledPast(dragEl, 'top', 'top'),
            scrollBefore = scrolledPastTop ? scrolledPastTop.scrollTop : void 0;

        if (lastTarget !== target) {
          targetBeforeFirstSwap = targetRect[side1];
          pastFirstInvertThresh = false;
          isCircumstantialInvert = !differentRowCol && options.invertSwap || differentLevel;
        }

        direction = _getSwapDirection(evt, target, targetRect, vertical, differentRowCol ? 1 : options.swapThreshold, options.invertedSwapThreshold == null ? options.swapThreshold : options.invertedSwapThreshold, isCircumstantialInvert, lastTarget === target);
        var sibling;

        if (direction !== 0) {
          // Check if target is beside dragEl in respective direction (ignoring hidden elements)
          var dragIndex = index(dragEl);

          do {
            dragIndex -= direction;
            sibling = parentEl.children[dragIndex];
          } while (sibling && (css(sibling, 'display') === 'none' || sibling === ghostEl));
        } // If dragEl is already beside target: Do not insert


        if (direction === 0 || sibling === target) {
          return completed(false);
        }

        lastTarget = target;
        lastDirection = direction;
        var nextSibling = target.nextElementSibling,
            after = false;
        after = direction === 1;

        var moveVector = _onMove(rootEl, el, dragEl, dragRect, target, targetRect, evt, after);

        if (moveVector !== false) {
          if (moveVector === 1 || moveVector === -1) {
            after = moveVector === 1;
          }

          _silent = true;
          setTimeout(_unsilent, 30);
          capture();

          if (after && !nextSibling) {
            el.appendChild(dragEl);
          } else {
            target.parentNode.insertBefore(dragEl, after ? nextSibling : target);
          } // Undo chrome's scroll adjustment (has no effect on other browsers)


          if (scrolledPastTop) {
            scrollBy(scrolledPastTop, 0, scrollBefore - scrolledPastTop.scrollTop);
          }

          parentEl = dragEl.parentNode; // actualization
          // must be done before animation

          if (targetBeforeFirstSwap !== undefined && !isCircumstantialInvert) {
            targetMoveDistance = Math.abs(targetBeforeFirstSwap - getRect$1(target)[side1]);
          }

          changed();
          return completed(true);
        }
      }

      if (el.contains(dragEl)) {
        return completed(false);
      }
    }

    return false;
  },
  _ignoreWhileAnimating: null,
  _offMoveEvents: function _offMoveEvents() {
    off(document, 'mousemove', this._onTouchMove);
    off(document, 'touchmove', this._onTouchMove);
    off(document, 'pointermove', this._onTouchMove);
    off(document, 'dragover', nearestEmptyInsertDetectEvent);
    off(document, 'mousemove', nearestEmptyInsertDetectEvent);
    off(document, 'touchmove', nearestEmptyInsertDetectEvent);
  },
  _offUpEvents: function _offUpEvents() {
    var ownerDocument = this.el.ownerDocument;
    off(ownerDocument, 'mouseup', this._onDrop);
    off(ownerDocument, 'touchend', this._onDrop);
    off(ownerDocument, 'pointerup', this._onDrop);
    off(ownerDocument, 'touchcancel', this._onDrop);
    off(document, 'selectstart', this);
  },
  _onDrop: function _onDrop(
  /**Event*/
  evt) {
    var el = this.el,
        options = this.options; // Get the index of the dragged element within its parent

    newIndex = index(dragEl);
    newDraggableIndex = index(dragEl, options.draggable);
    pluginEvent('drop', this, {
      evt: evt
    });
    parentEl = dragEl && dragEl.parentNode; // Get again after plugin event

    newIndex = index(dragEl);
    newDraggableIndex = index(dragEl, options.draggable);

    if (Sortable.eventCanceled) {
      this._nulling();

      return;
    }

    awaitingDragStarted = false;
    isCircumstantialInvert = false;
    pastFirstInvertThresh = false;
    clearInterval(this._loopId);
    clearTimeout(this._dragStartTimer);

    _cancelNextTick(this.cloneId);

    _cancelNextTick(this._dragStartId); // Unbind events


    if (this.nativeDraggable) {
      off(document, 'drop', this);
      off(el, 'dragstart', this._onDragStart);
    }

    this._offMoveEvents();

    this._offUpEvents();

    if (Safari) {
      css(document.body, 'user-select', '');
    }

    css(dragEl, 'transform', '');

    if (evt) {
      if (moved) {
        evt.cancelable && evt.preventDefault();
        !options.dropBubble && evt.stopPropagation();
      }

      ghostEl && ghostEl.parentNode && ghostEl.parentNode.removeChild(ghostEl);

      if (rootEl === parentEl || putSortable && putSortable.lastPutMode !== 'clone') {
        // Remove clone(s)
        cloneEl && cloneEl.parentNode && cloneEl.parentNode.removeChild(cloneEl);
      }

      if (dragEl) {
        if (this.nativeDraggable) {
          off(dragEl, 'dragend', this);
        }

        _disableDraggable(dragEl);

        dragEl.style['will-change'] = ''; // Remove classes
        // ghostClass is added in dragStarted

        if (moved && !awaitingDragStarted) {
          toggleClass$1(dragEl, putSortable ? putSortable.options.ghostClass : this.options.ghostClass, false);
        }

        toggleClass$1(dragEl, this.options.chosenClass, false); // Drag stop event

        _dispatchEvent({
          sortable: this,
          name: 'unchoose',
          toEl: parentEl,
          newIndex: null,
          newDraggableIndex: null,
          originalEvent: evt
        });

        if (rootEl !== parentEl) {
          if (newIndex >= 0) {
            // Add event
            _dispatchEvent({
              rootEl: parentEl,
              name: 'add',
              toEl: parentEl,
              fromEl: rootEl,
              originalEvent: evt
            }); // Remove event


            _dispatchEvent({
              sortable: this,
              name: 'remove',
              toEl: parentEl,
              originalEvent: evt
            }); // drag from one list and drop into another


            _dispatchEvent({
              rootEl: parentEl,
              name: 'sort',
              toEl: parentEl,
              fromEl: rootEl,
              originalEvent: evt
            });

            _dispatchEvent({
              sortable: this,
              name: 'sort',
              toEl: parentEl,
              originalEvent: evt
            });
          }

          putSortable && putSortable.save();
        } else {
          if (newIndex !== oldIndex) {
            if (newIndex >= 0) {
              // drag & drop within the same list
              _dispatchEvent({
                sortable: this,
                name: 'update',
                toEl: parentEl,
                originalEvent: evt
              });

              _dispatchEvent({
                sortable: this,
                name: 'sort',
                toEl: parentEl,
                originalEvent: evt
              });
            }
          }
        }

        if (Sortable.active) {
          /* jshint eqnull:true */
          if (newIndex == null || newIndex === -1) {
            newIndex = oldIndex;
            newDraggableIndex = oldDraggableIndex;
          }

          _dispatchEvent({
            sortable: this,
            name: 'end',
            toEl: parentEl,
            originalEvent: evt
          }); // Save sorting


          this.save();
        }
      }
    }

    this._nulling();
  },
  _nulling: function _nulling() {
    pluginEvent('nulling', this);
    rootEl = dragEl = parentEl = ghostEl = nextEl = cloneEl = lastDownEl = cloneHidden = tapEvt = touchEvt = moved = newIndex = newDraggableIndex = oldIndex = oldDraggableIndex = lastTarget = lastDirection = putSortable = activeGroup = Sortable.dragged = Sortable.ghost = Sortable.clone = Sortable.active = null;
    savedInputChecked.forEach(function (el) {
      el.checked = true;
    });
    savedInputChecked.length = lastDx = lastDy = 0;
  },
  handleEvent: function handleEvent(
  /**Event*/
  evt) {
    switch (evt.type) {
      case 'drop':
      case 'dragend':
        this._onDrop(evt);

        break;

      case 'dragenter':
      case 'dragover':
        if (dragEl) {
          this._onDragOver(evt);

          _globalDragOver(evt);
        }

        break;

      case 'selectstart':
        evt.preventDefault();
        break;
    }
  },

  /**
   * Serializes the item into an array of string.
   * @returns {String[]}
   */
  toArray: function toArray() {
    var order = [],
        el,
        children = this.el.children,
        i = 0,
        n = children.length,
        options = this.options;

    for (; i < n; i++) {
      el = children[i];

      if (closest(el, options.draggable, this.el, false)) {
        order.push(el.getAttribute(options.dataIdAttr) || _generateId(el));
      }
    }

    return order;
  },

  /**
   * Sorts the elements according to the array.
   * @param  {String[]}  order  order of the items
   */
  sort: function sort(order, useAnimation) {
    var items = {},
        rootEl = this.el;
    this.toArray().forEach(function (id, i) {
      var el = rootEl.children[i];

      if (closest(el, this.options.draggable, rootEl, false)) {
        items[id] = el;
      }
    }, this);
    useAnimation && this.captureAnimationState();
    order.forEach(function (id) {
      if (items[id]) {
        rootEl.removeChild(items[id]);
        rootEl.appendChild(items[id]);
      }
    });
    useAnimation && this.animateAll();
  },

  /**
   * Save the current sorting
   */
  save: function save() {
    var store = this.options.store;
    store && store.set && store.set(this);
  },

  /**
   * For each element in the set, get the first element that matches the selector by testing the element itself and traversing up through its ancestors in the DOM tree.
   * @param   {HTMLElement}  el
   * @param   {String}       [selector]  default: `options.draggable`
   * @returns {HTMLElement|null}
   */
  closest: function closest$1(el, selector) {
    return closest(el, selector || this.options.draggable, this.el, false);
  },

  /**
   * Set/get option
   * @param   {string} name
   * @param   {*}      [value]
   * @returns {*}
   */
  option: function option(name, value) {
    var options = this.options;

    if (value === void 0) {
      return options[name];
    } else {
      var modifiedValue = PluginManager.modifyOption(this, name, value);

      if (typeof modifiedValue !== 'undefined') {
        options[name] = modifiedValue;
      } else {
        options[name] = value;
      }

      if (name === 'group') {
        _prepareGroup(options);
      }
    }
  },

  /**
   * Destroy
   */
  destroy: function destroy() {
    pluginEvent('destroy', this);
    var el = this.el;
    el[expando] = null;
    off(el, 'mousedown', this._onTapStart);
    off(el, 'touchstart', this._onTapStart);
    off(el, 'pointerdown', this._onTapStart);

    if (this.nativeDraggable) {
      off(el, 'dragover', this);
      off(el, 'dragenter', this);
    } // Remove draggable attributes


    Array.prototype.forEach.call(el.querySelectorAll('[draggable]'), function (el) {
      el.removeAttribute('draggable');
    });

    this._onDrop();

    this._disableDelayedDragEvents();

    sortables.splice(sortables.indexOf(this.el), 1);
    this.el = el = null;
  },
  _hideClone: function _hideClone() {
    if (!cloneHidden) {
      pluginEvent('hideClone', this);
      if (Sortable.eventCanceled) return;
      css(cloneEl, 'display', 'none');

      if (this.options.removeCloneOnHide && cloneEl.parentNode) {
        cloneEl.parentNode.removeChild(cloneEl);
      }

      cloneHidden = true;
    }
  },
  _showClone: function _showClone(putSortable) {
    if (putSortable.lastPutMode !== 'clone') {
      this._hideClone();

      return;
    }

    if (cloneHidden) {
      pluginEvent('showClone', this);
      if (Sortable.eventCanceled) return; // show clone at dragEl or original position

      if (dragEl.parentNode == rootEl && !this.options.group.revertClone) {
        rootEl.insertBefore(cloneEl, dragEl);
      } else if (nextEl) {
        rootEl.insertBefore(cloneEl, nextEl);
      } else {
        rootEl.appendChild(cloneEl);
      }

      if (this.options.group.revertClone) {
        this.animate(dragEl, cloneEl);
      }

      css(cloneEl, 'display', '');
      cloneHidden = false;
    }
  }
};

function _globalDragOver(
/**Event*/
evt) {
  if (evt.dataTransfer) {
    evt.dataTransfer.dropEffect = 'move';
  }

  evt.cancelable && evt.preventDefault();
}

function _onMove(fromEl, toEl, dragEl, dragRect, targetEl, targetRect, originalEvent, willInsertAfter) {
  var evt,
      sortable = fromEl[expando],
      onMoveFn = sortable.options.onMove,
      retVal; // Support for new CustomEvent feature

  if (window.CustomEvent && !IE11OrLess && !Edge) {
    evt = new CustomEvent('move', {
      bubbles: true,
      cancelable: true
    });
  } else {
    evt = document.createEvent('Event');
    evt.initEvent('move', true, true);
  }

  evt.to = toEl;
  evt.from = fromEl;
  evt.dragged = dragEl;
  evt.draggedRect = dragRect;
  evt.related = targetEl || toEl;
  evt.relatedRect = targetRect || getRect$1(toEl);
  evt.willInsertAfter = willInsertAfter;
  evt.originalEvent = originalEvent;
  fromEl.dispatchEvent(evt);

  if (onMoveFn) {
    retVal = onMoveFn.call(sortable, evt, originalEvent);
  }

  return retVal;
}

function _disableDraggable(el) {
  el.draggable = false;
}

function _unsilent() {
  _silent = false;
}

function _ghostIsFirst(evt, vertical, sortable) {
  var rect = getRect$1(getChild(sortable.el, 0, sortable.options, true));
  var spacer = 10;
  return vertical ? evt.clientX < rect.left - spacer || evt.clientY < rect.top && evt.clientX < rect.right : evt.clientY < rect.top - spacer || evt.clientY < rect.bottom && evt.clientX < rect.left;
}

function _ghostIsLast(evt, vertical, sortable) {
  var rect = getRect$1(lastChild(sortable.el, sortable.options.draggable));
  var spacer = 10;
  return vertical ? evt.clientX > rect.right + spacer || evt.clientX <= rect.right && evt.clientY > rect.bottom && evt.clientX >= rect.left : evt.clientX > rect.right && evt.clientY > rect.top || evt.clientX <= rect.right && evt.clientY > rect.bottom + spacer;
}

function _getSwapDirection(evt, target, targetRect, vertical, swapThreshold, invertedSwapThreshold, invertSwap, isLastTarget) {
  var mouseOnAxis = vertical ? evt.clientY : evt.clientX,
      targetLength = vertical ? targetRect.height : targetRect.width,
      targetS1 = vertical ? targetRect.top : targetRect.left,
      targetS2 = vertical ? targetRect.bottom : targetRect.right,
      invert = false;

  if (!invertSwap) {
    // Never invert or create dragEl shadow when target movemenet causes mouse to move past the end of regular swapThreshold
    if (isLastTarget && targetMoveDistance < targetLength * swapThreshold) {
      // multiplied only by swapThreshold because mouse will already be inside target by (1 - threshold) * targetLength / 2
      // check if past first invert threshold on side opposite of lastDirection
      if (!pastFirstInvertThresh && (lastDirection === 1 ? mouseOnAxis > targetS1 + targetLength * invertedSwapThreshold / 2 : mouseOnAxis < targetS2 - targetLength * invertedSwapThreshold / 2)) {
        // past first invert threshold, do not restrict inverted threshold to dragEl shadow
        pastFirstInvertThresh = true;
      }

      if (!pastFirstInvertThresh) {
        // dragEl shadow (target move distance shadow)
        if (lastDirection === 1 ? mouseOnAxis < targetS1 + targetMoveDistance // over dragEl shadow
        : mouseOnAxis > targetS2 - targetMoveDistance) {
          return -lastDirection;
        }
      } else {
        invert = true;
      }
    } else {
      // Regular
      if (mouseOnAxis > targetS1 + targetLength * (1 - swapThreshold) / 2 && mouseOnAxis < targetS2 - targetLength * (1 - swapThreshold) / 2) {
        return _getInsertDirection(target);
      }
    }
  }

  invert = invert || invertSwap;

  if (invert) {
    // Invert of regular
    if (mouseOnAxis < targetS1 + targetLength * invertedSwapThreshold / 2 || mouseOnAxis > targetS2 - targetLength * invertedSwapThreshold / 2) {
      return mouseOnAxis > targetS1 + targetLength / 2 ? 1 : -1;
    }
  }

  return 0;
}
/**
 * Gets the direction dragEl must be swapped relative to target in order to make it
 * seem that dragEl has been "inserted" into that element's position
 * @param  {HTMLElement} target       The target whose position dragEl is being inserted at
 * @return {Number}                   Direction dragEl must be swapped
 */


function _getInsertDirection(target) {
  if (index(dragEl) < index(target)) {
    return 1;
  } else {
    return -1;
  }
}
/**
 * Generate id
 * @param   {HTMLElement} el
 * @returns {String}
 * @private
 */


function _generateId(el) {
  var str = el.tagName + el.className + el.src + el.href + el.textContent,
      i = str.length,
      sum = 0;

  while (i--) {
    sum += str.charCodeAt(i);
  }

  return sum.toString(36);
}

function _saveInputCheckedState(root) {
  savedInputChecked.length = 0;
  var inputs = root.getElementsByTagName('input');
  var idx = inputs.length;

  while (idx--) {
    var el = inputs[idx];
    el.checked && savedInputChecked.push(el);
  }
}

function _nextTick(fn) {
  return setTimeout(fn, 0);
}

function _cancelNextTick(id) {
  return clearTimeout(id);
} // Fixed #973:


if (documentExists) {
  on(document, 'touchmove', function (evt) {
    if ((Sortable.active || awaitingDragStarted) && evt.cancelable) {
      evt.preventDefault();
    }
  });
} // Export utils


Sortable.utils = {
  on: on,
  off: off,
  css: css,
  find: find$2,
  is: function is(el, selector) {
    return !!closest(el, selector, el, false);
  },
  extend: extend,
  throttle: throttle$1,
  closest: closest,
  toggleClass: toggleClass$1,
  clone: clone,
  index: index,
  nextTick: _nextTick,
  cancelNextTick: _cancelNextTick,
  detectDirection: _detectDirection,
  getChild: getChild
};
/**
 * Get the Sortable instance of an element
 * @param  {HTMLElement} element The element
 * @return {Sortable|undefined}         The instance of Sortable
 */

Sortable.get = function (element) {
  return element[expando];
};
/**
 * Mount a plugin to Sortable
 * @param  {...SortablePlugin|SortablePlugin[]} plugins       Plugins being mounted
 */


Sortable.mount = function () {
  for (var _len = arguments.length, plugins = new Array(_len), _key = 0; _key < _len; _key++) {
    plugins[_key] = arguments[_key];
  }

  if (plugins[0].constructor === Array) plugins = plugins[0];
  plugins.forEach(function (plugin) {
    if (!plugin.prototype || !plugin.prototype.constructor) {
      throw "Sortable: Mounted plugin must be a constructor function, not ".concat({}.toString.call(plugin));
    }

    if (plugin.utils) Sortable.utils = _objectSpread2$1(_objectSpread2$1({}, Sortable.utils), plugin.utils);
    PluginManager.mount(plugin);
  });
};
/**
 * Create sortable instance
 * @param {HTMLElement}  el
 * @param {Object}      [options]
 */


Sortable.create = function (el, options) {
  return new Sortable(el, options);
}; // Export


Sortable.version = version$1;

var autoScrolls = [],
    scrollEl,
    scrollRootEl,
    scrolling = false,
    lastAutoScrollX,
    lastAutoScrollY,
    touchEvt$1,
    pointerElemChangedInterval;

function AutoScrollPlugin() {
  function AutoScroll() {
    this.defaults = {
      scroll: true,
      forceAutoScrollFallback: false,
      scrollSensitivity: 30,
      scrollSpeed: 10,
      bubbleScroll: true
    }; // Bind all private methods

    for (var fn in this) {
      if (fn.charAt(0) === '_' && typeof this[fn] === 'function') {
        this[fn] = this[fn].bind(this);
      }
    }
  }

  AutoScroll.prototype = {
    dragStarted: function dragStarted(_ref) {
      var originalEvent = _ref.originalEvent;

      if (this.sortable.nativeDraggable) {
        on(document, 'dragover', this._handleAutoScroll);
      } else {
        if (this.options.supportPointer) {
          on(document, 'pointermove', this._handleFallbackAutoScroll);
        } else if (originalEvent.touches) {
          on(document, 'touchmove', this._handleFallbackAutoScroll);
        } else {
          on(document, 'mousemove', this._handleFallbackAutoScroll);
        }
      }
    },
    dragOverCompleted: function dragOverCompleted(_ref2) {
      var originalEvent = _ref2.originalEvent;

      // For when bubbling is canceled and using fallback (fallback 'touchmove' always reached)
      if (!this.options.dragOverBubble && !originalEvent.rootEl) {
        this._handleAutoScroll(originalEvent);
      }
    },
    drop: function drop() {
      if (this.sortable.nativeDraggable) {
        off(document, 'dragover', this._handleAutoScroll);
      } else {
        off(document, 'pointermove', this._handleFallbackAutoScroll);
        off(document, 'touchmove', this._handleFallbackAutoScroll);
        off(document, 'mousemove', this._handleFallbackAutoScroll);
      }

      clearPointerElemChangedInterval();
      clearAutoScrolls();
      cancelThrottle();
    },
    nulling: function nulling() {
      touchEvt$1 = scrollRootEl = scrollEl = scrolling = pointerElemChangedInterval = lastAutoScrollX = lastAutoScrollY = null;
      autoScrolls.length = 0;
    },
    _handleFallbackAutoScroll: function _handleFallbackAutoScroll(evt) {
      this._handleAutoScroll(evt, true);
    },
    _handleAutoScroll: function _handleAutoScroll(evt, fallback) {
      var _this = this;

      var x = (evt.touches ? evt.touches[0] : evt).clientX,
          y = (evt.touches ? evt.touches[0] : evt).clientY,
          elem = document.elementFromPoint(x, y);
      touchEvt$1 = evt; // IE does not seem to have native autoscroll,
      // Edge's autoscroll seems too conditional,
      // MACOS Safari does not have autoscroll,
      // Firefox and Chrome are good

      if (fallback || this.options.forceAutoScrollFallback || Edge || IE11OrLess || Safari) {
        autoScroll(evt, this.options, elem, fallback); // Listener for pointer element change

        var ogElemScroller = getParentAutoScrollElement(elem, true);

        if (scrolling && (!pointerElemChangedInterval || x !== lastAutoScrollX || y !== lastAutoScrollY)) {
          pointerElemChangedInterval && clearPointerElemChangedInterval(); // Detect for pointer elem change, emulating native DnD behaviour

          pointerElemChangedInterval = setInterval(function () {
            var newElem = getParentAutoScrollElement(document.elementFromPoint(x, y), true);

            if (newElem !== ogElemScroller) {
              ogElemScroller = newElem;
              clearAutoScrolls();
            }

            autoScroll(evt, _this.options, newElem, fallback);
          }, 10);
          lastAutoScrollX = x;
          lastAutoScrollY = y;
        }
      } else {
        // if DnD is enabled (and browser has good autoscrolling), first autoscroll will already scroll, so get parent autoscroll of first autoscroll
        if (!this.options.bubbleScroll || getParentAutoScrollElement(elem, true) === getWindowScrollingElement()) {
          clearAutoScrolls();
          return;
        }

        autoScroll(evt, this.options, getParentAutoScrollElement(elem, false), false);
      }
    }
  };
  return _extends(AutoScroll, {
    pluginName: 'scroll',
    initializeByDefault: true
  });
}

function clearAutoScrolls() {
  autoScrolls.forEach(function (autoScroll) {
    clearInterval(autoScroll.pid);
  });
  autoScrolls = [];
}

function clearPointerElemChangedInterval() {
  clearInterval(pointerElemChangedInterval);
}

var autoScroll = throttle$1(function (evt, options, rootEl, isFallback) {
  // Bug: https://bugzilla.mozilla.org/show_bug.cgi?id=505521
  if (!options.scroll) return;
  var x = (evt.touches ? evt.touches[0] : evt).clientX,
      y = (evt.touches ? evt.touches[0] : evt).clientY,
      sens = options.scrollSensitivity,
      speed = options.scrollSpeed,
      winScroller = getWindowScrollingElement();
  var scrollThisInstance = false,
      scrollCustomFn; // New scroll root, set scrollEl

  if (scrollRootEl !== rootEl) {
    scrollRootEl = rootEl;
    clearAutoScrolls();
    scrollEl = options.scroll;
    scrollCustomFn = options.scrollFn;

    if (scrollEl === true) {
      scrollEl = getParentAutoScrollElement(rootEl, true);
    }
  }

  var layersOut = 0;
  var currentParent = scrollEl;

  do {
    var el = currentParent,
        rect = getRect$1(el),
        top = rect.top,
        bottom = rect.bottom,
        left = rect.left,
        right = rect.right,
        width = rect.width,
        height = rect.height,
        canScrollX = void 0,
        canScrollY = void 0,
        scrollWidth = el.scrollWidth,
        scrollHeight = el.scrollHeight,
        elCSS = css(el),
        scrollPosX = el.scrollLeft,
        scrollPosY = el.scrollTop;

    if (el === winScroller) {
      canScrollX = width < scrollWidth && (elCSS.overflowX === 'auto' || elCSS.overflowX === 'scroll' || elCSS.overflowX === 'visible');
      canScrollY = height < scrollHeight && (elCSS.overflowY === 'auto' || elCSS.overflowY === 'scroll' || elCSS.overflowY === 'visible');
    } else {
      canScrollX = width < scrollWidth && (elCSS.overflowX === 'auto' || elCSS.overflowX === 'scroll');
      canScrollY = height < scrollHeight && (elCSS.overflowY === 'auto' || elCSS.overflowY === 'scroll');
    }

    var vx = canScrollX && (Math.abs(right - x) <= sens && scrollPosX + width < scrollWidth) - (Math.abs(left - x) <= sens && !!scrollPosX);
    var vy = canScrollY && (Math.abs(bottom - y) <= sens && scrollPosY + height < scrollHeight) - (Math.abs(top - y) <= sens && !!scrollPosY);

    if (!autoScrolls[layersOut]) {
      for (var i = 0; i <= layersOut; i++) {
        if (!autoScrolls[i]) {
          autoScrolls[i] = {};
        }
      }
    }

    if (autoScrolls[layersOut].vx != vx || autoScrolls[layersOut].vy != vy || autoScrolls[layersOut].el !== el) {
      autoScrolls[layersOut].el = el;
      autoScrolls[layersOut].vx = vx;
      autoScrolls[layersOut].vy = vy;
      clearInterval(autoScrolls[layersOut].pid);

      if (vx != 0 || vy != 0) {
        scrollThisInstance = true;
        /* jshint loopfunc:true */

        autoScrolls[layersOut].pid = setInterval(function () {
          // emulate drag over during autoscroll (fallback), emulating native DnD behaviour
          if (isFallback && this.layer === 0) {
            Sortable.active._onTouchMove(touchEvt$1); // To move ghost if it is positioned absolutely

          }

          var scrollOffsetY = autoScrolls[this.layer].vy ? autoScrolls[this.layer].vy * speed : 0;
          var scrollOffsetX = autoScrolls[this.layer].vx ? autoScrolls[this.layer].vx * speed : 0;

          if (typeof scrollCustomFn === 'function') {
            if (scrollCustomFn.call(Sortable.dragged.parentNode[expando], scrollOffsetX, scrollOffsetY, evt, touchEvt$1, autoScrolls[this.layer].el) !== 'continue') {
              return;
            }
          }

          scrollBy(autoScrolls[this.layer].el, scrollOffsetX, scrollOffsetY);
        }.bind({
          layer: layersOut
        }), 24);
      }
    }

    layersOut++;
  } while (options.bubbleScroll && currentParent !== winScroller && (currentParent = getParentAutoScrollElement(currentParent, false)));

  scrolling = scrollThisInstance; // in case another function catches scrolling as false in between when it is not
}, 30);

var drop = function drop(_ref) {
  var originalEvent = _ref.originalEvent,
      putSortable = _ref.putSortable,
      dragEl = _ref.dragEl,
      activeSortable = _ref.activeSortable,
      dispatchSortableEvent = _ref.dispatchSortableEvent,
      hideGhostForTarget = _ref.hideGhostForTarget,
      unhideGhostForTarget = _ref.unhideGhostForTarget;
  if (!originalEvent) return;
  var toSortable = putSortable || activeSortable;
  hideGhostForTarget();
  var touch = originalEvent.changedTouches && originalEvent.changedTouches.length ? originalEvent.changedTouches[0] : originalEvent;
  var target = document.elementFromPoint(touch.clientX, touch.clientY);
  unhideGhostForTarget();

  if (toSortable && !toSortable.el.contains(target)) {
    dispatchSortableEvent('spill');
    this.onSpill({
      dragEl: dragEl,
      putSortable: putSortable
    });
  }
};

function Revert() {}

Revert.prototype = {
  startIndex: null,
  dragStart: function dragStart(_ref2) {
    var oldDraggableIndex = _ref2.oldDraggableIndex;
    this.startIndex = oldDraggableIndex;
  },
  onSpill: function onSpill(_ref3) {
    var dragEl = _ref3.dragEl,
        putSortable = _ref3.putSortable;
    this.sortable.captureAnimationState();

    if (putSortable) {
      putSortable.captureAnimationState();
    }

    var nextSibling = getChild(this.sortable.el, this.startIndex, this.options);

    if (nextSibling) {
      this.sortable.el.insertBefore(dragEl, nextSibling);
    } else {
      this.sortable.el.appendChild(dragEl);
    }

    this.sortable.animateAll();

    if (putSortable) {
      putSortable.animateAll();
    }
  },
  drop: drop
};

_extends(Revert, {
  pluginName: 'revertOnSpill'
});

function Remove() {}

Remove.prototype = {
  onSpill: function onSpill(_ref4) {
    var dragEl = _ref4.dragEl,
        putSortable = _ref4.putSortable;
    var parentSortable = putSortable || this.sortable;
    parentSortable.captureAnimationState();
    dragEl.parentNode && dragEl.parentNode.removeChild(dragEl);
    parentSortable.animateAll();
  },
  drop: drop
};

_extends(Remove, {
  pluginName: 'removeOnSpill'
});

Sortable.mount(new AutoScrollPlugin());
Sortable.mount(Remove, Revert);

const dom$i = new Dom();

class Select {
  constructor(dropDown) {
    const [toggler, menu] = dropDown.children;

    const handleClickOut = e => {
      if (!dropDown) {
        return document.removeEventListener('click', handleClickOut);
      }

      if (!dropDown.contains(e.target)) {
        // click outside
        this.toggle(false);
      }
    };

    const setValue = item => {
      const text = item.textContent;
      const val = item.getAttribute('data-value');
      toggler.textContent = text;
      this.value = val;
      this.toggle(false);
      dropDown.dispatchEvent(new Event('change'));
      toggler.focus();
      let elms = dom$i.elementChildren(menu);
      elms.forEach(elm => {
        dom$i.removeClass(elm, 'selected');
      });
      let optionselected = menu.querySelector('[data-value="' + val + '"]');

      if (optionselected) {
        dom$i.addClass(optionselected, 'selected');
      }
    };

    const handleItemKeyDown = e => {
      e.preventDefault();

      if (e.keyCode === 38 && e.target.previousElementSibling) {
        // up
        e.target.previousElementSibling.focus();
      } else if (e.keyCode === 40 && e.target.nextElementSibling) {
        // down
        e.target.nextElementSibling.focus();
      } else if (e.keyCode === 27) {
        // escape key
        this.toggle(false);
        toggler.focus();
      } else if (e.keyCode === 13 || e.keyCode === 32) {
        // enter or spacebar key
        setValue(e.target);
      }
    };

    const handleToggleKeyPress = e => {
      if (e.keyCode === 27) {
        // escape key
        this.toggle(false);
        e.preventDefault();
      } else if (e.keyCode === 13 || e.keyCode === 32) {
        // enter or spacebar key
        this.toggle(true);
        e.preventDefault();
      }
    };

    toggler.addEventListener('keydown', handleToggleKeyPress);
    toggler.addEventListener('click', () => this.toggle());
    [...menu.children].forEach(item => {
      item.addEventListener('keydown', handleItemKeyDown);
      item.addEventListener('click', () => setValue(item));
    });
    this.element = dropDown; // this.value = toggler.textContent;

    this.value = toggler.getAttribute('data-value');

    this.toggle = (expand = null) => {
      expand = expand === null ? menu.getAttribute('aria-expanded') !== 'true' : expand;
      menu.setAttribute('aria-expanded', expand);

      if (expand) {
        toggler.classList.add('active'); // menu.children[0].focus();

        let optionselected = menu.querySelector('[data-value="' + this.value + '"]');

        if (optionselected) {
          dom$i.addClass(optionselected, 'selected');
          optionselected.focus();
        }

        document.addEventListener('click', handleClickOut);
        dropDown.dispatchEvent(new Event('opened'));
      } else {
        toggler.classList.remove('active');
        dropDown.dispatchEvent(new Event('closed'));
        document.removeEventListener('click', handleClickOut);
      }
    };
  }

}

const renderSnippetPanel = builder => {
  const util = builder.util;
  const builderStuff = builder.builderStuff;
  const dom = builder.dom;
  let hideHandle = '';
  let sidePanel = builder.opts.sidePanel;

  if (builder.opts.snippetList === '#divSnippetList') {
    const html = `<div id="divSnippetList" class="is-side ${sidePanel === 'right' ? '' : ' fromleft'} snippetlist scroll-darker" tabindex="-1" role="dialog" aria-modal="true" title="${util.out('Snippets')}">
            </div>`;
    dom.appendHtml(builderStuff, html);

    if (!builder.opts.snippetHandle) {
      hideHandle = 'display:none;';
    }
  } else {
    hideHandle = 'display:none;';
    sidePanel = 'left';
  }

  let defaultcat = '';
  let defaultcatval = ''; // let catoptions = '';

  let catitems = '';

  if (builder.opts.emailMode) {
    defaultcatval = builder.opts.defaultEmailSnippetCategory;

    for (let i = 0; i < builder.opts.emailSnippetCategories.length; i++) {
      // catoptions +=  '<div role="button" tabindex="0" data-value="' + builder.opts.emailSnippetCategories[i][0] + '">' + builder.opts.emailSnippetCategories[i][1] + '</div>';
      catitems += '<li role="option" tabindex="0" data-value="' + builder.opts.emailSnippetCategories[i][0] + '">' + builder.opts.emailSnippetCategories[i][1] + '</li>';
      if (builder.opts.emailSnippetCategories[i][0] === builder.opts.defaultEmailSnippetCategory) defaultcat = builder.opts.emailSnippetCategories[i][1];
    }
  } else {
    defaultcatval = builder.opts.defaultSnippetCategory;

    for (let i = 0; i < builder.opts.snippetCategories.length; i++) {
      // catoptions += '<div role="button" tabindex="0" data-value="' + builder.opts.snippetCategories[i][0] + '">' + builder.opts.snippetCategories[i][1] + '</div>';
      catitems += '<li role="option" tabindex="0" data-value="' + builder.opts.snippetCategories[i][0] + '">' + builder.opts.snippetCategories[i][1] + '</li>';
      if (builder.opts.snippetCategories[i][0] === builder.opts.defaultSnippetCategory) defaultcat = builder.opts.snippetCategories[i][1];
    }
  }

  let html_snippets = '' + '<div class="is-dropdown selectsnippet" style="position:absolute;top:0;right:0;padding: 0;width:100%;z-index:2;">' + '<button tabindex="0" class="dropdown-toggle no-outline" title="' + util.out('Snippet Categories') + '" type="button" aria-haspopup="true" data-value="' + defaultcatval + '">' + '<span>' + defaultcat + '</span>' + '</button>' + '<ul class="dropdown-menu" role="listbox" title="' + util.out('Snippets') + '" aria-expanded="false">' + catitems + '</ul>' + '</div>' + (sidePanel === 'right' ? '<div id="divSnippetScrollUp" role="button" tabindex="-1" style="top:calc(50% - 27px);right:25px;">&#9650;</div>' + '<div id="divSnippetScrollDown" role="button" tabindex="-1" style="top:calc(50% + 27px);right:25px;">&#9660;</div>' + '<div id="divSnippetHandle" role="button" tabindex="-1" title="' + util.out('Snippets') + '" data-title="' + util.out('Snippets') + '" style="' + hideHandle + '">' + '<svg class="is-icon-flex"><use xlink:href="#ion-ios-arrow-left"></use></svg>' + '</div>' : '<div id="divSnippetScrollUp" role="button" tabindex="-1" style="top:calc(50% - 27px);left:10px;">&#9650;</div>' + '<div id="divSnippetScrollDown" role="button" tabindex="-1" style="top:calc(50% + 27px);left:10px;">&#9660;</div>' + '<div id="divSnippetHandle" role="button" tabindex="-1" title="' + util.out('Snippets') + '" data-title="' + util.out('Snippets') + '" style="' + hideHandle + '">' + '<svg class="is-icon-flex"><use xlink:href="#ion-ios-arrow-right"></use></svg>' + '</div>') + '<div class="is-design-list" tabindex="0">' + '</div>';
  let snippetPanel = document.querySelector(builder.opts.snippetList);
  dom.appendHtml(snippetPanel, html_snippets);
  const dropDown = new Select(snippetPanel.querySelector('.selectsnippet'));
  dropDown.element.addEventListener('change', () => {
    let cat = dropDown.value;
    let elms = snippetlist.querySelectorAll('.snippet-item');
    let exist = false;
    Array.prototype.forEach.call(elms, elm => {
      if (elm.getAttribute('data-cat') === cat) exist = true;
    });

    if (!exist) {
      builder.opts.snippetJSON.snippets.forEach(item => {
        if (item.category === cat) {
          dom.appendHtml(snippetlist, '<div class="snippet-item" data-id="' + item.id + '" data-cat="' + item.category + '"><img alt="' + util.out('Snippet') + '" alt="" src="' + snippetPath + item.thumbnail + '"><span class="is-overlay"></span></div>');
        }
      });
    }

    if (cat) {
      //let elms = snippetlist.querySelectorAll('.snippet-item');
      Array.prototype.forEach.call(elms, elm => {
        dom.addClass(elm, 'hide');
      });
      Array.prototype.forEach.call(elms, elm => {
        if (elm.getAttribute('data-cat') === cat) {
          elm.className = elm.className.replace(/hide/g, '');
        }
      });
    }
  });
  dropDown.toggle(false);

  if (builder.opts.snippetList === '#divSnippetList') {
    snippetPanel.querySelector('.selectsnippet').style.display = 'none';
    snippetPanel.querySelector('.is-design-list').style.display = 'none'; // Hide snippet panel on content click

    builder.doc.addEventListener('click', builder.doHideSnippetHandler = e => {
      e = e || window.event;
      var target = e.target || e.srcElement;

      if (builder.opts.snippetsSidebarDisplay === 'auto') {
        if (dom.hasClass(snippetPanel, 'active')) {
          // let a = dom.parentsHasAttribute(target, 'contenteditable');
          // let b = dom.parentsHasClass(target, 'is-builder'); // builder area
          // if(a||b) {
          //     hideSnippets(builder);
          // }
          let a = dom.parentsHasClass(target, 'is-builder'); // builder area

          if (a) {
            hideSnippets(builder);
          }
        }
      }
    }, false);
  }

  const snippetlist = document.querySelector('.is-design-list');
  let snippetPath = builder.opts.snippetPath;
  /*
  Hide slider snippets (backward compatible)
  let bHideSliderSnippet = false;
  try{
      if (typeof jQuery.fn.slick === 'undefined') {
          bHideSliderSnippet = true;
      }
  } catch(e){
      bHideSliderSnippet = true;
  }
  */

  let snippets = builder.opts.snippetJSON.snippets;
  let slider = builder.opts.slider;

  if (typeof slider !== 'undefined' && slider !== null) {
    if (slider === 'slick') {
      //remove glide
      const predicate = item => item.type !== 'glide';

      builder.opts.snippetJSON.snippets = snippets.filter(predicate);
    } else if (slider === 'glide') {
      //remove slick
      const predicate = item => item.type !== 'slick';

      builder.opts.snippetJSON.snippets = snippets.filter(predicate);
    } else if (slider === 'all') ; else {
      // remove all slider (if incorrect settings)
      const predicate = item => item.type !== 'glide' && item.type !== 'slick';

      builder.opts.snippetJSON.snippets = snippets.filter(predicate);
    } // for (let i = 0; i < builder.opts.snippetJSON.snippets.length; i++) {
    //     console.log(builder.opts.snippetJSON.snippets[i].type)
    // }

  } else {
    // Backward compatible OR if slider param not set
    // Hide slider snippet if slick is not included
    var bHideSliderSnippet = true;

    if (builder.win.jQuery) {
      if (builder.win.jQuery.fn.slick) {
        bHideSliderSnippet = false;
      }
    }

    if (bHideSliderSnippet) {
      const result = snippets.filter(item => {
        return item.type !== 'slick';
      });
      snippets = [...result];
    }

    if (!builder.win.Glide) {
      const result = snippets.filter(item => {
        return !(item.glide || item.type === 'glide');
      });
      snippets = [...result];
    } // for (let i = 0; i < snippets.length; i++) {
    //     console.log(snippets[i].type)
    // }


    builder.opts.snippetJSON.snippets = [...snippets];
  }

  let index = 1;

  if (builder.opts.emailMode) {
    builder.opts.snippetJSON.snippets.forEach(item => {
      item.id = index; //Give id to each record

      if (item.category === builder.opts.defaultEmailSnippetCategory + '') {
        dom.appendHtml(snippetlist, '<div class="snippet-item" data-id="' + item.id + '" data-cat="' + item.category + '"><img alt="' + util.out('Snippet') + '" src="' + snippetPath + item.thumbnail + '"><span class="is-overlay"></span></div>');
      }

      index++;
    });
  } else {
    builder.opts.snippetJSON.snippets.forEach(item => {
      item.id = index; //Give id to each record

      if (item.category === builder.opts.defaultSnippetCategory + '') {
        dom.appendHtml(snippetlist, '<div class="snippet-item" data-id="' + item.id + '" data-cat="' + item.category + '"><img alt="' + util.out('Snippet') + '" src="' + snippetPath + item.thumbnail + '"><span class="is-overlay"></span></div>');
      }

      index++;
    });
  }
  /*
  let userAgentString = navigator.userAgent; 
  let safariAgent = userAgentString.indexOf('Safari') > -1; 
  let chromeAgent = userAgentString.indexOf('Chrome') > -1; 
  if ((chromeAgent) && (safariAgent)) safariAgent = false;
  */
  // let safariAgent = false;


  let activeBuilderArea;
  new Sortable(snippetlist, {
    // forceFallback: safariAgent,
    group: {
      name: 'shared',
      pull: 'clone',
      put: false // Do not allow items to be put into this list

    },
    sort: false,
    animation: 150,
    onChoose: () => {
      // Adjust temmporary
      const newCss = `
            <style id="css-scale">
                .sortable-ghost {
                    height: 50px;
                }
                .sortable-ghost * {
                    display: none
                }
            </style>
            `;

      if (builder.iframe) {
        const oldCss = builder.contentStuff.querySelector('#css-scale');
        if (oldCss) oldCss.parentNode.removeChild(oldCss);
        builder.contentStuff.insertAdjacentHTML('afterbegin', newCss);
      } else {
        const oldCss = builder.builderStuff.querySelector('#css-scale');
        if (oldCss) oldCss.parentNode.removeChild(oldCss);
        builder.builderStuff.insertAdjacentHTML('afterbegin', newCss);
      }

      builder.sectionDropSetup();
    },
    onMove: () => {
      let emptyinfo = builder.doc.querySelector('.row-add-initial'); // if there is empty info, remove it during snippet drag drop
      // if(emptyinfo) emptyinfo.parentNode.removeChild(emptyinfo);

      if (emptyinfo) emptyinfo.style.display = 'none'; // if(builder.sortableOnPage) if(evt.related.getBoundingClientRect().top<0 ||
      //     evt.related.getBoundingClientRect().top>window.innerHeight) {
      //     return false;
      // }
    },
    onStart: () => {
      // Remove .builder-active during dragging (because this class makes rows have border-right/left 120px)
      activeBuilderArea = null;
      const area = builder.doc.querySelector('.builder-active');

      if (area) {
        activeBuilderArea = area;
        dom.removeClass(activeBuilderArea, 'builder-active');
      }

      builder.uo.saveForUndo(); // Even if cancelled, saveForUndo will make sure not to save if there is no change 

      let elm = document.querySelector('.is-sidebar-overlay');
      if (elm) elm.style.display = 'none'; // LATER: ContentBox
    },
    onEnd: () => {
      let elm = document.querySelector('.is-sidebar-overlay');
      if (elm) elm.style.display = 'block'; // LATER: ContentBox

      util.checkEmpty(); // In case container is still empty (drag drop snippet cancelled)

      let emptyinfo = builder.doc.querySelector('.row-add-initial');
      if (emptyinfo) emptyinfo.style.display = ''; // Return back the .builder-active

      if (activeBuilderArea) {
        dom.addClass(activeBuilderArea, 'builder-active');
      }

      if (builder.sortableOnPage) builder.sortableOnPage.destroy();
    }
  });

  if (builder.opts.snippetList === '#divSnippetList') {
    const snippethandle = snippetPanel.querySelector('#divSnippetHandle');
    dom.addEventListener(snippethandle, 'click', () => {
      toggleSnippets(builder);
      util.clearActiveCell();
      util.clearControls();
    });
    const viewportWidth = window.innerWidth;

    if (builder.opts.snippetOpen && viewportWidth >= 960) {
      snippetPanel.style.cssText = 'transition: all ease 0.8s;';
      setTimeout(function () {
        toggleSnippets(builder);
      }, 100);
      setTimeout(function () {
        snippetPanel.style.cssText = '';
      }, 1300);
    }
  } //Scroll helper


  let scrollup = snippetPanel.querySelector('#divSnippetScrollUp');
  let scrolldown = snippetPanel.querySelector('#divSnippetScrollDown'); // if ((navigator.userAgent.match(/iPhone/i)) || (navigator.userAgent.match(/iPod/i)) || (navigator.userAgent.match(/iPad/i))) {
  // } else {
  //     scrollup.style.display = 'none';
  //     scrolldown.style.display = 'none';
  // }

  scrollup.style.display = 'none';
  scrolldown.style.display = 'none';
  /*
  TODO
  
  var maxScroll=100000000;       
  jQuery('#divSnippetScrollUp').css('display','none');
  jQuery('#divSnippetScrollUp').on("click touchup", function(e) { 
      jQuery(".is-design-list").animate({ scrollTop: (jQuery(".is-design-list").scrollTop() - (jQuery(".is-design-list").height()-150) ) + "px" },300, function(){
          if(jQuery(".is-design-list").scrollTop()!=0){
              jQuery('#divSnippetScrollUp').fadeIn(300);
          } else {
                  jQuery('#divSnippetScrollUp').fadeOut(300);
          }
          if(jQuery(".is-design-list").scrollTop() != maxScroll){
              jQuery('#divSnippetScrollDown').fadeIn(300);
          } else {
                  jQuery('#divSnippetScrollDown').fadeOut(300);
          }  
      });           
       e.preventDefault();
      e.stopImmediatePropagation();
      return false;
  });            
  jQuery('#divSnippetScrollDown').on("click touchup", function(e) {                         
      jQuery(".is-design-list").animate({ scrollTop: (jQuery(".is-design-list").scrollTop() + (jQuery(".is-design-list").height()-150) ) + "px" }, 300, function() {
          if(jQuery(".is-design-list").scrollTop()!=0){
              jQuery('#divSnippetScrollUp').fadeIn(300);
          } else {
              jQuery('#divSnippetScrollUp').fadeOut(300);
      
          }
          if(maxScroll===100000000){
              maxScroll = jQuery('.is-design-list').prop('scrollHeight') - jQuery('.is-design-list').height() - 30;
          }  
          
          if(jQuery(".is-design-list").scrollTop() != maxScroll){
              jQuery('#divSnippetScrollDown').fadeIn(300);
          } else {
              jQuery('#divSnippetScrollDown').fadeOut(300);
          }  
      });
       e.preventDefault();
      e.stopImmediatePropagation();
      return false;
  });
  */
};

function toggleSnippets(builder) {
  const dom = builder.dom;
  let snippetPanel = document.querySelector('#divSnippetList');
  const snippethandle = snippetPanel.querySelector('#divSnippetHandle');

  if (dom.hasClass(snippetPanel, 'active')) {
    dom.removeClass(snippetPanel, 'active');

    if (builder.opts.sidePanel === 'right') {
      snippethandle.innerHTML = '<svg class="is-icon-flex" style="width:17px;height:17px;"><use xlink:href="#ion-ios-arrow-left"></use></svg>';
    } else {
      snippethandle.innerHTML = '<svg class="is-icon-flex" style="width:17px;height:17px;"><use xlink:href="#ion-ios-arrow-right"></use></svg>';
    }

    setTimeout(() => {
      snippetPanel.querySelector('.selectsnippet').style.display = 'none';
      snippetPanel.querySelector('.is-design-list').style.display = 'none';
    }, 300);
  } else {
    snippetPanel.querySelector('.selectsnippet').style.display = 'block';
    snippetPanel.querySelector('.is-design-list').style.display = 'block';
    setTimeout(() => {
      dom.addClass(snippetPanel, 'active');

      if (builder.opts.sidePanel === 'right') {
        snippethandle.innerHTML = '<svg class="is-icon-flex" style="width:17px;height:17px;"><use xlink:href="#ion-ios-arrow-right"></use></svg>';
      } else {
        snippethandle.innerHTML = '<svg class="is-icon-flex" style="width:17px;height:17px;"><use xlink:href="#ion-ios-arrow-left"></use></svg>';
      }
    }, 10);
  }
}

function hideSnippets(builder) {
  const dom = builder.dom;
  let snippetPanel = document.querySelector('#divSnippetList');
  const snippethandle = snippetPanel.querySelector('#divSnippetHandle');
  dom.removeClass(snippetPanel, 'active');

  if (builder.opts.sidePanel === 'right') {
    snippethandle.innerHTML = '<svg class="is-icon-flex" style="width:17px;height:17px;"><use xlink:href="#ion-ios-arrow-left"></use></svg>';
  } else {
    snippethandle.innerHTML = '<svg class="is-icon-flex" style="width:17px;height:17px;"><use xlink:href="#ion-ios-arrow-right"></use></svg>';
  }

  setTimeout(() => {
    snippetPanel.querySelector('.selectsnippet').style.display = 'none';
    snippetPanel.querySelector('.is-design-list').style.display = 'none';
  }, 300);
}

/*!
 * Cropper.js v1.5.12
 * https://fengyuanchen.github.io/cropperjs
 *
 * Copyright 2015-present Chen Fengyuan
 * Released under the MIT license
 *
 * Date: 2021-06-12T08:00:17.411Z
 */

function ownKeys(object, enumerableOnly) {
  var keys = Object.keys(object);

  if (Object.getOwnPropertySymbols) {
    var symbols = Object.getOwnPropertySymbols(object);

    if (enumerableOnly) {
      symbols = symbols.filter(function (sym) {
        return Object.getOwnPropertyDescriptor(object, sym).enumerable;
      });
    }

    keys.push.apply(keys, symbols);
  }

  return keys;
}

function _objectSpread2(target) {
  for (var i = 1; i < arguments.length; i++) {
    var source = arguments[i] != null ? arguments[i] : {};

    if (i % 2) {
      ownKeys(Object(source), true).forEach(function (key) {
        _defineProperty(target, key, source[key]);
      });
    } else if (Object.getOwnPropertyDescriptors) {
      Object.defineProperties(target, Object.getOwnPropertyDescriptors(source));
    } else {
      ownKeys(Object(source)).forEach(function (key) {
        Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key));
      });
    }
  }

  return target;
}

function _typeof(obj) {
  "@babel/helpers - typeof";

  if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
    _typeof = function (obj) {
      return typeof obj;
    };
  } else {
    _typeof = function (obj) {
      return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
    };
  }

  return _typeof(obj);
}

function _classCallCheck(instance, Constructor) {
  if (!(instance instanceof Constructor)) {
    throw new TypeError("Cannot call a class as a function");
  }
}

function _defineProperties(target, props) {
  for (var i = 0; i < props.length; i++) {
    var descriptor = props[i];
    descriptor.enumerable = descriptor.enumerable || false;
    descriptor.configurable = true;
    if ("value" in descriptor) descriptor.writable = true;
    Object.defineProperty(target, descriptor.key, descriptor);
  }
}

function _createClass(Constructor, protoProps, staticProps) {
  if (protoProps) _defineProperties(Constructor.prototype, protoProps);
  if (staticProps) _defineProperties(Constructor, staticProps);
  return Constructor;
}

function _defineProperty(obj, key, value) {
  if (key in obj) {
    Object.defineProperty(obj, key, {
      value: value,
      enumerable: true,
      configurable: true,
      writable: true
    });
  } else {
    obj[key] = value;
  }

  return obj;
}

function _toConsumableArray(arr) {
  return _arrayWithoutHoles(arr) || _iterableToArray(arr) || _unsupportedIterableToArray(arr) || _nonIterableSpread();
}

function _arrayWithoutHoles(arr) {
  if (Array.isArray(arr)) return _arrayLikeToArray(arr);
}

function _iterableToArray(iter) {
  if (typeof Symbol !== "undefined" && iter[Symbol.iterator] != null || iter["@@iterator"] != null) return Array.from(iter);
}

function _unsupportedIterableToArray(o, minLen) {
  if (!o) return;
  if (typeof o === "string") return _arrayLikeToArray(o, minLen);
  var n = Object.prototype.toString.call(o).slice(8, -1);
  if (n === "Object" && o.constructor) n = o.constructor.name;
  if (n === "Map" || n === "Set") return Array.from(o);
  if (n === "Arguments" || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(n)) return _arrayLikeToArray(o, minLen);
}

function _arrayLikeToArray(arr, len) {
  if (len == null || len > arr.length) len = arr.length;

  for (var i = 0, arr2 = new Array(len); i < len; i++) arr2[i] = arr[i];

  return arr2;
}

function _nonIterableSpread() {
  throw new TypeError("Invalid attempt to spread non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.");
}

var IS_BROWSER = typeof window !== 'undefined' && typeof window.document !== 'undefined';
var WINDOW = IS_BROWSER ? window : {};
var IS_TOUCH_DEVICE = IS_BROWSER && WINDOW.document.documentElement ? 'ontouchstart' in WINDOW.document.documentElement : false;
var HAS_POINTER_EVENT = IS_BROWSER ? 'PointerEvent' in WINDOW : false;
var NAMESPACE = 'cropper'; // Actions

var ACTION_ALL = 'all';
var ACTION_CROP = 'crop';
var ACTION_MOVE = 'move';
var ACTION_ZOOM = 'zoom';
var ACTION_EAST = 'e';
var ACTION_WEST = 'w';
var ACTION_SOUTH = 's';
var ACTION_NORTH = 'n';
var ACTION_NORTH_EAST = 'ne';
var ACTION_NORTH_WEST = 'nw';
var ACTION_SOUTH_EAST = 'se';
var ACTION_SOUTH_WEST = 'sw'; // Classes

var CLASS_CROP = "".concat(NAMESPACE, "-crop");
var CLASS_DISABLED = "".concat(NAMESPACE, "-disabled");
var CLASS_HIDDEN = "".concat(NAMESPACE, "-hidden");
var CLASS_HIDE = "".concat(NAMESPACE, "-hide");
var CLASS_INVISIBLE = "".concat(NAMESPACE, "-invisible");
var CLASS_MODAL = "".concat(NAMESPACE, "-modal");
var CLASS_MOVE = "".concat(NAMESPACE, "-move"); // Data keys

var DATA_ACTION = "".concat(NAMESPACE, "Action");
var DATA_PREVIEW = "".concat(NAMESPACE, "Preview"); // Drag modes

var DRAG_MODE_CROP = 'crop';
var DRAG_MODE_MOVE = 'move';
var DRAG_MODE_NONE = 'none'; // Events

var EVENT_CROP = 'crop';
var EVENT_CROP_END = 'cropend';
var EVENT_CROP_MOVE = 'cropmove';
var EVENT_CROP_START = 'cropstart';
var EVENT_DBLCLICK = 'dblclick';
var EVENT_TOUCH_START = IS_TOUCH_DEVICE ? 'touchstart' : 'mousedown';
var EVENT_TOUCH_MOVE = IS_TOUCH_DEVICE ? 'touchmove' : 'mousemove';
var EVENT_TOUCH_END = IS_TOUCH_DEVICE ? 'touchend touchcancel' : 'mouseup';
var EVENT_POINTER_DOWN = HAS_POINTER_EVENT ? 'pointerdown' : EVENT_TOUCH_START;
var EVENT_POINTER_MOVE = HAS_POINTER_EVENT ? 'pointermove' : EVENT_TOUCH_MOVE;
var EVENT_POINTER_UP = HAS_POINTER_EVENT ? 'pointerup pointercancel' : EVENT_TOUCH_END;
var EVENT_READY = 'ready';
var EVENT_RESIZE = 'resize';
var EVENT_WHEEL = 'wheel';
var EVENT_ZOOM = 'zoom'; // Mime types

var MIME_TYPE_JPEG = 'image/jpeg'; // RegExps

var REGEXP_ACTIONS = /^e|w|s|n|se|sw|ne|nw|all|crop|move|zoom$/;
var REGEXP_DATA_URL = /^data:/;
var REGEXP_DATA_URL_JPEG = /^data:image\/jpeg;base64,/;
var REGEXP_TAG_NAME = /^img|canvas$/i; // Misc
// Inspired by the default width and height of a canvas element.

var MIN_CONTAINER_WIDTH = 200;
var MIN_CONTAINER_HEIGHT = 100;

var DEFAULTS = {
  // Define the view mode of the cropper
  viewMode: 0,
  // 0, 1, 2, 3
  // Define the dragging mode of the cropper
  dragMode: DRAG_MODE_CROP,
  // 'crop', 'move' or 'none'
  // Define the initial aspect ratio of the crop box
  initialAspectRatio: NaN,
  // Define the aspect ratio of the crop box
  aspectRatio: NaN,
  // An object with the previous cropping result data
  data: null,
  // A selector for adding extra containers to preview
  preview: '',
  // Re-render the cropper when resize the window
  responsive: true,
  // Restore the cropped area after resize the window
  restore: true,
  // Check if the current image is a cross-origin image
  checkCrossOrigin: true,
  // Check the current image's Exif Orientation information
  checkOrientation: true,
  // Show the black modal
  modal: true,
  // Show the dashed lines for guiding
  guides: true,
  // Show the center indicator for guiding
  center: true,
  // Show the white modal to highlight the crop box
  highlight: true,
  // Show the grid background
  background: true,
  // Enable to crop the image automatically when initialize
  autoCrop: true,
  // Define the percentage of automatic cropping area when initializes
  autoCropArea: 0.8,
  // Enable to move the image
  movable: true,
  // Enable to rotate the image
  rotatable: true,
  // Enable to scale the image
  scalable: true,
  // Enable to zoom the image
  zoomable: true,
  // Enable to zoom the image by dragging touch
  zoomOnTouch: true,
  // Enable to zoom the image by wheeling mouse
  zoomOnWheel: true,
  // Define zoom ratio when zoom the image by wheeling mouse
  wheelZoomRatio: 0.1,
  // Enable to move the crop box
  cropBoxMovable: true,
  // Enable to resize the crop box
  cropBoxResizable: true,
  // Toggle drag mode between "crop" and "move" when click twice on the cropper
  toggleDragModeOnDblclick: true,
  // Size limitation
  minCanvasWidth: 0,
  minCanvasHeight: 0,
  minCropBoxWidth: 0,
  minCropBoxHeight: 0,
  minContainerWidth: MIN_CONTAINER_WIDTH,
  minContainerHeight: MIN_CONTAINER_HEIGHT,
  // Shortcuts of events
  ready: null,
  cropstart: null,
  cropmove: null,
  cropend: null,
  crop: null,
  zoom: null
};

var TEMPLATE = '<div class="cropper-container" touch-action="none">' + '<div class="cropper-wrap-box">' + '<div class="cropper-canvas"></div>' + '</div>' + '<div class="cropper-drag-box"></div>' + '<div class="cropper-crop-box">' + '<span class="cropper-view-box"></span>' + '<span class="cropper-dashed dashed-h"></span>' + '<span class="cropper-dashed dashed-v"></span>' + '<span class="cropper-center"></span>' + '<span class="cropper-face"></span>' + '<span class="cropper-line line-e" data-cropper-action="e"></span>' + '<span class="cropper-line line-n" data-cropper-action="n"></span>' + '<span class="cropper-line line-w" data-cropper-action="w"></span>' + '<span class="cropper-line line-s" data-cropper-action="s"></span>' + '<span class="cropper-point point-e" data-cropper-action="e"></span>' + '<span class="cropper-point point-n" data-cropper-action="n"></span>' + '<span class="cropper-point point-w" data-cropper-action="w"></span>' + '<span class="cropper-point point-s" data-cropper-action="s"></span>' + '<span class="cropper-point point-ne" data-cropper-action="ne"></span>' + '<span class="cropper-point point-nw" data-cropper-action="nw"></span>' + '<span class="cropper-point point-sw" data-cropper-action="sw"></span>' + '<span class="cropper-point point-se" data-cropper-action="se"></span>' + '</div>' + '</div>';

/**
 * Check if the given value is not a number.
 */

var isNaN$1 = Number.isNaN || WINDOW.isNaN;
/**
 * Check if the given value is a number.
 * @param {*} value - The value to check.
 * @returns {boolean} Returns `true` if the given value is a number, else `false`.
 */

function isNumber$1(value) {
  return typeof value === 'number' && !isNaN$1(value);
}
/**
 * Check if the given value is a positive number.
 * @param {*} value - The value to check.
 * @returns {boolean} Returns `true` if the given value is a positive number, else `false`.
 */

var isPositiveNumber = function isPositiveNumber(value) {
  return value > 0 && value < Infinity;
};
/**
 * Check if the given value is undefined.
 * @param {*} value - The value to check.
 * @returns {boolean} Returns `true` if the given value is undefined, else `false`.
 */

function isUndefined$1(value) {
  return typeof value === 'undefined';
}
/**
 * Check if the given value is an object.
 * @param {*} value - The value to check.
 * @returns {boolean} Returns `true` if the given value is an object, else `false`.
 */

function isObject$1(value) {
  return _typeof(value) === 'object' && value !== null;
}
var hasOwnProperty = Object.prototype.hasOwnProperty;
/**
 * Check if the given value is a plain object.
 * @param {*} value - The value to check.
 * @returns {boolean} Returns `true` if the given value is a plain object, else `false`.
 */

function isPlainObject(value) {
  if (!isObject$1(value)) {
    return false;
  }

  try {
    var _constructor = value.constructor;
    var prototype = _constructor.prototype;
    return _constructor && prototype && hasOwnProperty.call(prototype, 'isPrototypeOf');
  } catch (error) {
    return false;
  }
}
/**
 * Check if the given value is a function.
 * @param {*} value - The value to check.
 * @returns {boolean} Returns `true` if the given value is a function, else `false`.
 */

function isFunction$1(value) {
  return typeof value === 'function';
}
var slice = Array.prototype.slice;
/**
 * Convert array-like or iterable object to an array.
 * @param {*} value - The value to convert.
 * @returns {Array} Returns a new array.
 */

function toArray(value) {
  return Array.from ? Array.from(value) : slice.call(value);
}
/**
 * Iterate the given data.
 * @param {*} data - The data to iterate.
 * @param {Function} callback - The process function for each element.
 * @returns {*} The original data.
 */

function forEach(data, callback) {
  if (data && isFunction$1(callback)) {
    if (Array.isArray(data) || isNumber$1(data.length)
    /* array-like */
    ) {
        toArray(data).forEach(function (value, key) {
          callback.call(data, value, key, data);
        });
      } else if (isObject$1(data)) {
      Object.keys(data).forEach(function (key) {
        callback.call(data, data[key], key, data);
      });
    }
  }

  return data;
}
/**
 * Extend the given object.
 * @param {*} target - The target object to extend.
 * @param {*} args - The rest objects for merging to the target object.
 * @returns {Object} The extended object.
 */

var assign = Object.assign || function assign(target) {
  for (var _len = arguments.length, args = new Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
    args[_key - 1] = arguments[_key];
  }

  if (isObject$1(target) && args.length > 0) {
    args.forEach(function (arg) {
      if (isObject$1(arg)) {
        Object.keys(arg).forEach(function (key) {
          target[key] = arg[key];
        });
      }
    });
  }

  return target;
};
var REGEXP_DECIMALS = /\.\d*(?:0|9){12}\d*$/;
/**
 * Normalize decimal number.
 * Check out {@link https://0.30000000000000004.com/}
 * @param {number} value - The value to normalize.
 * @param {number} [times=100000000000] - The times for normalizing.
 * @returns {number} Returns the normalized number.
 */

function normalizeDecimalNumber(value) {
  var times = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 100000000000;
  return REGEXP_DECIMALS.test(value) ? Math.round(value * times) / times : value;
}
var REGEXP_SUFFIX = /^width|height|left|top|marginLeft|marginTop$/;
/**
 * Apply styles to the given element.
 * @param {Element} element - The target element.
 * @param {Object} styles - The styles for applying.
 */

function setStyle(element, styles) {
  var style = element.style;
  forEach(styles, function (value, property) {
    if (REGEXP_SUFFIX.test(property) && isNumber$1(value)) {
      value = "".concat(value, "px");
    }

    style[property] = value;
  });
}
/**
 * Check if the given element has a special class.
 * @param {Element} element - The element to check.
 * @param {string} value - The class to search.
 * @returns {boolean} Returns `true` if the special class was found.
 */

function hasClass$1(element, value) {
  return element.classList ? element.classList.contains(value) : element.className.indexOf(value) > -1;
}
/**
 * Add classes to the given element.
 * @param {Element} element - The target element.
 * @param {string} value - The classes to be added.
 */

function addClass$1(element, value) {
  if (!value) {
    return;
  }

  if (isNumber$1(element.length)) {
    forEach(element, function (elem) {
      addClass$1(elem, value);
    });
    return;
  }

  if (element.classList) {
    element.classList.add(value);
    return;
  }

  var className = element.className.trim();

  if (!className) {
    element.className = value;
  } else if (className.indexOf(value) < 0) {
    element.className = "".concat(className, " ").concat(value);
  }
}
/**
 * Remove classes from the given element.
 * @param {Element} element - The target element.
 * @param {string} value - The classes to be removed.
 */

function removeClass$1(element, value) {
  if (!value) {
    return;
  }

  if (isNumber$1(element.length)) {
    forEach(element, function (elem) {
      removeClass$1(elem, value);
    });
    return;
  }

  if (element.classList) {
    element.classList.remove(value);
    return;
  }

  if (element.className.indexOf(value) >= 0) {
    element.className = element.className.replace(value, '');
  }
}
/**
 * Add or remove classes from the given element.
 * @param {Element} element - The target element.
 * @param {string} value - The classes to be toggled.
 * @param {boolean} added - Add only.
 */

function toggleClass(element, value, added) {
  if (!value) {
    return;
  }

  if (isNumber$1(element.length)) {
    forEach(element, function (elem) {
      toggleClass(elem, value, added);
    });
    return;
  } // IE10-11 doesn't support the second parameter of `classList.toggle`


  if (added) {
    addClass$1(element, value);
  } else {
    removeClass$1(element, value);
  }
}
var REGEXP_CAMEL_CASE = /([a-z\d])([A-Z])/g;
/**
 * Transform the given string from camelCase to kebab-case
 * @param {string} value - The value to transform.
 * @returns {string} The transformed value.
 */

function toParamCase(value) {
  return value.replace(REGEXP_CAMEL_CASE, '$1-$2').toLowerCase();
}
/**
 * Get data from the given element.
 * @param {Element} element - The target element.
 * @param {string} name - The data key to get.
 * @returns {string} The data value.
 */

function getData(element, name) {
  if (isObject$1(element[name])) {
    return element[name];
  }

  if (element.dataset) {
    return element.dataset[name];
  }

  return element.getAttribute("data-".concat(toParamCase(name)));
}
/**
 * Set data to the given element.
 * @param {Element} element - The target element.
 * @param {string} name - The data key to set.
 * @param {string} data - The data value.
 */

function setData(element, name, data) {
  if (isObject$1(data)) {
    element[name] = data;
  } else if (element.dataset) {
    element.dataset[name] = data;
  } else {
    element.setAttribute("data-".concat(toParamCase(name)), data);
  }
}
/**
 * Remove data from the given element.
 * @param {Element} element - The target element.
 * @param {string} name - The data key to remove.
 */

function removeData(element, name) {
  if (isObject$1(element[name])) {
    try {
      delete element[name];
    } catch (error) {
      element[name] = undefined;
    }
  } else if (element.dataset) {
    // #128 Safari not allows to delete dataset property
    try {
      delete element.dataset[name];
    } catch (error) {
      element.dataset[name] = undefined;
    }
  } else {
    element.removeAttribute("data-".concat(toParamCase(name)));
  }
}
var REGEXP_SPACES = /\s\s*/;

var onceSupported = function () {
  var supported = false;

  if (IS_BROWSER) {
    var once = false;

    var listener = function listener() {};

    var options = Object.defineProperty({}, 'once', {
      get: function get() {
        supported = true;
        return once;
      },

      /**
       * This setter can fix a `TypeError` in strict mode
       * {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Errors/Getter_only}
       * @param {boolean} value - The value to set
       */
      set: function set(value) {
        once = value;
      }
    });
    WINDOW.addEventListener('test', listener, options);
    WINDOW.removeEventListener('test', listener, options);
  }

  return supported;
}();
/**
 * Remove event listener from the target element.
 * @param {Element} element - The event target.
 * @param {string} type - The event type(s).
 * @param {Function} listener - The event listener.
 * @param {Object} options - The event options.
 */


function removeListener(element, type, listener) {
  var options = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
  var handler = listener;
  type.trim().split(REGEXP_SPACES).forEach(function (event) {
    if (!onceSupported) {
      var listeners = element.listeners;

      if (listeners && listeners[event] && listeners[event][listener]) {
        handler = listeners[event][listener];
        delete listeners[event][listener];

        if (Object.keys(listeners[event]).length === 0) {
          delete listeners[event];
        }

        if (Object.keys(listeners).length === 0) {
          delete element.listeners;
        }
      }
    }

    element.removeEventListener(event, handler, options);
  });
}
/**
 * Add event listener to the target element.
 * @param {Element} element - The event target.
 * @param {string} type - The event type(s).
 * @param {Function} listener - The event listener.
 * @param {Object} options - The event options.
 */

function addListener(element, type, listener) {
  var options = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : {};
  var _handler = listener;
  type.trim().split(REGEXP_SPACES).forEach(function (event) {
    if (options.once && !onceSupported) {
      var _element$listeners = element.listeners,
          listeners = _element$listeners === void 0 ? {} : _element$listeners;

      _handler = function handler() {
        delete listeners[event][listener];
        element.removeEventListener(event, _handler, options);

        for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
          args[_key2] = arguments[_key2];
        }

        listener.apply(element, args);
      };

      if (!listeners[event]) {
        listeners[event] = {};
      }

      if (listeners[event][listener]) {
        element.removeEventListener(event, listeners[event][listener], options);
      }

      listeners[event][listener] = _handler;
      element.listeners = listeners;
    }

    element.addEventListener(event, _handler, options);
  });
}
/**
 * Dispatch event on the target element.
 * @param {Element} element - The event target.
 * @param {string} type - The event type(s).
 * @param {Object} data - The additional event data.
 * @returns {boolean} Indicate if the event is default prevented or not.
 */

function dispatchEvent(element, type, data) {
  var event; // Event and CustomEvent on IE9-11 are global objects, not constructors

  if (isFunction$1(Event) && isFunction$1(CustomEvent)) {
    event = new CustomEvent(type, {
      detail: data,
      bubbles: true,
      cancelable: true
    });
  } else {
    event = document.createEvent('CustomEvent');
    event.initCustomEvent(type, true, true, data);
  }

  return element.dispatchEvent(event);
}
/**
 * Get the offset base on the document.
 * @param {Element} element - The target element.
 * @returns {Object} The offset data.
 */

function getOffset(element) {
  var box = element.getBoundingClientRect();
  return {
    left: box.left + (window.pageXOffset - document.documentElement.clientLeft),
    top: box.top + (window.pageYOffset - document.documentElement.clientTop)
  };
}
var location$1 = WINDOW.location;
var REGEXP_ORIGINS = /^(\w+:)\/\/([^:/?#]*):?(\d*)/i;
/**
 * Check if the given URL is a cross origin URL.
 * @param {string} url - The target URL.
 * @returns {boolean} Returns `true` if the given URL is a cross origin URL, else `false`.
 */

function isCrossOriginURL(url) {
  var parts = url.match(REGEXP_ORIGINS);
  return parts !== null && (parts[1] !== location$1.protocol || parts[2] !== location$1.hostname || parts[3] !== location$1.port);
}
/**
 * Add timestamp to the given URL.
 * @param {string} url - The target URL.
 * @returns {string} The result URL.
 */

function addTimestamp(url) {
  var timestamp = "timestamp=".concat(new Date().getTime());
  return url + (url.indexOf('?') === -1 ? '?' : '&') + timestamp;
}
/**
 * Get transforms base on the given object.
 * @param {Object} obj - The target object.
 * @returns {string} A string contains transform values.
 */

function getTransforms(_ref) {
  var rotate = _ref.rotate,
      scaleX = _ref.scaleX,
      scaleY = _ref.scaleY,
      translateX = _ref.translateX,
      translateY = _ref.translateY;
  var values = [];

  if (isNumber$1(translateX) && translateX !== 0) {
    values.push("translateX(".concat(translateX, "px)"));
  }

  if (isNumber$1(translateY) && translateY !== 0) {
    values.push("translateY(".concat(translateY, "px)"));
  } // Rotate should come first before scale to match orientation transform


  if (isNumber$1(rotate) && rotate !== 0) {
    values.push("rotate(".concat(rotate, "deg)"));
  }

  if (isNumber$1(scaleX) && scaleX !== 1) {
    values.push("scaleX(".concat(scaleX, ")"));
  }

  if (isNumber$1(scaleY) && scaleY !== 1) {
    values.push("scaleY(".concat(scaleY, ")"));
  }

  var transform = values.length ? values.join(' ') : 'none';
  return {
    WebkitTransform: transform,
    msTransform: transform,
    transform: transform
  };
}
/**
 * Get the max ratio of a group of pointers.
 * @param {string} pointers - The target pointers.
 * @returns {number} The result ratio.
 */

function getMaxZoomRatio(pointers) {
  var pointers2 = _objectSpread2({}, pointers);

  var maxRatio = 0;
  forEach(pointers, function (pointer, pointerId) {
    delete pointers2[pointerId];
    forEach(pointers2, function (pointer2) {
      var x1 = Math.abs(pointer.startX - pointer2.startX);
      var y1 = Math.abs(pointer.startY - pointer2.startY);
      var x2 = Math.abs(pointer.endX - pointer2.endX);
      var y2 = Math.abs(pointer.endY - pointer2.endY);
      var z1 = Math.sqrt(x1 * x1 + y1 * y1);
      var z2 = Math.sqrt(x2 * x2 + y2 * y2);
      var ratio = (z2 - z1) / z1;

      if (Math.abs(ratio) > Math.abs(maxRatio)) {
        maxRatio = ratio;
      }
    });
  });
  return maxRatio;
}
/**
 * Get a pointer from an event object.
 * @param {Object} event - The target event object.
 * @param {boolean} endOnly - Indicates if only returns the end point coordinate or not.
 * @returns {Object} The result pointer contains start and/or end point coordinates.
 */

function getPointer(_ref2, endOnly) {
  var pageX = _ref2.pageX,
      pageY = _ref2.pageY;
  var end = {
    endX: pageX,
    endY: pageY
  };
  return endOnly ? end : _objectSpread2({
    startX: pageX,
    startY: pageY
  }, end);
}
/**
 * Get the center point coordinate of a group of pointers.
 * @param {Object} pointers - The target pointers.
 * @returns {Object} The center point coordinate.
 */

function getPointersCenter(pointers) {
  var pageX = 0;
  var pageY = 0;
  var count = 0;
  forEach(pointers, function (_ref3) {
    var startX = _ref3.startX,
        startY = _ref3.startY;
    pageX += startX;
    pageY += startY;
    count += 1;
  });
  pageX /= count;
  pageY /= count;
  return {
    pageX: pageX,
    pageY: pageY
  };
}
/**
 * Get the max sizes in a rectangle under the given aspect ratio.
 * @param {Object} data - The original sizes.
 * @param {string} [type='contain'] - The adjust type.
 * @returns {Object} The result sizes.
 */

function getAdjustedSizes(_ref4) // or 'cover'
{
  var aspectRatio = _ref4.aspectRatio,
      height = _ref4.height,
      width = _ref4.width;
  var type = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'contain';
  var isValidWidth = isPositiveNumber(width);
  var isValidHeight = isPositiveNumber(height);

  if (isValidWidth && isValidHeight) {
    var adjustedWidth = height * aspectRatio;

    if (type === 'contain' && adjustedWidth > width || type === 'cover' && adjustedWidth < width) {
      height = width / aspectRatio;
    } else {
      width = height * aspectRatio;
    }
  } else if (isValidWidth) {
    height = width / aspectRatio;
  } else if (isValidHeight) {
    width = height * aspectRatio;
  }

  return {
    width: width,
    height: height
  };
}
/**
 * Get the new sizes of a rectangle after rotated.
 * @param {Object} data - The original sizes.
 * @returns {Object} The result sizes.
 */

function getRotatedSizes(_ref5) {
  var width = _ref5.width,
      height = _ref5.height,
      degree = _ref5.degree;
  degree = Math.abs(degree) % 180;

  if (degree === 90) {
    return {
      width: height,
      height: width
    };
  }

  var arc = degree % 90 * Math.PI / 180;
  var sinArc = Math.sin(arc);
  var cosArc = Math.cos(arc);
  var newWidth = width * cosArc + height * sinArc;
  var newHeight = width * sinArc + height * cosArc;
  return degree > 90 ? {
    width: newHeight,
    height: newWidth
  } : {
    width: newWidth,
    height: newHeight
  };
}
/**
 * Get a canvas which drew the given image.
 * @param {HTMLImageElement} image - The image for drawing.
 * @param {Object} imageData - The image data.
 * @param {Object} canvasData - The canvas data.
 * @param {Object} options - The options.
 * @returns {HTMLCanvasElement} The result canvas.
 */

function getSourceCanvas(image, _ref6, _ref7, _ref8) {
  var imageAspectRatio = _ref6.aspectRatio,
      imageNaturalWidth = _ref6.naturalWidth,
      imageNaturalHeight = _ref6.naturalHeight,
      _ref6$rotate = _ref6.rotate,
      rotate = _ref6$rotate === void 0 ? 0 : _ref6$rotate,
      _ref6$scaleX = _ref6.scaleX,
      scaleX = _ref6$scaleX === void 0 ? 1 : _ref6$scaleX,
      _ref6$scaleY = _ref6.scaleY,
      scaleY = _ref6$scaleY === void 0 ? 1 : _ref6$scaleY;
  var aspectRatio = _ref7.aspectRatio,
      naturalWidth = _ref7.naturalWidth,
      naturalHeight = _ref7.naturalHeight;
  var _ref8$fillColor = _ref8.fillColor,
      fillColor = _ref8$fillColor === void 0 ? 'transparent' : _ref8$fillColor,
      _ref8$imageSmoothingE = _ref8.imageSmoothingEnabled,
      imageSmoothingEnabled = _ref8$imageSmoothingE === void 0 ? true : _ref8$imageSmoothingE,
      _ref8$imageSmoothingQ = _ref8.imageSmoothingQuality,
      imageSmoothingQuality = _ref8$imageSmoothingQ === void 0 ? 'low' : _ref8$imageSmoothingQ,
      _ref8$maxWidth = _ref8.maxWidth,
      maxWidth = _ref8$maxWidth === void 0 ? Infinity : _ref8$maxWidth,
      _ref8$maxHeight = _ref8.maxHeight,
      maxHeight = _ref8$maxHeight === void 0 ? Infinity : _ref8$maxHeight,
      _ref8$minWidth = _ref8.minWidth,
      minWidth = _ref8$minWidth === void 0 ? 0 : _ref8$minWidth,
      _ref8$minHeight = _ref8.minHeight,
      minHeight = _ref8$minHeight === void 0 ? 0 : _ref8$minHeight;
  var canvas = document.createElement('canvas');
  var context = canvas.getContext('2d');
  var maxSizes = getAdjustedSizes({
    aspectRatio: aspectRatio,
    width: maxWidth,
    height: maxHeight
  });
  var minSizes = getAdjustedSizes({
    aspectRatio: aspectRatio,
    width: minWidth,
    height: minHeight
  }, 'cover');
  var width = Math.min(maxSizes.width, Math.max(minSizes.width, naturalWidth));
  var height = Math.min(maxSizes.height, Math.max(minSizes.height, naturalHeight)); // Note: should always use image's natural sizes for drawing as
  // imageData.naturalWidth === canvasData.naturalHeight when rotate % 180 === 90

  var destMaxSizes = getAdjustedSizes({
    aspectRatio: imageAspectRatio,
    width: maxWidth,
    height: maxHeight
  });
  var destMinSizes = getAdjustedSizes({
    aspectRatio: imageAspectRatio,
    width: minWidth,
    height: minHeight
  }, 'cover');
  var destWidth = Math.min(destMaxSizes.width, Math.max(destMinSizes.width, imageNaturalWidth));
  var destHeight = Math.min(destMaxSizes.height, Math.max(destMinSizes.height, imageNaturalHeight));
  var params = [-destWidth / 2, -destHeight / 2, destWidth, destHeight];
  canvas.width = normalizeDecimalNumber(width);
  canvas.height = normalizeDecimalNumber(height);
  context.fillStyle = fillColor;
  context.fillRect(0, 0, width, height);
  context.save();
  context.translate(width / 2, height / 2);
  context.rotate(rotate * Math.PI / 180);
  context.scale(scaleX, scaleY);
  context.imageSmoothingEnabled = imageSmoothingEnabled;
  context.imageSmoothingQuality = imageSmoothingQuality;
  context.drawImage.apply(context, [image].concat(_toConsumableArray(params.map(function (param) {
    return Math.floor(normalizeDecimalNumber(param));
  }))));
  context.restore();
  return canvas;
}
var fromCharCode = String.fromCharCode;
/**
 * Get string from char code in data view.
 * @param {DataView} dataView - The data view for read.
 * @param {number} start - The start index.
 * @param {number} length - The read length.
 * @returns {string} The read result.
 */

function getStringFromCharCode(dataView, start, length) {
  var str = '';
  length += start;

  for (var i = start; i < length; i += 1) {
    str += fromCharCode(dataView.getUint8(i));
  }

  return str;
}
var REGEXP_DATA_URL_HEAD = /^data:.*,/;
/**
 * Transform Data URL to array buffer.
 * @param {string} dataURL - The Data URL to transform.
 * @returns {ArrayBuffer} The result array buffer.
 */

function dataURLToArrayBuffer(dataURL) {
  var base64 = dataURL.replace(REGEXP_DATA_URL_HEAD, '');
  var binary = atob(base64);
  var arrayBuffer = new ArrayBuffer(binary.length);
  var uint8 = new Uint8Array(arrayBuffer);
  forEach(uint8, function (value, i) {
    uint8[i] = binary.charCodeAt(i);
  });
  return arrayBuffer;
}
/**
 * Transform array buffer to Data URL.
 * @param {ArrayBuffer} arrayBuffer - The array buffer to transform.
 * @param {string} mimeType - The mime type of the Data URL.
 * @returns {string} The result Data URL.
 */

function arrayBufferToDataURL(arrayBuffer, mimeType) {
  var chunks = []; // Chunk Typed Array for better performance (#435)

  var chunkSize = 8192;
  var uint8 = new Uint8Array(arrayBuffer);

  while (uint8.length > 0) {
    // XXX: Babel's `toConsumableArray` helper will throw error in IE or Safari 9
    // eslint-disable-next-line prefer-spread
    chunks.push(fromCharCode.apply(null, toArray(uint8.subarray(0, chunkSize))));
    uint8 = uint8.subarray(chunkSize);
  }

  return "data:".concat(mimeType, ";base64,").concat(btoa(chunks.join('')));
}
/**
 * Get orientation value from given array buffer.
 * @param {ArrayBuffer} arrayBuffer - The array buffer to read.
 * @returns {number} The read orientation value.
 */

function resetAndGetOrientation(arrayBuffer) {
  var dataView = new DataView(arrayBuffer);
  var orientation; // Ignores range error when the image does not have correct Exif information

  try {
    var littleEndian;
    var app1Start;
    var ifdStart; // Only handle JPEG image (start by 0xFFD8)

    if (dataView.getUint8(0) === 0xFF && dataView.getUint8(1) === 0xD8) {
      var length = dataView.byteLength;
      var offset = 2;

      while (offset + 1 < length) {
        if (dataView.getUint8(offset) === 0xFF && dataView.getUint8(offset + 1) === 0xE1) {
          app1Start = offset;
          break;
        }

        offset += 1;
      }
    }

    if (app1Start) {
      var exifIDCode = app1Start + 4;
      var tiffOffset = app1Start + 10;

      if (getStringFromCharCode(dataView, exifIDCode, 4) === 'Exif') {
        var endianness = dataView.getUint16(tiffOffset);
        littleEndian = endianness === 0x4949;

        if (littleEndian || endianness === 0x4D4D
        /* bigEndian */
        ) {
            if (dataView.getUint16(tiffOffset + 2, littleEndian) === 0x002A) {
              var firstIFDOffset = dataView.getUint32(tiffOffset + 4, littleEndian);

              if (firstIFDOffset >= 0x00000008) {
                ifdStart = tiffOffset + firstIFDOffset;
              }
            }
          }
      }
    }

    if (ifdStart) {
      var _length = dataView.getUint16(ifdStart, littleEndian);

      var _offset;

      var i;

      for (i = 0; i < _length; i += 1) {
        _offset = ifdStart + i * 12 + 2;

        if (dataView.getUint16(_offset, littleEndian) === 0x0112
        /* Orientation */
        ) {
            // 8 is the offset of the current tag's value
            _offset += 8; // Get the original orientation value

            orientation = dataView.getUint16(_offset, littleEndian); // Override the orientation with its default value

            dataView.setUint16(_offset, 1, littleEndian);
            break;
          }
      }
    }
  } catch (error) {
    orientation = 1;
  }

  return orientation;
}
/**
 * Parse Exif Orientation value.
 * @param {number} orientation - The orientation to parse.
 * @returns {Object} The parsed result.
 */

function parseOrientation(orientation) {
  var rotate = 0;
  var scaleX = 1;
  var scaleY = 1;

  switch (orientation) {
    // Flip horizontal
    case 2:
      scaleX = -1;
      break;
    // Rotate left 180°

    case 3:
      rotate = -180;
      break;
    // Flip vertical

    case 4:
      scaleY = -1;
      break;
    // Flip vertical and rotate right 90°

    case 5:
      rotate = 90;
      scaleY = -1;
      break;
    // Rotate right 90°

    case 6:
      rotate = 90;
      break;
    // Flip horizontal and rotate right 90°

    case 7:
      rotate = 90;
      scaleX = -1;
      break;
    // Rotate left 90°

    case 8:
      rotate = -90;
      break;
  }

  return {
    rotate: rotate,
    scaleX: scaleX,
    scaleY: scaleY
  };
}

var render$1 = {
  render: function render() {
    this.initContainer();
    this.initCanvas();
    this.initCropBox();
    this.renderCanvas();

    if (this.cropped) {
      this.renderCropBox();
    }
  },
  initContainer: function initContainer() {
    var element = this.element,
        options = this.options,
        container = this.container,
        cropper = this.cropper;
    var minWidth = Number(options.minContainerWidth);
    var minHeight = Number(options.minContainerHeight);
    addClass$1(cropper, CLASS_HIDDEN);
    removeClass$1(element, CLASS_HIDDEN);
    var containerData = {
      width: Math.max(container.offsetWidth, minWidth >= 0 ? minWidth : MIN_CONTAINER_WIDTH),
      height: Math.max(container.offsetHeight, minHeight >= 0 ? minHeight : MIN_CONTAINER_HEIGHT)
    };
    this.containerData = containerData;
    setStyle(cropper, {
      width: containerData.width,
      height: containerData.height
    });
    addClass$1(element, CLASS_HIDDEN);
    removeClass$1(cropper, CLASS_HIDDEN);
  },
  // Canvas (image wrapper)
  initCanvas: function initCanvas() {
    var containerData = this.containerData,
        imageData = this.imageData;
    var viewMode = this.options.viewMode;
    var rotated = Math.abs(imageData.rotate) % 180 === 90;
    var naturalWidth = rotated ? imageData.naturalHeight : imageData.naturalWidth;
    var naturalHeight = rotated ? imageData.naturalWidth : imageData.naturalHeight;
    var aspectRatio = naturalWidth / naturalHeight;
    var canvasWidth = containerData.width;
    var canvasHeight = containerData.height;

    if (containerData.height * aspectRatio > containerData.width) {
      if (viewMode === 3) {
        canvasWidth = containerData.height * aspectRatio;
      } else {
        canvasHeight = containerData.width / aspectRatio;
      }
    } else if (viewMode === 3) {
      canvasHeight = containerData.width / aspectRatio;
    } else {
      canvasWidth = containerData.height * aspectRatio;
    }

    var canvasData = {
      aspectRatio: aspectRatio,
      naturalWidth: naturalWidth,
      naturalHeight: naturalHeight,
      width: canvasWidth,
      height: canvasHeight
    };
    this.canvasData = canvasData;
    this.limited = viewMode === 1 || viewMode === 2;
    this.limitCanvas(true, true);
    canvasData.width = Math.min(Math.max(canvasData.width, canvasData.minWidth), canvasData.maxWidth);
    canvasData.height = Math.min(Math.max(canvasData.height, canvasData.minHeight), canvasData.maxHeight);
    canvasData.left = (containerData.width - canvasData.width) / 2;
    canvasData.top = (containerData.height - canvasData.height) / 2;
    canvasData.oldLeft = canvasData.left;
    canvasData.oldTop = canvasData.top;
    this.initialCanvasData = assign({}, canvasData);
  },
  limitCanvas: function limitCanvas(sizeLimited, positionLimited) {
    var options = this.options,
        containerData = this.containerData,
        canvasData = this.canvasData,
        cropBoxData = this.cropBoxData;
    var viewMode = options.viewMode;
    var aspectRatio = canvasData.aspectRatio;
    var cropped = this.cropped && cropBoxData;

    if (sizeLimited) {
      var minCanvasWidth = Number(options.minCanvasWidth) || 0;
      var minCanvasHeight = Number(options.minCanvasHeight) || 0;

      if (viewMode > 1) {
        minCanvasWidth = Math.max(minCanvasWidth, containerData.width);
        minCanvasHeight = Math.max(minCanvasHeight, containerData.height);

        if (viewMode === 3) {
          if (minCanvasHeight * aspectRatio > minCanvasWidth) {
            minCanvasWidth = minCanvasHeight * aspectRatio;
          } else {
            minCanvasHeight = minCanvasWidth / aspectRatio;
          }
        }
      } else if (viewMode > 0) {
        if (minCanvasWidth) {
          minCanvasWidth = Math.max(minCanvasWidth, cropped ? cropBoxData.width : 0);
        } else if (minCanvasHeight) {
          minCanvasHeight = Math.max(minCanvasHeight, cropped ? cropBoxData.height : 0);
        } else if (cropped) {
          minCanvasWidth = cropBoxData.width;
          minCanvasHeight = cropBoxData.height;

          if (minCanvasHeight * aspectRatio > minCanvasWidth) {
            minCanvasWidth = minCanvasHeight * aspectRatio;
          } else {
            minCanvasHeight = minCanvasWidth / aspectRatio;
          }
        }
      }

      var _getAdjustedSizes = getAdjustedSizes({
        aspectRatio: aspectRatio,
        width: minCanvasWidth,
        height: minCanvasHeight
      });

      minCanvasWidth = _getAdjustedSizes.width;
      minCanvasHeight = _getAdjustedSizes.height;
      canvasData.minWidth = minCanvasWidth;
      canvasData.minHeight = minCanvasHeight;
      canvasData.maxWidth = Infinity;
      canvasData.maxHeight = Infinity;
    }

    if (positionLimited) {
      if (viewMode > (cropped ? 0 : 1)) {
        var newCanvasLeft = containerData.width - canvasData.width;
        var newCanvasTop = containerData.height - canvasData.height;
        canvasData.minLeft = Math.min(0, newCanvasLeft);
        canvasData.minTop = Math.min(0, newCanvasTop);
        canvasData.maxLeft = Math.max(0, newCanvasLeft);
        canvasData.maxTop = Math.max(0, newCanvasTop);

        if (cropped && this.limited) {
          canvasData.minLeft = Math.min(cropBoxData.left, cropBoxData.left + (cropBoxData.width - canvasData.width));
          canvasData.minTop = Math.min(cropBoxData.top, cropBoxData.top + (cropBoxData.height - canvasData.height));
          canvasData.maxLeft = cropBoxData.left;
          canvasData.maxTop = cropBoxData.top;

          if (viewMode === 2) {
            if (canvasData.width >= containerData.width) {
              canvasData.minLeft = Math.min(0, newCanvasLeft);
              canvasData.maxLeft = Math.max(0, newCanvasLeft);
            }

            if (canvasData.height >= containerData.height) {
              canvasData.minTop = Math.min(0, newCanvasTop);
              canvasData.maxTop = Math.max(0, newCanvasTop);
            }
          }
        }
      } else {
        canvasData.minLeft = -canvasData.width;
        canvasData.minTop = -canvasData.height;
        canvasData.maxLeft = containerData.width;
        canvasData.maxTop = containerData.height;
      }
    }
  },
  renderCanvas: function renderCanvas(changed, transformed) {
    var canvasData = this.canvasData,
        imageData = this.imageData;

    if (transformed) {
      var _getRotatedSizes = getRotatedSizes({
        width: imageData.naturalWidth * Math.abs(imageData.scaleX || 1),
        height: imageData.naturalHeight * Math.abs(imageData.scaleY || 1),
        degree: imageData.rotate || 0
      }),
          naturalWidth = _getRotatedSizes.width,
          naturalHeight = _getRotatedSizes.height;

      var width = canvasData.width * (naturalWidth / canvasData.naturalWidth);
      var height = canvasData.height * (naturalHeight / canvasData.naturalHeight);
      canvasData.left -= (width - canvasData.width) / 2;
      canvasData.top -= (height - canvasData.height) / 2;
      canvasData.width = width;
      canvasData.height = height;
      canvasData.aspectRatio = naturalWidth / naturalHeight;
      canvasData.naturalWidth = naturalWidth;
      canvasData.naturalHeight = naturalHeight;
      this.limitCanvas(true, false);
    }

    if (canvasData.width > canvasData.maxWidth || canvasData.width < canvasData.minWidth) {
      canvasData.left = canvasData.oldLeft;
    }

    if (canvasData.height > canvasData.maxHeight || canvasData.height < canvasData.minHeight) {
      canvasData.top = canvasData.oldTop;
    }

    canvasData.width = Math.min(Math.max(canvasData.width, canvasData.minWidth), canvasData.maxWidth);
    canvasData.height = Math.min(Math.max(canvasData.height, canvasData.minHeight), canvasData.maxHeight);
    this.limitCanvas(false, true);
    canvasData.left = Math.min(Math.max(canvasData.left, canvasData.minLeft), canvasData.maxLeft);
    canvasData.top = Math.min(Math.max(canvasData.top, canvasData.minTop), canvasData.maxTop);
    canvasData.oldLeft = canvasData.left;
    canvasData.oldTop = canvasData.top;
    setStyle(this.canvas, assign({
      width: canvasData.width,
      height: canvasData.height
    }, getTransforms({
      translateX: canvasData.left,
      translateY: canvasData.top
    })));
    this.renderImage(changed);

    if (this.cropped && this.limited) {
      this.limitCropBox(true, true);
    }
  },
  renderImage: function renderImage(changed) {
    var canvasData = this.canvasData,
        imageData = this.imageData;
    var width = imageData.naturalWidth * (canvasData.width / canvasData.naturalWidth);
    var height = imageData.naturalHeight * (canvasData.height / canvasData.naturalHeight);
    assign(imageData, {
      width: width,
      height: height,
      left: (canvasData.width - width) / 2,
      top: (canvasData.height - height) / 2
    });
    setStyle(this.image, assign({
      width: imageData.width,
      height: imageData.height
    }, getTransforms(assign({
      translateX: imageData.left,
      translateY: imageData.top
    }, imageData))));

    if (changed) {
      this.output();
    }
  },
  initCropBox: function initCropBox() {
    var options = this.options,
        canvasData = this.canvasData;
    var aspectRatio = options.aspectRatio || options.initialAspectRatio;
    var autoCropArea = Number(options.autoCropArea) || 0.8;
    var cropBoxData = {
      width: canvasData.width,
      height: canvasData.height
    };

    if (aspectRatio) {
      if (canvasData.height * aspectRatio > canvasData.width) {
        cropBoxData.height = cropBoxData.width / aspectRatio;
      } else {
        cropBoxData.width = cropBoxData.height * aspectRatio;
      }
    }

    this.cropBoxData = cropBoxData;
    this.limitCropBox(true, true); // Initialize auto crop area

    cropBoxData.width = Math.min(Math.max(cropBoxData.width, cropBoxData.minWidth), cropBoxData.maxWidth);
    cropBoxData.height = Math.min(Math.max(cropBoxData.height, cropBoxData.minHeight), cropBoxData.maxHeight); // The width/height of auto crop area must large than "minWidth/Height"

    cropBoxData.width = Math.max(cropBoxData.minWidth, cropBoxData.width * autoCropArea);
    cropBoxData.height = Math.max(cropBoxData.minHeight, cropBoxData.height * autoCropArea);
    cropBoxData.left = canvasData.left + (canvasData.width - cropBoxData.width) / 2;
    cropBoxData.top = canvasData.top + (canvasData.height - cropBoxData.height) / 2;
    cropBoxData.oldLeft = cropBoxData.left;
    cropBoxData.oldTop = cropBoxData.top;
    this.initialCropBoxData = assign({}, cropBoxData);
  },
  limitCropBox: function limitCropBox(sizeLimited, positionLimited) {
    var options = this.options,
        containerData = this.containerData,
        canvasData = this.canvasData,
        cropBoxData = this.cropBoxData,
        limited = this.limited;
    var aspectRatio = options.aspectRatio;

    if (sizeLimited) {
      var minCropBoxWidth = Number(options.minCropBoxWidth) || 0;
      var minCropBoxHeight = Number(options.minCropBoxHeight) || 0;
      var maxCropBoxWidth = limited ? Math.min(containerData.width, canvasData.width, canvasData.width + canvasData.left, containerData.width - canvasData.left) : containerData.width;
      var maxCropBoxHeight = limited ? Math.min(containerData.height, canvasData.height, canvasData.height + canvasData.top, containerData.height - canvasData.top) : containerData.height; // The min/maxCropBoxWidth/Height must be less than container's width/height

      minCropBoxWidth = Math.min(minCropBoxWidth, containerData.width);
      minCropBoxHeight = Math.min(minCropBoxHeight, containerData.height);

      if (aspectRatio) {
        if (minCropBoxWidth && minCropBoxHeight) {
          if (minCropBoxHeight * aspectRatio > minCropBoxWidth) {
            minCropBoxHeight = minCropBoxWidth / aspectRatio;
          } else {
            minCropBoxWidth = minCropBoxHeight * aspectRatio;
          }
        } else if (minCropBoxWidth) {
          minCropBoxHeight = minCropBoxWidth / aspectRatio;
        } else if (minCropBoxHeight) {
          minCropBoxWidth = minCropBoxHeight * aspectRatio;
        }

        if (maxCropBoxHeight * aspectRatio > maxCropBoxWidth) {
          maxCropBoxHeight = maxCropBoxWidth / aspectRatio;
        } else {
          maxCropBoxWidth = maxCropBoxHeight * aspectRatio;
        }
      } // The minWidth/Height must be less than maxWidth/Height


      cropBoxData.minWidth = Math.min(minCropBoxWidth, maxCropBoxWidth);
      cropBoxData.minHeight = Math.min(minCropBoxHeight, maxCropBoxHeight);
      cropBoxData.maxWidth = maxCropBoxWidth;
      cropBoxData.maxHeight = maxCropBoxHeight;
    }

    if (positionLimited) {
      if (limited) {
        cropBoxData.minLeft = Math.max(0, canvasData.left);
        cropBoxData.minTop = Math.max(0, canvasData.top);
        cropBoxData.maxLeft = Math.min(containerData.width, canvasData.left + canvasData.width) - cropBoxData.width;
        cropBoxData.maxTop = Math.min(containerData.height, canvasData.top + canvasData.height) - cropBoxData.height;
      } else {
        cropBoxData.minLeft = 0;
        cropBoxData.minTop = 0;
        cropBoxData.maxLeft = con